/*

Copyright (C) 2000 - 2006 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_packet_recycler_h
#define __libnd_packet_recycler_h

#include <libnd_types.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*
 * This recycler implements a memory reuse strategy for packets.
 * Instead of freeing allocated memory, the packets are put on
 * queues based on the size of payload they can handle. Later on,
 * packets can be requested from the recycler that need to be able
 * to handle a specific amount of data.
 *
 * Pretty straightforward.
 *
 * The maximum size of the packet recycler is configured via the
 * num_recycled_packets configuration item in the LND_DOM_NETDUDE
 * configuration domain. The default value is 1000 packets. A
 * maximum size provided via num_recycled_packets is only honoured
 * at libnetdude initialization and not dynamically updated at
 * run time.
 */


/**
 * libnd_prec_init - initializes the recycler's data structures.
 */
void         libnd_prec_init();

/**
 * libnd_prec_put - puts packet into recycler.
 * @packet: packet to recycle.
 *
 * NOTE: DO NOT USE THIS FUNCTION. Use libnd_packet_free().
 * The function puts a packet into the recycler. It cleans it
 * up, looking at the amount of payload it can hold, and makes
 * it available when someone needs a packet that can hold up to
 * an equal amount.
 *
 * Returns: %TRUE when the packet is recycled, %FALSE if the recycler
 * is full. In that case the packet is unused and needs to be handled
 * differently, for example fully released.
 */
gboolean     libnd_prec_put(LND_Packet *packet);

/**
 * libnd_prec_get - returns a recycled packet, or a new one.
 * @mem_needed: amount of memory that new packet will need to carry.
 *
 * NOTE: DO NOT USE THIS FUNCTION. Use libnd_packet_new().
 * The function returns a recycled packet that is able to hold
 * @mem_needed bytes of payload.
 *
 * Returns: unused packet.
 */
LND_Packet  *libnd_prec_get(guint mem_needed);

/**
 * libnd_prec_calc_mem - returns actual memory used for given request.
 * guint @mem_req: requested amount of memory.
 *
 * This function returns the amount of memory the packet recycler
 * will allocate if a packet of @mem_req bytes is needed.
 *
 * Returns: allocated packet size.
 */
guint        libnd_prec_calc_mem(guint mem_req);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
