# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

"""
Models managed by the UPnP resource provider.
"""

from elisa.core.components.model import Model
from elisa.plugins.base.models import audio, video, image

class UpnpModel(Model):
    """
    Model always returned by the UpnpResource provider. It represents
    the results of a content-directory browse request.

    @ivar items: browse results
    @type items: list containing L{UpnpContainerModel} and/or L{UpnpItemModel} instances
    """

    def __init__(self):
        super(UpnpModel, self).__init__()
        self.items = []

class UpnpContainerModel(Model):
    """
    Model representing an UPnP container. Each container can has one
    or zero items.

    @ivar container_id: UPnP id of the container
    @type container_id: unicode
    @ivar name:         container name, human readable
    @type name:         unicode
    @ivar items_count:  number of items contained in the container
    @type items_count:  int
    @ivar elisa_uri:    upnp URI of the container
    @type elisa_uri:    L{elisa.core.media_uri.MediaUri}
    """

    def __init__(self):
        super(UpnpContainerModel, self).__init__()
        self.container_id = None
        self.name = None
        self.items_count = None

class UpnpItemModel(Model):
    """
    Model representing an UPnP container item. It can be playable.. or not.

    @ivar name:           item name, human readable
    @type name:           unicode
    @ivar playable_uri:   upnp URI set if the media's playable_model can't be built
    @type playable_uri:   L{elisa.core.media_uri.MediaUri}
    @ivar playable_model: model representing the playable media
    @type playable_model: L{elisa.plugins.base.models.media.PlayableModel}
    """

    def __init__(self):
        super(UpnpItemModel, self).__init__()
        self.name = None
        self.playable_model = None
        self.playable_uri = None

class UpnpArtistModel(UpnpContainerModel, audio.ArtistModel):
    """
    An Artist in UPnP world is considered as a container.
    """
    pass

class UpnpAlbumModel(UpnpContainerModel, audio.AlbumModel):
    """
    An Album in UPnP world is considered as a container.

    Additionnally we make the album name an alias to the container
    name.
    """

    def _set_name(self, name):
        self._name = name
        self.album = name

    def _get_name(self):
        return self._name

    name = property(fget=_get_name, fset=_set_name)

class UpnpTrackModel(UpnpItemModel, audio.TrackModel):
    """
    A Track in UPnP world is considered as an item.

    Additionnally we make the track title an alias to the item name.
    """

    def _set_name(self, name):
        self._name = name
        self.title = name

    def _get_name(self):
        return self._name

    name = property(fget=_get_name, fset=_set_name)

class UpnpVideoModel(UpnpItemModel, video.VideoModel):
    """
    A Video in UPnP world is considered as an item
    """
    pass

class UpnpImageModel(UpnpItemModel, image.ImageModel):
    """
    An Image in UPnP world is considered as an item
    """
    pass

def create_model_from_upnp_class(upnp_class):
    """
    Given a UPnP container or item class name, instantiate the correct
    L{UpnpContainerModel} or L{UpnpItemModel} class.

    @param upnp_class: container or item class name
    @type upnp_class:  string
    @rtype:            L{UpnpContainerModel} or L{UpnpItemModel} instance
    """
    if upnp_class.startswith('object.container'):
        if upnp_class == 'object.container.person.musicArtist':
            model = UpnpArtistModel()
        elif upnp_class == 'object.container.album.musicAlbum':
            model = UpnpAlbumModel()
        else:
            model = UpnpContainerModel()

    else:
        if upnp_class.startswith('object.item.audioItem'):
            model = UpnpTrackModel()
        elif upnp_class.startswith('object.item.videoItem'):
            model = UpnpVideoModel()
        elif upnp_class.startswith('object.item.imageItem'):
            model = UpnpImageModel()
        else:
            model = UpnpItemModel()
    return model
