/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

/*
 * dump-info - start of a tool to dump the information on a dump tape for
 * debugging purposes
 */

#include <config.h>
#include "faketape-lib.h"

#include "bswap_header.h"

#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <iostream>
#include <vector>
#include <cstring>
#include <iomanip>
#include <fcntl.h>
#include <limits.h>
#include <err.h>
#include <unistd.h>
#include <getopt.h>
#include <zlib.h>
#include <bzlib.h>
#include <lzo/lzo1x.h>
#include <ext2fs/ext2_fs.h>
#include <ext2fs/ext2_ext_attr.h>
#include <optional>

// Messy. Ideally we'd use readblock_tape from restore/readtape.c but there are
// too many issues with global variables that we don't use but need to link in.

// Manually copied from protocols/dumprestore.h
#define ROOTINO              EXT2_ROOT_INO

/*
 * extattributes inode info
 */
#define EXT_REGULAR		0
#define EXT_MACOSFNDRINFO	1
#define EXT_MACOSRESFORK	2
#define EXT_XATTR		3

#define COMPRESS_ZLIB	0
#define COMPRESS_BZLIB	1
#define COMPRESS_LZO	2

std::string convert_type(uint32_t type) {
#define C(X) case X: return std::string(#X "        ").substr(0,8); break
	switch (type) {
		C(TS_TAPE);
		C(TS_INODE);
		C(TS_ADDR);
		C(TS_BITS);
		C(TS_CLRI);
		C(TS_END);
	default:
		return "unknown " + std::to_string(type);
	}
#undef C
}

std::string convert_flags(int32_t flags) {
#define C(X,P) if(flags&(X)) { if(P) ret += "|" #X; flags &= ~(X); }
	std::string ret;
	C(DR_NEWHEADER, false);
	C(DR_NEWINODEFMT, false);
	C(DR_COMPRESSED, true);
	C(DR_METAONLY, true);
	C(DR_EXTATTRIBUTES, true);
	if (flags)
		ret += "|" + std::to_string(flags);
	if(ret.size())
		return ret.substr(1);
	return ret;
#undef C
}

std::vector<uint8_t> databuffer;

[[noreturn]] void usage(const char* prog) {
	std::cerr << "Usage " << prog << " dump-img" << std::endl;
	exit(1);
}

[[noreturn]] void fatal(const char* msg, const char* prog) {
	std::cerr << "FATAL: " << msg << std::endl << std::endl;
	usage(prog);
}

std::string getmode(uint16_t mode) {
	std::string result;
	if (mode & S_ISUID) result += 'r'; else result += '-';

	if (mode & S_IRUSR) result += 'r'; else result += '-';
	if (mode & S_IWUSR) result += 'w'; else result += '-';
	if (mode & S_ISGID) {
		if (mode & S_IXUSR) result += 's'; else result += 'S';
	} else
		if (mode & S_IXUSR) result += 'x'; else result += '-';

	if (mode & S_IRGRP) result += 'r'; else result += '-';
	if (mode & S_IWGRP) result += 'w'; else result += '-';

	if (mode & S_ISGID) {
		if (mode & S_IXGRP) result += 's'; else result += 'S';
	} else
		if (mode & S_IXGRP) result += 'x'; else result += '-';

	if (mode & S_IROTH) result += 'r'; else result += '-';
	if (mode & S_IWOTH) result += 'w'; else result += '-';

	if (mode & S_ISVTX) {
		if (mode & S_IXOTH) result += 't'; else result += 'T';
	} else
		if (mode & S_IXOTH) result += 'x'; else result += '-';

	switch(mode & S_IFMT) {
		case S_IFDIR: result += " directory"; break;
		case S_IFCHR: result += " character special file "; break;
		case S_IFBLK: result += " block special file"; break;
		case S_IFREG: result += " regular file"; break;
		case S_IFIFO: result += " FIFO special file"; break;
		case S_IFLNK: result += " symbolic link"; break;
		case S_IFSOCK: result += " socket"; break;
		default:       result += " UNKNOWN"; break;
	}
	return result;
}

uint32_t crc32(const uint8_t* data, size_t size, uint32_t crc) {
	static constexpr uint32_t crcTable[256] = {
		0x00000000, 0xedb88320, 0x36c98560, 0xdb710640, 0x6d930ac0, 0x802b89e0, 0x5b5a8fa0, 0xb6e20c80,
		0xdb261580, 0x369e96a0, 0xedef90e0, 0x005713c0, 0xb6b51f40, 0x5b0d9c60, 0x807c9a20, 0x6dc41900,
		0x5bf4a820, 0xb64c2b00, 0x6d3d2d40, 0x8085ae60, 0x3667a2e0, 0xdbdf21c0, 0x00ae2780, 0xed16a4a0,
		0x80d2bda0, 0x6d6a3e80, 0xb61b38c0, 0x5ba3bbe0, 0xed41b760, 0x00f93440, 0xdb883200, 0x3630b120,
		0xb7e95040, 0x5a51d360, 0x8120d520, 0x6c985600, 0xda7a5a80, 0x37c2d9a0, 0xecb3dfe0, 0x010b5cc0,
		0x6ccf45c0, 0x8177c6e0, 0x5a06c0a0, 0xb7be4380, 0x015c4f00, 0xece4cc20, 0x3795ca60, 0xda2d4940,
		0xec1df860, 0x01a57b40, 0xdad47d00, 0x376cfe20, 0x818ef2a0, 0x6c367180, 0xb74777c0, 0x5afff4e0,
		0x373bede0, 0xda836ec0, 0x01f26880, 0xec4aeba0, 0x5aa8e720, 0xb7106400, 0x6c616240, 0x81d9e160,
		0x826a23a0, 0x6fd2a080, 0xb4a3a6c0, 0x591b25e0, 0xeff92960, 0x0241aa40, 0xd930ac00, 0x34882f20,
		0x594c3620, 0xb4f4b500, 0x6f85b340, 0x823d3060, 0x34df3ce0, 0xd967bfc0, 0x0216b980, 0xefae3aa0,
		0xd99e8b80, 0x342608a0, 0xef570ee0, 0x02ef8dc0, 0xb40d8140, 0x59b50260, 0x82c40420, 0x6f7c8700,
		0x02b89e00, 0xef001d20, 0x34711b60, 0xd9c99840, 0x6f2b94c0, 0x829317e0, 0x59e211a0, 0xb45a9280,
		0x358373e0, 0xd83bf0c0, 0x034af680, 0xeef275a0, 0x58107920, 0xb5a8fa00, 0x6ed9fc40, 0x83617f60,
		0xeea56660, 0x031de540, 0xd86ce300, 0x35d46020, 0x83366ca0, 0x6e8eef80, 0xb5ffe9c0, 0x58476ae0,
		0x6e77dbc0, 0x83cf58e0, 0x58be5ea0, 0xb506dd80, 0x03e4d100, 0xee5c5220, 0x352d5460, 0xd895d740,
		0xb551ce40, 0x58e94d60, 0x83984b20, 0x6e20c800, 0xd8c2c480, 0x357a47a0, 0xee0b41e0, 0x03b3c2c0,
		0xe96cc460, 0x04d44740, 0xdfa54100, 0x321dc220, 0x84ffcea0, 0x69474d80, 0xb2364bc0, 0x5f8ec8e0,
		0x324ad1e0, 0xdff252c0, 0x04835480, 0xe93bd7a0, 0x5fd9db20, 0xb2615800, 0x69105e40, 0x84a8dd60,
		0xb2986c40, 0x5f20ef60, 0x8451e920, 0x69e96a00, 0xdf0b6680, 0x32b3e5a0, 0xe9c2e3e0, 0x047a60c0,
		0x69be79c0, 0x8406fae0, 0x5f77fca0, 0xb2cf7f80, 0x042d7300, 0xe995f020, 0x32e4f660, 0xdf5c7540,
		0x5e859420, 0xb33d1700, 0x684c1140, 0x85f49260, 0x33169ee0, 0xdeae1dc0, 0x05df1b80, 0xe86798a0,
		0x85a381a0, 0x681b0280, 0xb36a04c0, 0x5ed287e0, 0xe8308b60, 0x05880840, 0xdef90e00, 0x33418d20,
		0x05713c00, 0xe8c9bf20, 0x33b8b960, 0xde003a40, 0x68e236c0, 0x855ab5e0, 0x5e2bb3a0, 0xb3933080,
		0xde572980, 0x33efaaa0, 0xe89eace0, 0x05262fc0, 0xb3c42340, 0x5e7ca060, 0x850da620, 0x68b52500,
		0x6b06e7c0, 0x86be64e0, 0x5dcf62a0, 0xb077e180, 0x0695ed00, 0xeb2d6e20, 0x305c6860, 0xdde4eb40,
		0xb020f240, 0x5d987160, 0x86e97720, 0x6b51f400, 0xddb3f880, 0x300b7ba0, 0xeb7a7de0, 0x06c2fec0,
		0x30f24fe0, 0xdd4accc0, 0x063bca80, 0xeb8349a0, 0x5d614520, 0xb0d9c600, 0x6ba8c040, 0x86104360,
		0xebd45a60, 0x066cd940, 0xdd1ddf00, 0x30a55c20, 0x864750a0, 0x6bffd380, 0xb08ed5c0, 0x5d3656e0,
		0xdcefb780, 0x315734a0, 0xea2632e0, 0x079eb1c0, 0xb17cbd40, 0x5cc43e60, 0x87b53820, 0x6a0dbb00,
		0x07c9a200, 0xea712120, 0x31002760, 0xdcb8a440, 0x6a5aa8c0, 0x87e22be0, 0x5c932da0, 0xb12bae80,
		0x871b1fa0, 0x6aa39c80, 0xb1d29ac0, 0x5c6a19e0, 0xea881560, 0x07309640, 0xdc419000, 0x31f91320,
		0x5c3d0a20, 0xb1858900, 0x6af48f40, 0x874c0c60, 0x31ae00e0, 0xdc1683c0, 0x07678580, 0xeadf06a0,
	};

	for (size_t i = 0; i < size; ++i) {
		const uint32_t lookupIndex = (crc ^ data[i]) & 0xff;
		crc = (crc >> 8) ^ crcTable[lookupIndex];
//		std::cerr << std::hex;
//		if (i && !(i%32))
//			std::cerr << std::endl;
//		std::cerr << std::setw(2) << (int)data[i] << " ";
	}
//	if (size)
//		std::cerr << std::endl;
//	std::cerr << std::dec;

	if (!size) {
		// Finalize the CRC-32 value by inverting all the bits
		crc ^= 0xFFFFFFFFu;
	}
	return crc;
}


void print_tape(uint8_t* data, size_t bytes) {

	static uint8_t s_addr[512];
	static uint8_t* s_addr_ptr;
	static uint32_t subcount;
	static int state = 0;
	static size_t count = 0;
	static bool inode_is_ea = false;
	static bool inode_is_directory = false;
	static uint64_t inode_data_size = 0;
	static std::string indent = "";
	static size_t blockno = 1;
	static bool seen_end = true;
	static bool new_vol = false;
	static uint32_t crc;

	if(!data || !bytes) {
		if(seen_end) {
			subcount = 0;
			state = 0;
			count = 0;
			inode_is_directory = false;
			inode_data_size = 0;
			indent = "";
			blockno = 1;
		} else
			new_vol = true;
		return;
	}
	if (bytes % sizeof(header) == 4) {
		bytes -= 4;
		data += 4;
	}
	header* hdr = reinterpret_cast<header*>(data);
	bytes /= sizeof *hdr;

	auto skip_holes = []() {
		static size_t data_indexes = 0;
		static size_t data_blocks = 0;
		size_t holes_skipped = 0;
		size_t indexes = 0;
		while (count && !(*s_addr_ptr&1)) {
			holes_skipped += get_s_addr_length(s_addr_ptr);
			size_t hole_size = get_s_addr_length(s_addr_ptr) * 1024;
			if (hole_size > inode_data_size)
				hole_size = inode_data_size;
			inode_data_size -= hole_size;
			indexes++;
			if(get_s_addr_length(s_addr_ptr) > 64)
				s_addr_ptr += 2;
			else
				s_addr_ptr++;
			count--;
		}
		if (!count) {
			if (data_blocks) {
				crc = crc32(nullptr, 0, crc);
				std::cerr << indent << "    Processed " << data_blocks << " data blocks in " << data_indexes << " indexes" << " crc=" << std::hex << crc << std::dec << std::endl;
			}
			subcount = 0;
			state = 0;
			data_indexes = 0;
			data_blocks = 0;
			crc = 0xffffffff;
		} else {
			subcount = get_s_addr_length(s_addr_ptr);
			data_blocks += subcount;
			data_indexes++;
		}
		if (holes_skipped)
			std::cerr << indent << "Skipped " << holes_skipped << " holes in " << indexes << " indexes" << std::endl;
	};
#define EXT2_XATTR_MAGIC		0xEA020000	// block EA
#define EXT2_XATTR_MAGIC2		0xEA020001	// in inode EA

	auto print_ea = [&]() {
		static std::vector<uint8_t> eabuffer;
		if (count) {
			if (!eabuffer.size()) {
				/* h_checksum includes the blocknum in the
				 * calculation so it's useless in the dump and
				 * causes dumps to be different just based on
				 * the block that the EAs were stored in on the
				 * disk */
				ext2_ext_attr_header* header = reinterpret_cast<ext2_ext_attr_header*>(hdr);
				if (header->h_magic == EXT2_EXT_ATTR_MAGIC)
					header->h_checksum = 0;
			}
			eabuffer.insert(eabuffer.end(), reinterpret_cast<uint8_t*>(hdr), reinterpret_cast<uint8_t*>(hdr)+sizeof *hdr);
			return;
		}
		const uint8_t* data_start;
		const ext2_ext_attr_entry* entry;
		const ext2_ext_attr_header* header = reinterpret_cast<const ext2_ext_attr_header*>(eabuffer.data());
		switch(header->h_magic) {
			case EXT2_EXT_ATTR_MAGIC:
				data_start = reinterpret_cast<const uint8_t*>(header);
				entry = reinterpret_cast<const ext2_ext_attr_entry*>(header+1);
				break;
			case EXT2_EXT_ATTR_MAGIC+1:
				data_start = reinterpret_cast<const uint8_t*>(header) + sizeof header->h_magic;
				entry = reinterpret_cast<const ext2_ext_attr_entry*>(data_start);
				break;
			default:
				errx(1, "Unknown magic %x in print_ea", header->h_magic);
		}
		while(entry->e_name_len || entry->e_name_index || entry->e_value_offs) {
			std::cerr << "entry->e_name_len=" << (int)entry->e_name_len << " entry->e_name_index=" << (int)entry->e_name_index << " entry->e_value_offs=" << entry->e_value_offs << std::endl;
			std::string_view nametxt{ reinterpret_cast<const char*>(entry)+sizeof *entry, entry->e_name_len};
			std::string name;
			switch (entry->e_name_index) {
				case 10:
					name = std::string("gnu.") + std::string(nametxt);
					break;
				case 3:
					name = std::string("system.posix_acl_default.") + std::string(nametxt);
					break;
				case 2:
					name = std::string("system.posix_acl_access.") + std::string(nametxt);
					break;
				case 8:
					name = std::string("system.richacl.") + std::string(nametxt);
					break;
				case 6:
					name = std::string("security.") + std::string(nametxt);
					break;
				case 4:
					name = std::string("trusted.") + std::string(nametxt);
					break;
				case 7:
					name = std::string("system.") + std::string(nametxt);
					break;
				case 1:
					name = std::string("user.") + std::string(nametxt);
					break;
				default:
					errx(1, "unknown attribute");
			}
			std::vector<uint8_t> value;
			value.insert(value.end(), data_start + entry->e_value_offs, data_start + entry->e_value_offs + entry->e_value_size);
			std::cerr << indent << "    EA: " << name << " value size is " << value.size() << std::endl;

			size_t size = ((sizeof *entry + entry->e_name_len + EXT2_EXT_ATTR_PAD - 1) / EXT2_EXT_ATTR_PAD) * EXT2_EXT_ATTR_PAD;
			entry = reinterpret_cast<const ext2_ext_attr_entry*>(reinterpret_cast<const uint8_t*>(entry) + size);
		}
		eabuffer.clear();
	};

	auto print_direct = [&]() {
		uint8_t* data = reinterpret_cast<uint8_t*>(hdr);
		size_t consumed = 0;
		const direct* ptr = reinterpret_cast<const direct*>(data);

		auto strtype = [](uint8_t type) -> std::string {
			switch((uint32_t)type << 12) {
				case S_IFDIR: return "directory"; break;
				case S_IFCHR: return "character special file "; break;
				case S_IFBLK: return "block special file"; break;
				case S_IFREG: return "regular file"; break;
				case S_IFIFO: return "FIFO special file"; break;
				case S_IFLNK: return "symbolic link"; break;
				case S_IFSOCK: return "socket"; break;
				default:       return "UNKNOWN:" + std::to_string(type);  break;
			}
		};
		while(consumed < sizeof *hdr && inode_data_size) {
			uint16_t reclen = ptr->d_reclen;
			std::cerr << indent << "  ino=" << ptr->d_ino << " reclen=" << ptr->d_reclen << " type=" << strtype(ptr->d_type) << " namlen=" << (int)ptr->d_namlen << " : " << ptr->d_name << std::endl;
			data += reclen;
			consumed += reclen;
			inode_data_size -= reclen;
			ptr = reinterpret_cast<const direct*>(data);
		}
	};

	auto process_block = [&]() {
		switch(state) {
			case 0: {
				indent = "";
				if (hdr->c_magic != NFS_MAGIC || !checksum(hdr)) {
					std::cerr << "SKIPPING UNKNOWN at blockno " << blockno <<  " hdr->c_magic=" << hdr->c_magic << "NFS_MAGIC=" << NFS_MAGIC << " csum=" << checksum(hdr) << std::endl;
					hdr++;
					blockno++;
					bytes--;
					subcount = 0;
					count = 0;
					inode_is_directory = false;
					inode_data_size = 0;
					break;
				}
#if 0
				subcount = 0;
				count = 0;
				inode_is_directory = false;
				inode_data_size = 0;
#endif
				std::cerr << indent << convert_type(hdr->c_type) << " at blockno " << blockno << " inode=" << hdr->c_inumber << " mode=" << getmode(hdr->c_dinode.di_mode) << " size=" << hdr->c_dinode.di_size << " flags=" << convert_flags(hdr->c_flags) << std::endl;
				indent = "  ";
				seen_end = false;
				switch (hdr->c_type) {
					case TS_END:
						seen_end = true;
						[[fallthrough]];
					case TS_TAPE: {
						if (hdr->c_data.s_inos[1] == ROOTINO)
							for (int i=1; i<128; i++) {
								if (hdr->c_data.s_inos[i])
									std::cerr << indent << "volinfo[" << i << "] = " << hdr->c_data.s_inos[i] << std::endl;
							}
						break;
					}
					case TS_CLRI:
						count = hdr->c_count;
						state = count?TS_CLRI:0;
						std::cerr << indent << count << " data blocks with CLR-inode bitmap" << std::endl;
						break;
					case TS_BITS:
						count = hdr->c_count;
						state = count?TS_BITS:0;
						std::cerr << indent << count << " data blocks with HAS-inode bitmap" << std::endl;
						break;
					case TS_INODE:
						std::cerr << indent << "INODE " << hdr->c_inumber << " mode=" << getmode(hdr->c_dinode.di_mode) << " size=" << hdr->c_dinode.di_size << " flags=" << convert_flags(hdr->c_flags) << std::endl;
						inode_is_ea = hdr->c_flags & DR_EXTATTRIBUTES;
						inode_is_directory = !inode_is_ea && (hdr->c_dinode.di_mode & S_IFMT) == S_IFDIR;
						inode_data_size = hdr->c_dinode.di_size;
						[[fallthrough]];
					case TS_ADDR:
						count = hdr->c_count;
						/* TODO - there was a problem with EAs sometimes not having s_addr populated properly */
						std::memcpy(s_addr, hdr->c_data.s_addrs, 512);
						s_addr_ptr = s_addr;
						skip_holes();
						if (count) {
							crc = 0xffffffff;
							state = TS_INODE;
						}
						break;
					default:
						errx(1, "unknown state on tape");
						break;
				}
				hdr++;
				blockno++;
				bytes--;
				break;
			}
			case TS_INODE:
				if (inode_is_directory)
					print_direct();
				else if (inode_is_ea)
					print_ea();
				crc = crc32(reinterpret_cast<const uint8_t*>(hdr), sizeof *hdr, crc);
				hdr++;
				blockno++;
				bytes--;
				if(--subcount == 0) {
					if(--count == 0) {
						state = 0;
						if (inode_is_ea)
							print_ea();
					} else {
						if(get_s_addr_length(s_addr_ptr) > 64)
							s_addr_ptr += 2;
						else
							s_addr_ptr++;
					}
					skip_holes();
				}
				break;
			case TS_BITS:
			case TS_CLRI:
				hdr++;
				blockno++;
				bytes--;
				if(--count == 0)
					state = 0;
				break;
			default:
				errx(1, "unexpected state while reading tape");
				break;
		}
	};
	if (new_vol) {
		if (hdr->c_magic != NFS_MAGIC || !checksum(hdr) || hdr->c_type != TS_TAPE) {
			std::cerr << "EXPECTED TS_TAPE at start of volume hdr->c_magic=" << std::hex << hdr->c_magic << std::endl;
		} else {
			int saved_state = state;
			state = 0;
			process_block();
			state = saved_state;
		}
		new_vol = false;
	}
	while (bytes)
		process_block();
}

int main(int argc, char* argv[]) {

	FakeTape	tapein;
	const char* prog = argv[0];

	const char* opts = "dhv";
	struct option lopts[] = {
		{ .name = "version", .has_arg = 0, .flag = nullptr, .val = 'v' },
		{ .name = "debug", .has_arg = 0, .flag = nullptr, .val = 'd' },
		{},
	};
	std::string tapedev;

	int ch;

	while (( ch = getopt_long(argc, argv, opts, lopts, nullptr)) != -1)
		switch(ch) {
			case 'd':
				tapein.debug = std::make_unique<std::ostream>(std::cerr.rdbuf());
				tapein.debug->copyfmt(std::cerr);
				tapein.debug->clear(std::cerr.rdstate());
				break;
			case 'v':
				std::cerr << prog << " version 1.0" << std::endl;
				break;
			case 'h':
				usage(prog);
			default:
				usage(prog);
		}
	argc -= optind;
	argv += optind;

	if (argc != 1)
		usage(prog);

	/* Now open the dump image */
	warnx("Starting %s", prog);
	if(tapein.open(argv[0], O_RDONLY) != FakeTape::Response::OK) {
		err(1, "%s cannot be opened as a tape.",  argv[0]);
	}

	std::vector<uint8_t> buffer(FakeTape::getMaxBlockSize());
	if (lzo_init() != LZO_E_OK) {
		fatal("internal error - lzo_init failed \n", prog);
	}

	bswap_context inctx;
	do {
		bswap_header(nullptr, 0, &inctx);
		print_tape(nullptr, 0);
		/* Read the first block */
		size_t readlen;
		if(tapein.readData(buffer.data(), buffer.size(), &readlen) != FakeTape::Response::OK)
			err(1, "Failed to read first block from dump image");
		if (readlen == 0)
			break;

		header* hdr = reinterpret_cast<header*>(buffer.data());

		if (!checksum(hdr))
			errx(1, "This doesn't appear to be a dump tape, or was written on a system with byteswapped values");

		if (hdr->c_magic != NFS_MAGIC) {
			inctx.to_cpu = true;
			warnx("Input tape appears to be byteswapped");
		}
		bswap_header(buffer.data(), readlen, &inctx);

		if (hdr->c_magic != NFS_MAGIC)
			errx(1, "magic failure");

		bool compressed = hdr->c_flags & DR_COMPRESSED;
		uint32_t ntrec = hdr->c_ntrec;

		std::cerr << "Tape has ntrec of " << ntrec << std::endl;

		print_tape(buffer.data(), readlen);

		std::optional<bool> demangle_bitfields;
		while(true) {

			if(tapein.readData(buffer.data(), buffer.size(), &readlen) != FakeTape::Response::OK)
				err(1, "Failed to read block from dump image");
			if (readlen == 0)
				break;
			uint32_t bsize = readlen;
			uint32_t ctype = 0;
			uint32_t cflag = 0;

			if (compressed) {
				if (inctx.to_cpu) {
					auto data = buffer.data();
					swap4(1, data);
				}
				uint32_t clen = *reinterpret_cast<uint32_t*>(buffer.data());
				if (!demangle_bitfields.has_value()) {
					if(clen>>31)
						// Case 2b. Only way mc can be set.
						demangle_bitfields = true;
					/* blocks written without compression must be exactly ntrec*TP_BSIZE in size */
					else if ((clen & 0xfffffff) != ntrec * TP_BSIZE)
						// This covers case 2a and 1a. 2a because length will be odd when read mangled, and 1a because length will be 16x too big.
						demangle_bitfields = false;
					else
						// Only case 1b remains.
						demangle_bitfields = true;
					if (*demangle_bitfields)
						std::cerr << "Warning: Tape appears to have been written with different bitfield ordering" << std::endl;
				}
				if (*demangle_bitfields) {
					bsize = clen & 0xfffffff;
					ctype = (clen&0x70000000)>>28;
					cflag = clen>>31;
				} else {
					bsize = clen >> 4;
					ctype = (clen&0xe)>>1;
					cflag = clen&1;
				}
				if (bsize == readlen) {
					size_t extra;
					if(tapein.readData(buffer.data() + readlen, 4, &extra) != FakeTape::Response::OK)
						err(1, "Failed to read continuation block from dump image");
					readlen += extra;
					bswap_header(buffer.data()+4, readlen-4, &inctx);
				} else if (bsize != readlen - 4) {
					err(1, "Unexpected size of compressed block bsize=%d readlen=%zd clen=%x demangle=%d", bsize, readlen, clen, *demangle_bitfields);
				}
				/* TODO check ctype and use correct decompressor */
				if (cflag) {
					switch(ctype) {
						case COMPRESS_ZLIB: {
							std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
							std::swap(cmprbuf, buffer);
							uLongf decmprlen = buffer.size();
							int status = uncompress(buffer.data()+4, &decmprlen, cmprbuf.data()+4, bsize);
							switch (status) {
								case Z_OK: break;
								case Z_MEM_ERROR: errx(1, "Memory error while decompressing");
								case Z_BUF_ERROR: errx(1, "BUF error while decompressing");;
								case Z_DATA_ERROR: errx(1, "Corrupt compressed data while decompressing");
								default: errx(1, "Unexpected return from uncompress");
							}
							bsize = decmprlen;
							break;
						}
						case COMPRESS_BZLIB: {
							std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
							std::swap(cmprbuf, buffer);
							unsigned int decmprlen = buffer.size();
							int status = BZ2_bzBuffToBuffDecompress((char*)buffer.data()+4, &decmprlen, (char*)cmprbuf.data()+4, bsize, 0, 0);

							switch (status) {
								case BZ_OK: break;
								case BZ_MEM_ERROR: errx(1, "Memory error while decompressing");
								case BZ_OUTBUFF_FULL: errx(1, "BUF error while decompressing");;
								case BZ_DATA_ERROR:
								case BZ_DATA_ERROR_MAGIC:
								case BZ_UNEXPECTED_EOF:
									errx(1, "Corrupt compressed data while decompressing");
								default: errx(1, "Unexpected return from uncompress");
							}
							bsize = decmprlen;
							break;
						}
						case COMPRESS_LZO: {
							std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
							std::swap(cmprbuf, buffer);
							lzo_uint decmprlen = buffer.size();
							int status = lzo1x_decompress(cmprbuf.data()+4, bsize, buffer.data()+4, &decmprlen, nullptr);

							switch (status) {
								case LZO_E_OK: break;
								case LZO_E_ERROR: errx(1, "LZO error while decompressing");
								case LZO_E_EOF_NOT_FOUND: errx(1, "LZO No EOF error while decompressing");;
								default: errx(1, "Unexpected return from LZO uncompress");
							}
							bsize = decmprlen;
							break;
						}
					}
				}
				bswap_header(buffer.data()+4, bsize, &inctx);
				readlen = bsize + 4;
			} else {
				bswap_header(buffer.data(), readlen, &inctx);
				std::memmove(buffer.data()+4, buffer.data(), readlen);
				readlen += 4;
			}
			/* At this point we have uncompressed data with a compression header (header contains garbage) */
			cflag = 0;
			ctype = 0;
			print_tape(buffer.data(), readlen);
		}
	} while(true);

	warnx("Finished");
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
