# SPDX-License-Identifier: AGPL-3.0-or-later
"""FreedomBox app to configure OpenVPN server."""

from django.urls import reverse_lazy
from django.utils.translation import gettext_lazy as _

from plinth import app as app_module
from plinth import cfg, frontpage, menu
from plinth.daemon import Daemon
from plinth.modules.backups.components import BackupRestore
from plinth.modules.firewall.components import Firewall
from plinth.modules.users.components import UsersAndGroups
from plinth.package import Packages
from plinth.utils import format_lazy

from . import manifest, privileged

_description = [
    format_lazy(
        _('Virtual Private Network (VPN) is a technique for securely '
          'connecting two devices in order to access resources of a '
          'private network.  While you are away from home, you can connect '
          'to your {box_name} in order to join your home network and '
          'access private/internal services provided by {box_name}. '
          'You can also access the rest of the Internet via {box_name} '
          'for added security and anonymity.'), box_name=_(cfg.box_name))
]

SERVER_CONFIGURATION_FILE = '/etc/openvpn/server/freedombox.conf'


class OpenVPNApp(app_module.App):
    """FreedomBox app for OpenVPN."""

    app_id = 'openvpn'

    _version = 5

    def __init__(self) -> None:
        """Create components for the app."""
        super().__init__()

        self.groups = {'vpn': _('Connect to VPN services')}

        info = app_module.Info(app_id=self.app_id, version=self._version,
                               name=_('OpenVPN'), icon_filename='openvpn',
                               description=_description, manual_page='OpenVPN',
                               clients=manifest.clients, tags=manifest.tags)
        self.add(info)

        menu_item = menu.Menu('menu-openvpn', info.name, info.icon_filename,
                              info.tags, 'openvpn:index',
                              parent_url_name='apps')
        self.add(menu_item)

        download_profile = \
            format_lazy(_('<a class="btn btn-primary btn-sm" href="{link}">'
                          'Download Profile</a>'),
                        link=reverse_lazy('openvpn:profile'))
        shortcut = frontpage.Shortcut(
            'shortcut-openvpn', info.name, icon=info.icon_filename,
            description=(info.description or []) + [download_profile],
            manual_page=info.manual_page,
            configure_url=reverse_lazy('openvpn:index'), tags=info.tags,
            login_required=True, allowed_groups=['vpn'])
        self.add(shortcut)

        packages = Packages('packages-openvpn',
                            ['openvpn', 'easy-rsa', 'openssl'])
        self.add(packages)

        firewall = Firewall('firewall-openvpn', info.name, ports=['openvpn'],
                            is_external=True)
        self.add(firewall)

        daemon = Daemon('daemon-openvpn', 'openvpn-server@freedombox',
                        listen_ports=[(1194, 'udp4'), (1194, 'udp6')])
        self.add(daemon)

        users_and_groups = UsersAndGroups('users-and-groups-openvpn',
                                          groups=self.groups)
        self.add(users_and_groups)

        backup_restore = BackupRestore('backup-restore-openvpn',
                                       **manifest.backup)
        self.add(backup_restore)

    def setup(self, old_version):
        """Install and configure the app."""
        super().setup(old_version)
        privileged.setup()
        if not old_version:
            self.enable()

    def uninstall(self):
        """De-configure and uninstall the app."""
        super().uninstall()
        privileged.uninstall()
