/* gtkui.c - GTK+ GUI implementation
 *
 * Copyright 2010, 2013, 2016 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gtkui.h"
#include "test.h"
#include <stdlib.h>

int current_trial;              /* Current test trial, or -1 if there
                                 * is no test in progress. */
Metadata metadata_a;            /* Metadata for sample A. */
Metadata metadata_b;            /* Metadata for sample B. */
gchar *basename_a;              /* Basename for file A. */
gchar *basename_b;              /* Basename for file B. */

static gint ntrials = 16;
static gint outdev = -1;
static gint latency = 100;

/*
 * Parse command line options.
 */
static void
parse_options(int *argc, char **argv[])
{
    GError *error = NULL;
    GOptionContext *context;

    GOptionEntry entries[] = {
        { "num-trials", 'i', 0, G_OPTION_ARG_INT, &ntrials,
          "Do N trials", "N" },
        { "audio-device", 'd', 0, G_OPTION_ARG_INT, &outdev,
          "Use PortAudio audio device ID", "ID" },
        { "latency", 'l', 0, G_OPTION_ARG_INT, &latency,
          "Set suggested output latency to L milliseconds", "L" },
        { NULL }
    };

    context = g_option_context_new("[SAMPLE-A SAMPLE-B]");
    g_option_context_set_summary
        (context, "Test for audible differences in fidelity between "
         "two audio samples\nwith ABX test.");
    g_option_context_set_description
        (context, "Report bugs to <petterih@iki.fi>.");
    g_option_context_add_main_entries(context, entries, NULL);
    g_option_context_add_group(context, gtk_get_option_group(TRUE));
    if (!g_option_context_parse(context, argc, argv, &error)) {
        g_error("option parsing failed: %s\n", error->message);
    }

    /* ntrials and latency must be greater than zero. */
    if (ntrials <= 0) {
        g_error("the number of trials must be greater than zero");
    }
    if (latency <= 0) {
        g_error("the suggested latency must be greater than zero");
    }    
}

/*
 * Initialize GTK+, init a new test with command-line arguments (if
 * needed), and pass control for GTK+.
 */
int
ui_main(int argc, char *argv[])
{
    parse_options(&argc, &argv);
    gtk_init(&argc, &argv);
    if (init_audio(outdev, latency) != 0) {
        error_dialog(main_window, "Cannot initialize audio subsystem.");
        return EXIT_FAILURE;
    }
    if (argc == 2) {
        error_dialog(main_window,
                     "Only one sample filename given in the command line. "
                     "Two samples are required.");
            return EXIT_FAILURE;
    } else if (argc >= 3) {
        switch (init_test(argv[1], argv[2], ntrials)) {
        case 0:
            /* test init successful */
            show_main_window();
            reset_test();
            break;
        case 1:
            error_dialog(main_window,
                         "An error occurred while opening "
                         "the first sample.");
            return EXIT_FAILURE;
        case 2:
            error_dialog(main_window,
                         "An error occurred while opening "
                         "the second sample files.");
            return EXIT_FAILURE;
        case 3:
            error_dialog(main_window, "The given samples have "
                         "different durations.  Both samples must have "
                         "exactly the same duration.");
            return EXIT_FAILURE;
        case 4:
            error_dialog(main_window, "Invalid number of trials given in "
                         "the command line.");
            return EXIT_FAILURE;
        }
    } else {
        /* no current test available */
        current_trial = -1;
        show_main_window();
    }
    gtk_main();
    return EXIT_SUCCESS;
}

/*
 * Display a simple error dialog with the given parent window and
 * error message.
 */
void
error_dialog(GtkWidget *parent_win, char *msg)
{
    GtkWidget *errdialog;
    errdialog = gtk_message_dialog_new
        (GTK_WINDOW(parent_win), GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "%s", msg);
    gtk_window_set_title(GTK_WINDOW(errdialog), "Error");
    gtk_dialog_run(GTK_DIALOG(errdialog));
    gtk_widget_destroy(errdialog);
}

/*
 * Reset test parameters (current trial and metadata).  Close and
 * destroy possibly open windows, and update the main window.  This
 * function is usually called after a new test has been successfully
 * initialized.
 */
void
reset_test(void)
{
    if (basename_a) g_free(basename_a);
    if (basename_b) g_free(basename_b);
    current_trial = 0;
    get_sample_metadatas(&metadata_a, &metadata_b);
    basename_a = g_path_get_basename(metadata_a.filename);
    basename_b = g_path_get_basename(metadata_b.filename);
    hide_decide_dialog();
    hide_result_window();
    update_main_window();
}
