// Note: The license below is based on the template at:
// http://opensource.org/licenses/BSD-3-Clause
// Copyright (C) 2020 Regents of the University of Texas
//All rights reserved.

// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:

// o Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.

// o Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.

// o Neither the name of the copyright holders nor the names of its
//   contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.

// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// Original Author(s):
// Mertcan Temel         <mert@utexas.edu>

// This file is partly generated by a multiplier generator 
// which can be downloaded at https://github.com/temelmertcan/multgen.


// TOP MODULE
module Integrated_Multiplier
  (input logic clk,
   input logic [127:0] 	IN1,
   input logic [127:0] 	IN2,
   input logic [127:0] 	IN3,
   input logic [4:0] 	mode, 
   output reg [127:0] result);

   //  This  module can  perform  1  64x64-bit multiply-add,  4  32x32-bit
   //  multiply-add  in parallel,  and 1  4-32x32-bit dot-product-add  in a
   //  single clock cycle. The module also has an accumulater register that
   //  can be updated sequentially.
    
   //  This  purpose of this  module is to mimic  some of the  behaviour in
   //  industrial  designs by reusing multipliers  for different operations
   //  for  real-world applications.  It  can work with  various multiplier
   //  sizes  by allocating only  4 33x33-bit signed multiplier,  one final
   //  stage adder, and some smaller reduction trees to merge the results.

   // The module has different modes as determined by mode[4:0]:
   //  
   // 1. acc_on  (value to be added is from acc)  (mode[0] == 0)
   //    acc_off (value to be added is from IN3)  (mode[0] == 1)
   //            (and value in acc remains the same)
   // 2. (synchronous) acc=IN3 (reset ACC to IN3) (mode[1] == 0)
   //                  acc=result                 (mode[1] == 1)
   //    - This is the only synchronous element in the module.
   // 3. Signed    (mode[2] == 0)
   //    Unsigned  (mode[2] == 1)
   // 4. Integer Multiplier operations:
   // 4a. Dot product (32x32 + 32x32 + 32x32 + 32x32 + acc/in3)
   //     result[127:0]  = IN1[31:0]*IN2[31:0]
   //                      + IN1[63:32]*IN2[63:32]
   //                      + ... + ... + ACC/IN3[127:0]
   //     - overflow is truncated.
   //     - mode[4:3] == 00
   // 4b. 4 lanes of 32x32 multiplication;
   //     result[31:0]   = IN1[31:0]*IN2[31:0]+ACC/IN3[31:0]
   //     result[63:32]  = IN1[63:32]*IN2[63:32]+ACC/IN3[63:32]
   //     result[95:64]  = ...
   //     result[127:96] = ...
   //     - Either truncated on the lower half or higher half
   //     - mode[4:3] == 01 to save the lower half of multiplication
   //     - mode[4:3] == 10 to save the higher half of multiplication
   // 4c. 1 lane of 64x64-bit multiplication.
   //     result[127:0] = IN1[63:0] * IN2[63:0] + acc/IN3[127:0]
   //     - IN1[127:64] and IN2[127:64] are ignored.
   //     - overflow is truncated.
   //     - mode[4:3] == 11

   logic acc_on, reload_acc, signed_mult, truncate_low, truncate_high,
	 not_truncate, dot_product,four_lanes, one_lane;
   
   assign acc_on = mode[0] == 0;
   assign reload_acc = mode[1] == 0;
   assign signed_mult = mode[2] == 0;
   assign dot_product = mode[4:3] == 2'b00;
   assign four_lanes_lo =  mode[4:3] == 2'b01;
   assign four_lanes_hi =  mode[4:3] == 2'b10;
   assign one_lane = mode[4:3] == 2'b11;

   wire  logic [65:0] m0_0;
   wire  logic [65:0] m0_1;
   wire  logic [65:0] m1_0;
   wire  logic [65:0] m1_1;
   wire  logic [65:0] m2_0;
   wire  logic [65:0] m2_1;
   wire  logic [65:0] m3_0;
   wire  logic [65:0] m3_1;
   wire  logic [128:0] number_to_be_added;
   wire  logic [127:0] dot_result0;
   wire  logic [127:0] dot_result1; 
   wire  logic [127:0] one_lane_result0;
   wire  logic [127:0] one_lane_result1;
   wire  logic [130:0] adder0_input0, adder0_input1;
   wire  logic [131:0] adder0_output;
   wire  logic [130:0] adder1_input0, adder1_input1;
   wire  logic [131:0] adder1_output;
   wire  logic [63:0] four_lane_merger0_in3, four_lane_merger1_in3, four_lane_merger2_in3, four_lane_merger3_in3;
   wire  logic [63:0] four_lane_merger0_res0, four_lane_merger1_res0, four_lane_merger2_res0, four_lane_merger3_res0;
   wire  logic [63:0] four_lane_merger0_res1, four_lane_merger1_res1, four_lane_merger2_res1, four_lane_merger3_res1;

   reg [32:0] m0_in1;
   reg [32:0] m0_in2;
   reg [32:0] m1_in1;
   reg [32:0] m1_in2;
   reg [32:0] m2_in1;
   reg [32:0] m2_in2;
   reg [32:0] m3_in1;
   reg [32:0] m3_in2;
   reg [127:0] acc;

   // Depending on the mode, determine the number to be added.
   assign number_to_be_added[127:0] = acc_on ? acc : IN3;
   assign number_to_be_added[128]   = signed_mult ? (acc_on ? acc[127] : IN3[127]) : 0;


   // Allocate for small multipliers
   DT_SB4_33x33 m0 (m0_in1, m0_in2, m0_0, m0_1);
   DT_SB4_33x33 m1 (m1_in1, m1_in2, m1_0, m1_1);
   DT_SB4_33x33 m2 (m2_in1, m2_in2, m2_0, m2_1);
   DT_SB4_33x33 m3 (m3_in1, m3_in2, m3_0, m3_1);

   // Allocate reduction tree for the result of DOT product.
   // This will sum all the results.
   DOT_Merger_Signed_4_33x33_129_128 dot_merger 
     (m0_0, m0_1, m1_0, m1_1, m2_0, m2_1, m3_0, m3_1, number_to_be_added, 
      dot_result0, dot_result1);
   // Same for one lane 64x64-bit multiplication
   FourMultMerger_Signed_64x64_plus_129_128 one_lane_merger
     (m0_0, m0_1, m1_0, m1_1, m2_0, m2_1, m3_0, m3_1, number_to_be_added,
      one_lane_result0, one_lane_result1);
   // Same for four-lane multiplication.
   FMA_Merger_Signed_33x33_64_64 four_lane_merger0
     (m0_0[63:0], m0_1[63:0], four_lane_merger0_in3, four_lane_merger0_res0, four_lane_merger0_res1);
   FMA_Merger_Signed_33x33_64_64 four_lane_merger1
     (m1_0[63:0], m1_1[63:0], four_lane_merger1_in3, four_lane_merger1_res0, four_lane_merger1_res1);
   FMA_Merger_Signed_33x33_64_64 four_lane_merger2
     (m2_0[63:0], m2_1[63:0], four_lane_merger2_in3, four_lane_merger2_res0, four_lane_merger2_res1);
   FMA_Merger_Signed_33x33_64_64 four_lane_merger3
     (m3_0[63:0], m3_1[63:0], four_lane_merger3_in3, four_lane_merger3_res0, four_lane_merger3_res1);
   // Some of  these reduction  trees  (above modules)  could probably  be
   // merged together to save more space. But we do not bother with that.

   // Summation trees  in above  modules reduce  the summation  height to
   // two. Use a  vector adder to get the final result.  The inputs of this
   // adder module depends on the selected mode.
   LF_131 adder0 (adder0_input0, adder0_input1, adder0_output);
   // the second adder is only needed for four-lanes-hi mode.
   LF_131 adder1 (adder1_input0, adder1_input1, adder1_output);


   // Determine the behaviour of acc with respect to the current mode.
   always @(posedge clk)
     begin
	if (reload_acc) begin
	   acc <= IN3;
	end else if (acc_on) begin
	   acc <= result;
	end else begin
	   acc <= acc;
	end
     end

   // Determine the values of intermittant signals with respect to the current mode.
   // These will help utilize the same multiplier and adder module(s) for different operations.
   always @(*)
     begin
	adder0_input0 <= 0;
	adder0_input1 <= 0;
	adder1_input0 <= 0;
	adder1_input1 <= 0;
	result <= 0;
	four_lane_merger0_in3 <= 0;
	four_lane_merger1_in3 <= 0;
	four_lane_merger2_in3 <= 0;
	four_lane_merger3_in3 <= 0;
	
	m0_in1[31:0] <= IN1[31:0];
	m0_in2[31:0] <= IN2[31:0];
	m1_in1[31:0] <= IN1[63:32];
	m1_in2[31:0] <= IN2[63:32];
	m2_in1[31:0] <= IN1[95:64];
	m2_in2[31:0] <= IN2[95:64];
	m3_in1[31:0] <= IN1[127:96];
	m3_in2[31:0] <= IN2[127:96];
	if (signed_mult) begin
	   m0_in1[32] <= IN1[31];
	   m0_in2[32] <= IN2[31];
	   m1_in1[32] <= IN1[63];
	   m1_in2[32] <= IN2[63];
	   m2_in1[32] <= IN1[95];
	   m2_in2[32] <= IN2[95];
	   m3_in1[32] <= IN1[127];
	   m3_in2[32] <= IN2[127];
	end else begin
	   m0_in1[32] <= 0;
	   m0_in2[32] <= 0;
	   m1_in1[32] <= 0;
	   m1_in2[32] <= 0;
	   m2_in1[32] <= 0;
	   m2_in2[32] <= 0;
	   m3_in1[32] <= 0;
	   m3_in2[32] <= 0;
	end // else: !ifsigned_mult

	
	if (dot_product) begin
	   adder0_input0[127:0] <= dot_result0;
	   adder0_input1[127:0] <= dot_result1;
	   result <= adder0_output[127:0];
	   
	end else if (four_lanes_lo) begin
	   four_lane_merger0_in3[31:0] <= number_to_be_added[31:0];
	   four_lane_merger1_in3[31:0] <= number_to_be_added[63:32];
	   four_lane_merger2_in3[31:0] <= number_to_be_added[95:64];
	   four_lane_merger3_in3[31:0] <= number_to_be_added[127:96];

	   // feed to the adder by skipping one bit so it doesn't mess with each other's results.
	   adder0_input0[31:0] <= four_lane_merger0_res0[31:0];
	   adder0_input1[31:0] <= four_lane_merger0_res1[31:0];

	   adder0_input0[64:33] <= four_lane_merger1_res0[31:0];
	   adder0_input1[64:33] <= four_lane_merger1_res1[31:0];

	   adder0_input0[97:66] <= four_lane_merger2_res0[31:0];
	   adder0_input1[97:66] <= four_lane_merger2_res1[31:0];

	   adder0_input0[130:99] <= four_lane_merger3_res0[31:0];
	   adder0_input1[130:99] <= four_lane_merger3_res1[31:0];

	   result[31:0] <= adder0_output[31:0];
	   result[63:32] <= adder0_output[64:33];
	   result[95:64] <= adder0_output[97:66];
	   result[127:96] <= adder0_output[130:99];

	end else if (four_lanes_hi) begin 
	   // add the number to the higher portion of the multiplication result.
	   four_lane_merger0_in3[63:32] <= number_to_be_added[31:0];
	   four_lane_merger1_in3[63:32] <= number_to_be_added[63:32];
	   four_lane_merger2_in3[63:32] <= number_to_be_added[95:64];
	   four_lane_merger3_in3[63:32] <= number_to_be_added[127:96];

	   // Select      the      higher     portion      of      the
	   // multiplication+number_to_be_added and feed it to the adder.
	   
	   // feed to the adder by skipping one bit so it doesn't mess with
	   // each other's results.
	   adder0_input0[63:0]     <= four_lane_merger0_res0[63:0];
	   adder0_input1[63:0]     <= four_lane_merger0_res1[63:0];
	   adder0_input0[128:65]   <= four_lane_merger1_res0[63:0];
	   adder0_input1[128:65]   <= four_lane_merger1_res1[63:0];
	   adder1_input0[63:0]     <= four_lane_merger2_res0[63:0];
	   adder1_input1[63:0]     <= four_lane_merger2_res1[63:0];
	   adder1_input0[128:65]   <= four_lane_merger3_res0[63:0];
	   adder1_input1[128:65]   <= four_lane_merger3_res1[63:0];

	   result[31:0]   <= adder0_output[63:32];
	   result[63:32]  <= adder0_output[128:97];
	   result[95:64]  <= adder1_output[63:32];
	   result[127:96] <= adder1_output[128:97];
	   
	end else begin 
	   // if selected mode is one_lane.
	   adder0_input0[127:0] <= one_lane_result0;
	   adder0_input1[127:0] <= one_lane_result1;
	   result <= adder0_output[127:0];
	   
	   m0_in1 <= {1'b0, IN1[31:0]};
	   m0_in2 <= {1'b0, IN2[31:0]};
	   m1_in1 <= {1'b0, IN1[31:0]};
	   m2_in2 <= {1'b0, IN2[31:0]};
	   if (signed_mult) begin
	      m1_in2 <= {IN2[63], IN2[63:32]};
	      m2_in1 <= {IN1[63], IN1[63:32]};
	      m3_in1 <= {IN1[63], IN1[63:32]};
	      m3_in2 <= {IN2[63], IN2[63:32]};
	   end else begin
	      m1_in2 <= {1'b0, IN2[63:32]};
	      m2_in1 <= {1'b0, IN1[63:32]};
	      m3_in1 <= {1'b0, IN1[63:32]};
	      m3_in2 <= {1'b0, IN2[63:32]};
	   end 
	end
      
     end


endmodule



// HALF ADDER
module ha (
        input logic a,
        input logic b,
        output logic s,
        output logic c);
    
    assign s = a ^ b;
    assign c = a & b;
    endmodule


// FULL ADDER
module fa (
        input logic x,
        input logic y,
        input logic z,
        output logic s,
        output logic c);
    
    assign s = x ^ y ^ z;
    assign c = (x & y) | (x & z) | (y & z);
    endmodule


// INCOMPLETE MULTIPLIER WITH DADDA TREE (FSA STAGE SKIPPED)
module DT_SB4_33x33(
        input logic [32:0] IN1,
        input logic [32:0] IN2,
        output logic [65:0] result0,
        output logic [65:0] result1);
    
    
// Creating Partial Products 

    wire logic const1;
    assign const1 = 1'b1;
    wire [34:0] IN2_1x;
    wire [34:0] IN2_2x;
    assign IN2_1x = {IN2[32],  IN2[32],  IN2};
    assign IN2_2x = {IN2[32],  IN2, 1'b0};
    
    // Signed Booth Radix-4 Partial Products Row 1
    wire logic select_e_0, select_ne_0, select_2x_0, tcomp0, select_n2x_0;
    assign select_e_0 = ((~ IN1[1]) & (IN1[0] ^ 1'b0));
    assign select_ne_0 = IN1[1] &  (IN1[0] ^ 1'b0);
    assign select_2x_0 = (~ IN1[1]) & IN1[0] & 1'b0;
    assign select_n2x_0 = IN1[1] & (~ IN1[0]) & (~ 1'b0);
    wire [34:0] pp_0;
    assign pp_0 = (1<<34) ^ ((select_e_0 ? IN2_1x : 0) | (select_2x_0 ? IN2_2x : 0) | (select_n2x_0 ? (~ IN2_2x) : 0) | (select_ne_0 ? (~ IN2_1x) : 0));
    assign tcomp0 = select_ne_0 | select_n2x_0;
    
    // Signed Booth Radix-4 Partial Products Row 2
    wire logic select_e_1, select_ne_1, select_2x_1, tcomp1, select_n2x_1;
    assign select_e_1 = ((~ IN1[3]) & (IN1[2] ^ IN1[1]));
    assign select_ne_1 = IN1[3] &  (IN1[2] ^ IN1[1]);
    assign select_2x_1 = (~ IN1[3]) & IN1[2] & IN1[1];
    assign select_n2x_1 = IN1[3] & (~ IN1[2]) & (~ IN1[1]);
    wire [34:0] pp_1;
    assign pp_1 = (1<<34) ^ ((select_e_1 ? IN2_1x : 0) | (select_2x_1 ? IN2_2x : 0) | (select_n2x_1 ? (~ IN2_2x) : 0) | (select_ne_1 ? (~ IN2_1x) : 0));
    assign tcomp1 = select_ne_1 | select_n2x_1;
    
    // Signed Booth Radix-4 Partial Products Row 3
    wire logic select_e_2, select_ne_2, select_2x_2, tcomp2, select_n2x_2;
    assign select_e_2 = ((~ IN1[5]) & (IN1[4] ^ IN1[3]));
    assign select_ne_2 = IN1[5] &  (IN1[4] ^ IN1[3]);
    assign select_2x_2 = (~ IN1[5]) & IN1[4] & IN1[3];
    assign select_n2x_2 = IN1[5] & (~ IN1[4]) & (~ IN1[3]);
    wire [34:0] pp_2;
    assign pp_2 = (1<<34) ^ ((select_e_2 ? IN2_1x : 0) | (select_2x_2 ? IN2_2x : 0) | (select_n2x_2 ? (~ IN2_2x) : 0) | (select_ne_2 ? (~ IN2_1x) : 0));
    assign tcomp2 = select_ne_2 | select_n2x_2;
    
    // Signed Booth Radix-4 Partial Products Row 4
    wire logic select_e_3, select_ne_3, select_2x_3, tcomp3, select_n2x_3;
    assign select_e_3 = ((~ IN1[7]) & (IN1[6] ^ IN1[5]));
    assign select_ne_3 = IN1[7] &  (IN1[6] ^ IN1[5]);
    assign select_2x_3 = (~ IN1[7]) & IN1[6] & IN1[5];
    assign select_n2x_3 = IN1[7] & (~ IN1[6]) & (~ IN1[5]);
    wire [34:0] pp_3;
    assign pp_3 = (1<<34) ^ ((select_e_3 ? IN2_1x : 0) | (select_2x_3 ? IN2_2x : 0) | (select_n2x_3 ? (~ IN2_2x) : 0) | (select_ne_3 ? (~ IN2_1x) : 0));
    assign tcomp3 = select_ne_3 | select_n2x_3;
    
    // Signed Booth Radix-4 Partial Products Row 5
    wire logic select_e_4, select_ne_4, select_2x_4, tcomp4, select_n2x_4;
    assign select_e_4 = ((~ IN1[9]) & (IN1[8] ^ IN1[7]));
    assign select_ne_4 = IN1[9] &  (IN1[8] ^ IN1[7]);
    assign select_2x_4 = (~ IN1[9]) & IN1[8] & IN1[7];
    assign select_n2x_4 = IN1[9] & (~ IN1[8]) & (~ IN1[7]);
    wire [34:0] pp_4;
    assign pp_4 = (1<<34) ^ ((select_e_4 ? IN2_1x : 0) | (select_2x_4 ? IN2_2x : 0) | (select_n2x_4 ? (~ IN2_2x) : 0) | (select_ne_4 ? (~ IN2_1x) : 0));
    assign tcomp4 = select_ne_4 | select_n2x_4;
    
    // Signed Booth Radix-4 Partial Products Row 6
    wire logic select_e_5, select_ne_5, select_2x_5, tcomp5, select_n2x_5;
    assign select_e_5 = ((~ IN1[11]) & (IN1[10] ^ IN1[9]));
    assign select_ne_5 = IN1[11] &  (IN1[10] ^ IN1[9]);
    assign select_2x_5 = (~ IN1[11]) & IN1[10] & IN1[9];
    assign select_n2x_5 = IN1[11] & (~ IN1[10]) & (~ IN1[9]);
    wire [34:0] pp_5;
    assign pp_5 = (1<<34) ^ ((select_e_5 ? IN2_1x : 0) | (select_2x_5 ? IN2_2x : 0) | (select_n2x_5 ? (~ IN2_2x) : 0) | (select_ne_5 ? (~ IN2_1x) : 0));
    assign tcomp5 = select_ne_5 | select_n2x_5;
    
    // Signed Booth Radix-4 Partial Products Row 7
    wire logic select_e_6, select_ne_6, select_2x_6, tcomp6, select_n2x_6;
    assign select_e_6 = ((~ IN1[13]) & (IN1[12] ^ IN1[11]));
    assign select_ne_6 = IN1[13] &  (IN1[12] ^ IN1[11]);
    assign select_2x_6 = (~ IN1[13]) & IN1[12] & IN1[11];
    assign select_n2x_6 = IN1[13] & (~ IN1[12]) & (~ IN1[11]);
    wire [34:0] pp_6;
    assign pp_6 = (1<<34) ^ ((select_e_6 ? IN2_1x : 0) | (select_2x_6 ? IN2_2x : 0) | (select_n2x_6 ? (~ IN2_2x) : 0) | (select_ne_6 ? (~ IN2_1x) : 0));
    assign tcomp6 = select_ne_6 | select_n2x_6;
    
    // Signed Booth Radix-4 Partial Products Row 8
    wire logic select_e_7, select_ne_7, select_2x_7, tcomp7, select_n2x_7;
    assign select_e_7 = ((~ IN1[15]) & (IN1[14] ^ IN1[13]));
    assign select_ne_7 = IN1[15] &  (IN1[14] ^ IN1[13]);
    assign select_2x_7 = (~ IN1[15]) & IN1[14] & IN1[13];
    assign select_n2x_7 = IN1[15] & (~ IN1[14]) & (~ IN1[13]);
    wire [34:0] pp_7;
    assign pp_7 = (1<<34) ^ ((select_e_7 ? IN2_1x : 0) | (select_2x_7 ? IN2_2x : 0) | (select_n2x_7 ? (~ IN2_2x) : 0) | (select_ne_7 ? (~ IN2_1x) : 0));
    assign tcomp7 = select_ne_7 | select_n2x_7;
    
    // Signed Booth Radix-4 Partial Products Row 9
    wire logic select_e_8, select_ne_8, select_2x_8, tcomp8, select_n2x_8;
    assign select_e_8 = ((~ IN1[17]) & (IN1[16] ^ IN1[15]));
    assign select_ne_8 = IN1[17] &  (IN1[16] ^ IN1[15]);
    assign select_2x_8 = (~ IN1[17]) & IN1[16] & IN1[15];
    assign select_n2x_8 = IN1[17] & (~ IN1[16]) & (~ IN1[15]);
    wire [34:0] pp_8;
    assign pp_8 = (1<<34) ^ ((select_e_8 ? IN2_1x : 0) | (select_2x_8 ? IN2_2x : 0) | (select_n2x_8 ? (~ IN2_2x) : 0) | (select_ne_8 ? (~ IN2_1x) : 0));
    assign tcomp8 = select_ne_8 | select_n2x_8;
    
    // Signed Booth Radix-4 Partial Products Row 10
    wire logic select_e_9, select_ne_9, select_2x_9, tcomp9, select_n2x_9;
    assign select_e_9 = ((~ IN1[19]) & (IN1[18] ^ IN1[17]));
    assign select_ne_9 = IN1[19] &  (IN1[18] ^ IN1[17]);
    assign select_2x_9 = (~ IN1[19]) & IN1[18] & IN1[17];
    assign select_n2x_9 = IN1[19] & (~ IN1[18]) & (~ IN1[17]);
    wire [34:0] pp_9;
    assign pp_9 = (1<<34) ^ ((select_e_9 ? IN2_1x : 0) | (select_2x_9 ? IN2_2x : 0) | (select_n2x_9 ? (~ IN2_2x) : 0) | (select_ne_9 ? (~ IN2_1x) : 0));
    assign tcomp9 = select_ne_9 | select_n2x_9;
    
    // Signed Booth Radix-4 Partial Products Row 11
    wire logic select_e_10, select_ne_10, select_2x_10, tcomp10, select_n2x_10;
    assign select_e_10 = ((~ IN1[21]) & (IN1[20] ^ IN1[19]));
    assign select_ne_10 = IN1[21] &  (IN1[20] ^ IN1[19]);
    assign select_2x_10 = (~ IN1[21]) & IN1[20] & IN1[19];
    assign select_n2x_10 = IN1[21] & (~ IN1[20]) & (~ IN1[19]);
    wire [34:0] pp_10;
    assign pp_10 = (1<<34) ^ ((select_e_10 ? IN2_1x : 0) | (select_2x_10 ? IN2_2x : 0) | (select_n2x_10 ? (~ IN2_2x) : 0) | (select_ne_10 ? (~ IN2_1x) : 0));
    assign tcomp10 = select_ne_10 | select_n2x_10;
    
    // Signed Booth Radix-4 Partial Products Row 12
    wire logic select_e_11, select_ne_11, select_2x_11, tcomp11, select_n2x_11;
    assign select_e_11 = ((~ IN1[23]) & (IN1[22] ^ IN1[21]));
    assign select_ne_11 = IN1[23] &  (IN1[22] ^ IN1[21]);
    assign select_2x_11 = (~ IN1[23]) & IN1[22] & IN1[21];
    assign select_n2x_11 = IN1[23] & (~ IN1[22]) & (~ IN1[21]);
    wire [34:0] pp_11;
    assign pp_11 = (1<<34) ^ ((select_e_11 ? IN2_1x : 0) | (select_2x_11 ? IN2_2x : 0) | (select_n2x_11 ? (~ IN2_2x) : 0) | (select_ne_11 ? (~ IN2_1x) : 0));
    assign tcomp11 = select_ne_11 | select_n2x_11;
    
    // Signed Booth Radix-4 Partial Products Row 13
    wire logic select_e_12, select_ne_12, select_2x_12, tcomp12, select_n2x_12;
    assign select_e_12 = ((~ IN1[25]) & (IN1[24] ^ IN1[23]));
    assign select_ne_12 = IN1[25] &  (IN1[24] ^ IN1[23]);
    assign select_2x_12 = (~ IN1[25]) & IN1[24] & IN1[23];
    assign select_n2x_12 = IN1[25] & (~ IN1[24]) & (~ IN1[23]);
    wire [34:0] pp_12;
    assign pp_12 = (1<<34) ^ ((select_e_12 ? IN2_1x : 0) | (select_2x_12 ? IN2_2x : 0) | (select_n2x_12 ? (~ IN2_2x) : 0) | (select_ne_12 ? (~ IN2_1x) : 0));
    assign tcomp12 = select_ne_12 | select_n2x_12;
    
    // Signed Booth Radix-4 Partial Products Row 14
    wire logic select_e_13, select_ne_13, select_2x_13, tcomp13, select_n2x_13;
    assign select_e_13 = ((~ IN1[27]) & (IN1[26] ^ IN1[25]));
    assign select_ne_13 = IN1[27] &  (IN1[26] ^ IN1[25]);
    assign select_2x_13 = (~ IN1[27]) & IN1[26] & IN1[25];
    assign select_n2x_13 = IN1[27] & (~ IN1[26]) & (~ IN1[25]);
    wire [34:0] pp_13;
    assign pp_13 = (1<<34) ^ ((select_e_13 ? IN2_1x : 0) | (select_2x_13 ? IN2_2x : 0) | (select_n2x_13 ? (~ IN2_2x) : 0) | (select_ne_13 ? (~ IN2_1x) : 0));
    assign tcomp13 = select_ne_13 | select_n2x_13;
    
    // Signed Booth Radix-4 Partial Products Row 15
    wire logic select_e_14, select_ne_14, select_2x_14, tcomp14, select_n2x_14;
    assign select_e_14 = ((~ IN1[29]) & (IN1[28] ^ IN1[27]));
    assign select_ne_14 = IN1[29] &  (IN1[28] ^ IN1[27]);
    assign select_2x_14 = (~ IN1[29]) & IN1[28] & IN1[27];
    assign select_n2x_14 = IN1[29] & (~ IN1[28]) & (~ IN1[27]);
    wire [34:0] pp_14;
    assign pp_14 = (1<<34) ^ ((select_e_14 ? IN2_1x : 0) | (select_2x_14 ? IN2_2x : 0) | (select_n2x_14 ? (~ IN2_2x) : 0) | (select_ne_14 ? (~ IN2_1x) : 0));
    assign tcomp14 = select_ne_14 | select_n2x_14;
    
    // Signed Booth Radix-4 Partial Products Row 16
    wire logic select_e_15, select_ne_15, select_2x_15, tcomp15, select_n2x_15;
    assign select_e_15 = ((~ IN1[31]) & (IN1[30] ^ IN1[29]));
    assign select_ne_15 = IN1[31] &  (IN1[30] ^ IN1[29]);
    assign select_2x_15 = (~ IN1[31]) & IN1[30] & IN1[29];
    assign select_n2x_15 = IN1[31] & (~ IN1[30]) & (~ IN1[29]);
    wire [34:0] pp_15;
    assign pp_15 = (1<<34) ^ ((select_e_15 ? IN2_1x : 0) | (select_2x_15 ? IN2_2x : 0) | (select_n2x_15 ? (~ IN2_2x) : 0) | (select_ne_15 ? (~ IN2_1x) : 0));
    assign tcomp15 = select_ne_15 | select_n2x_15;
    
    // Signed Booth Radix-4 Partial Products Row 17
    wire logic select_e_16, select_ne_16, select_2x_16, tcomp16, select_n2x_16;
    assign select_e_16 = ((~ IN1[32]) & (IN1[32] ^ IN1[31]));
    assign select_ne_16 = IN1[32] &  (IN1[32] ^ IN1[31]);
    assign select_2x_16 = (~ IN1[32]) & IN1[32] & IN1[31];
    assign select_n2x_16 = IN1[32] & (~ IN1[32]) & (~ IN1[31]);
    wire [34:0] pp_16;
    assign pp_16 = (1<<34) ^ ((select_e_16 ? IN2_1x : 0) | (select_2x_16 ? IN2_2x : 0) | (select_n2x_16 ? (~ IN2_2x) : 0) | (select_ne_16 ? (~ IN2_1x) : 0));
    assign tcomp16 = select_ne_16 | select_n2x_16;
    
// Creating Summation Tree 

    
    // Dadda Summation Stage 1
    logic s0 ,c0;
    ha ha0 (pp_0[24], pp_1[22], s0, c0);
    logic s1 ,c1;
    ha ha1 (pp_0[25], pp_1[23], s1, c1);
    logic s2 ,c2; 
    fa fa2 (pp_0[26], pp_1[24], pp_2[22], s2, c2);
    logic s3 ,c3;
    ha ha3 (pp_3[20], pp_4[18], s3, c3);
    logic s4 ,c4; 
    fa fa4 (pp_0[27], pp_1[25], pp_2[23], s4, c4);
    logic s5 ,c5;
    ha ha5 (pp_3[21], pp_4[19], s5, c5);
    logic s6 ,c6; 
    fa fa6 (pp_0[28], pp_1[26], pp_2[24], s6, c6);
    logic s7 ,c7; 
    fa fa7 (pp_3[22], pp_4[20], pp_5[18], s7, c7);
    logic s8 ,c8;
    ha ha8 (pp_6[16], pp_7[14], s8, c8);
    logic s9 ,c9; 
    fa fa9 (pp_0[29], pp_1[27], pp_2[25], s9, c9);
    logic s10 ,c10; 
    fa fa10 (pp_3[23], pp_4[21], pp_5[19], s10, c10);
    logic s11 ,c11;
    ha ha11 (pp_6[17], pp_7[15], s11, c11);
    logic s12 ,c12; 
    fa fa12 (pp_0[30], pp_1[28], pp_2[26], s12, c12);
    logic s13 ,c13; 
    fa fa13 (pp_3[24], pp_4[22], pp_5[20], s13, c13);
    logic s14 ,c14; 
    fa fa14 (pp_6[18], pp_7[16], pp_8[14], s14, c14);
    logic s15 ,c15;
    ha ha15 (pp_9[12], pp_10[10], s15, c15);
    logic s16 ,c16; 
    fa fa16 (pp_0[31], pp_1[29], pp_2[27], s16, c16);
    logic s17 ,c17; 
    fa fa17 (pp_3[25], pp_4[23], pp_5[21], s17, c17);
    logic s18 ,c18; 
    fa fa18 (pp_6[19], pp_7[17], pp_8[15], s18, c18);
    logic s19 ,c19;
    ha ha19 (pp_9[13], pp_10[11], s19, c19);
    logic s20 ,c20; 
    fa fa20 (pp_0[32], pp_1[30], pp_2[28], s20, c20);
    logic s21 ,c21; 
    fa fa21 (pp_3[26], pp_4[24], pp_5[22], s21, c21);
    logic s22 ,c22; 
    fa fa22 (pp_6[20], pp_7[18], pp_8[16], s22, c22);
    logic s23 ,c23; 
    fa fa23 (pp_9[14], pp_10[12], pp_11[10], s23, c23);
    logic s24 ,c24;
    ha ha24 (pp_12[8], pp_13[6], s24, c24);
    logic s25 ,c25; 
    fa fa25 (pp_0[33], pp_1[31], pp_2[29], s25, c25);
    logic s26 ,c26; 
    fa fa26 (pp_3[27], pp_4[25], pp_5[23], s26, c26);
    logic s27 ,c27; 
    fa fa27 (pp_6[21], pp_7[19], pp_8[17], s27, c27);
    logic s28 ,c28; 
    fa fa28 (pp_9[15], pp_10[13], pp_11[11], s28, c28);
    logic s29 ,c29;
    ha ha29 (pp_12[9], pp_13[7], s29, c29);
    logic s30 ,c30; 
    fa fa30 (pp_0[34], pp_1[32], pp_2[30], s30, c30);
    logic s31 ,c31; 
    fa fa31 (pp_3[28], pp_4[26], pp_5[24], s31, c31);
    logic s32 ,c32; 
    fa fa32 (pp_6[22], pp_7[20], pp_8[18], s32, c32);
    logic s33 ,c33; 
    fa fa33 (pp_9[16], pp_10[14], pp_11[12], s33, c33);
    logic s34 ,c34; 
    fa fa34 (pp_12[10], pp_13[8], pp_14[6], s34, c34);
    logic s35 ,c35; 
    fa fa35 (const1, pp_1[33], pp_2[31], s35, c35);
    logic s36 ,c36; 
    fa fa36 (pp_3[29], pp_4[27], pp_5[25], s36, c36);
    logic s37 ,c37; 
    fa fa37 (pp_6[23], pp_7[21], pp_8[19], s37, c37);
    logic s38 ,c38; 
    fa fa38 (pp_9[17], pp_10[15], pp_11[13], s38, c38);
    logic s39 ,c39;
    ha ha39 (pp_12[11], pp_13[9], s39, c39);
    logic s40 ,c40; 
    fa fa40 (pp_1[34], pp_2[32], pp_3[30], s40, c40);
    logic s41 ,c41; 
    fa fa41 (pp_4[28], pp_5[26], pp_6[24], s41, c41);
    logic s42 ,c42; 
    fa fa42 (pp_7[22], pp_8[20], pp_9[18], s42, c42);
    logic s43 ,c43; 
    fa fa43 (pp_10[16], pp_11[14], pp_12[12], s43, c43);
    logic s44 ,c44; 
    fa fa44 (const1, pp_2[33], pp_3[31], s44, c44);
    logic s45 ,c45; 
    fa fa45 (pp_4[29], pp_5[27], pp_6[25], s45, c45);
    logic s46 ,c46; 
    fa fa46 (pp_7[23], pp_8[21], pp_9[19], s46, c46);
    logic s47 ,c47;
    ha ha47 (pp_10[17], pp_11[15], s47, c47);
    logic s48 ,c48; 
    fa fa48 (pp_2[34], pp_3[32], pp_4[30], s48, c48);
    logic s49 ,c49; 
    fa fa49 (pp_5[28], pp_6[26], pp_7[24], s49, c49);
    logic s50 ,c50; 
    fa fa50 (pp_8[22], pp_9[20], pp_10[18], s50, c50);
    logic s51 ,c51; 
    fa fa51 (const1, pp_3[33], pp_4[31], s51, c51);
    logic s52 ,c52; 
    fa fa52 (pp_5[29], pp_6[27], pp_7[25], s52, c52);
    logic s53 ,c53;
    ha ha53 (pp_8[23], pp_9[21], s53, c53);
    logic s54 ,c54; 
    fa fa54 (pp_3[34], pp_4[32], pp_5[30], s54, c54);
    logic s55 ,c55; 
    fa fa55 (pp_6[28], pp_7[26], pp_8[24], s55, c55);
    logic s56 ,c56; 
    fa fa56 (const1, pp_4[33], pp_5[31], s56, c56);
    logic s57 ,c57;
    ha ha57 (pp_6[29], pp_7[27], s57, c57);
    logic s58 ,c58; 
    fa fa58 (pp_4[34], pp_5[32], pp_6[30], s58, c58);
    logic s59 ,c59;
    ha ha59 (const1, pp_5[33], s59, c59);
    
    // Dadda Summation Stage 2
    logic s60 ,c60;
    ha ha60 (pp_0[16], pp_1[14], s60, c60);
    logic s61 ,c61;
    ha ha61 (pp_0[17], pp_1[15], s61, c61);
    logic s62 ,c62; 
    fa fa62 (pp_0[18], pp_1[16], pp_2[14], s62, c62);
    logic s63 ,c63;
    ha ha63 (pp_3[12], pp_4[10], s63, c63);
    logic s64 ,c64; 
    fa fa64 (pp_0[19], pp_1[17], pp_2[15], s64, c64);
    logic s65 ,c65;
    ha ha65 (pp_3[13], pp_4[11], s65, c65);
    logic s66 ,c66; 
    fa fa66 (pp_0[20], pp_1[18], pp_2[16], s66, c66);
    logic s67 ,c67; 
    fa fa67 (pp_3[14], pp_4[12], pp_5[10], s67, c67);
    logic s68 ,c68;
    ha ha68 (pp_6[8], pp_7[6], s68, c68);
    logic s69 ,c69; 
    fa fa69 (pp_0[21], pp_1[19], pp_2[17], s69, c69);
    logic s70 ,c70; 
    fa fa70 (pp_3[15], pp_4[13], pp_5[11], s70, c70);
    logic s71 ,c71;
    ha ha71 (pp_6[9], pp_7[7], s71, c71);
    logic s72 ,c72; 
    fa fa72 (pp_0[22], pp_1[20], pp_2[18], s72, c72);
    logic s73 ,c73; 
    fa fa73 (pp_3[16], pp_4[14], pp_5[12], s73, c73);
    logic s74 ,c74; 
    fa fa74 (pp_6[10], pp_7[8], pp_8[6], s74, c74);
    logic s75 ,c75;
    ha ha75 (pp_9[4], pp_10[2], s75, c75);
    logic s76 ,c76; 
    fa fa76 (pp_0[23], pp_1[21], pp_2[19], s76, c76);
    logic s77 ,c77; 
    fa fa77 (pp_3[17], pp_4[15], pp_5[13], s77, c77);
    logic s78 ,c78; 
    fa fa78 (pp_6[11], pp_7[9], pp_8[7], s78, c78);
    logic s79 ,c79;
    ha ha79 (pp_9[5], pp_10[3], s79, c79);
    logic s80 ,c80; 
    fa fa80 (pp_2[20], pp_3[18], pp_4[16], s80, c80);
    logic s81 ,c81; 
    fa fa81 (pp_5[14], pp_6[12], pp_7[10], s81, c81);
    logic s82 ,c82; 
    fa fa82 (pp_8[8], pp_9[6], pp_10[4], s82, c82);
    logic s83 ,c83; 
    fa fa83 (pp_11[2], pp_12[0], tcomp12, s83, c83);
    logic s84 ,c84; 
    fa fa84 (pp_2[21], pp_3[19], pp_4[17], s84, c84);
    logic s85 ,c85; 
    fa fa85 (pp_5[15], pp_6[13], pp_7[11], s85, c85);
    logic s86 ,c86; 
    fa fa86 (pp_8[9], pp_9[7], pp_10[5], s86, c86);
    logic s87 ,c87; 
    fa fa87 (pp_11[3], pp_12[1], c0, s87, c87);
    logic s88 ,c88; 
    fa fa88 (pp_5[16], pp_6[14], pp_7[12], s88, c88);
    logic s89 ,c89; 
    fa fa89 (pp_8[10], pp_9[8], pp_10[6], s89, c89);
    logic s90 ,c90; 
    fa fa90 (pp_11[4], pp_12[2], pp_13[0], s90, c90);
    logic s91 ,c91; 
    fa fa91 (tcomp13, c1, s2, s91, c91);
    logic s92 ,c92; 
    fa fa92 (pp_5[17], pp_6[15], pp_7[13], s92, c92);
    logic s93 ,c93; 
    fa fa93 (pp_8[11], pp_9[9], pp_10[7], s93, c93);
    logic s94 ,c94; 
    fa fa94 (pp_11[5], pp_12[3], pp_13[1], s94, c94);
    logic s95 ,c95; 
    fa fa95 (c2, c3, s4, s95, c95);
    logic s96 ,c96; 
    fa fa96 (pp_8[12], pp_9[10], pp_10[8], s96, c96);
    logic s97 ,c97; 
    fa fa97 (pp_11[6], pp_12[4], pp_13[2], s97, c97);
    logic s98 ,c98; 
    fa fa98 (pp_14[0], tcomp14, c4, s98, c98);
    logic s99 ,c99; 
    fa fa99 (c5, s6, s7, s99, c99);
    logic s100 ,c100; 
    fa fa100 (pp_8[13], pp_9[11], pp_10[9], s100, c100);
    logic s101 ,c101; 
    fa fa101 (pp_11[7], pp_12[5], pp_13[3], s101, c101);
    logic s102 ,c102; 
    fa fa102 (pp_14[1], c6, c7, s102, c102);
    logic s103 ,c103; 
    fa fa103 (c8, s9, s10, s103, c103);
    logic s104 ,c104; 
    fa fa104 (pp_11[8], pp_12[6], pp_13[4], s104, c104);
    logic s105 ,c105; 
    fa fa105 (pp_14[2], pp_15[0], tcomp15, s105, c105);
    logic s106 ,c106; 
    fa fa106 (c9, c10, c11, s106, c106);
    logic s107 ,c107; 
    fa fa107 (s12, s13, s14, s107, c107);
    logic s108 ,c108; 
    fa fa108 (pp_11[9], pp_12[7], pp_13[5], s108, c108);
    logic s109 ,c109; 
    fa fa109 (pp_14[3], pp_15[1], c12, s109, c109);
    logic s110 ,c110; 
    fa fa110 (c13, c14, c15, s110, c110);
    logic s111 ,c111; 
    fa fa111 (s16, s17, s18, s111, c111);
    logic s112 ,c112; 
    fa fa112 (pp_14[4], pp_15[2], pp_16[0], s112, c112);
    logic s113 ,c113; 
    fa fa113 (tcomp16, c16, c17, s113, c113);
    logic s114 ,c114; 
    fa fa114 (c18, c19, s20, s114, c114);
    logic s115 ,c115; 
    fa fa115 (s21, s22, s23, s115, c115);
    logic s116 ,c116; 
    fa fa116 (pp_14[5], pp_15[3], pp_16[1], s116, c116);
    logic s117 ,c117; 
    fa fa117 (c20, c21, c22, s117, c117);
    logic s118 ,c118; 
    fa fa118 (c23, c24, s25, s118, c118);
    logic s119 ,c119; 
    fa fa119 (s26, s27, s28, s119, c119);
    logic s120 ,c120; 
    fa fa120 (pp_15[4], pp_16[2], const1, s120, c120);
    logic s121 ,c121; 
    fa fa121 (c25, c26, c27, s121, c121);
    logic s122 ,c122; 
    fa fa122 (c28, c29, s30, s122, c122);
    logic s123 ,c123; 
    fa fa123 (s31, s32, s33, s123, c123);
    logic s124 ,c124; 
    fa fa124 (pp_14[7], pp_15[5], pp_16[3], s124, c124);
    logic s125 ,c125; 
    fa fa125 (c30, c31, c32, s125, c125);
    logic s126 ,c126; 
    fa fa126 (c33, c34, s35, s126, c126);
    logic s127 ,c127; 
    fa fa127 (s36, s37, s38, s127, c127);
    logic s128 ,c128; 
    fa fa128 (pp_13[10], pp_14[8], pp_15[6], s128, c128);
    logic s129 ,c129; 
    fa fa129 (pp_16[4], c35, c36, s129, c129);
    logic s130 ,c130; 
    fa fa130 (c37, c38, c39, s130, c130);
    logic s131 ,c131; 
    fa fa131 (s40, s41, s42, s131, c131);
    logic s132 ,c132; 
    fa fa132 (pp_12[13], pp_13[11], pp_14[9], s132, c132);
    logic s133 ,c133; 
    fa fa133 (pp_15[7], pp_16[5], c40, s133, c133);
    logic s134 ,c134; 
    fa fa134 (c41, c42, c43, s134, c134);
    logic s135 ,c135; 
    fa fa135 (s44, s45, s46, s135, c135);
    logic s136 ,c136; 
    fa fa136 (pp_11[16], pp_12[14], pp_13[12], s136, c136);
    logic s137 ,c137; 
    fa fa137 (pp_14[10], pp_15[8], pp_16[6], s137, c137);
    logic s138 ,c138; 
    fa fa138 (c44, c45, c46, s138, c138);
    logic s139 ,c139; 
    fa fa139 (c47, s48, s49, s139, c139);
    logic s140 ,c140; 
    fa fa140 (pp_10[19], pp_11[17], pp_12[15], s140, c140);
    logic s141 ,c141; 
    fa fa141 (pp_13[13], pp_14[11], pp_15[9], s141, c141);
    logic s142 ,c142; 
    fa fa142 (pp_16[7], c48, c49, s142, c142);
    logic s143 ,c143; 
    fa fa143 (c50, s51, s52, s143, c143);
    logic s144 ,c144; 
    fa fa144 (pp_9[22], pp_10[20], pp_11[18], s144, c144);
    logic s145 ,c145; 
    fa fa145 (pp_12[16], pp_13[14], pp_14[12], s145, c145);
    logic s146 ,c146; 
    fa fa146 (pp_15[10], pp_16[8], c51, s146, c146);
    logic s147 ,c147; 
    fa fa147 (c52, c53, s54, s147, c147);
    logic s148 ,c148; 
    fa fa148 (pp_8[25], pp_9[23], pp_10[21], s148, c148);
    logic s149 ,c149; 
    fa fa149 (pp_11[19], pp_12[17], pp_13[15], s149, c149);
    logic s150 ,c150; 
    fa fa150 (pp_14[13], pp_15[11], pp_16[9], s150, c150);
    logic s151 ,c151; 
    fa fa151 (c54, c55, s56, s151, c151);
    logic s152 ,c152; 
    fa fa152 (pp_7[28], pp_8[26], pp_9[24], s152, c152);
    logic s153 ,c153; 
    fa fa153 (pp_10[22], pp_11[20], pp_12[18], s153, c153);
    logic s154 ,c154; 
    fa fa154 (pp_13[16], pp_14[14], pp_15[12], s154, c154);
    logic s155 ,c155; 
    fa fa155 (pp_16[10], c56, c57, s155, c155);
    logic s156 ,c156; 
    fa fa156 (pp_6[31], pp_7[29], pp_8[27], s156, c156);
    logic s157 ,c157; 
    fa fa157 (pp_9[25], pp_10[23], pp_11[21], s157, c157);
    logic s158 ,c158; 
    fa fa158 (pp_12[19], pp_13[17], pp_14[15], s158, c158);
    logic s159 ,c159; 
    fa fa159 (pp_15[13], pp_16[11], c58, s159, c159);
    logic s160 ,c160; 
    fa fa160 (pp_5[34], pp_6[32], pp_7[30], s160, c160);
    logic s161 ,c161; 
    fa fa161 (pp_8[28], pp_9[26], pp_10[24], s161, c161);
    logic s162 ,c162; 
    fa fa162 (pp_11[22], pp_12[20], pp_13[18], s162, c162);
    logic s163 ,c163; 
    fa fa163 (pp_14[16], pp_15[14], pp_16[12], s163, c163);
    logic s164 ,c164; 
    fa fa164 (const1, pp_6[33], pp_7[31], s164, c164);
    logic s165 ,c165; 
    fa fa165 (pp_8[29], pp_9[27], pp_10[25], s165, c165);
    logic s166 ,c166; 
    fa fa166 (pp_11[23], pp_12[21], pp_13[19], s166, c166);
    logic s167 ,c167;
    ha ha167 (pp_14[17], pp_15[15], s167, c167);
    logic s168 ,c168; 
    fa fa168 (pp_6[34], pp_7[32], pp_8[30], s168, c168);
    logic s169 ,c169; 
    fa fa169 (pp_9[28], pp_10[26], pp_11[24], s169, c169);
    logic s170 ,c170; 
    fa fa170 (pp_12[22], pp_13[20], pp_14[18], s170, c170);
    logic s171 ,c171; 
    fa fa171 (const1, pp_7[33], pp_8[31], s171, c171);
    logic s172 ,c172; 
    fa fa172 (pp_9[29], pp_10[27], pp_11[25], s172, c172);
    logic s173 ,c173;
    ha ha173 (pp_12[23], pp_13[21], s173, c173);
    logic s174 ,c174; 
    fa fa174 (pp_7[34], pp_8[32], pp_9[30], s174, c174);
    logic s175 ,c175; 
    fa fa175 (pp_10[28], pp_11[26], pp_12[24], s175, c175);
    logic s176 ,c176; 
    fa fa176 (const1, pp_8[33], pp_9[31], s176, c176);
    logic s177 ,c177;
    ha ha177 (pp_10[29], pp_11[27], s177, c177);
    logic s178 ,c178; 
    fa fa178 (pp_8[34], pp_9[32], pp_10[30], s178, c178);
    logic s179 ,c179;
    ha ha179 (const1, pp_9[33], s179, c179);
    
    // Dadda Summation Stage 3
    logic s180 ,c180;
    ha ha180 (pp_0[10], pp_1[8], s180, c180);
    logic s181 ,c181;
    ha ha181 (pp_0[11], pp_1[9], s181, c181);
    logic s182 ,c182; 
    fa fa182 (pp_0[12], pp_1[10], pp_2[8], s182, c182);
    logic s183 ,c183;
    ha ha183 (pp_3[6], pp_4[4], s183, c183);
    logic s184 ,c184; 
    fa fa184 (pp_0[13], pp_1[11], pp_2[9], s184, c184);
    logic s185 ,c185;
    ha ha185 (pp_3[7], pp_4[5], s185, c185);
    logic s186 ,c186; 
    fa fa186 (pp_0[14], pp_1[12], pp_2[10], s186, c186);
    logic s187 ,c187; 
    fa fa187 (pp_3[8], pp_4[6], pp_5[4], s187, c187);
    logic s188 ,c188;
    ha ha188 (pp_6[2], pp_7[0], s188, c188);
    logic s189 ,c189; 
    fa fa189 (pp_0[15], pp_1[13], pp_2[11], s189, c189);
    logic s190 ,c190; 
    fa fa190 (pp_3[9], pp_4[7], pp_5[5], s190, c190);
    logic s191 ,c191;
    ha ha191 (pp_6[3], pp_7[1], s191, c191);
    logic s192 ,c192; 
    fa fa192 (pp_2[12], pp_3[10], pp_4[8], s192, c192);
    logic s193 ,c193; 
    fa fa193 (pp_5[6], pp_6[4], pp_7[2], s193, c193);
    logic s194 ,c194; 
    fa fa194 (pp_8[0], tcomp8, s60, s194, c194);
    logic s195 ,c195; 
    fa fa195 (pp_2[13], pp_3[11], pp_4[9], s195, c195);
    logic s196 ,c196; 
    fa fa196 (pp_5[7], pp_6[5], pp_7[3], s196, c196);
    logic s197 ,c197; 
    fa fa197 (pp_8[1], c60, s61, s197, c197);
    logic s198 ,c198; 
    fa fa198 (pp_5[8], pp_6[6], pp_7[4], s198, c198);
    logic s199 ,c199; 
    fa fa199 (pp_8[2], pp_9[0], tcomp9, s199, c199);
    logic s200 ,c200; 
    fa fa200 (c61, s62, s63, s200, c200);
    logic s201 ,c201; 
    fa fa201 (pp_5[9], pp_6[7], pp_7[5], s201, c201);
    logic s202 ,c202; 
    fa fa202 (pp_8[3], pp_9[1], c62, s202, c202);
    logic s203 ,c203; 
    fa fa203 (c63, s64, s65, s203, c203);
    logic s204 ,c204; 
    fa fa204 (pp_8[4], pp_9[2], pp_10[0], s204, c204);
    logic s205 ,c205; 
    fa fa205 (tcomp10, c64, c65, s205, c205);
    logic s206 ,c206; 
    fa fa206 (s66, s67, s68, s206, c206);
    logic s207 ,c207; 
    fa fa207 (pp_8[5], pp_9[3], pp_10[1], s207, c207);
    logic s208 ,c208; 
    fa fa208 (c66, c67, c68, s208, c208);
    logic s209 ,c209; 
    fa fa209 (s69, s70, s71, s209, c209);
    logic s210 ,c210; 
    fa fa210 (pp_11[0], tcomp11, c69, s210, c210);
    logic s211 ,c211; 
    fa fa211 (c70, c71, s72, s211, c211);
    logic s212 ,c212; 
    fa fa212 (s73, s74, s75, s212, c212);
    logic s213 ,c213; 
    fa fa213 (pp_11[1], c72, c73, s213, c213);
    logic s214 ,c214; 
    fa fa214 (c74, c75, s76, s214, c214);
    logic s215 ,c215; 
    fa fa215 (s77, s78, s79, s215, c215);
    logic s216 ,c216; 
    fa fa216 (s0, c76, c77, s216, c216);
    logic s217 ,c217; 
    fa fa217 (c78, c79, s80, s217, c217);
    logic s218 ,c218; 
    fa fa218 (s81, s82, s83, s218, c218);
    logic s219 ,c219; 
    fa fa219 (s1, c80, c81, s219, c219);
    logic s220 ,c220; 
    fa fa220 (c82, c83, s84, s220, c220);
    logic s221 ,c221; 
    fa fa221 (s85, s86, s87, s221, c221);
    logic s222 ,c222; 
    fa fa222 (s3, c84, c85, s222, c222);
    logic s223 ,c223; 
    fa fa223 (c86, c87, s88, s223, c223);
    logic s224 ,c224; 
    fa fa224 (s89, s90, s91, s224, c224);
    logic s225 ,c225; 
    fa fa225 (s5, c88, c89, s225, c225);
    logic s226 ,c226; 
    fa fa226 (c90, c91, s92, s226, c226);
    logic s227 ,c227; 
    fa fa227 (s93, s94, s95, s227, c227);
    logic s228 ,c228; 
    fa fa228 (s8, c92, c93, s228, c228);
    logic s229 ,c229; 
    fa fa229 (c94, c95, s96, s229, c229);
    logic s230 ,c230; 
    fa fa230 (s97, s98, s99, s230, c230);
    logic s231 ,c231; 
    fa fa231 (s11, c96, c97, s231, c231);
    logic s232 ,c232; 
    fa fa232 (c98, c99, s100, s232, c232);
    logic s233 ,c233; 
    fa fa233 (s101, s102, s103, s233, c233);
    logic s234 ,c234; 
    fa fa234 (s15, c100, c101, s234, c234);
    logic s235 ,c235; 
    fa fa235 (c102, c103, s104, s235, c235);
    logic s236 ,c236; 
    fa fa236 (s105, s106, s107, s236, c236);
    logic s237 ,c237; 
    fa fa237 (s19, c104, c105, s237, c237);
    logic s238 ,c238; 
    fa fa238 (c106, c107, s108, s238, c238);
    logic s239 ,c239; 
    fa fa239 (s109, s110, s111, s239, c239);
    logic s240 ,c240; 
    fa fa240 (s24, c108, c109, s240, c240);
    logic s241 ,c241; 
    fa fa241 (c110, c111, s112, s241, c241);
    logic s242 ,c242; 
    fa fa242 (s113, s114, s115, s242, c242);
    logic s243 ,c243; 
    fa fa243 (s29, c112, c113, s243, c243);
    logic s244 ,c244; 
    fa fa244 (c114, c115, s116, s244, c244);
    logic s245 ,c245; 
    fa fa245 (s117, s118, s119, s245, c245);
    logic s246 ,c246; 
    fa fa246 (s34, c116, c117, s246, c246);
    logic s247 ,c247; 
    fa fa247 (c118, c119, s120, s247, c247);
    logic s248 ,c248; 
    fa fa248 (s121, s122, s123, s248, c248);
    logic s249 ,c249; 
    fa fa249 (s39, c120, c121, s249, c249);
    logic s250 ,c250; 
    fa fa250 (c122, c123, s124, s250, c250);
    logic s251 ,c251; 
    fa fa251 (s125, s126, s127, s251, c251);
    logic s252 ,c252; 
    fa fa252 (s43, c124, c125, s252, c252);
    logic s253 ,c253; 
    fa fa253 (c126, c127, s128, s253, c253);
    logic s254 ,c254; 
    fa fa254 (s129, s130, s131, s254, c254);
    logic s255 ,c255; 
    fa fa255 (s47, c128, c129, s255, c255);
    logic s256 ,c256; 
    fa fa256 (c130, c131, s132, s256, c256);
    logic s257 ,c257; 
    fa fa257 (s133, s134, s135, s257, c257);
    logic s258 ,c258; 
    fa fa258 (s50, c132, c133, s258, c258);
    logic s259 ,c259; 
    fa fa259 (c134, c135, s136, s259, c259);
    logic s260 ,c260; 
    fa fa260 (s137, s138, s139, s260, c260);
    logic s261 ,c261; 
    fa fa261 (s53, c136, c137, s261, c261);
    logic s262 ,c262; 
    fa fa262 (c138, c139, s140, s262, c262);
    logic s263 ,c263; 
    fa fa263 (s141, s142, s143, s263, c263);
    logic s264 ,c264; 
    fa fa264 (s55, c140, c141, s264, c264);
    logic s265 ,c265; 
    fa fa265 (c142, c143, s144, s265, c265);
    logic s266 ,c266; 
    fa fa266 (s145, s146, s147, s266, c266);
    logic s267 ,c267; 
    fa fa267 (s57, c144, c145, s267, c267);
    logic s268 ,c268; 
    fa fa268 (c146, c147, s148, s268, c268);
    logic s269 ,c269; 
    fa fa269 (s149, s150, s151, s269, c269);
    logic s270 ,c270; 
    fa fa270 (s58, c148, c149, s270, c270);
    logic s271 ,c271; 
    fa fa271 (c150, c151, s152, s271, c271);
    logic s272 ,c272; 
    fa fa272 (s153, s154, s155, s272, c272);
    logic s273 ,c273; 
    fa fa273 (s59, c152, c153, s273, c273);
    logic s274 ,c274; 
    fa fa274 (c154, c155, s156, s274, c274);
    logic s275 ,c275; 
    fa fa275 (s157, s158, s159, s275, c275);
    logic s276 ,c276; 
    fa fa276 (c59, c156, c157, s276, c276);
    logic s277 ,c277; 
    fa fa277 (c158, c159, s160, s277, c277);
    logic s278 ,c278; 
    fa fa278 (s161, s162, s163, s278, c278);
    logic s279 ,c279; 
    fa fa279 (pp_16[13], c160, c161, s279, c279);
    logic s280 ,c280; 
    fa fa280 (c162, c163, s164, s280, c280);
    logic s281 ,c281; 
    fa fa281 (s165, s166, s167, s281, c281);
    logic s282 ,c282; 
    fa fa282 (pp_15[16], pp_16[14], c164, s282, c282);
    logic s283 ,c283; 
    fa fa283 (c165, c166, c167, s283, c283);
    logic s284 ,c284; 
    fa fa284 (s168, s169, s170, s284, c284);
    logic s285 ,c285; 
    fa fa285 (pp_14[19], pp_15[17], pp_16[15], s285, c285);
    logic s286 ,c286; 
    fa fa286 (c168, c169, c170, s286, c286);
    logic s287 ,c287; 
    fa fa287 (s171, s172, s173, s287, c287);
    logic s288 ,c288; 
    fa fa288 (pp_13[22], pp_14[20], pp_15[18], s288, c288);
    logic s289 ,c289; 
    fa fa289 (pp_16[16], c171, c172, s289, c289);
    logic s290 ,c290; 
    fa fa290 (c173, s174, s175, s290, c290);
    logic s291 ,c291; 
    fa fa291 (pp_12[25], pp_13[23], pp_14[21], s291, c291);
    logic s292 ,c292; 
    fa fa292 (pp_15[19], pp_16[17], c174, s292, c292);
    logic s293 ,c293; 
    fa fa293 (c175, s176, s177, s293, c293);
    logic s294 ,c294; 
    fa fa294 (pp_11[28], pp_12[26], pp_13[24], s294, c294);
    logic s295 ,c295; 
    fa fa295 (pp_14[22], pp_15[20], pp_16[18], s295, c295);
    logic s296 ,c296; 
    fa fa296 (c176, c177, s178, s296, c296);
    logic s297 ,c297; 
    fa fa297 (pp_10[31], pp_11[29], pp_12[27], s297, c297);
    logic s298 ,c298; 
    fa fa298 (pp_13[25], pp_14[23], pp_15[21], s298, c298);
    logic s299 ,c299; 
    fa fa299 (pp_16[19], c178, s179, s299, c299);
    logic s300 ,c300; 
    fa fa300 (pp_9[34], pp_10[32], pp_11[30], s300, c300);
    logic s301 ,c301; 
    fa fa301 (pp_12[28], pp_13[26], pp_14[24], s301, c301);
    logic s302 ,c302; 
    fa fa302 (pp_15[22], pp_16[20], c179, s302, c302);
    logic s303 ,c303; 
    fa fa303 (const1, pp_10[33], pp_11[31], s303, c303);
    logic s304 ,c304; 
    fa fa304 (pp_12[29], pp_13[27], pp_14[25], s304, c304);
    logic s305 ,c305;
    ha ha305 (pp_15[23], pp_16[21], s305, c305);
    logic s306 ,c306; 
    fa fa306 (pp_10[34], pp_11[32], pp_12[30], s306, c306);
    logic s307 ,c307; 
    fa fa307 (pp_13[28], pp_14[26], pp_15[24], s307, c307);
    logic s308 ,c308; 
    fa fa308 (const1, pp_11[33], pp_12[31], s308, c308);
    logic s309 ,c309;
    ha ha309 (pp_13[29], pp_14[27], s309, c309);
    logic s310 ,c310; 
    fa fa310 (pp_11[34], pp_12[32], pp_13[30], s310, c310);
    logic s311 ,c311;
    ha ha311 (const1, pp_12[33], s311, c311);
    
    // Dadda Summation Stage 4
    logic s312 ,c312;
    ha ha312 (pp_0[6], pp_1[4], s312, c312);
    logic s313 ,c313;
    ha ha313 (pp_0[7], pp_1[5], s313, c313);
    logic s314 ,c314; 
    fa fa314 (pp_0[8], pp_1[6], pp_2[4], s314, c314);
    logic s315 ,c315;
    ha ha315 (pp_3[2], pp_4[0], s315, c315);
    logic s316 ,c316; 
    fa fa316 (pp_0[9], pp_1[7], pp_2[5], s316, c316);
    logic s317 ,c317;
    ha ha317 (pp_3[3], pp_4[1], s317, c317);
    logic s318 ,c318; 
    fa fa318 (pp_2[6], pp_3[4], pp_4[2], s318, c318);
    logic s319 ,c319; 
    fa fa319 (pp_5[0], tcomp5, s180, s319, c319);
    logic s320 ,c320; 
    fa fa320 (pp_2[7], pp_3[5], pp_4[3], s320, c320);
    logic s321 ,c321; 
    fa fa321 (pp_5[1], c180, s181, s321, c321);
    logic s322 ,c322; 
    fa fa322 (pp_5[2], pp_6[0], tcomp6, s322, c322);
    logic s323 ,c323; 
    fa fa323 (c181, s182, s183, s323, c323);
    logic s324 ,c324; 
    fa fa324 (pp_5[3], pp_6[1], c182, s324, c324);
    logic s325 ,c325; 
    fa fa325 (c183, s184, s185, s325, c325);
    logic s326 ,c326; 
    fa fa326 (tcomp7, c184, c185, s326, c326);
    logic s327 ,c327; 
    fa fa327 (s186, s187, s188, s327, c327);
    logic s328 ,c328; 
    fa fa328 (c186, c187, c188, s328, c328);
    logic s329 ,c329; 
    fa fa329 (s189, s190, s191, s329, c329);
    logic s330 ,c330; 
    fa fa330 (c189, c190, c191, s330, c330);
    logic s331 ,c331; 
    fa fa331 (s192, s193, s194, s331, c331);
    logic s332 ,c332; 
    fa fa332 (c192, c193, c194, s332, c332);
    logic s333 ,c333; 
    fa fa333 (s195, s196, s197, s333, c333);
    logic s334 ,c334; 
    fa fa334 (c195, c196, c197, s334, c334);
    logic s335 ,c335; 
    fa fa335 (s198, s199, s200, s335, c335);
    logic s336 ,c336; 
    fa fa336 (c198, c199, c200, s336, c336);
    logic s337 ,c337; 
    fa fa337 (s201, s202, s203, s337, c337);
    logic s338 ,c338; 
    fa fa338 (c201, c202, c203, s338, c338);
    logic s339 ,c339; 
    fa fa339 (s204, s205, s206, s339, c339);
    logic s340 ,c340; 
    fa fa340 (c204, c205, c206, s340, c340);
    logic s341 ,c341; 
    fa fa341 (s207, s208, s209, s341, c341);
    logic s342 ,c342; 
    fa fa342 (c207, c208, c209, s342, c342);
    logic s343 ,c343; 
    fa fa343 (s210, s211, s212, s343, c343);
    logic s344 ,c344; 
    fa fa344 (c210, c211, c212, s344, c344);
    logic s345 ,c345; 
    fa fa345 (s213, s214, s215, s345, c345);
    logic s346 ,c346; 
    fa fa346 (c213, c214, c215, s346, c346);
    logic s347 ,c347; 
    fa fa347 (s216, s217, s218, s347, c347);
    logic s348 ,c348; 
    fa fa348 (c216, c217, c218, s348, c348);
    logic s349 ,c349; 
    fa fa349 (s219, s220, s221, s349, c349);
    logic s350 ,c350; 
    fa fa350 (c219, c220, c221, s350, c350);
    logic s351 ,c351; 
    fa fa351 (s222, s223, s224, s351, c351);
    logic s352 ,c352; 
    fa fa352 (c222, c223, c224, s352, c352);
    logic s353 ,c353; 
    fa fa353 (s225, s226, s227, s353, c353);
    logic s354 ,c354; 
    fa fa354 (c225, c226, c227, s354, c354);
    logic s355 ,c355; 
    fa fa355 (s228, s229, s230, s355, c355);
    logic s356 ,c356; 
    fa fa356 (c228, c229, c230, s356, c356);
    logic s357 ,c357; 
    fa fa357 (s231, s232, s233, s357, c357);
    logic s358 ,c358; 
    fa fa358 (c231, c232, c233, s358, c358);
    logic s359 ,c359; 
    fa fa359 (s234, s235, s236, s359, c359);
    logic s360 ,c360; 
    fa fa360 (c234, c235, c236, s360, c360);
    logic s361 ,c361; 
    fa fa361 (s237, s238, s239, s361, c361);
    logic s362 ,c362; 
    fa fa362 (c237, c238, c239, s362, c362);
    logic s363 ,c363; 
    fa fa363 (s240, s241, s242, s363, c363);
    logic s364 ,c364; 
    fa fa364 (c240, c241, c242, s364, c364);
    logic s365 ,c365; 
    fa fa365 (s243, s244, s245, s365, c365);
    logic s366 ,c366; 
    fa fa366 (c243, c244, c245, s366, c366);
    logic s367 ,c367; 
    fa fa367 (s246, s247, s248, s367, c367);
    logic s368 ,c368; 
    fa fa368 (c246, c247, c248, s368, c368);
    logic s369 ,c369; 
    fa fa369 (s249, s250, s251, s369, c369);
    logic s370 ,c370; 
    fa fa370 (c249, c250, c251, s370, c370);
    logic s371 ,c371; 
    fa fa371 (s252, s253, s254, s371, c371);
    logic s372 ,c372; 
    fa fa372 (c252, c253, c254, s372, c372);
    logic s373 ,c373; 
    fa fa373 (s255, s256, s257, s373, c373);
    logic s374 ,c374; 
    fa fa374 (c255, c256, c257, s374, c374);
    logic s375 ,c375; 
    fa fa375 (s258, s259, s260, s375, c375);
    logic s376 ,c376; 
    fa fa376 (c258, c259, c260, s376, c376);
    logic s377 ,c377; 
    fa fa377 (s261, s262, s263, s377, c377);
    logic s378 ,c378; 
    fa fa378 (c261, c262, c263, s378, c378);
    logic s379 ,c379; 
    fa fa379 (s264, s265, s266, s379, c379);
    logic s380 ,c380; 
    fa fa380 (c264, c265, c266, s380, c380);
    logic s381 ,c381; 
    fa fa381 (s267, s268, s269, s381, c381);
    logic s382 ,c382; 
    fa fa382 (c267, c268, c269, s382, c382);
    logic s383 ,c383; 
    fa fa383 (s270, s271, s272, s383, c383);
    logic s384 ,c384; 
    fa fa384 (c270, c271, c272, s384, c384);
    logic s385 ,c385; 
    fa fa385 (s273, s274, s275, s385, c385);
    logic s386 ,c386; 
    fa fa386 (c273, c274, c275, s386, c386);
    logic s387 ,c387; 
    fa fa387 (s276, s277, s278, s387, c387);
    logic s388 ,c388; 
    fa fa388 (c276, c277, c278, s388, c388);
    logic s389 ,c389; 
    fa fa389 (s279, s280, s281, s389, c389);
    logic s390 ,c390; 
    fa fa390 (c279, c280, c281, s390, c390);
    logic s391 ,c391; 
    fa fa391 (s282, s283, s284, s391, c391);
    logic s392 ,c392; 
    fa fa392 (c282, c283, c284, s392, c392);
    logic s393 ,c393; 
    fa fa393 (s285, s286, s287, s393, c393);
    logic s394 ,c394; 
    fa fa394 (c285, c286, c287, s394, c394);
    logic s395 ,c395; 
    fa fa395 (s288, s289, s290, s395, c395);
    logic s396 ,c396; 
    fa fa396 (c288, c289, c290, s396, c396);
    logic s397 ,c397; 
    fa fa397 (s291, s292, s293, s397, c397);
    logic s398 ,c398; 
    fa fa398 (c291, c292, c293, s398, c398);
    logic s399 ,c399; 
    fa fa399 (s294, s295, s296, s399, c399);
    logic s400 ,c400; 
    fa fa400 (c294, c295, c296, s400, c400);
    logic s401 ,c401; 
    fa fa401 (s297, s298, s299, s401, c401);
    logic s402 ,c402; 
    fa fa402 (c297, c298, c299, s402, c402);
    logic s403 ,c403; 
    fa fa403 (s300, s301, s302, s403, c403);
    logic s404 ,c404; 
    fa fa404 (c300, c301, c302, s404, c404);
    logic s405 ,c405; 
    fa fa405 (s303, s304, s305, s405, c405);
    logic s406 ,c406; 
    fa fa406 (pp_16[22], c303, c304, s406, c406);
    logic s407 ,c407; 
    fa fa407 (c305, s306, s307, s407, c407);
    logic s408 ,c408; 
    fa fa408 (pp_15[25], pp_16[23], c306, s408, c408);
    logic s409 ,c409; 
    fa fa409 (c307, s308, s309, s409, c409);
    logic s410 ,c410; 
    fa fa410 (pp_14[28], pp_15[26], pp_16[24], s410, c410);
    logic s411 ,c411; 
    fa fa411 (c308, c309, s310, s411, c411);
    logic s412 ,c412; 
    fa fa412 (pp_13[31], pp_14[29], pp_15[27], s412, c412);
    logic s413 ,c413; 
    fa fa413 (pp_16[25], c310, s311, s413, c413);
    logic s414 ,c414; 
    fa fa414 (pp_12[34], pp_13[32], pp_14[30], s414, c414);
    logic s415 ,c415; 
    fa fa415 (pp_15[28], pp_16[26], c311, s415, c415);
    logic s416 ,c416; 
    fa fa416 (const1, pp_13[33], pp_14[31], s416, c416);
    logic s417 ,c417;
    ha ha417 (pp_15[29], pp_16[27], s417, c417);
    logic s418 ,c418; 
    fa fa418 (pp_13[34], pp_14[32], pp_15[30], s418, c418);
    logic s419 ,c419;
    ha ha419 (const1, pp_14[33], s419, c419);
    
    // Dadda Summation Stage 5
    logic s420 ,c420;
    ha ha420 (pp_0[4], pp_1[2], s420, c420);
    logic s421 ,c421;
    ha ha421 (pp_0[5], pp_1[3], s421, c421);
    logic s422 ,c422; 
    fa fa422 (pp_2[2], pp_3[0], tcomp3, s422, c422);
    logic s423 ,c423; 
    fa fa423 (pp_2[3], pp_3[1], c312, s423, c423);
    logic s424 ,c424; 
    fa fa424 (tcomp4, c313, s314, s424, c424);
    logic s425 ,c425; 
    fa fa425 (c314, c315, s316, s425, c425);
    logic s426 ,c426; 
    fa fa426 (c316, c317, s318, s426, c426);
    logic s427 ,c427; 
    fa fa427 (c318, c319, s320, s427, c427);
    logic s428 ,c428; 
    fa fa428 (c320, c321, s322, s428, c428);
    logic s429 ,c429; 
    fa fa429 (c322, c323, s324, s429, c429);
    logic s430 ,c430; 
    fa fa430 (c324, c325, s326, s430, c430);
    logic s431 ,c431; 
    fa fa431 (c326, c327, s328, s431, c431);
    logic s432 ,c432; 
    fa fa432 (c328, c329, s330, s432, c432);
    logic s433 ,c433; 
    fa fa433 (c330, c331, s332, s433, c433);
    logic s434 ,c434; 
    fa fa434 (c332, c333, s334, s434, c434);
    logic s435 ,c435; 
    fa fa435 (c334, c335, s336, s435, c435);
    logic s436 ,c436; 
    fa fa436 (c336, c337, s338, s436, c436);
    logic s437 ,c437; 
    fa fa437 (c338, c339, s340, s437, c437);
    logic s438 ,c438; 
    fa fa438 (c340, c341, s342, s438, c438);
    logic s439 ,c439; 
    fa fa439 (c342, c343, s344, s439, c439);
    logic s440 ,c440; 
    fa fa440 (c344, c345, s346, s440, c440);
    logic s441 ,c441; 
    fa fa441 (c346, c347, s348, s441, c441);
    logic s442 ,c442; 
    fa fa442 (c348, c349, s350, s442, c442);
    logic s443 ,c443; 
    fa fa443 (c350, c351, s352, s443, c443);
    logic s444 ,c444; 
    fa fa444 (c352, c353, s354, s444, c444);
    logic s445 ,c445; 
    fa fa445 (c354, c355, s356, s445, c445);
    logic s446 ,c446; 
    fa fa446 (c356, c357, s358, s446, c446);
    logic s447 ,c447; 
    fa fa447 (c358, c359, s360, s447, c447);
    logic s448 ,c448; 
    fa fa448 (c360, c361, s362, s448, c448);
    logic s449 ,c449; 
    fa fa449 (c362, c363, s364, s449, c449);
    logic s450 ,c450; 
    fa fa450 (c364, c365, s366, s450, c450);
    logic s451 ,c451; 
    fa fa451 (c366, c367, s368, s451, c451);
    logic s452 ,c452; 
    fa fa452 (c368, c369, s370, s452, c452);
    logic s453 ,c453; 
    fa fa453 (c370, c371, s372, s453, c453);
    logic s454 ,c454; 
    fa fa454 (c372, c373, s374, s454, c454);
    logic s455 ,c455; 
    fa fa455 (c374, c375, s376, s455, c455);
    logic s456 ,c456; 
    fa fa456 (c376, c377, s378, s456, c456);
    logic s457 ,c457; 
    fa fa457 (c378, c379, s380, s457, c457);
    logic s458 ,c458; 
    fa fa458 (c380, c381, s382, s458, c458);
    logic s459 ,c459; 
    fa fa459 (c382, c383, s384, s459, c459);
    logic s460 ,c460; 
    fa fa460 (c384, c385, s386, s460, c460);
    logic s461 ,c461; 
    fa fa461 (c386, c387, s388, s461, c461);
    logic s462 ,c462; 
    fa fa462 (c388, c389, s390, s462, c462);
    logic s463 ,c463; 
    fa fa463 (c390, c391, s392, s463, c463);
    logic s464 ,c464; 
    fa fa464 (c392, c393, s394, s464, c464);
    logic s465 ,c465; 
    fa fa465 (c394, c395, s396, s465, c465);
    logic s466 ,c466; 
    fa fa466 (c396, c397, s398, s466, c466);
    logic s467 ,c467; 
    fa fa467 (c398, c399, s400, s467, c467);
    logic s468 ,c468; 
    fa fa468 (c400, c401, s402, s468, c468);
    logic s469 ,c469; 
    fa fa469 (c402, c403, s404, s469, c469);
    logic s470 ,c470; 
    fa fa470 (c404, c405, s406, s470, c470);
    logic s471 ,c471; 
    fa fa471 (c406, c407, s408, s471, c471);
    logic s472 ,c472; 
    fa fa472 (c408, c409, s410, s472, c472);
    logic s473 ,c473; 
    fa fa473 (c410, c411, s412, s473, c473);
    logic s474 ,c474; 
    fa fa474 (c412, c413, s414, s474, c474);
    logic s475 ,c475; 
    fa fa475 (c414, c415, s416, s475, c475);
    logic s476 ,c476; 
    fa fa476 (pp_16[28], c416, c417, s476, c476);
    logic s477 ,c477; 
    fa fa477 (pp_15[31], pp_16[29], c418, s477, c477);
    logic s478 ,c478; 
    fa fa478 (pp_14[34], pp_15[32], pp_16[30], s478, c478);
    logic s479 ,c479;
    ha ha479 (const1, pp_15[33], s479, c479);
    
    // Dadda Summation Stage 6
    logic s480 ,c480;
    ha ha480 (pp_0[2], pp_1[0], s480, c480);
    logic s481 ,c481;
    ha ha481 (pp_0[3], pp_1[1], s481, c481);
    logic s482 ,c482; 
    fa fa482 (pp_2[0], tcomp2, s420, s482, c482);
    logic s483 ,c483; 
    fa fa483 (pp_2[1], c420, s421, s483, c483);
    logic s484 ,c484; 
    fa fa484 (s312, c421, s422, s484, c484);
    logic s485 ,c485; 
    fa fa485 (s313, c422, s423, s485, c485);
    logic s486 ,c486; 
    fa fa486 (s315, c423, s424, s486, c486);
    logic s487 ,c487; 
    fa fa487 (s317, c424, s425, s487, c487);
    logic s488 ,c488; 
    fa fa488 (s319, c425, s426, s488, c488);
    logic s489 ,c489; 
    fa fa489 (s321, c426, s427, s489, c489);
    logic s490 ,c490; 
    fa fa490 (s323, c427, s428, s490, c490);
    logic s491 ,c491; 
    fa fa491 (s325, c428, s429, s491, c491);
    logic s492 ,c492; 
    fa fa492 (s327, c429, s430, s492, c492);
    logic s493 ,c493; 
    fa fa493 (s329, c430, s431, s493, c493);
    logic s494 ,c494; 
    fa fa494 (s331, c431, s432, s494, c494);
    logic s495 ,c495; 
    fa fa495 (s333, c432, s433, s495, c495);
    logic s496 ,c496; 
    fa fa496 (s335, c433, s434, s496, c496);
    logic s497 ,c497; 
    fa fa497 (s337, c434, s435, s497, c497);
    logic s498 ,c498; 
    fa fa498 (s339, c435, s436, s498, c498);
    logic s499 ,c499; 
    fa fa499 (s341, c436, s437, s499, c499);
    logic s500 ,c500; 
    fa fa500 (s343, c437, s438, s500, c500);
    logic s501 ,c501; 
    fa fa501 (s345, c438, s439, s501, c501);
    logic s502 ,c502; 
    fa fa502 (s347, c439, s440, s502, c502);
    logic s503 ,c503; 
    fa fa503 (s349, c440, s441, s503, c503);
    logic s504 ,c504; 
    fa fa504 (s351, c441, s442, s504, c504);
    logic s505 ,c505; 
    fa fa505 (s353, c442, s443, s505, c505);
    logic s506 ,c506; 
    fa fa506 (s355, c443, s444, s506, c506);
    logic s507 ,c507; 
    fa fa507 (s357, c444, s445, s507, c507);
    logic s508 ,c508; 
    fa fa508 (s359, c445, s446, s508, c508);
    logic s509 ,c509; 
    fa fa509 (s361, c446, s447, s509, c509);
    logic s510 ,c510; 
    fa fa510 (s363, c447, s448, s510, c510);
    logic s511 ,c511; 
    fa fa511 (s365, c448, s449, s511, c511);
    logic s512 ,c512; 
    fa fa512 (s367, c449, s450, s512, c512);
    logic s513 ,c513; 
    fa fa513 (s369, c450, s451, s513, c513);
    logic s514 ,c514; 
    fa fa514 (s371, c451, s452, s514, c514);
    logic s515 ,c515; 
    fa fa515 (s373, c452, s453, s515, c515);
    logic s516 ,c516; 
    fa fa516 (s375, c453, s454, s516, c516);
    logic s517 ,c517; 
    fa fa517 (s377, c454, s455, s517, c517);
    logic s518 ,c518; 
    fa fa518 (s379, c455, s456, s518, c518);
    logic s519 ,c519; 
    fa fa519 (s381, c456, s457, s519, c519);
    logic s520 ,c520; 
    fa fa520 (s383, c457, s458, s520, c520);
    logic s521 ,c521; 
    fa fa521 (s385, c458, s459, s521, c521);
    logic s522 ,c522; 
    fa fa522 (s387, c459, s460, s522, c522);
    logic s523 ,c523; 
    fa fa523 (s389, c460, s461, s523, c523);
    logic s524 ,c524; 
    fa fa524 (s391, c461, s462, s524, c524);
    logic s525 ,c525; 
    fa fa525 (s393, c462, s463, s525, c525);
    logic s526 ,c526; 
    fa fa526 (s395, c463, s464, s526, c526);
    logic s527 ,c527; 
    fa fa527 (s397, c464, s465, s527, c527);
    logic s528 ,c528; 
    fa fa528 (s399, c465, s466, s528, c528);
    logic s529 ,c529; 
    fa fa529 (s401, c466, s467, s529, c529);
    logic s530 ,c530; 
    fa fa530 (s403, c467, s468, s530, c530);
    logic s531 ,c531; 
    fa fa531 (s405, c468, s469, s531, c531);
    logic s532 ,c532; 
    fa fa532 (s407, c469, s470, s532, c532);
    logic s533 ,c533; 
    fa fa533 (s409, c470, s471, s533, c533);
    logic s534 ,c534; 
    fa fa534 (s411, c471, s472, s534, c534);
    logic s535 ,c535; 
    fa fa535 (s413, c472, s473, s535, c535);
    logic s536 ,c536; 
    fa fa536 (s415, c473, s474, s536, c536);
    logic s537 ,c537; 
    fa fa537 (s417, c474, s475, s537, c537);
    logic s538 ,c538; 
    fa fa538 (s418, c475, s476, s538, c538);
    logic s539 ,c539; 
    fa fa539 (s419, c476, s477, s539, c539);
    logic s540 ,c540; 
    fa fa540 (c419, c477, s478, s540, c540);
    logic s541 ,c541; 
    fa fa541 (pp_16[31], c478, s479, s541, c541);
    logic s542 ,c542; 
    fa fa542 (pp_15[34], pp_16[32], c479, s542, c542);
    logic s543 ,c543;
    ha ha543 (const1, pp_16[33], s543, c543);
    
    
    assign result0[65:0] = {c542, c541, c540, c539, c538, c537, c536, c535, c534, c533, c532, c531, c530, c529, c528, c527, c526, c525, c524, c523, c522, c521, c520, c519, c518, c517, c516, c515, c514, c513, c512, c511, c510, c509, c508, c507, c506, c505, c504, c503, c502, c501, c500, c499, c498, c497, c496, c495, c494, c493, c492, c491, c490, c489, c488, c487, c486, c485, c484, c483, c482, c481, c480, tcomp1, pp_0[1], pp_0[0] };
    assign result1[65:0] = {s543, s542, s541, s540, s539, s538, s537, s536, s535, s534, s533, s532, s531, s530, s529, s528, s527, s526, s525, s524, s523, s522, s521, s520, s519, s518, s517, s516, s515, s514, s513, s512, s511, s510, s509, s508, s507, s506, s505, s504, s503, s502, s501, s500, s499, s498, s497, s496, s495, s494, s493, s492, s491, s490, s489, s488, s487, s486, s485, s484, s483, s482, s481, s480, 1'b0, tcomp0 };
endmodule




module FourMultMerger_Signed_64x64_plus_129_128(
        input logic [65:0]   m1_0,
        input logic [65:0]   m1_1,
        input logic [65:0]   m2_0,
        input logic [65:0]   m2_1,
        input logic [65:0]   m3_0,
        input logic [65:0]   m3_1,
        input logic [65:0]   m4_0,
        input logic [65:0]   m4_1,
        input logic [128:0]  IN3,
	output logic [127:0] result0,
        output logic [127:0] result1);
    wire logic const1;
    assign const_1 = 1'b1;
    
    // Dadda Summation Stage 1
    logic s0 ,c0;
    ha ha0 (m1_0[32], m1_1[32], s0, c0);
    logic s1 ,c1; 
    fa fa1 (m1_0[33], m1_1[33], m2_0[1], s1, c1);
    logic s2 ,c2; 
    fa fa2 (m1_0[34], m1_1[34], m2_0[2], s2, c2);
    logic s3 ,c3; 
    fa fa3 (m1_0[35], m1_1[35], m2_0[3], s3, c3);
    logic s4 ,c4; 
    fa fa4 (m1_0[36], m1_1[36], m2_0[4], s4, c4);
    logic s5 ,c5; 
    fa fa5 (m1_0[37], m1_1[37], m2_0[5], s5, c5);
    logic s6 ,c6; 
    fa fa6 (m1_0[38], m1_1[38], m2_0[6], s6, c6);
    logic s7 ,c7; 
    fa fa7 (m1_0[39], m1_1[39], m2_0[7], s7, c7);
    logic s8 ,c8; 
    fa fa8 (m1_0[40], m1_1[40], m2_0[8], s8, c8);
    logic s9 ,c9; 
    fa fa9 (m1_0[41], m1_1[41], m2_0[9], s9, c9);
    logic s10 ,c10; 
    fa fa10 (m1_0[42], m1_1[42], m2_0[10], s10, c10);
    logic s11 ,c11; 
    fa fa11 (m1_0[43], m1_1[43], m2_0[11], s11, c11);
    logic s12 ,c12; 
    fa fa12 (m1_0[44], m1_1[44], m2_0[12], s12, c12);
    logic s13 ,c13; 
    fa fa13 (m1_0[45], m1_1[45], m2_0[13], s13, c13);
    logic s14 ,c14; 
    fa fa14 (m1_0[46], m1_1[46], m2_0[14], s14, c14);
    logic s15 ,c15; 
    fa fa15 (m1_0[47], m1_1[47], m2_0[15], s15, c15);
    logic s16 ,c16; 
    fa fa16 (m1_0[48], m1_1[48], m2_0[16], s16, c16);
    logic s17 ,c17; 
    fa fa17 (m1_0[49], m1_1[49], m2_0[17], s17, c17);
    logic s18 ,c18; 
    fa fa18 (m1_0[50], m1_1[50], m2_0[18], s18, c18);
    logic s19 ,c19; 
    fa fa19 (m1_0[51], m1_1[51], m2_0[19], s19, c19);
    logic s20 ,c20; 
    fa fa20 (m1_0[52], m1_1[52], m2_0[20], s20, c20);
    logic s21 ,c21; 
    fa fa21 (m1_0[53], m1_1[53], m2_0[21], s21, c21);
    logic s22 ,c22; 
    fa fa22 (m1_0[54], m1_1[54], m2_0[22], s22, c22);
    logic s23 ,c23; 
    fa fa23 (m1_0[55], m1_1[55], m2_0[23], s23, c23);
    logic s24 ,c24; 
    fa fa24 (m1_0[56], m1_1[56], m2_0[24], s24, c24);
    logic s25 ,c25; 
    fa fa25 (m1_0[57], m1_1[57], m2_0[25], s25, c25);
    logic s26 ,c26; 
    fa fa26 (m1_0[58], m1_1[58], m2_0[26], s26, c26);
    logic s27 ,c27; 
    fa fa27 (m1_0[59], m1_1[59], m2_0[27], s27, c27);
    logic s28 ,c28; 
    fa fa28 (m1_0[60], m1_1[60], m2_0[28], s28, c28);
    logic s29 ,c29; 
    fa fa29 (m1_0[61], m1_1[61], m2_0[29], s29, c29);
    logic s30 ,c30; 
    fa fa30 (m1_0[62], m1_1[62], m2_0[30], s30, c30);
    logic s31 ,c31; 
    fa fa31 (m1_0[63], m1_1[63], m2_0[31], s31, c31);
    logic s32 ,c32; 
    fa fa32 (m1_0[64], m1_1[64], m2_0[32], s32, c32);
    logic s33 ,c33; 
    fa fa33 (m2_1[32], m3_0[32], m3_1[32], s33, c33);
    logic s34 ,c34; 
    fa fa34 (m1_0[65], m1_1[65], m2_0[33], s34, c34);
    logic s35 ,c35; 
    fa fa35 (m2_1[33], m3_0[33], m3_1[33], s35, c35);
    logic s36 ,c36;
    ha ha36 (m4_0[1], m4_1[1], s36, c36);
    logic s37 ,c37; 
    fa fa37 (m2_0[34], m2_1[34], m3_0[34], s37, c37);
    logic s38 ,c38; 
    fa fa38 (m3_1[34], m4_0[2], m4_1[2], s38, c38);
    logic s39 ,c39;
    ha ha39 (IN3[66], const_1, s39, c39);
    logic s40 ,c40; 
    fa fa40 (m2_0[35], m2_1[35], m3_0[35], s40, c40);
    logic s41 ,c41; 
    fa fa41 (m3_1[35], m4_0[3], m4_1[3], s41, c41);
    logic s42 ,c42;
    ha ha42 (IN3[67], const_1, s42, c42);
    logic s43 ,c43; 
    fa fa43 (m2_0[36], m2_1[36], m3_0[36], s43, c43);
    logic s44 ,c44; 
    fa fa44 (m3_1[36], m4_0[4], m4_1[4], s44, c44);
    logic s45 ,c45;
    ha ha45 (IN3[68], const_1, s45, c45);
    logic s46 ,c46; 
    fa fa46 (m2_0[37], m2_1[37], m3_0[37], s46, c46);
    logic s47 ,c47; 
    fa fa47 (m3_1[37], m4_0[5], m4_1[5], s47, c47);
    logic s48 ,c48;
    ha ha48 (IN3[69], const_1, s48, c48);
    logic s49 ,c49; 
    fa fa49 (m2_0[38], m2_1[38], m3_0[38], s49, c49);
    logic s50 ,c50; 
    fa fa50 (m3_1[38], m4_0[6], m4_1[6], s50, c50);
    logic s51 ,c51;
    ha ha51 (IN3[70], const_1, s51, c51);
    logic s52 ,c52; 
    fa fa52 (m2_0[39], m2_1[39], m3_0[39], s52, c52);
    logic s53 ,c53; 
    fa fa53 (m3_1[39], m4_0[7], m4_1[7], s53, c53);
    logic s54 ,c54;
    ha ha54 (IN3[71], const_1, s54, c54);
    logic s55 ,c55; 
    fa fa55 (m2_0[40], m2_1[40], m3_0[40], s55, c55);
    logic s56 ,c56; 
    fa fa56 (m3_1[40], m4_0[8], m4_1[8], s56, c56);
    logic s57 ,c57;
    ha ha57 (IN3[72], const_1, s57, c57);
    logic s58 ,c58; 
    fa fa58 (m2_0[41], m2_1[41], m3_0[41], s58, c58);
    logic s59 ,c59; 
    fa fa59 (m3_1[41], m4_0[9], m4_1[9], s59, c59);
    logic s60 ,c60;
    ha ha60 (IN3[73], const_1, s60, c60);
    logic s61 ,c61; 
    fa fa61 (m2_0[42], m2_1[42], m3_0[42], s61, c61);
    logic s62 ,c62; 
    fa fa62 (m3_1[42], m4_0[10], m4_1[10], s62, c62);
    logic s63 ,c63;
    ha ha63 (IN3[74], const_1, s63, c63);
    logic s64 ,c64; 
    fa fa64 (m2_0[43], m2_1[43], m3_0[43], s64, c64);
    logic s65 ,c65; 
    fa fa65 (m3_1[43], m4_0[11], m4_1[11], s65, c65);
    logic s66 ,c66;
    ha ha66 (IN3[75], const_1, s66, c66);
    logic s67 ,c67; 
    fa fa67 (m2_0[44], m2_1[44], m3_0[44], s67, c67);
    logic s68 ,c68; 
    fa fa68 (m3_1[44], m4_0[12], m4_1[12], s68, c68);
    logic s69 ,c69;
    ha ha69 (IN3[76], const_1, s69, c69);
    logic s70 ,c70; 
    fa fa70 (m2_0[45], m2_1[45], m3_0[45], s70, c70);
    logic s71 ,c71; 
    fa fa71 (m3_1[45], m4_0[13], m4_1[13], s71, c71);
    logic s72 ,c72;
    ha ha72 (IN3[77], const_1, s72, c72);
    logic s73 ,c73; 
    fa fa73 (m2_0[46], m2_1[46], m3_0[46], s73, c73);
    logic s74 ,c74; 
    fa fa74 (m3_1[46], m4_0[14], m4_1[14], s74, c74);
    logic s75 ,c75;
    ha ha75 (IN3[78], const_1, s75, c75);
    logic s76 ,c76; 
    fa fa76 (m2_0[47], m2_1[47], m3_0[47], s76, c76);
    logic s77 ,c77; 
    fa fa77 (m3_1[47], m4_0[15], m4_1[15], s77, c77);
    logic s78 ,c78;
    ha ha78 (IN3[79], const_1, s78, c78);
    logic s79 ,c79; 
    fa fa79 (m2_0[48], m2_1[48], m3_0[48], s79, c79);
    logic s80 ,c80; 
    fa fa80 (m3_1[48], m4_0[16], m4_1[16], s80, c80);
    logic s81 ,c81;
    ha ha81 (IN3[80], const_1, s81, c81);
    logic s82 ,c82; 
    fa fa82 (m2_0[49], m2_1[49], m3_0[49], s82, c82);
    logic s83 ,c83; 
    fa fa83 (m3_1[49], m4_0[17], m4_1[17], s83, c83);
    logic s84 ,c84;
    ha ha84 (IN3[81], const_1, s84, c84);
    logic s85 ,c85; 
    fa fa85 (m2_0[50], m2_1[50], m3_0[50], s85, c85);
    logic s86 ,c86; 
    fa fa86 (m3_1[50], m4_0[18], m4_1[18], s86, c86);
    logic s87 ,c87;
    ha ha87 (IN3[82], const_1, s87, c87);
    logic s88 ,c88; 
    fa fa88 (m2_0[51], m2_1[51], m3_0[51], s88, c88);
    logic s89 ,c89; 
    fa fa89 (m3_1[51], m4_0[19], m4_1[19], s89, c89);
    logic s90 ,c90;
    ha ha90 (IN3[83], const_1, s90, c90);
    logic s91 ,c91; 
    fa fa91 (m2_0[52], m2_1[52], m3_0[52], s91, c91);
    logic s92 ,c92; 
    fa fa92 (m3_1[52], m4_0[20], m4_1[20], s92, c92);
    logic s93 ,c93;
    ha ha93 (IN3[84], const_1, s93, c93);
    logic s94 ,c94; 
    fa fa94 (m2_0[53], m2_1[53], m3_0[53], s94, c94);
    logic s95 ,c95; 
    fa fa95 (m3_1[53], m4_0[21], m4_1[21], s95, c95);
    logic s96 ,c96;
    ha ha96 (IN3[85], const_1, s96, c96);
    logic s97 ,c97; 
    fa fa97 (m2_0[54], m2_1[54], m3_0[54], s97, c97);
    logic s98 ,c98; 
    fa fa98 (m3_1[54], m4_0[22], m4_1[22], s98, c98);
    logic s99 ,c99;
    ha ha99 (IN3[86], const_1, s99, c99);
    logic s100 ,c100; 
    fa fa100 (m2_0[55], m2_1[55], m3_0[55], s100, c100);
    logic s101 ,c101; 
    fa fa101 (m3_1[55], m4_0[23], m4_1[23], s101, c101);
    logic s102 ,c102;
    ha ha102 (IN3[87], const_1, s102, c102);
    logic s103 ,c103; 
    fa fa103 (m2_0[56], m2_1[56], m3_0[56], s103, c103);
    logic s104 ,c104; 
    fa fa104 (m3_1[56], m4_0[24], m4_1[24], s104, c104);
    logic s105 ,c105;
    ha ha105 (IN3[88], const_1, s105, c105);
    logic s106 ,c106; 
    fa fa106 (m2_0[57], m2_1[57], m3_0[57], s106, c106);
    logic s107 ,c107; 
    fa fa107 (m3_1[57], m4_0[25], m4_1[25], s107, c107);
    logic s108 ,c108;
    ha ha108 (IN3[89], const_1, s108, c108);
    logic s109 ,c109; 
    fa fa109 (m2_0[58], m2_1[58], m3_0[58], s109, c109);
    logic s110 ,c110; 
    fa fa110 (m3_1[58], m4_0[26], m4_1[26], s110, c110);
    logic s111 ,c111;
    ha ha111 (IN3[90], const_1, s111, c111);
    logic s112 ,c112; 
    fa fa112 (m2_0[59], m2_1[59], m3_0[59], s112, c112);
    logic s113 ,c113; 
    fa fa113 (m3_1[59], m4_0[27], m4_1[27], s113, c113);
    logic s114 ,c114;
    ha ha114 (IN3[91], const_1, s114, c114);
    logic s115 ,c115; 
    fa fa115 (m2_0[60], m2_1[60], m3_0[60], s115, c115);
    logic s116 ,c116; 
    fa fa116 (m3_1[60], m4_0[28], m4_1[28], s116, c116);
    logic s117 ,c117;
    ha ha117 (IN3[92], const_1, s117, c117);
    logic s118 ,c118; 
    fa fa118 (m2_0[61], m2_1[61], m3_0[61], s118, c118);
    logic s119 ,c119; 
    fa fa119 (m3_1[61], m4_0[29], m4_1[29], s119, c119);
    logic s120 ,c120;
    ha ha120 (IN3[93], const_1, s120, c120);
    logic s121 ,c121; 
    fa fa121 (m2_0[62], m2_1[62], m3_0[62], s121, c121);
    logic s122 ,c122; 
    fa fa122 (m3_1[62], m4_0[30], m4_1[30], s122, c122);
    logic s123 ,c123;
    ha ha123 (IN3[94], const_1, s123, c123);
    logic s124 ,c124; 
    fa fa124 (m2_0[63], m2_1[63], m3_0[63], s124, c124);
    logic s125 ,c125; 
    fa fa125 (m3_1[63], m4_0[31], m4_1[31], s125, c125);
    logic s126 ,c126;
    ha ha126 (IN3[95], const_1, s126, c126);
    logic s127 ,c127; 
    fa fa127 (m2_0[64], m2_1[64], m3_0[64], s127, c127);
    logic s128 ,c128; 
    fa fa128 (m3_1[64], m4_0[32], m4_1[32], s128, c128);
    logic s129 ,c129;
    ha ha129 (IN3[96], const_1, s129, c129);
    logic s130 ,c130; 
    fa fa130 (m2_0[65], m2_1[65], m3_0[65], s130, c130);
    logic s131 ,c131; 
    fa fa131 (m3_1[65], m4_0[33], m4_1[33], s131, c131);
    logic s132 ,c132;
    ha ha132 (IN3[97], const_1, s132, c132);
    logic s133 ,c133;
    ha ha133 (m4_0[34], m4_1[34], s133, c133);
    
    // Dadda Summation Stage 2
    logic s134 ,c134; 
    fa fa134 (m2_0[0], m2_1[0], m3_0[0], s134, c134);
    logic s135 ,c135; 
    fa fa135 (m2_1[1], m3_0[1], m3_1[1], s135, c135);
    logic s136 ,c136;
    ha ha136 (IN3[33], c0, s136, c136);
    logic s137 ,c137; 
    fa fa137 (m2_1[2], m3_0[2], m3_1[2], s137, c137);
    logic s138 ,c138; 
    fa fa138 (IN3[34], c1, s2, s138, c138);
    logic s139 ,c139; 
    fa fa139 (m2_1[3], m3_0[3], m3_1[3], s139, c139);
    logic s140 ,c140; 
    fa fa140 (IN3[35], c2, s3, s140, c140);
    logic s141 ,c141; 
    fa fa141 (m2_1[4], m3_0[4], m3_1[4], s141, c141);
    logic s142 ,c142; 
    fa fa142 (IN3[36], c3, s4, s142, c142);
    logic s143 ,c143; 
    fa fa143 (m2_1[5], m3_0[5], m3_1[5], s143, c143);
    logic s144 ,c144; 
    fa fa144 (IN3[37], c4, s5, s144, c144);
    logic s145 ,c145; 
    fa fa145 (m2_1[6], m3_0[6], m3_1[6], s145, c145);
    logic s146 ,c146; 
    fa fa146 (IN3[38], c5, s6, s146, c146);
    logic s147 ,c147; 
    fa fa147 (m2_1[7], m3_0[7], m3_1[7], s147, c147);
    logic s148 ,c148; 
    fa fa148 (IN3[39], c6, s7, s148, c148);
    logic s149 ,c149; 
    fa fa149 (m2_1[8], m3_0[8], m3_1[8], s149, c149);
    logic s150 ,c150; 
    fa fa150 (IN3[40], c7, s8, s150, c150);
    logic s151 ,c151; 
    fa fa151 (m2_1[9], m3_0[9], m3_1[9], s151, c151);
    logic s152 ,c152; 
    fa fa152 (IN3[41], c8, s9, s152, c152);
    logic s153 ,c153; 
    fa fa153 (m2_1[10], m3_0[10], m3_1[10], s153, c153);
    logic s154 ,c154; 
    fa fa154 (IN3[42], c9, s10, s154, c154);
    logic s155 ,c155; 
    fa fa155 (m2_1[11], m3_0[11], m3_1[11], s155, c155);
    logic s156 ,c156; 
    fa fa156 (IN3[43], c10, s11, s156, c156);
    logic s157 ,c157; 
    fa fa157 (m2_1[12], m3_0[12], m3_1[12], s157, c157);
    logic s158 ,c158; 
    fa fa158 (IN3[44], c11, s12, s158, c158);
    logic s159 ,c159; 
    fa fa159 (m2_1[13], m3_0[13], m3_1[13], s159, c159);
    logic s160 ,c160; 
    fa fa160 (IN3[45], c12, s13, s160, c160);
    logic s161 ,c161; 
    fa fa161 (m2_1[14], m3_0[14], m3_1[14], s161, c161);
    logic s162 ,c162; 
    fa fa162 (IN3[46], c13, s14, s162, c162);
    logic s163 ,c163; 
    fa fa163 (m2_1[15], m3_0[15], m3_1[15], s163, c163);
    logic s164 ,c164; 
    fa fa164 (IN3[47], c14, s15, s164, c164);
    logic s165 ,c165; 
    fa fa165 (m2_1[16], m3_0[16], m3_1[16], s165, c165);
    logic s166 ,c166; 
    fa fa166 (IN3[48], c15, s16, s166, c166);
    logic s167 ,c167; 
    fa fa167 (m2_1[17], m3_0[17], m3_1[17], s167, c167);
    logic s168 ,c168; 
    fa fa168 (IN3[49], c16, s17, s168, c168);
    logic s169 ,c169; 
    fa fa169 (m2_1[18], m3_0[18], m3_1[18], s169, c169);
    logic s170 ,c170; 
    fa fa170 (IN3[50], c17, s18, s170, c170);
    logic s171 ,c171; 
    fa fa171 (m2_1[19], m3_0[19], m3_1[19], s171, c171);
    logic s172 ,c172; 
    fa fa172 (IN3[51], c18, s19, s172, c172);
    logic s173 ,c173; 
    fa fa173 (m2_1[20], m3_0[20], m3_1[20], s173, c173);
    logic s174 ,c174; 
    fa fa174 (IN3[52], c19, s20, s174, c174);
    logic s175 ,c175; 
    fa fa175 (m2_1[21], m3_0[21], m3_1[21], s175, c175);
    logic s176 ,c176; 
    fa fa176 (IN3[53], c20, s21, s176, c176);
    logic s177 ,c177; 
    fa fa177 (m2_1[22], m3_0[22], m3_1[22], s177, c177);
    logic s178 ,c178; 
    fa fa178 (IN3[54], c21, s22, s178, c178);
    logic s179 ,c179; 
    fa fa179 (m2_1[23], m3_0[23], m3_1[23], s179, c179);
    logic s180 ,c180; 
    fa fa180 (IN3[55], c22, s23, s180, c180);
    logic s181 ,c181; 
    fa fa181 (m2_1[24], m3_0[24], m3_1[24], s181, c181);
    logic s182 ,c182; 
    fa fa182 (IN3[56], c23, s24, s182, c182);
    logic s183 ,c183; 
    fa fa183 (m2_1[25], m3_0[25], m3_1[25], s183, c183);
    logic s184 ,c184; 
    fa fa184 (IN3[57], c24, s25, s184, c184);
    logic s185 ,c185; 
    fa fa185 (m2_1[26], m3_0[26], m3_1[26], s185, c185);
    logic s186 ,c186; 
    fa fa186 (IN3[58], c25, s26, s186, c186);
    logic s187 ,c187; 
    fa fa187 (m2_1[27], m3_0[27], m3_1[27], s187, c187);
    logic s188 ,c188; 
    fa fa188 (IN3[59], c26, s27, s188, c188);
    logic s189 ,c189; 
    fa fa189 (m2_1[28], m3_0[28], m3_1[28], s189, c189);
    logic s190 ,c190; 
    fa fa190 (IN3[60], c27, s28, s190, c190);
    logic s191 ,c191; 
    fa fa191 (m2_1[29], m3_0[29], m3_1[29], s191, c191);
    logic s192 ,c192; 
    fa fa192 (IN3[61], c28, s29, s192, c192);
    logic s193 ,c193; 
    fa fa193 (m2_1[30], m3_0[30], m3_1[30], s193, c193);
    logic s194 ,c194; 
    fa fa194 (IN3[62], c29, s30, s194, c194);
    logic s195 ,c195; 
    fa fa195 (m2_1[31], m3_0[31], m3_1[31], s195, c195);
    logic s196 ,c196; 
    fa fa196 (IN3[63], c30, s31, s196, c196);
    logic s197 ,c197; 
    fa fa197 (m4_0[0], m4_1[0], IN3[64], s197, c197);
    logic s198 ,c198; 
    fa fa198 (c31, s32, s33, s198, c198);
    logic s199 ,c199; 
    fa fa199 (IN3[65], c32, c33, s199, c199);
    logic s200 ,c200; 
    fa fa200 (s34, s35, s36, s200, c200);
    logic s201 ,c201; 
    fa fa201 (c34, c35, c36, s201, c201);
    logic s202 ,c202; 
    fa fa202 (s37, s38, s39, s202, c202);
    logic s203 ,c203; 
    fa fa203 (c37, c38, c39, s203, c203);
    logic s204 ,c204; 
    fa fa204 (s40, s41, s42, s204, c204);
    logic s205 ,c205; 
    fa fa205 (c40, c41, c42, s205, c205);
    logic s206 ,c206; 
    fa fa206 (s43, s44, s45, s206, c206);
    logic s207 ,c207; 
    fa fa207 (c43, c44, c45, s207, c207);
    logic s208 ,c208; 
    fa fa208 (s46, s47, s48, s208, c208);
    logic s209 ,c209; 
    fa fa209 (c46, c47, c48, s209, c209);
    logic s210 ,c210; 
    fa fa210 (s49, s50, s51, s210, c210);
    logic s211 ,c211; 
    fa fa211 (c49, c50, c51, s211, c211);
    logic s212 ,c212; 
    fa fa212 (s52, s53, s54, s212, c212);
    logic s213 ,c213; 
    fa fa213 (c52, c53, c54, s213, c213);
    logic s214 ,c214; 
    fa fa214 (s55, s56, s57, s214, c214);
    logic s215 ,c215; 
    fa fa215 (c55, c56, c57, s215, c215);
    logic s216 ,c216; 
    fa fa216 (s58, s59, s60, s216, c216);
    logic s217 ,c217; 
    fa fa217 (c58, c59, c60, s217, c217);
    logic s218 ,c218; 
    fa fa218 (s61, s62, s63, s218, c218);
    logic s219 ,c219; 
    fa fa219 (c61, c62, c63, s219, c219);
    logic s220 ,c220; 
    fa fa220 (s64, s65, s66, s220, c220);
    logic s221 ,c221; 
    fa fa221 (c64, c65, c66, s221, c221);
    logic s222 ,c222; 
    fa fa222 (s67, s68, s69, s222, c222);
    logic s223 ,c223; 
    fa fa223 (c67, c68, c69, s223, c223);
    logic s224 ,c224; 
    fa fa224 (s70, s71, s72, s224, c224);
    logic s225 ,c225; 
    fa fa225 (c70, c71, c72, s225, c225);
    logic s226 ,c226; 
    fa fa226 (s73, s74, s75, s226, c226);
    logic s227 ,c227; 
    fa fa227 (c73, c74, c75, s227, c227);
    logic s228 ,c228; 
    fa fa228 (s76, s77, s78, s228, c228);
    logic s229 ,c229; 
    fa fa229 (c76, c77, c78, s229, c229);
    logic s230 ,c230; 
    fa fa230 (s79, s80, s81, s230, c230);
    logic s231 ,c231; 
    fa fa231 (c79, c80, c81, s231, c231);
    logic s232 ,c232; 
    fa fa232 (s82, s83, s84, s232, c232);
    logic s233 ,c233; 
    fa fa233 (c82, c83, c84, s233, c233);
    logic s234 ,c234; 
    fa fa234 (s85, s86, s87, s234, c234);
    logic s235 ,c235; 
    fa fa235 (c85, c86, c87, s235, c235);
    logic s236 ,c236; 
    fa fa236 (s88, s89, s90, s236, c236);
    logic s237 ,c237; 
    fa fa237 (c88, c89, c90, s237, c237);
    logic s238 ,c238; 
    fa fa238 (s91, s92, s93, s238, c238);
    logic s239 ,c239; 
    fa fa239 (c91, c92, c93, s239, c239);
    logic s240 ,c240; 
    fa fa240 (s94, s95, s96, s240, c240);
    logic s241 ,c241; 
    fa fa241 (c94, c95, c96, s241, c241);
    logic s242 ,c242; 
    fa fa242 (s97, s98, s99, s242, c242);
    logic s243 ,c243; 
    fa fa243 (c97, c98, c99, s243, c243);
    logic s244 ,c244; 
    fa fa244 (s100, s101, s102, s244, c244);
    logic s245 ,c245; 
    fa fa245 (c100, c101, c102, s245, c245);
    logic s246 ,c246; 
    fa fa246 (s103, s104, s105, s246, c246);
    logic s247 ,c247; 
    fa fa247 (c103, c104, c105, s247, c247);
    logic s248 ,c248; 
    fa fa248 (s106, s107, s108, s248, c248);
    logic s249 ,c249; 
    fa fa249 (c106, c107, c108, s249, c249);
    logic s250 ,c250; 
    fa fa250 (s109, s110, s111, s250, c250);
    logic s251 ,c251; 
    fa fa251 (c109, c110, c111, s251, c251);
    logic s252 ,c252; 
    fa fa252 (s112, s113, s114, s252, c252);
    logic s253 ,c253; 
    fa fa253 (c112, c113, c114, s253, c253);
    logic s254 ,c254; 
    fa fa254 (s115, s116, s117, s254, c254);
    logic s255 ,c255; 
    fa fa255 (c115, c116, c117, s255, c255);
    logic s256 ,c256; 
    fa fa256 (s118, s119, s120, s256, c256);
    logic s257 ,c257; 
    fa fa257 (c118, c119, c120, s257, c257);
    logic s258 ,c258; 
    fa fa258 (s121, s122, s123, s258, c258);
    logic s259 ,c259; 
    fa fa259 (c121, c122, c123, s259, c259);
    logic s260 ,c260; 
    fa fa260 (s124, s125, s126, s260, c260);
    logic s261 ,c261; 
    fa fa261 (c124, c125, c126, s261, c261);
    logic s262 ,c262; 
    fa fa262 (s127, s128, s129, s262, c262);
    logic s263 ,c263; 
    fa fa263 (c127, c128, c129, s263, c263);
    logic s264 ,c264; 
    fa fa264 (s130, s131, s132, s264, c264);
    logic s265 ,c265; 
    fa fa265 (IN3[98], const_1, c130, s265, c265);
    logic s266 ,c266; 
    fa fa266 (c131, c132, s133, s266, c266);
    logic s267 ,c267; 
    fa fa267 (m4_0[35], m4_1[35], IN3[99], s267, c267);
    logic s268 ,c268;
    ha ha268 (m4_0[36], m4_1[36], s268, c268);
    logic s269 ,c269;
    ha ha269 (m4_0[37], m4_1[37], s269, c269);
    logic s270 ,c270;
    ha ha270 (m4_0[38], m4_1[38], s270, c270);
    logic s271 ,c271;
    ha ha271 (m4_0[39], m4_1[39], s271, c271);
    logic s272 ,c272;
    ha ha272 (m4_0[40], m4_1[40], s272, c272);
    logic s273 ,c273;
    ha ha273 (m4_0[41], m4_1[41], s273, c273);
    logic s274 ,c274;
    ha ha274 (m4_0[42], m4_1[42], s274, c274);
    logic s275 ,c275;
    ha ha275 (m4_0[43], m4_1[43], s275, c275);
    logic s276 ,c276;
    ha ha276 (m4_0[44], m4_1[44], s276, c276);
    logic s277 ,c277;
    ha ha277 (m4_0[45], m4_1[45], s277, c277);
    logic s278 ,c278;
    ha ha278 (m4_0[46], m4_1[46], s278, c278);
    logic s279 ,c279;
    ha ha279 (m4_0[47], m4_1[47], s279, c279);
    logic s280 ,c280;
    ha ha280 (m4_0[48], m4_1[48], s280, c280);
    logic s281 ,c281;
    ha ha281 (m4_0[49], m4_1[49], s281, c281);
    logic s282 ,c282;
    ha ha282 (m4_0[50], m4_1[50], s282, c282);
    logic s283 ,c283;
    ha ha283 (m4_0[51], m4_1[51], s283, c283);
    logic s284 ,c284;
    ha ha284 (m4_0[52], m4_1[52], s284, c284);
    logic s285 ,c285;
    ha ha285 (m4_0[53], m4_1[53], s285, c285);
    logic s286 ,c286;
    ha ha286 (m4_0[54], m4_1[54], s286, c286);
    logic s287 ,c287;
    ha ha287 (m4_0[55], m4_1[55], s287, c287);
    logic s288 ,c288;
    ha ha288 (m4_0[56], m4_1[56], s288, c288);
    logic s289 ,c289;
    ha ha289 (m4_0[57], m4_1[57], s289, c289);
    logic s290 ,c290;
    ha ha290 (m4_0[58], m4_1[58], s290, c290);
    logic s291 ,c291;
    ha ha291 (m4_0[59], m4_1[59], s291, c291);
    logic s292 ,c292;
    ha ha292 (m4_0[60], m4_1[60], s292, c292);
    logic s293 ,c293;
    ha ha293 (m4_0[61], m4_1[61], s293, c293);
    logic s294 ,c294;
    ha ha294 (m4_0[62], m4_1[62], s294, c294);
    logic s295 ,c295;
    ha ha295 (m4_0[63], m4_1[63], s295, c295);
    
    // Dadda Summation Stage 3
    logic s296 ,c296;
    ha ha296 (m3_1[0], IN3[32], s296, c296);
    logic s297 ,c297; 
    fa fa297 (s1, c134, s135, s297, c297);
    logic s298 ,c298; 
    fa fa298 (c135, c136, s137, s298, c298);
    logic s299 ,c299; 
    fa fa299 (c137, c138, s139, s299, c299);
    logic s300 ,c300; 
    fa fa300 (c139, c140, s141, s300, c300);
    logic s301 ,c301; 
    fa fa301 (c141, c142, s143, s301, c301);
    logic s302 ,c302; 
    fa fa302 (c143, c144, s145, s302, c302);
    logic s303 ,c303; 
    fa fa303 (c145, c146, s147, s303, c303);
    logic s304 ,c304; 
    fa fa304 (c147, c148, s149, s304, c304);
    logic s305 ,c305; 
    fa fa305 (c149, c150, s151, s305, c305);
    logic s306 ,c306; 
    fa fa306 (c151, c152, s153, s306, c306);
    logic s307 ,c307; 
    fa fa307 (c153, c154, s155, s307, c307);
    logic s308 ,c308; 
    fa fa308 (c155, c156, s157, s308, c308);
    logic s309 ,c309; 
    fa fa309 (c157, c158, s159, s309, c309);
    logic s310 ,c310; 
    fa fa310 (c159, c160, s161, s310, c310);
    logic s311 ,c311; 
    fa fa311 (c161, c162, s163, s311, c311);
    logic s312 ,c312; 
    fa fa312 (c163, c164, s165, s312, c312);
    logic s313 ,c313; 
    fa fa313 (c165, c166, s167, s313, c313);
    logic s314 ,c314; 
    fa fa314 (c167, c168, s169, s314, c314);
    logic s315 ,c315; 
    fa fa315 (c169, c170, s171, s315, c315);
    logic s316 ,c316; 
    fa fa316 (c171, c172, s173, s316, c316);
    logic s317 ,c317; 
    fa fa317 (c173, c174, s175, s317, c317);
    logic s318 ,c318; 
    fa fa318 (c175, c176, s177, s318, c318);
    logic s319 ,c319; 
    fa fa319 (c177, c178, s179, s319, c319);
    logic s320 ,c320; 
    fa fa320 (c179, c180, s181, s320, c320);
    logic s321 ,c321; 
    fa fa321 (c181, c182, s183, s321, c321);
    logic s322 ,c322; 
    fa fa322 (c183, c184, s185, s322, c322);
    logic s323 ,c323; 
    fa fa323 (c185, c186, s187, s323, c323);
    logic s324 ,c324; 
    fa fa324 (c187, c188, s189, s324, c324);
    logic s325 ,c325; 
    fa fa325 (c189, c190, s191, s325, c325);
    logic s326 ,c326; 
    fa fa326 (c191, c192, s193, s326, c326);
    logic s327 ,c327; 
    fa fa327 (c193, c194, s195, s327, c327);
    logic s328 ,c328; 
    fa fa328 (c195, c196, s197, s328, c328);
    logic s329 ,c329; 
    fa fa329 (c197, c198, s199, s329, c329);
    logic s330 ,c330; 
    fa fa330 (c199, c200, s201, s330, c330);
    logic s331 ,c331; 
    fa fa331 (c201, c202, s203, s331, c331);
    logic s332 ,c332; 
    fa fa332 (c203, c204, s205, s332, c332);
    logic s333 ,c333; 
    fa fa333 (c205, c206, s207, s333, c333);
    logic s334 ,c334; 
    fa fa334 (c207, c208, s209, s334, c334);
    logic s335 ,c335; 
    fa fa335 (c209, c210, s211, s335, c335);
    logic s336 ,c336; 
    fa fa336 (c211, c212, s213, s336, c336);
    logic s337 ,c337; 
    fa fa337 (c213, c214, s215, s337, c337);
    logic s338 ,c338; 
    fa fa338 (c215, c216, s217, s338, c338);
    logic s339 ,c339; 
    fa fa339 (c217, c218, s219, s339, c339);
    logic s340 ,c340; 
    fa fa340 (c219, c220, s221, s340, c340);
    logic s341 ,c341; 
    fa fa341 (c221, c222, s223, s341, c341);
    logic s342 ,c342; 
    fa fa342 (c223, c224, s225, s342, c342);
    logic s343 ,c343; 
    fa fa343 (c225, c226, s227, s343, c343);
    logic s344 ,c344; 
    fa fa344 (c227, c228, s229, s344, c344);
    logic s345 ,c345; 
    fa fa345 (c229, c230, s231, s345, c345);
    logic s346 ,c346; 
    fa fa346 (c231, c232, s233, s346, c346);
    logic s347 ,c347; 
    fa fa347 (c233, c234, s235, s347, c347);
    logic s348 ,c348; 
    fa fa348 (c235, c236, s237, s348, c348);
    logic s349 ,c349; 
    fa fa349 (c237, c238, s239, s349, c349);
    logic s350 ,c350; 
    fa fa350 (c239, c240, s241, s350, c350);
    logic s351 ,c351; 
    fa fa351 (c241, c242, s243, s351, c351);
    logic s352 ,c352; 
    fa fa352 (c243, c244, s245, s352, c352);
    logic s353 ,c353; 
    fa fa353 (c245, c246, s247, s353, c353);
    logic s354 ,c354; 
    fa fa354 (c247, c248, s249, s354, c354);
    logic s355 ,c355; 
    fa fa355 (c249, c250, s251, s355, c355);
    logic s356 ,c356; 
    fa fa356 (c251, c252, s253, s356, c356);
    logic s357 ,c357; 
    fa fa357 (c253, c254, s255, s357, c357);
    logic s358 ,c358; 
    fa fa358 (c255, c256, s257, s358, c358);
    logic s359 ,c359; 
    fa fa359 (c257, c258, s259, s359, c359);
    logic s360 ,c360; 
    fa fa360 (c259, c260, s261, s360, c360);
    logic s361 ,c361; 
    fa fa361 (c261, c262, s263, s361, c361);
    logic s362 ,c362; 
    fa fa362 (c263, c264, s265, s362, c362);
    logic s363 ,c363; 
    fa fa363 (c133, c265, c266, s363, c363);
    logic s364 ,c364; 
    fa fa364 (IN3[100], const_1, c267, s364, c364);
    logic s365 ,c365; 
    fa fa365 (IN3[101], const_1, c268, s365, c365);
    logic s366 ,c366; 
    fa fa366 (IN3[102], const_1, c269, s366, c366);
    logic s367 ,c367; 
    fa fa367 (IN3[103], const_1, c270, s367, c367);
    logic s368 ,c368; 
    fa fa368 (IN3[104], const_1, c271, s368, c368);
    logic s369 ,c369; 
    fa fa369 (IN3[105], const_1, c272, s369, c369);
    logic s370 ,c370; 
    fa fa370 (IN3[106], const_1, c273, s370, c370);
    logic s371 ,c371; 
    fa fa371 (IN3[107], const_1, c274, s371, c371);
    logic s372 ,c372; 
    fa fa372 (IN3[108], const_1, c275, s372, c372);
    logic s373 ,c373; 
    fa fa373 (IN3[109], const_1, c276, s373, c373);
    logic s374 ,c374; 
    fa fa374 (IN3[110], const_1, c277, s374, c374);
    logic s375 ,c375; 
    fa fa375 (IN3[111], const_1, c278, s375, c375);
    logic s376 ,c376; 
    fa fa376 (IN3[112], const_1, c279, s376, c376);
    logic s377 ,c377; 
    fa fa377 (IN3[113], const_1, c280, s377, c377);
    logic s378 ,c378; 
    fa fa378 (IN3[114], const_1, c281, s378, c378);
    logic s379 ,c379; 
    fa fa379 (IN3[115], const_1, c282, s379, c379);
    logic s380 ,c380; 
    fa fa380 (IN3[116], const_1, c283, s380, c380);
    logic s381 ,c381; 
    fa fa381 (IN3[117], const_1, c284, s381, c381);
    logic s382 ,c382; 
    fa fa382 (IN3[118], const_1, c285, s382, c382);
    logic s383 ,c383; 
    fa fa383 (IN3[119], const_1, c286, s383, c383);
    logic s384 ,c384; 
    fa fa384 (IN3[120], const_1, c287, s384, c384);
    logic s385 ,c385; 
    fa fa385 (IN3[121], const_1, c288, s385, c385);
    logic s386 ,c386; 
    fa fa386 (IN3[122], const_1, c289, s386, c386);
    logic s387 ,c387; 
    fa fa387 (IN3[123], const_1, c290, s387, c387);
    logic s388 ,c388; 
    fa fa388 (IN3[124], const_1, c291, s388, c388);
    logic s389 ,c389; 
    fa fa389 (IN3[125], const_1, c292, s389, c389);
    logic s390 ,c390; 
    fa fa390 (IN3[126], const_1, c293, s390, c390);
    logic s391 ,c391; 
    fa fa391 (IN3[127], const_1, c294, s391, c391);
    
    // Dadda Summation Stage 4
    logic s392 ,c392;
    ha ha392 (m1_0[0], m1_1[0], s392, c392);
    logic s393 ,c393; 
    fa fa393 (m1_0[1], m1_1[1], IN3[1], s393, c393);
    logic s394 ,c394; 
    fa fa394 (m1_0[2], m1_1[2], IN3[2], s394, c394);
    logic s395 ,c395; 
    fa fa395 (m1_0[3], m1_1[3], IN3[3], s395, c395);
    logic s396 ,c396; 
    fa fa396 (m1_0[4], m1_1[4], IN3[4], s396, c396);
    logic s397 ,c397; 
    fa fa397 (m1_0[5], m1_1[5], IN3[5], s397, c397);
    logic s398 ,c398; 
    fa fa398 (m1_0[6], m1_1[6], IN3[6], s398, c398);
    logic s399 ,c399; 
    fa fa399 (m1_0[7], m1_1[7], IN3[7], s399, c399);
    logic s400 ,c400; 
    fa fa400 (m1_0[8], m1_1[8], IN3[8], s400, c400);
    logic s401 ,c401; 
    fa fa401 (m1_0[9], m1_1[9], IN3[9], s401, c401);
    logic s402 ,c402; 
    fa fa402 (m1_0[10], m1_1[10], IN3[10], s402, c402);
    logic s403 ,c403; 
    fa fa403 (m1_0[11], m1_1[11], IN3[11], s403, c403);
    logic s404 ,c404; 
    fa fa404 (m1_0[12], m1_1[12], IN3[12], s404, c404);
    logic s405 ,c405; 
    fa fa405 (m1_0[13], m1_1[13], IN3[13], s405, c405);
    logic s406 ,c406; 
    fa fa406 (m1_0[14], m1_1[14], IN3[14], s406, c406);
    logic s407 ,c407; 
    fa fa407 (m1_0[15], m1_1[15], IN3[15], s407, c407);
    logic s408 ,c408; 
    fa fa408 (m1_0[16], m1_1[16], IN3[16], s408, c408);
    logic s409 ,c409; 
    fa fa409 (m1_0[17], m1_1[17], IN3[17], s409, c409);
    logic s410 ,c410; 
    fa fa410 (m1_0[18], m1_1[18], IN3[18], s410, c410);
    logic s411 ,c411; 
    fa fa411 (m1_0[19], m1_1[19], IN3[19], s411, c411);
    logic s412 ,c412; 
    fa fa412 (m1_0[20], m1_1[20], IN3[20], s412, c412);
    logic s413 ,c413; 
    fa fa413 (m1_0[21], m1_1[21], IN3[21], s413, c413);
    logic s414 ,c414; 
    fa fa414 (m1_0[22], m1_1[22], IN3[22], s414, c414);
    logic s415 ,c415; 
    fa fa415 (m1_0[23], m1_1[23], IN3[23], s415, c415);
    logic s416 ,c416; 
    fa fa416 (m1_0[24], m1_1[24], IN3[24], s416, c416);
    logic s417 ,c417; 
    fa fa417 (m1_0[25], m1_1[25], IN3[25], s417, c417);
    logic s418 ,c418; 
    fa fa418 (m1_0[26], m1_1[26], IN3[26], s418, c418);
    logic s419 ,c419; 
    fa fa419 (m1_0[27], m1_1[27], IN3[27], s419, c419);
    logic s420 ,c420; 
    fa fa420 (m1_0[28], m1_1[28], IN3[28], s420, c420);
    logic s421 ,c421; 
    fa fa421 (m1_0[29], m1_1[29], IN3[29], s421, c421);
    logic s422 ,c422; 
    fa fa422 (m1_0[30], m1_1[30], IN3[30], s422, c422);
    logic s423 ,c423; 
    fa fa423 (m1_0[31], m1_1[31], IN3[31], s423, c423);
    logic s424 ,c424; 
    fa fa424 (s0, s134, s296, s424, c424);
    logic s425 ,c425; 
    fa fa425 (s136, c296, s297, s425, c425);
    logic s426 ,c426; 
    fa fa426 (s138, c297, s298, s426, c426);
    logic s427 ,c427; 
    fa fa427 (s140, c298, s299, s427, c427);
    logic s428 ,c428; 
    fa fa428 (s142, c299, s300, s428, c428);
    logic s429 ,c429; 
    fa fa429 (s144, c300, s301, s429, c429);
    logic s430 ,c430; 
    fa fa430 (s146, c301, s302, s430, c430);
    logic s431 ,c431; 
    fa fa431 (s148, c302, s303, s431, c431);
    logic s432 ,c432; 
    fa fa432 (s150, c303, s304, s432, c432);
    logic s433 ,c433; 
    fa fa433 (s152, c304, s305, s433, c433);
    logic s434 ,c434; 
    fa fa434 (s154, c305, s306, s434, c434);
    logic s435 ,c435; 
    fa fa435 (s156, c306, s307, s435, c435);
    logic s436 ,c436; 
    fa fa436 (s158, c307, s308, s436, c436);
    logic s437 ,c437; 
    fa fa437 (s160, c308, s309, s437, c437);
    logic s438 ,c438; 
    fa fa438 (s162, c309, s310, s438, c438);
    logic s439 ,c439; 
    fa fa439 (s164, c310, s311, s439, c439);
    logic s440 ,c440; 
    fa fa440 (s166, c311, s312, s440, c440);
    logic s441 ,c441; 
    fa fa441 (s168, c312, s313, s441, c441);
    logic s442 ,c442; 
    fa fa442 (s170, c313, s314, s442, c442);
    logic s443 ,c443; 
    fa fa443 (s172, c314, s315, s443, c443);
    logic s444 ,c444; 
    fa fa444 (s174, c315, s316, s444, c444);
    logic s445 ,c445; 
    fa fa445 (s176, c316, s317, s445, c445);
    logic s446 ,c446; 
    fa fa446 (s178, c317, s318, s446, c446);
    logic s447 ,c447; 
    fa fa447 (s180, c318, s319, s447, c447);
    logic s448 ,c448; 
    fa fa448 (s182, c319, s320, s448, c448);
    logic s449 ,c449; 
    fa fa449 (s184, c320, s321, s449, c449);
    logic s450 ,c450; 
    fa fa450 (s186, c321, s322, s450, c450);
    logic s451 ,c451; 
    fa fa451 (s188, c322, s323, s451, c451);
    logic s452 ,c452; 
    fa fa452 (s190, c323, s324, s452, c452);
    logic s453 ,c453; 
    fa fa453 (s192, c324, s325, s453, c453);
    logic s454 ,c454; 
    fa fa454 (s194, c325, s326, s454, c454);
    logic s455 ,c455; 
    fa fa455 (s196, c326, s327, s455, c455);
    logic s456 ,c456; 
    fa fa456 (s198, c327, s328, s456, c456);
    logic s457 ,c457; 
    fa fa457 (s200, c328, s329, s457, c457);
    logic s458 ,c458; 
    fa fa458 (s202, c329, s330, s458, c458);
    logic s459 ,c459; 
    fa fa459 (s204, c330, s331, s459, c459);
    logic s460 ,c460; 
    fa fa460 (s206, c331, s332, s460, c460);
    logic s461 ,c461; 
    fa fa461 (s208, c332, s333, s461, c461);
    logic s462 ,c462; 
    fa fa462 (s210, c333, s334, s462, c462);
    logic s463 ,c463; 
    fa fa463 (s212, c334, s335, s463, c463);
    logic s464 ,c464; 
    fa fa464 (s214, c335, s336, s464, c464);
    logic s465 ,c465; 
    fa fa465 (s216, c336, s337, s465, c465);
    logic s466 ,c466; 
    fa fa466 (s218, c337, s338, s466, c466);
    logic s467 ,c467; 
    fa fa467 (s220, c338, s339, s467, c467);
    logic s468 ,c468; 
    fa fa468 (s222, c339, s340, s468, c468);
    logic s469 ,c469; 
    fa fa469 (s224, c340, s341, s469, c469);
    logic s470 ,c470; 
    fa fa470 (s226, c341, s342, s470, c470);
    logic s471 ,c471; 
    fa fa471 (s228, c342, s343, s471, c471);
    logic s472 ,c472; 
    fa fa472 (s230, c343, s344, s472, c472);
    logic s473 ,c473; 
    fa fa473 (s232, c344, s345, s473, c473);
    logic s474 ,c474; 
    fa fa474 (s234, c345, s346, s474, c474);
    logic s475 ,c475; 
    fa fa475 (s236, c346, s347, s475, c475);
    logic s476 ,c476; 
    fa fa476 (s238, c347, s348, s476, c476);
    logic s477 ,c477; 
    fa fa477 (s240, c348, s349, s477, c477);
    logic s478 ,c478; 
    fa fa478 (s242, c349, s350, s478, c478);
    logic s479 ,c479; 
    fa fa479 (s244, c350, s351, s479, c479);
    logic s480 ,c480; 
    fa fa480 (s246, c351, s352, s480, c480);
    logic s481 ,c481; 
    fa fa481 (s248, c352, s353, s481, c481);
    logic s482 ,c482; 
    fa fa482 (s250, c353, s354, s482, c482);
    logic s483 ,c483; 
    fa fa483 (s252, c354, s355, s483, c483);
    logic s484 ,c484; 
    fa fa484 (s254, c355, s356, s484, c484);
    logic s485 ,c485; 
    fa fa485 (s256, c356, s357, s485, c485);
    logic s486 ,c486; 
    fa fa486 (s258, c357, s358, s486, c486);
    logic s487 ,c487; 
    fa fa487 (s260, c358, s359, s487, c487);
    logic s488 ,c488; 
    fa fa488 (s262, c359, s360, s488, c488);
    logic s489 ,c489; 
    fa fa489 (s264, c360, s361, s489, c489);
    logic s490 ,c490; 
    fa fa490 (s266, c361, s362, s490, c490);
    logic s491 ,c491; 
    fa fa491 (s267, c362, s363, s491, c491);
    logic s492 ,c492; 
    fa fa492 (s268, c363, s364, s492, c492);
    logic s493 ,c493; 
    fa fa493 (s269, c364, s365, s493, c493);
    logic s494 ,c494; 
    fa fa494 (s270, c365, s366, s494, c494);
    logic s495 ,c495; 
    fa fa495 (s271, c366, s367, s495, c495);
    logic s496 ,c496; 
    fa fa496 (s272, c367, s368, s496, c496);
    logic s497 ,c497; 
    fa fa497 (s273, c368, s369, s497, c497);
    logic s498 ,c498; 
    fa fa498 (s274, c369, s370, s498, c498);
    logic s499 ,c499; 
    fa fa499 (s275, c370, s371, s499, c499);
    logic s500 ,c500; 
    fa fa500 (s276, c371, s372, s500, c500);
    logic s501 ,c501; 
    fa fa501 (s277, c372, s373, s501, c501);
    logic s502 ,c502; 
    fa fa502 (s278, c373, s374, s502, c502);
    logic s503 ,c503; 
    fa fa503 (s279, c374, s375, s503, c503);
    logic s504 ,c504; 
    fa fa504 (s280, c375, s376, s504, c504);
    logic s505 ,c505; 
    fa fa505 (s281, c376, s377, s505, c505);
    logic s506 ,c506; 
    fa fa506 (s282, c377, s378, s506, c506);
    logic s507 ,c507; 
    fa fa507 (s283, c378, s379, s507, c507);
    logic s508 ,c508; 
    fa fa508 (s284, c379, s380, s508, c508);
    logic s509 ,c509; 
    fa fa509 (s285, c380, s381, s509, c509);
    logic s510 ,c510; 
    fa fa510 (s286, c381, s382, s510, c510);
    logic s511 ,c511; 
    fa fa511 (s287, c382, s383, s511, c511);
    logic s512 ,c512; 
    fa fa512 (s288, c383, s384, s512, c512);
    logic s513 ,c513; 
    fa fa513 (s289, c384, s385, s513, c513);
    logic s514 ,c514; 
    fa fa514 (s290, c385, s386, s514, c514);
    logic s515 ,c515; 
    fa fa515 (s291, c386, s387, s515, c515);
    logic s516 ,c516; 
    fa fa516 (s292, c387, s388, s516, c516);
    logic s517 ,c517; 
    fa fa517 (s293, c388, s389, s517, c517);
    logic s518 ,c518; 
    fa fa518 (s294, c389, s390, s518, c518);
    logic s519 ,c519; 
    fa fa519 (s295, c390, s391, s519, c519);
    
   

   assign result0 = {c518, c517, c516, c515, c514, c513, c512, c511, c510, c509, c508, c507, c506, c505, c504, c503, c502, c501, c500, c499, c498, c497, c496, c495, c494, c493, c492, c491, c490, c489, c488, c487, c486, c485, c484, c483, c482, c481, c480, c479, c478, c477, c476, c475, c474, c473, c472, c471, c470, c469, c468, c467, c466, c465, c464, c463, c462, c461, c460, c459, c458, c457, c456, c455, c454, c453, c452, c451, c450, c449, c448, c447, c446, c445, c444, c443, c442, c441, c440, c439, c438, c437, c436, c435, c434, c433, c432, c431, c430, c429, c428, c427, c426, c425, c424, c423, c422, c421, c420, c419, c418, c417, c416, c415, c414, c413, c412, c411, c410, c409, c408, c407, c406, c405, c404, c403, c402, c401, c400, c399, c398, c397, c396, c395, c394, c393, c392, IN3[0] };

   assign result1 = {s519, s518, s517, s516, s515, s514, s513, s512, s511, s510, s509, s508, s507, s506, s505, s504, s503, s502, s501, s500, s499, s498, s497, s496, s495, s494, s493, s492, s491, s490, s489, s488, s487, s486, s485, s484, s483, s482, s481, s480, s479, s478, s477, s476, s475, s474, s473, s472, s471, s470, s469, s468, s467, s466, s465, s464, s463, s462, s461, s460, s459, s458, s457, s456, s455, s454, s453, s452, s451, s450, s449, s448, s447, s446, s445, s444, s443, s442, s441, s440, s439, s438, s437, s436, s435, s434, s433, s432, s431, s430, s429, s428, s427, s426, s425, s424, s423, s422, s421, s420, s419, s418, s417, s416, s415, s414, s413, s412, s411, s410, s409, s408, s407, s406, s405, s404, s403, s402, s401, s400, s399, s398, s397, s396, s395, s394, s393, s392 };
   

endmodule // FourMultMerger_Signed_64x64_plus_129_128





// LADNER-FISCHER VECTOR ADDER
module LF_131 ( 
        input logic [130:0] IN1,
        input logic [130:0] IN2,
        output logic [131:0] OUT);
    
    wire logic [130:0] p_0;
    wire logic [130:0] g_0;
    assign g_0 = IN1 & IN2;
    assign p_0 = IN1 ^ IN2;
    
    // LF stage 1
    wire logic p_1_1;
    wire logic g_1_1;
    assign p_1_1 = p_0[1] & p_0[0];
    assign g_1_1 = (p_0[1] & g_0[0]) | g_0[1];
    wire logic p_1_3;
    wire logic g_1_3;
    assign p_1_3 = p_0[3] & p_0[2];
    assign g_1_3 = (p_0[3] & g_0[2]) | g_0[3];
    wire logic p_1_5;
    wire logic g_1_5;
    assign p_1_5 = p_0[5] & p_0[4];
    assign g_1_5 = (p_0[5] & g_0[4]) | g_0[5];
    wire logic p_1_7;
    wire logic g_1_7;
    assign p_1_7 = p_0[7] & p_0[6];
    assign g_1_7 = (p_0[7] & g_0[6]) | g_0[7];
    wire logic p_1_9;
    wire logic g_1_9;
    assign p_1_9 = p_0[9] & p_0[8];
    assign g_1_9 = (p_0[9] & g_0[8]) | g_0[9];
    wire logic p_1_11;
    wire logic g_1_11;
    assign p_1_11 = p_0[11] & p_0[10];
    assign g_1_11 = (p_0[11] & g_0[10]) | g_0[11];
    wire logic p_1_13;
    wire logic g_1_13;
    assign p_1_13 = p_0[13] & p_0[12];
    assign g_1_13 = (p_0[13] & g_0[12]) | g_0[13];
    wire logic p_1_15;
    wire logic g_1_15;
    assign p_1_15 = p_0[15] & p_0[14];
    assign g_1_15 = (p_0[15] & g_0[14]) | g_0[15];
    wire logic p_1_17;
    wire logic g_1_17;
    assign p_1_17 = p_0[17] & p_0[16];
    assign g_1_17 = (p_0[17] & g_0[16]) | g_0[17];
    wire logic p_1_19;
    wire logic g_1_19;
    assign p_1_19 = p_0[19] & p_0[18];
    assign g_1_19 = (p_0[19] & g_0[18]) | g_0[19];
    wire logic p_1_21;
    wire logic g_1_21;
    assign p_1_21 = p_0[21] & p_0[20];
    assign g_1_21 = (p_0[21] & g_0[20]) | g_0[21];
    wire logic p_1_23;
    wire logic g_1_23;
    assign p_1_23 = p_0[23] & p_0[22];
    assign g_1_23 = (p_0[23] & g_0[22]) | g_0[23];
    wire logic p_1_25;
    wire logic g_1_25;
    assign p_1_25 = p_0[25] & p_0[24];
    assign g_1_25 = (p_0[25] & g_0[24]) | g_0[25];
    wire logic p_1_27;
    wire logic g_1_27;
    assign p_1_27 = p_0[27] & p_0[26];
    assign g_1_27 = (p_0[27] & g_0[26]) | g_0[27];
    wire logic p_1_29;
    wire logic g_1_29;
    assign p_1_29 = p_0[29] & p_0[28];
    assign g_1_29 = (p_0[29] & g_0[28]) | g_0[29];
    wire logic p_1_31;
    wire logic g_1_31;
    assign p_1_31 = p_0[31] & p_0[30];
    assign g_1_31 = (p_0[31] & g_0[30]) | g_0[31];
    wire logic p_1_33;
    wire logic g_1_33;
    assign p_1_33 = p_0[33] & p_0[32];
    assign g_1_33 = (p_0[33] & g_0[32]) | g_0[33];
    wire logic p_1_35;
    wire logic g_1_35;
    assign p_1_35 = p_0[35] & p_0[34];
    assign g_1_35 = (p_0[35] & g_0[34]) | g_0[35];
    wire logic p_1_37;
    wire logic g_1_37;
    assign p_1_37 = p_0[37] & p_0[36];
    assign g_1_37 = (p_0[37] & g_0[36]) | g_0[37];
    wire logic p_1_39;
    wire logic g_1_39;
    assign p_1_39 = p_0[39] & p_0[38];
    assign g_1_39 = (p_0[39] & g_0[38]) | g_0[39];
    wire logic p_1_41;
    wire logic g_1_41;
    assign p_1_41 = p_0[41] & p_0[40];
    assign g_1_41 = (p_0[41] & g_0[40]) | g_0[41];
    wire logic p_1_43;
    wire logic g_1_43;
    assign p_1_43 = p_0[43] & p_0[42];
    assign g_1_43 = (p_0[43] & g_0[42]) | g_0[43];
    wire logic p_1_45;
    wire logic g_1_45;
    assign p_1_45 = p_0[45] & p_0[44];
    assign g_1_45 = (p_0[45] & g_0[44]) | g_0[45];
    wire logic p_1_47;
    wire logic g_1_47;
    assign p_1_47 = p_0[47] & p_0[46];
    assign g_1_47 = (p_0[47] & g_0[46]) | g_0[47];
    wire logic p_1_49;
    wire logic g_1_49;
    assign p_1_49 = p_0[49] & p_0[48];
    assign g_1_49 = (p_0[49] & g_0[48]) | g_0[49];
    wire logic p_1_51;
    wire logic g_1_51;
    assign p_1_51 = p_0[51] & p_0[50];
    assign g_1_51 = (p_0[51] & g_0[50]) | g_0[51];
    wire logic p_1_53;
    wire logic g_1_53;
    assign p_1_53 = p_0[53] & p_0[52];
    assign g_1_53 = (p_0[53] & g_0[52]) | g_0[53];
    wire logic p_1_55;
    wire logic g_1_55;
    assign p_1_55 = p_0[55] & p_0[54];
    assign g_1_55 = (p_0[55] & g_0[54]) | g_0[55];
    wire logic p_1_57;
    wire logic g_1_57;
    assign p_1_57 = p_0[57] & p_0[56];
    assign g_1_57 = (p_0[57] & g_0[56]) | g_0[57];
    wire logic p_1_59;
    wire logic g_1_59;
    assign p_1_59 = p_0[59] & p_0[58];
    assign g_1_59 = (p_0[59] & g_0[58]) | g_0[59];
    wire logic p_1_61;
    wire logic g_1_61;
    assign p_1_61 = p_0[61] & p_0[60];
    assign g_1_61 = (p_0[61] & g_0[60]) | g_0[61];
    wire logic p_1_63;
    wire logic g_1_63;
    assign p_1_63 = p_0[63] & p_0[62];
    assign g_1_63 = (p_0[63] & g_0[62]) | g_0[63];
    wire logic p_1_65;
    wire logic g_1_65;
    assign p_1_65 = p_0[65] & p_0[64];
    assign g_1_65 = (p_0[65] & g_0[64]) | g_0[65];
    wire logic p_1_67;
    wire logic g_1_67;
    assign p_1_67 = p_0[67] & p_0[66];
    assign g_1_67 = (p_0[67] & g_0[66]) | g_0[67];
    wire logic p_1_69;
    wire logic g_1_69;
    assign p_1_69 = p_0[69] & p_0[68];
    assign g_1_69 = (p_0[69] & g_0[68]) | g_0[69];
    wire logic p_1_71;
    wire logic g_1_71;
    assign p_1_71 = p_0[71] & p_0[70];
    assign g_1_71 = (p_0[71] & g_0[70]) | g_0[71];
    wire logic p_1_73;
    wire logic g_1_73;
    assign p_1_73 = p_0[73] & p_0[72];
    assign g_1_73 = (p_0[73] & g_0[72]) | g_0[73];
    wire logic p_1_75;
    wire logic g_1_75;
    assign p_1_75 = p_0[75] & p_0[74];
    assign g_1_75 = (p_0[75] & g_0[74]) | g_0[75];
    wire logic p_1_77;
    wire logic g_1_77;
    assign p_1_77 = p_0[77] & p_0[76];
    assign g_1_77 = (p_0[77] & g_0[76]) | g_0[77];
    wire logic p_1_79;
    wire logic g_1_79;
    assign p_1_79 = p_0[79] & p_0[78];
    assign g_1_79 = (p_0[79] & g_0[78]) | g_0[79];
    wire logic p_1_81;
    wire logic g_1_81;
    assign p_1_81 = p_0[81] & p_0[80];
    assign g_1_81 = (p_0[81] & g_0[80]) | g_0[81];
    wire logic p_1_83;
    wire logic g_1_83;
    assign p_1_83 = p_0[83] & p_0[82];
    assign g_1_83 = (p_0[83] & g_0[82]) | g_0[83];
    wire logic p_1_85;
    wire logic g_1_85;
    assign p_1_85 = p_0[85] & p_0[84];
    assign g_1_85 = (p_0[85] & g_0[84]) | g_0[85];
    wire logic p_1_87;
    wire logic g_1_87;
    assign p_1_87 = p_0[87] & p_0[86];
    assign g_1_87 = (p_0[87] & g_0[86]) | g_0[87];
    wire logic p_1_89;
    wire logic g_1_89;
    assign p_1_89 = p_0[89] & p_0[88];
    assign g_1_89 = (p_0[89] & g_0[88]) | g_0[89];
    wire logic p_1_91;
    wire logic g_1_91;
    assign p_1_91 = p_0[91] & p_0[90];
    assign g_1_91 = (p_0[91] & g_0[90]) | g_0[91];
    wire logic p_1_93;
    wire logic g_1_93;
    assign p_1_93 = p_0[93] & p_0[92];
    assign g_1_93 = (p_0[93] & g_0[92]) | g_0[93];
    wire logic p_1_95;
    wire logic g_1_95;
    assign p_1_95 = p_0[95] & p_0[94];
    assign g_1_95 = (p_0[95] & g_0[94]) | g_0[95];
    wire logic p_1_97;
    wire logic g_1_97;
    assign p_1_97 = p_0[97] & p_0[96];
    assign g_1_97 = (p_0[97] & g_0[96]) | g_0[97];
    wire logic p_1_99;
    wire logic g_1_99;
    assign p_1_99 = p_0[99] & p_0[98];
    assign g_1_99 = (p_0[99] & g_0[98]) | g_0[99];
    wire logic p_1_101;
    wire logic g_1_101;
    assign p_1_101 = p_0[101] & p_0[100];
    assign g_1_101 = (p_0[101] & g_0[100]) | g_0[101];
    wire logic p_1_103;
    wire logic g_1_103;
    assign p_1_103 = p_0[103] & p_0[102];
    assign g_1_103 = (p_0[103] & g_0[102]) | g_0[103];
    wire logic p_1_105;
    wire logic g_1_105;
    assign p_1_105 = p_0[105] & p_0[104];
    assign g_1_105 = (p_0[105] & g_0[104]) | g_0[105];
    wire logic p_1_107;
    wire logic g_1_107;
    assign p_1_107 = p_0[107] & p_0[106];
    assign g_1_107 = (p_0[107] & g_0[106]) | g_0[107];
    wire logic p_1_109;
    wire logic g_1_109;
    assign p_1_109 = p_0[109] & p_0[108];
    assign g_1_109 = (p_0[109] & g_0[108]) | g_0[109];
    wire logic p_1_111;
    wire logic g_1_111;
    assign p_1_111 = p_0[111] & p_0[110];
    assign g_1_111 = (p_0[111] & g_0[110]) | g_0[111];
    wire logic p_1_113;
    wire logic g_1_113;
    assign p_1_113 = p_0[113] & p_0[112];
    assign g_1_113 = (p_0[113] & g_0[112]) | g_0[113];
    wire logic p_1_115;
    wire logic g_1_115;
    assign p_1_115 = p_0[115] & p_0[114];
    assign g_1_115 = (p_0[115] & g_0[114]) | g_0[115];
    wire logic p_1_117;
    wire logic g_1_117;
    assign p_1_117 = p_0[117] & p_0[116];
    assign g_1_117 = (p_0[117] & g_0[116]) | g_0[117];
    wire logic p_1_119;
    wire logic g_1_119;
    assign p_1_119 = p_0[119] & p_0[118];
    assign g_1_119 = (p_0[119] & g_0[118]) | g_0[119];
    wire logic p_1_121;
    wire logic g_1_121;
    assign p_1_121 = p_0[121] & p_0[120];
    assign g_1_121 = (p_0[121] & g_0[120]) | g_0[121];
    wire logic p_1_123;
    wire logic g_1_123;
    assign p_1_123 = p_0[123] & p_0[122];
    assign g_1_123 = (p_0[123] & g_0[122]) | g_0[123];
    wire logic p_1_125;
    wire logic g_1_125;
    assign p_1_125 = p_0[125] & p_0[124];
    assign g_1_125 = (p_0[125] & g_0[124]) | g_0[125];
    wire logic p_1_127;
    wire logic g_1_127;
    assign p_1_127 = p_0[127] & p_0[126];
    assign g_1_127 = (p_0[127] & g_0[126]) | g_0[127];
    wire logic p_1_129;
    wire logic g_1_129;
    assign p_1_129 = p_0[129] & p_0[128];
    assign g_1_129 = (p_0[129] & g_0[128]) | g_0[129];
    
    // LF stage 2
    wire logic p_2_2;
    wire logic g_2_2;
    assign p_2_2 = p_0[2] & p_1_1;
    assign g_2_2 = (p_0[2] & g_1_1) | g_0[2];
    wire logic p_2_3;
    wire logic g_2_3;
    assign p_2_3 = p_1_3 & p_1_1;
    assign g_2_3 = (p_1_3 & g_1_1) | g_1_3;
    wire logic p_2_6;
    wire logic g_2_6;
    assign p_2_6 = p_0[6] & p_1_5;
    assign g_2_6 = (p_0[6] & g_1_5) | g_0[6];
    wire logic p_2_7;
    wire logic g_2_7;
    assign p_2_7 = p_1_7 & p_1_5;
    assign g_2_7 = (p_1_7 & g_1_5) | g_1_7;
    wire logic p_2_10;
    wire logic g_2_10;
    assign p_2_10 = p_0[10] & p_1_9;
    assign g_2_10 = (p_0[10] & g_1_9) | g_0[10];
    wire logic p_2_11;
    wire logic g_2_11;
    assign p_2_11 = p_1_11 & p_1_9;
    assign g_2_11 = (p_1_11 & g_1_9) | g_1_11;
    wire logic p_2_14;
    wire logic g_2_14;
    assign p_2_14 = p_0[14] & p_1_13;
    assign g_2_14 = (p_0[14] & g_1_13) | g_0[14];
    wire logic p_2_15;
    wire logic g_2_15;
    assign p_2_15 = p_1_15 & p_1_13;
    assign g_2_15 = (p_1_15 & g_1_13) | g_1_15;
    wire logic p_2_18;
    wire logic g_2_18;
    assign p_2_18 = p_0[18] & p_1_17;
    assign g_2_18 = (p_0[18] & g_1_17) | g_0[18];
    wire logic p_2_19;
    wire logic g_2_19;
    assign p_2_19 = p_1_19 & p_1_17;
    assign g_2_19 = (p_1_19 & g_1_17) | g_1_19;
    wire logic p_2_22;
    wire logic g_2_22;
    assign p_2_22 = p_0[22] & p_1_21;
    assign g_2_22 = (p_0[22] & g_1_21) | g_0[22];
    wire logic p_2_23;
    wire logic g_2_23;
    assign p_2_23 = p_1_23 & p_1_21;
    assign g_2_23 = (p_1_23 & g_1_21) | g_1_23;
    wire logic p_2_26;
    wire logic g_2_26;
    assign p_2_26 = p_0[26] & p_1_25;
    assign g_2_26 = (p_0[26] & g_1_25) | g_0[26];
    wire logic p_2_27;
    wire logic g_2_27;
    assign p_2_27 = p_1_27 & p_1_25;
    assign g_2_27 = (p_1_27 & g_1_25) | g_1_27;
    wire logic p_2_30;
    wire logic g_2_30;
    assign p_2_30 = p_0[30] & p_1_29;
    assign g_2_30 = (p_0[30] & g_1_29) | g_0[30];
    wire logic p_2_31;
    wire logic g_2_31;
    assign p_2_31 = p_1_31 & p_1_29;
    assign g_2_31 = (p_1_31 & g_1_29) | g_1_31;
    wire logic p_2_34;
    wire logic g_2_34;
    assign p_2_34 = p_0[34] & p_1_33;
    assign g_2_34 = (p_0[34] & g_1_33) | g_0[34];
    wire logic p_2_35;
    wire logic g_2_35;
    assign p_2_35 = p_1_35 & p_1_33;
    assign g_2_35 = (p_1_35 & g_1_33) | g_1_35;
    wire logic p_2_38;
    wire logic g_2_38;
    assign p_2_38 = p_0[38] & p_1_37;
    assign g_2_38 = (p_0[38] & g_1_37) | g_0[38];
    wire logic p_2_39;
    wire logic g_2_39;
    assign p_2_39 = p_1_39 & p_1_37;
    assign g_2_39 = (p_1_39 & g_1_37) | g_1_39;
    wire logic p_2_42;
    wire logic g_2_42;
    assign p_2_42 = p_0[42] & p_1_41;
    assign g_2_42 = (p_0[42] & g_1_41) | g_0[42];
    wire logic p_2_43;
    wire logic g_2_43;
    assign p_2_43 = p_1_43 & p_1_41;
    assign g_2_43 = (p_1_43 & g_1_41) | g_1_43;
    wire logic p_2_46;
    wire logic g_2_46;
    assign p_2_46 = p_0[46] & p_1_45;
    assign g_2_46 = (p_0[46] & g_1_45) | g_0[46];
    wire logic p_2_47;
    wire logic g_2_47;
    assign p_2_47 = p_1_47 & p_1_45;
    assign g_2_47 = (p_1_47 & g_1_45) | g_1_47;
    wire logic p_2_50;
    wire logic g_2_50;
    assign p_2_50 = p_0[50] & p_1_49;
    assign g_2_50 = (p_0[50] & g_1_49) | g_0[50];
    wire logic p_2_51;
    wire logic g_2_51;
    assign p_2_51 = p_1_51 & p_1_49;
    assign g_2_51 = (p_1_51 & g_1_49) | g_1_51;
    wire logic p_2_54;
    wire logic g_2_54;
    assign p_2_54 = p_0[54] & p_1_53;
    assign g_2_54 = (p_0[54] & g_1_53) | g_0[54];
    wire logic p_2_55;
    wire logic g_2_55;
    assign p_2_55 = p_1_55 & p_1_53;
    assign g_2_55 = (p_1_55 & g_1_53) | g_1_55;
    wire logic p_2_58;
    wire logic g_2_58;
    assign p_2_58 = p_0[58] & p_1_57;
    assign g_2_58 = (p_0[58] & g_1_57) | g_0[58];
    wire logic p_2_59;
    wire logic g_2_59;
    assign p_2_59 = p_1_59 & p_1_57;
    assign g_2_59 = (p_1_59 & g_1_57) | g_1_59;
    wire logic p_2_62;
    wire logic g_2_62;
    assign p_2_62 = p_0[62] & p_1_61;
    assign g_2_62 = (p_0[62] & g_1_61) | g_0[62];
    wire logic p_2_63;
    wire logic g_2_63;
    assign p_2_63 = p_1_63 & p_1_61;
    assign g_2_63 = (p_1_63 & g_1_61) | g_1_63;
    wire logic p_2_66;
    wire logic g_2_66;
    assign p_2_66 = p_0[66] & p_1_65;
    assign g_2_66 = (p_0[66] & g_1_65) | g_0[66];
    wire logic p_2_67;
    wire logic g_2_67;
    assign p_2_67 = p_1_67 & p_1_65;
    assign g_2_67 = (p_1_67 & g_1_65) | g_1_67;
    wire logic p_2_70;
    wire logic g_2_70;
    assign p_2_70 = p_0[70] & p_1_69;
    assign g_2_70 = (p_0[70] & g_1_69) | g_0[70];
    wire logic p_2_71;
    wire logic g_2_71;
    assign p_2_71 = p_1_71 & p_1_69;
    assign g_2_71 = (p_1_71 & g_1_69) | g_1_71;
    wire logic p_2_74;
    wire logic g_2_74;
    assign p_2_74 = p_0[74] & p_1_73;
    assign g_2_74 = (p_0[74] & g_1_73) | g_0[74];
    wire logic p_2_75;
    wire logic g_2_75;
    assign p_2_75 = p_1_75 & p_1_73;
    assign g_2_75 = (p_1_75 & g_1_73) | g_1_75;
    wire logic p_2_78;
    wire logic g_2_78;
    assign p_2_78 = p_0[78] & p_1_77;
    assign g_2_78 = (p_0[78] & g_1_77) | g_0[78];
    wire logic p_2_79;
    wire logic g_2_79;
    assign p_2_79 = p_1_79 & p_1_77;
    assign g_2_79 = (p_1_79 & g_1_77) | g_1_79;
    wire logic p_2_82;
    wire logic g_2_82;
    assign p_2_82 = p_0[82] & p_1_81;
    assign g_2_82 = (p_0[82] & g_1_81) | g_0[82];
    wire logic p_2_83;
    wire logic g_2_83;
    assign p_2_83 = p_1_83 & p_1_81;
    assign g_2_83 = (p_1_83 & g_1_81) | g_1_83;
    wire logic p_2_86;
    wire logic g_2_86;
    assign p_2_86 = p_0[86] & p_1_85;
    assign g_2_86 = (p_0[86] & g_1_85) | g_0[86];
    wire logic p_2_87;
    wire logic g_2_87;
    assign p_2_87 = p_1_87 & p_1_85;
    assign g_2_87 = (p_1_87 & g_1_85) | g_1_87;
    wire logic p_2_90;
    wire logic g_2_90;
    assign p_2_90 = p_0[90] & p_1_89;
    assign g_2_90 = (p_0[90] & g_1_89) | g_0[90];
    wire logic p_2_91;
    wire logic g_2_91;
    assign p_2_91 = p_1_91 & p_1_89;
    assign g_2_91 = (p_1_91 & g_1_89) | g_1_91;
    wire logic p_2_94;
    wire logic g_2_94;
    assign p_2_94 = p_0[94] & p_1_93;
    assign g_2_94 = (p_0[94] & g_1_93) | g_0[94];
    wire logic p_2_95;
    wire logic g_2_95;
    assign p_2_95 = p_1_95 & p_1_93;
    assign g_2_95 = (p_1_95 & g_1_93) | g_1_95;
    wire logic p_2_98;
    wire logic g_2_98;
    assign p_2_98 = p_0[98] & p_1_97;
    assign g_2_98 = (p_0[98] & g_1_97) | g_0[98];
    wire logic p_2_99;
    wire logic g_2_99;
    assign p_2_99 = p_1_99 & p_1_97;
    assign g_2_99 = (p_1_99 & g_1_97) | g_1_99;
    wire logic p_2_102;
    wire logic g_2_102;
    assign p_2_102 = p_0[102] & p_1_101;
    assign g_2_102 = (p_0[102] & g_1_101) | g_0[102];
    wire logic p_2_103;
    wire logic g_2_103;
    assign p_2_103 = p_1_103 & p_1_101;
    assign g_2_103 = (p_1_103 & g_1_101) | g_1_103;
    wire logic p_2_106;
    wire logic g_2_106;
    assign p_2_106 = p_0[106] & p_1_105;
    assign g_2_106 = (p_0[106] & g_1_105) | g_0[106];
    wire logic p_2_107;
    wire logic g_2_107;
    assign p_2_107 = p_1_107 & p_1_105;
    assign g_2_107 = (p_1_107 & g_1_105) | g_1_107;
    wire logic p_2_110;
    wire logic g_2_110;
    assign p_2_110 = p_0[110] & p_1_109;
    assign g_2_110 = (p_0[110] & g_1_109) | g_0[110];
    wire logic p_2_111;
    wire logic g_2_111;
    assign p_2_111 = p_1_111 & p_1_109;
    assign g_2_111 = (p_1_111 & g_1_109) | g_1_111;
    wire logic p_2_114;
    wire logic g_2_114;
    assign p_2_114 = p_0[114] & p_1_113;
    assign g_2_114 = (p_0[114] & g_1_113) | g_0[114];
    wire logic p_2_115;
    wire logic g_2_115;
    assign p_2_115 = p_1_115 & p_1_113;
    assign g_2_115 = (p_1_115 & g_1_113) | g_1_115;
    wire logic p_2_118;
    wire logic g_2_118;
    assign p_2_118 = p_0[118] & p_1_117;
    assign g_2_118 = (p_0[118] & g_1_117) | g_0[118];
    wire logic p_2_119;
    wire logic g_2_119;
    assign p_2_119 = p_1_119 & p_1_117;
    assign g_2_119 = (p_1_119 & g_1_117) | g_1_119;
    wire logic p_2_122;
    wire logic g_2_122;
    assign p_2_122 = p_0[122] & p_1_121;
    assign g_2_122 = (p_0[122] & g_1_121) | g_0[122];
    wire logic p_2_123;
    wire logic g_2_123;
    assign p_2_123 = p_1_123 & p_1_121;
    assign g_2_123 = (p_1_123 & g_1_121) | g_1_123;
    wire logic p_2_126;
    wire logic g_2_126;
    assign p_2_126 = p_0[126] & p_1_125;
    assign g_2_126 = (p_0[126] & g_1_125) | g_0[126];
    wire logic p_2_127;
    wire logic g_2_127;
    assign p_2_127 = p_1_127 & p_1_125;
    assign g_2_127 = (p_1_127 & g_1_125) | g_1_127;
    wire logic p_2_130;
    wire logic g_2_130;
    assign p_2_130 = p_0[130] & p_1_129;
    assign g_2_130 = (p_0[130] & g_1_129) | g_0[130];
    
    // LF stage 3
    wire logic p_3_4;
    wire logic g_3_4;
    assign p_3_4 = p_0[4] & p_2_3;
    assign g_3_4 = (p_0[4] & g_2_3) | g_0[4];
    wire logic p_3_5;
    wire logic g_3_5;
    assign p_3_5 = p_1_5 & p_2_3;
    assign g_3_5 = (p_1_5 & g_2_3) | g_1_5;
    wire logic p_3_6;
    wire logic g_3_6;
    assign p_3_6 = p_2_6 & p_2_3;
    assign g_3_6 = (p_2_6 & g_2_3) | g_2_6;
    wire logic p_3_7;
    wire logic g_3_7;
    assign p_3_7 = p_2_7 & p_2_3;
    assign g_3_7 = (p_2_7 & g_2_3) | g_2_7;
    wire logic p_3_12;
    wire logic g_3_12;
    assign p_3_12 = p_0[12] & p_2_11;
    assign g_3_12 = (p_0[12] & g_2_11) | g_0[12];
    wire logic p_3_13;
    wire logic g_3_13;
    assign p_3_13 = p_1_13 & p_2_11;
    assign g_3_13 = (p_1_13 & g_2_11) | g_1_13;
    wire logic p_3_14;
    wire logic g_3_14;
    assign p_3_14 = p_2_14 & p_2_11;
    assign g_3_14 = (p_2_14 & g_2_11) | g_2_14;
    wire logic p_3_15;
    wire logic g_3_15;
    assign p_3_15 = p_2_15 & p_2_11;
    assign g_3_15 = (p_2_15 & g_2_11) | g_2_15;
    wire logic p_3_20;
    wire logic g_3_20;
    assign p_3_20 = p_0[20] & p_2_19;
    assign g_3_20 = (p_0[20] & g_2_19) | g_0[20];
    wire logic p_3_21;
    wire logic g_3_21;
    assign p_3_21 = p_1_21 & p_2_19;
    assign g_3_21 = (p_1_21 & g_2_19) | g_1_21;
    wire logic p_3_22;
    wire logic g_3_22;
    assign p_3_22 = p_2_22 & p_2_19;
    assign g_3_22 = (p_2_22 & g_2_19) | g_2_22;
    wire logic p_3_23;
    wire logic g_3_23;
    assign p_3_23 = p_2_23 & p_2_19;
    assign g_3_23 = (p_2_23 & g_2_19) | g_2_23;
    wire logic p_3_28;
    wire logic g_3_28;
    assign p_3_28 = p_0[28] & p_2_27;
    assign g_3_28 = (p_0[28] & g_2_27) | g_0[28];
    wire logic p_3_29;
    wire logic g_3_29;
    assign p_3_29 = p_1_29 & p_2_27;
    assign g_3_29 = (p_1_29 & g_2_27) | g_1_29;
    wire logic p_3_30;
    wire logic g_3_30;
    assign p_3_30 = p_2_30 & p_2_27;
    assign g_3_30 = (p_2_30 & g_2_27) | g_2_30;
    wire logic p_3_31;
    wire logic g_3_31;
    assign p_3_31 = p_2_31 & p_2_27;
    assign g_3_31 = (p_2_31 & g_2_27) | g_2_31;
    wire logic p_3_36;
    wire logic g_3_36;
    assign p_3_36 = p_0[36] & p_2_35;
    assign g_3_36 = (p_0[36] & g_2_35) | g_0[36];
    wire logic p_3_37;
    wire logic g_3_37;
    assign p_3_37 = p_1_37 & p_2_35;
    assign g_3_37 = (p_1_37 & g_2_35) | g_1_37;
    wire logic p_3_38;
    wire logic g_3_38;
    assign p_3_38 = p_2_38 & p_2_35;
    assign g_3_38 = (p_2_38 & g_2_35) | g_2_38;
    wire logic p_3_39;
    wire logic g_3_39;
    assign p_3_39 = p_2_39 & p_2_35;
    assign g_3_39 = (p_2_39 & g_2_35) | g_2_39;
    wire logic p_3_44;
    wire logic g_3_44;
    assign p_3_44 = p_0[44] & p_2_43;
    assign g_3_44 = (p_0[44] & g_2_43) | g_0[44];
    wire logic p_3_45;
    wire logic g_3_45;
    assign p_3_45 = p_1_45 & p_2_43;
    assign g_3_45 = (p_1_45 & g_2_43) | g_1_45;
    wire logic p_3_46;
    wire logic g_3_46;
    assign p_3_46 = p_2_46 & p_2_43;
    assign g_3_46 = (p_2_46 & g_2_43) | g_2_46;
    wire logic p_3_47;
    wire logic g_3_47;
    assign p_3_47 = p_2_47 & p_2_43;
    assign g_3_47 = (p_2_47 & g_2_43) | g_2_47;
    wire logic p_3_52;
    wire logic g_3_52;
    assign p_3_52 = p_0[52] & p_2_51;
    assign g_3_52 = (p_0[52] & g_2_51) | g_0[52];
    wire logic p_3_53;
    wire logic g_3_53;
    assign p_3_53 = p_1_53 & p_2_51;
    assign g_3_53 = (p_1_53 & g_2_51) | g_1_53;
    wire logic p_3_54;
    wire logic g_3_54;
    assign p_3_54 = p_2_54 & p_2_51;
    assign g_3_54 = (p_2_54 & g_2_51) | g_2_54;
    wire logic p_3_55;
    wire logic g_3_55;
    assign p_3_55 = p_2_55 & p_2_51;
    assign g_3_55 = (p_2_55 & g_2_51) | g_2_55;
    wire logic p_3_60;
    wire logic g_3_60;
    assign p_3_60 = p_0[60] & p_2_59;
    assign g_3_60 = (p_0[60] & g_2_59) | g_0[60];
    wire logic p_3_61;
    wire logic g_3_61;
    assign p_3_61 = p_1_61 & p_2_59;
    assign g_3_61 = (p_1_61 & g_2_59) | g_1_61;
    wire logic p_3_62;
    wire logic g_3_62;
    assign p_3_62 = p_2_62 & p_2_59;
    assign g_3_62 = (p_2_62 & g_2_59) | g_2_62;
    wire logic p_3_63;
    wire logic g_3_63;
    assign p_3_63 = p_2_63 & p_2_59;
    assign g_3_63 = (p_2_63 & g_2_59) | g_2_63;
    wire logic p_3_68;
    wire logic g_3_68;
    assign p_3_68 = p_0[68] & p_2_67;
    assign g_3_68 = (p_0[68] & g_2_67) | g_0[68];
    wire logic p_3_69;
    wire logic g_3_69;
    assign p_3_69 = p_1_69 & p_2_67;
    assign g_3_69 = (p_1_69 & g_2_67) | g_1_69;
    wire logic p_3_70;
    wire logic g_3_70;
    assign p_3_70 = p_2_70 & p_2_67;
    assign g_3_70 = (p_2_70 & g_2_67) | g_2_70;
    wire logic p_3_71;
    wire logic g_3_71;
    assign p_3_71 = p_2_71 & p_2_67;
    assign g_3_71 = (p_2_71 & g_2_67) | g_2_71;
    wire logic p_3_76;
    wire logic g_3_76;
    assign p_3_76 = p_0[76] & p_2_75;
    assign g_3_76 = (p_0[76] & g_2_75) | g_0[76];
    wire logic p_3_77;
    wire logic g_3_77;
    assign p_3_77 = p_1_77 & p_2_75;
    assign g_3_77 = (p_1_77 & g_2_75) | g_1_77;
    wire logic p_3_78;
    wire logic g_3_78;
    assign p_3_78 = p_2_78 & p_2_75;
    assign g_3_78 = (p_2_78 & g_2_75) | g_2_78;
    wire logic p_3_79;
    wire logic g_3_79;
    assign p_3_79 = p_2_79 & p_2_75;
    assign g_3_79 = (p_2_79 & g_2_75) | g_2_79;
    wire logic p_3_84;
    wire logic g_3_84;
    assign p_3_84 = p_0[84] & p_2_83;
    assign g_3_84 = (p_0[84] & g_2_83) | g_0[84];
    wire logic p_3_85;
    wire logic g_3_85;
    assign p_3_85 = p_1_85 & p_2_83;
    assign g_3_85 = (p_1_85 & g_2_83) | g_1_85;
    wire logic p_3_86;
    wire logic g_3_86;
    assign p_3_86 = p_2_86 & p_2_83;
    assign g_3_86 = (p_2_86 & g_2_83) | g_2_86;
    wire logic p_3_87;
    wire logic g_3_87;
    assign p_3_87 = p_2_87 & p_2_83;
    assign g_3_87 = (p_2_87 & g_2_83) | g_2_87;
    wire logic p_3_92;
    wire logic g_3_92;
    assign p_3_92 = p_0[92] & p_2_91;
    assign g_3_92 = (p_0[92] & g_2_91) | g_0[92];
    wire logic p_3_93;
    wire logic g_3_93;
    assign p_3_93 = p_1_93 & p_2_91;
    assign g_3_93 = (p_1_93 & g_2_91) | g_1_93;
    wire logic p_3_94;
    wire logic g_3_94;
    assign p_3_94 = p_2_94 & p_2_91;
    assign g_3_94 = (p_2_94 & g_2_91) | g_2_94;
    wire logic p_3_95;
    wire logic g_3_95;
    assign p_3_95 = p_2_95 & p_2_91;
    assign g_3_95 = (p_2_95 & g_2_91) | g_2_95;
    wire logic p_3_100;
    wire logic g_3_100;
    assign p_3_100 = p_0[100] & p_2_99;
    assign g_3_100 = (p_0[100] & g_2_99) | g_0[100];
    wire logic p_3_101;
    wire logic g_3_101;
    assign p_3_101 = p_1_101 & p_2_99;
    assign g_3_101 = (p_1_101 & g_2_99) | g_1_101;
    wire logic p_3_102;
    wire logic g_3_102;
    assign p_3_102 = p_2_102 & p_2_99;
    assign g_3_102 = (p_2_102 & g_2_99) | g_2_102;
    wire logic p_3_103;
    wire logic g_3_103;
    assign p_3_103 = p_2_103 & p_2_99;
    assign g_3_103 = (p_2_103 & g_2_99) | g_2_103;
    wire logic p_3_108;
    wire logic g_3_108;
    assign p_3_108 = p_0[108] & p_2_107;
    assign g_3_108 = (p_0[108] & g_2_107) | g_0[108];
    wire logic p_3_109;
    wire logic g_3_109;
    assign p_3_109 = p_1_109 & p_2_107;
    assign g_3_109 = (p_1_109 & g_2_107) | g_1_109;
    wire logic p_3_110;
    wire logic g_3_110;
    assign p_3_110 = p_2_110 & p_2_107;
    assign g_3_110 = (p_2_110 & g_2_107) | g_2_110;
    wire logic p_3_111;
    wire logic g_3_111;
    assign p_3_111 = p_2_111 & p_2_107;
    assign g_3_111 = (p_2_111 & g_2_107) | g_2_111;
    wire logic p_3_116;
    wire logic g_3_116;
    assign p_3_116 = p_0[116] & p_2_115;
    assign g_3_116 = (p_0[116] & g_2_115) | g_0[116];
    wire logic p_3_117;
    wire logic g_3_117;
    assign p_3_117 = p_1_117 & p_2_115;
    assign g_3_117 = (p_1_117 & g_2_115) | g_1_117;
    wire logic p_3_118;
    wire logic g_3_118;
    assign p_3_118 = p_2_118 & p_2_115;
    assign g_3_118 = (p_2_118 & g_2_115) | g_2_118;
    wire logic p_3_119;
    wire logic g_3_119;
    assign p_3_119 = p_2_119 & p_2_115;
    assign g_3_119 = (p_2_119 & g_2_115) | g_2_119;
    wire logic p_3_124;
    wire logic g_3_124;
    assign p_3_124 = p_0[124] & p_2_123;
    assign g_3_124 = (p_0[124] & g_2_123) | g_0[124];
    wire logic p_3_125;
    wire logic g_3_125;
    assign p_3_125 = p_1_125 & p_2_123;
    assign g_3_125 = (p_1_125 & g_2_123) | g_1_125;
    wire logic p_3_126;
    wire logic g_3_126;
    assign p_3_126 = p_2_126 & p_2_123;
    assign g_3_126 = (p_2_126 & g_2_123) | g_2_126;
    wire logic p_3_127;
    wire logic g_3_127;
    assign p_3_127 = p_2_127 & p_2_123;
    assign g_3_127 = (p_2_127 & g_2_123) | g_2_127;
    
    // LF stage 4
    wire logic p_4_8;
    wire logic g_4_8;
    assign p_4_8 = p_0[8] & p_3_7;
    assign g_4_8 = (p_0[8] & g_3_7) | g_0[8];
    wire logic p_4_9;
    wire logic g_4_9;
    assign p_4_9 = p_1_9 & p_3_7;
    assign g_4_9 = (p_1_9 & g_3_7) | g_1_9;
    wire logic p_4_10;
    wire logic g_4_10;
    assign p_4_10 = p_2_10 & p_3_7;
    assign g_4_10 = (p_2_10 & g_3_7) | g_2_10;
    wire logic p_4_11;
    wire logic g_4_11;
    assign p_4_11 = p_2_11 & p_3_7;
    assign g_4_11 = (p_2_11 & g_3_7) | g_2_11;
    wire logic p_4_12;
    wire logic g_4_12;
    assign p_4_12 = p_3_12 & p_3_7;
    assign g_4_12 = (p_3_12 & g_3_7) | g_3_12;
    wire logic p_4_13;
    wire logic g_4_13;
    assign p_4_13 = p_3_13 & p_3_7;
    assign g_4_13 = (p_3_13 & g_3_7) | g_3_13;
    wire logic p_4_14;
    wire logic g_4_14;
    assign p_4_14 = p_3_14 & p_3_7;
    assign g_4_14 = (p_3_14 & g_3_7) | g_3_14;
    wire logic p_4_15;
    wire logic g_4_15;
    assign p_4_15 = p_3_15 & p_3_7;
    assign g_4_15 = (p_3_15 & g_3_7) | g_3_15;
    wire logic p_4_24;
    wire logic g_4_24;
    assign p_4_24 = p_0[24] & p_3_23;
    assign g_4_24 = (p_0[24] & g_3_23) | g_0[24];
    wire logic p_4_25;
    wire logic g_4_25;
    assign p_4_25 = p_1_25 & p_3_23;
    assign g_4_25 = (p_1_25 & g_3_23) | g_1_25;
    wire logic p_4_26;
    wire logic g_4_26;
    assign p_4_26 = p_2_26 & p_3_23;
    assign g_4_26 = (p_2_26 & g_3_23) | g_2_26;
    wire logic p_4_27;
    wire logic g_4_27;
    assign p_4_27 = p_2_27 & p_3_23;
    assign g_4_27 = (p_2_27 & g_3_23) | g_2_27;
    wire logic p_4_28;
    wire logic g_4_28;
    assign p_4_28 = p_3_28 & p_3_23;
    assign g_4_28 = (p_3_28 & g_3_23) | g_3_28;
    wire logic p_4_29;
    wire logic g_4_29;
    assign p_4_29 = p_3_29 & p_3_23;
    assign g_4_29 = (p_3_29 & g_3_23) | g_3_29;
    wire logic p_4_30;
    wire logic g_4_30;
    assign p_4_30 = p_3_30 & p_3_23;
    assign g_4_30 = (p_3_30 & g_3_23) | g_3_30;
    wire logic p_4_31;
    wire logic g_4_31;
    assign p_4_31 = p_3_31 & p_3_23;
    assign g_4_31 = (p_3_31 & g_3_23) | g_3_31;
    wire logic p_4_40;
    wire logic g_4_40;
    assign p_4_40 = p_0[40] & p_3_39;
    assign g_4_40 = (p_0[40] & g_3_39) | g_0[40];
    wire logic p_4_41;
    wire logic g_4_41;
    assign p_4_41 = p_1_41 & p_3_39;
    assign g_4_41 = (p_1_41 & g_3_39) | g_1_41;
    wire logic p_4_42;
    wire logic g_4_42;
    assign p_4_42 = p_2_42 & p_3_39;
    assign g_4_42 = (p_2_42 & g_3_39) | g_2_42;
    wire logic p_4_43;
    wire logic g_4_43;
    assign p_4_43 = p_2_43 & p_3_39;
    assign g_4_43 = (p_2_43 & g_3_39) | g_2_43;
    wire logic p_4_44;
    wire logic g_4_44;
    assign p_4_44 = p_3_44 & p_3_39;
    assign g_4_44 = (p_3_44 & g_3_39) | g_3_44;
    wire logic p_4_45;
    wire logic g_4_45;
    assign p_4_45 = p_3_45 & p_3_39;
    assign g_4_45 = (p_3_45 & g_3_39) | g_3_45;
    wire logic p_4_46;
    wire logic g_4_46;
    assign p_4_46 = p_3_46 & p_3_39;
    assign g_4_46 = (p_3_46 & g_3_39) | g_3_46;
    wire logic p_4_47;
    wire logic g_4_47;
    assign p_4_47 = p_3_47 & p_3_39;
    assign g_4_47 = (p_3_47 & g_3_39) | g_3_47;
    wire logic p_4_56;
    wire logic g_4_56;
    assign p_4_56 = p_0[56] & p_3_55;
    assign g_4_56 = (p_0[56] & g_3_55) | g_0[56];
    wire logic p_4_57;
    wire logic g_4_57;
    assign p_4_57 = p_1_57 & p_3_55;
    assign g_4_57 = (p_1_57 & g_3_55) | g_1_57;
    wire logic p_4_58;
    wire logic g_4_58;
    assign p_4_58 = p_2_58 & p_3_55;
    assign g_4_58 = (p_2_58 & g_3_55) | g_2_58;
    wire logic p_4_59;
    wire logic g_4_59;
    assign p_4_59 = p_2_59 & p_3_55;
    assign g_4_59 = (p_2_59 & g_3_55) | g_2_59;
    wire logic p_4_60;
    wire logic g_4_60;
    assign p_4_60 = p_3_60 & p_3_55;
    assign g_4_60 = (p_3_60 & g_3_55) | g_3_60;
    wire logic p_4_61;
    wire logic g_4_61;
    assign p_4_61 = p_3_61 & p_3_55;
    assign g_4_61 = (p_3_61 & g_3_55) | g_3_61;
    wire logic p_4_62;
    wire logic g_4_62;
    assign p_4_62 = p_3_62 & p_3_55;
    assign g_4_62 = (p_3_62 & g_3_55) | g_3_62;
    wire logic p_4_63;
    wire logic g_4_63;
    assign p_4_63 = p_3_63 & p_3_55;
    assign g_4_63 = (p_3_63 & g_3_55) | g_3_63;
    wire logic p_4_72;
    wire logic g_4_72;
    assign p_4_72 = p_0[72] & p_3_71;
    assign g_4_72 = (p_0[72] & g_3_71) | g_0[72];
    wire logic p_4_73;
    wire logic g_4_73;
    assign p_4_73 = p_1_73 & p_3_71;
    assign g_4_73 = (p_1_73 & g_3_71) | g_1_73;
    wire logic p_4_74;
    wire logic g_4_74;
    assign p_4_74 = p_2_74 & p_3_71;
    assign g_4_74 = (p_2_74 & g_3_71) | g_2_74;
    wire logic p_4_75;
    wire logic g_4_75;
    assign p_4_75 = p_2_75 & p_3_71;
    assign g_4_75 = (p_2_75 & g_3_71) | g_2_75;
    wire logic p_4_76;
    wire logic g_4_76;
    assign p_4_76 = p_3_76 & p_3_71;
    assign g_4_76 = (p_3_76 & g_3_71) | g_3_76;
    wire logic p_4_77;
    wire logic g_4_77;
    assign p_4_77 = p_3_77 & p_3_71;
    assign g_4_77 = (p_3_77 & g_3_71) | g_3_77;
    wire logic p_4_78;
    wire logic g_4_78;
    assign p_4_78 = p_3_78 & p_3_71;
    assign g_4_78 = (p_3_78 & g_3_71) | g_3_78;
    wire logic p_4_79;
    wire logic g_4_79;
    assign p_4_79 = p_3_79 & p_3_71;
    assign g_4_79 = (p_3_79 & g_3_71) | g_3_79;
    wire logic p_4_88;
    wire logic g_4_88;
    assign p_4_88 = p_0[88] & p_3_87;
    assign g_4_88 = (p_0[88] & g_3_87) | g_0[88];
    wire logic p_4_89;
    wire logic g_4_89;
    assign p_4_89 = p_1_89 & p_3_87;
    assign g_4_89 = (p_1_89 & g_3_87) | g_1_89;
    wire logic p_4_90;
    wire logic g_4_90;
    assign p_4_90 = p_2_90 & p_3_87;
    assign g_4_90 = (p_2_90 & g_3_87) | g_2_90;
    wire logic p_4_91;
    wire logic g_4_91;
    assign p_4_91 = p_2_91 & p_3_87;
    assign g_4_91 = (p_2_91 & g_3_87) | g_2_91;
    wire logic p_4_92;
    wire logic g_4_92;
    assign p_4_92 = p_3_92 & p_3_87;
    assign g_4_92 = (p_3_92 & g_3_87) | g_3_92;
    wire logic p_4_93;
    wire logic g_4_93;
    assign p_4_93 = p_3_93 & p_3_87;
    assign g_4_93 = (p_3_93 & g_3_87) | g_3_93;
    wire logic p_4_94;
    wire logic g_4_94;
    assign p_4_94 = p_3_94 & p_3_87;
    assign g_4_94 = (p_3_94 & g_3_87) | g_3_94;
    wire logic p_4_95;
    wire logic g_4_95;
    assign p_4_95 = p_3_95 & p_3_87;
    assign g_4_95 = (p_3_95 & g_3_87) | g_3_95;
    wire logic p_4_104;
    wire logic g_4_104;
    assign p_4_104 = p_0[104] & p_3_103;
    assign g_4_104 = (p_0[104] & g_3_103) | g_0[104];
    wire logic p_4_105;
    wire logic g_4_105;
    assign p_4_105 = p_1_105 & p_3_103;
    assign g_4_105 = (p_1_105 & g_3_103) | g_1_105;
    wire logic p_4_106;
    wire logic g_4_106;
    assign p_4_106 = p_2_106 & p_3_103;
    assign g_4_106 = (p_2_106 & g_3_103) | g_2_106;
    wire logic p_4_107;
    wire logic g_4_107;
    assign p_4_107 = p_2_107 & p_3_103;
    assign g_4_107 = (p_2_107 & g_3_103) | g_2_107;
    wire logic p_4_108;
    wire logic g_4_108;
    assign p_4_108 = p_3_108 & p_3_103;
    assign g_4_108 = (p_3_108 & g_3_103) | g_3_108;
    wire logic p_4_109;
    wire logic g_4_109;
    assign p_4_109 = p_3_109 & p_3_103;
    assign g_4_109 = (p_3_109 & g_3_103) | g_3_109;
    wire logic p_4_110;
    wire logic g_4_110;
    assign p_4_110 = p_3_110 & p_3_103;
    assign g_4_110 = (p_3_110 & g_3_103) | g_3_110;
    wire logic p_4_111;
    wire logic g_4_111;
    assign p_4_111 = p_3_111 & p_3_103;
    assign g_4_111 = (p_3_111 & g_3_103) | g_3_111;
    wire logic p_4_120;
    wire logic g_4_120;
    assign p_4_120 = p_0[120] & p_3_119;
    assign g_4_120 = (p_0[120] & g_3_119) | g_0[120];
    wire logic p_4_121;
    wire logic g_4_121;
    assign p_4_121 = p_1_121 & p_3_119;
    assign g_4_121 = (p_1_121 & g_3_119) | g_1_121;
    wire logic p_4_122;
    wire logic g_4_122;
    assign p_4_122 = p_2_122 & p_3_119;
    assign g_4_122 = (p_2_122 & g_3_119) | g_2_122;
    wire logic p_4_123;
    wire logic g_4_123;
    assign p_4_123 = p_2_123 & p_3_119;
    assign g_4_123 = (p_2_123 & g_3_119) | g_2_123;
    wire logic p_4_124;
    wire logic g_4_124;
    assign p_4_124 = p_3_124 & p_3_119;
    assign g_4_124 = (p_3_124 & g_3_119) | g_3_124;
    wire logic p_4_125;
    wire logic g_4_125;
    assign p_4_125 = p_3_125 & p_3_119;
    assign g_4_125 = (p_3_125 & g_3_119) | g_3_125;
    wire logic p_4_126;
    wire logic g_4_126;
    assign p_4_126 = p_3_126 & p_3_119;
    assign g_4_126 = (p_3_126 & g_3_119) | g_3_126;
    wire logic p_4_127;
    wire logic g_4_127;
    assign p_4_127 = p_3_127 & p_3_119;
    assign g_4_127 = (p_3_127 & g_3_119) | g_3_127;
    
    // LF stage 5
    wire logic p_5_16;
    wire logic g_5_16;
    assign p_5_16 = p_0[16] & p_4_15;
    assign g_5_16 = (p_0[16] & g_4_15) | g_0[16];
    wire logic p_5_17;
    wire logic g_5_17;
    assign p_5_17 = p_1_17 & p_4_15;
    assign g_5_17 = (p_1_17 & g_4_15) | g_1_17;
    wire logic p_5_18;
    wire logic g_5_18;
    assign p_5_18 = p_2_18 & p_4_15;
    assign g_5_18 = (p_2_18 & g_4_15) | g_2_18;
    wire logic p_5_19;
    wire logic g_5_19;
    assign p_5_19 = p_2_19 & p_4_15;
    assign g_5_19 = (p_2_19 & g_4_15) | g_2_19;
    wire logic p_5_20;
    wire logic g_5_20;
    assign p_5_20 = p_3_20 & p_4_15;
    assign g_5_20 = (p_3_20 & g_4_15) | g_3_20;
    wire logic p_5_21;
    wire logic g_5_21;
    assign p_5_21 = p_3_21 & p_4_15;
    assign g_5_21 = (p_3_21 & g_4_15) | g_3_21;
    wire logic p_5_22;
    wire logic g_5_22;
    assign p_5_22 = p_3_22 & p_4_15;
    assign g_5_22 = (p_3_22 & g_4_15) | g_3_22;
    wire logic p_5_23;
    wire logic g_5_23;
    assign p_5_23 = p_3_23 & p_4_15;
    assign g_5_23 = (p_3_23 & g_4_15) | g_3_23;
    wire logic p_5_24;
    wire logic g_5_24;
    assign p_5_24 = p_4_24 & p_4_15;
    assign g_5_24 = (p_4_24 & g_4_15) | g_4_24;
    wire logic p_5_25;
    wire logic g_5_25;
    assign p_5_25 = p_4_25 & p_4_15;
    assign g_5_25 = (p_4_25 & g_4_15) | g_4_25;
    wire logic p_5_26;
    wire logic g_5_26;
    assign p_5_26 = p_4_26 & p_4_15;
    assign g_5_26 = (p_4_26 & g_4_15) | g_4_26;
    wire logic p_5_27;
    wire logic g_5_27;
    assign p_5_27 = p_4_27 & p_4_15;
    assign g_5_27 = (p_4_27 & g_4_15) | g_4_27;
    wire logic p_5_28;
    wire logic g_5_28;
    assign p_5_28 = p_4_28 & p_4_15;
    assign g_5_28 = (p_4_28 & g_4_15) | g_4_28;
    wire logic p_5_29;
    wire logic g_5_29;
    assign p_5_29 = p_4_29 & p_4_15;
    assign g_5_29 = (p_4_29 & g_4_15) | g_4_29;
    wire logic p_5_30;
    wire logic g_5_30;
    assign p_5_30 = p_4_30 & p_4_15;
    assign g_5_30 = (p_4_30 & g_4_15) | g_4_30;
    wire logic p_5_31;
    wire logic g_5_31;
    assign p_5_31 = p_4_31 & p_4_15;
    assign g_5_31 = (p_4_31 & g_4_15) | g_4_31;
    wire logic p_5_48;
    wire logic g_5_48;
    assign p_5_48 = p_0[48] & p_4_47;
    assign g_5_48 = (p_0[48] & g_4_47) | g_0[48];
    wire logic p_5_49;
    wire logic g_5_49;
    assign p_5_49 = p_1_49 & p_4_47;
    assign g_5_49 = (p_1_49 & g_4_47) | g_1_49;
    wire logic p_5_50;
    wire logic g_5_50;
    assign p_5_50 = p_2_50 & p_4_47;
    assign g_5_50 = (p_2_50 & g_4_47) | g_2_50;
    wire logic p_5_51;
    wire logic g_5_51;
    assign p_5_51 = p_2_51 & p_4_47;
    assign g_5_51 = (p_2_51 & g_4_47) | g_2_51;
    wire logic p_5_52;
    wire logic g_5_52;
    assign p_5_52 = p_3_52 & p_4_47;
    assign g_5_52 = (p_3_52 & g_4_47) | g_3_52;
    wire logic p_5_53;
    wire logic g_5_53;
    assign p_5_53 = p_3_53 & p_4_47;
    assign g_5_53 = (p_3_53 & g_4_47) | g_3_53;
    wire logic p_5_54;
    wire logic g_5_54;
    assign p_5_54 = p_3_54 & p_4_47;
    assign g_5_54 = (p_3_54 & g_4_47) | g_3_54;
    wire logic p_5_55;
    wire logic g_5_55;
    assign p_5_55 = p_3_55 & p_4_47;
    assign g_5_55 = (p_3_55 & g_4_47) | g_3_55;
    wire logic p_5_56;
    wire logic g_5_56;
    assign p_5_56 = p_4_56 & p_4_47;
    assign g_5_56 = (p_4_56 & g_4_47) | g_4_56;
    wire logic p_5_57;
    wire logic g_5_57;
    assign p_5_57 = p_4_57 & p_4_47;
    assign g_5_57 = (p_4_57 & g_4_47) | g_4_57;
    wire logic p_5_58;
    wire logic g_5_58;
    assign p_5_58 = p_4_58 & p_4_47;
    assign g_5_58 = (p_4_58 & g_4_47) | g_4_58;
    wire logic p_5_59;
    wire logic g_5_59;
    assign p_5_59 = p_4_59 & p_4_47;
    assign g_5_59 = (p_4_59 & g_4_47) | g_4_59;
    wire logic p_5_60;
    wire logic g_5_60;
    assign p_5_60 = p_4_60 & p_4_47;
    assign g_5_60 = (p_4_60 & g_4_47) | g_4_60;
    wire logic p_5_61;
    wire logic g_5_61;
    assign p_5_61 = p_4_61 & p_4_47;
    assign g_5_61 = (p_4_61 & g_4_47) | g_4_61;
    wire logic p_5_62;
    wire logic g_5_62;
    assign p_5_62 = p_4_62 & p_4_47;
    assign g_5_62 = (p_4_62 & g_4_47) | g_4_62;
    wire logic p_5_63;
    wire logic g_5_63;
    assign p_5_63 = p_4_63 & p_4_47;
    assign g_5_63 = (p_4_63 & g_4_47) | g_4_63;
    wire logic p_5_80;
    wire logic g_5_80;
    assign p_5_80 = p_0[80] & p_4_79;
    assign g_5_80 = (p_0[80] & g_4_79) | g_0[80];
    wire logic p_5_81;
    wire logic g_5_81;
    assign p_5_81 = p_1_81 & p_4_79;
    assign g_5_81 = (p_1_81 & g_4_79) | g_1_81;
    wire logic p_5_82;
    wire logic g_5_82;
    assign p_5_82 = p_2_82 & p_4_79;
    assign g_5_82 = (p_2_82 & g_4_79) | g_2_82;
    wire logic p_5_83;
    wire logic g_5_83;
    assign p_5_83 = p_2_83 & p_4_79;
    assign g_5_83 = (p_2_83 & g_4_79) | g_2_83;
    wire logic p_5_84;
    wire logic g_5_84;
    assign p_5_84 = p_3_84 & p_4_79;
    assign g_5_84 = (p_3_84 & g_4_79) | g_3_84;
    wire logic p_5_85;
    wire logic g_5_85;
    assign p_5_85 = p_3_85 & p_4_79;
    assign g_5_85 = (p_3_85 & g_4_79) | g_3_85;
    wire logic p_5_86;
    wire logic g_5_86;
    assign p_5_86 = p_3_86 & p_4_79;
    assign g_5_86 = (p_3_86 & g_4_79) | g_3_86;
    wire logic p_5_87;
    wire logic g_5_87;
    assign p_5_87 = p_3_87 & p_4_79;
    assign g_5_87 = (p_3_87 & g_4_79) | g_3_87;
    wire logic p_5_88;
    wire logic g_5_88;
    assign p_5_88 = p_4_88 & p_4_79;
    assign g_5_88 = (p_4_88 & g_4_79) | g_4_88;
    wire logic p_5_89;
    wire logic g_5_89;
    assign p_5_89 = p_4_89 & p_4_79;
    assign g_5_89 = (p_4_89 & g_4_79) | g_4_89;
    wire logic p_5_90;
    wire logic g_5_90;
    assign p_5_90 = p_4_90 & p_4_79;
    assign g_5_90 = (p_4_90 & g_4_79) | g_4_90;
    wire logic p_5_91;
    wire logic g_5_91;
    assign p_5_91 = p_4_91 & p_4_79;
    assign g_5_91 = (p_4_91 & g_4_79) | g_4_91;
    wire logic p_5_92;
    wire logic g_5_92;
    assign p_5_92 = p_4_92 & p_4_79;
    assign g_5_92 = (p_4_92 & g_4_79) | g_4_92;
    wire logic p_5_93;
    wire logic g_5_93;
    assign p_5_93 = p_4_93 & p_4_79;
    assign g_5_93 = (p_4_93 & g_4_79) | g_4_93;
    wire logic p_5_94;
    wire logic g_5_94;
    assign p_5_94 = p_4_94 & p_4_79;
    assign g_5_94 = (p_4_94 & g_4_79) | g_4_94;
    wire logic p_5_95;
    wire logic g_5_95;
    assign p_5_95 = p_4_95 & p_4_79;
    assign g_5_95 = (p_4_95 & g_4_79) | g_4_95;
    wire logic p_5_112;
    wire logic g_5_112;
    assign p_5_112 = p_0[112] & p_4_111;
    assign g_5_112 = (p_0[112] & g_4_111) | g_0[112];
    wire logic p_5_113;
    wire logic g_5_113;
    assign p_5_113 = p_1_113 & p_4_111;
    assign g_5_113 = (p_1_113 & g_4_111) | g_1_113;
    wire logic p_5_114;
    wire logic g_5_114;
    assign p_5_114 = p_2_114 & p_4_111;
    assign g_5_114 = (p_2_114 & g_4_111) | g_2_114;
    wire logic p_5_115;
    wire logic g_5_115;
    assign p_5_115 = p_2_115 & p_4_111;
    assign g_5_115 = (p_2_115 & g_4_111) | g_2_115;
    wire logic p_5_116;
    wire logic g_5_116;
    assign p_5_116 = p_3_116 & p_4_111;
    assign g_5_116 = (p_3_116 & g_4_111) | g_3_116;
    wire logic p_5_117;
    wire logic g_5_117;
    assign p_5_117 = p_3_117 & p_4_111;
    assign g_5_117 = (p_3_117 & g_4_111) | g_3_117;
    wire logic p_5_118;
    wire logic g_5_118;
    assign p_5_118 = p_3_118 & p_4_111;
    assign g_5_118 = (p_3_118 & g_4_111) | g_3_118;
    wire logic p_5_119;
    wire logic g_5_119;
    assign p_5_119 = p_3_119 & p_4_111;
    assign g_5_119 = (p_3_119 & g_4_111) | g_3_119;
    wire logic p_5_120;
    wire logic g_5_120;
    assign p_5_120 = p_4_120 & p_4_111;
    assign g_5_120 = (p_4_120 & g_4_111) | g_4_120;
    wire logic p_5_121;
    wire logic g_5_121;
    assign p_5_121 = p_4_121 & p_4_111;
    assign g_5_121 = (p_4_121 & g_4_111) | g_4_121;
    wire logic p_5_122;
    wire logic g_5_122;
    assign p_5_122 = p_4_122 & p_4_111;
    assign g_5_122 = (p_4_122 & g_4_111) | g_4_122;
    wire logic p_5_123;
    wire logic g_5_123;
    assign p_5_123 = p_4_123 & p_4_111;
    assign g_5_123 = (p_4_123 & g_4_111) | g_4_123;
    wire logic p_5_124;
    wire logic g_5_124;
    assign p_5_124 = p_4_124 & p_4_111;
    assign g_5_124 = (p_4_124 & g_4_111) | g_4_124;
    wire logic p_5_125;
    wire logic g_5_125;
    assign p_5_125 = p_4_125 & p_4_111;
    assign g_5_125 = (p_4_125 & g_4_111) | g_4_125;
    wire logic p_5_126;
    wire logic g_5_126;
    assign p_5_126 = p_4_126 & p_4_111;
    assign g_5_126 = (p_4_126 & g_4_111) | g_4_126;
    wire logic p_5_127;
    wire logic g_5_127;
    assign p_5_127 = p_4_127 & p_4_111;
    assign g_5_127 = (p_4_127 & g_4_111) | g_4_127;
    
    // LF stage 6
    wire logic p_6_32;
    wire logic g_6_32;
    assign p_6_32 = p_0[32] & p_5_31;
    assign g_6_32 = (p_0[32] & g_5_31) | g_0[32];
    wire logic p_6_33;
    wire logic g_6_33;
    assign p_6_33 = p_1_33 & p_5_31;
    assign g_6_33 = (p_1_33 & g_5_31) | g_1_33;
    wire logic p_6_34;
    wire logic g_6_34;
    assign p_6_34 = p_2_34 & p_5_31;
    assign g_6_34 = (p_2_34 & g_5_31) | g_2_34;
    wire logic p_6_35;
    wire logic g_6_35;
    assign p_6_35 = p_2_35 & p_5_31;
    assign g_6_35 = (p_2_35 & g_5_31) | g_2_35;
    wire logic p_6_36;
    wire logic g_6_36;
    assign p_6_36 = p_3_36 & p_5_31;
    assign g_6_36 = (p_3_36 & g_5_31) | g_3_36;
    wire logic p_6_37;
    wire logic g_6_37;
    assign p_6_37 = p_3_37 & p_5_31;
    assign g_6_37 = (p_3_37 & g_5_31) | g_3_37;
    wire logic p_6_38;
    wire logic g_6_38;
    assign p_6_38 = p_3_38 & p_5_31;
    assign g_6_38 = (p_3_38 & g_5_31) | g_3_38;
    wire logic p_6_39;
    wire logic g_6_39;
    assign p_6_39 = p_3_39 & p_5_31;
    assign g_6_39 = (p_3_39 & g_5_31) | g_3_39;
    wire logic p_6_40;
    wire logic g_6_40;
    assign p_6_40 = p_4_40 & p_5_31;
    assign g_6_40 = (p_4_40 & g_5_31) | g_4_40;
    wire logic p_6_41;
    wire logic g_6_41;
    assign p_6_41 = p_4_41 & p_5_31;
    assign g_6_41 = (p_4_41 & g_5_31) | g_4_41;
    wire logic p_6_42;
    wire logic g_6_42;
    assign p_6_42 = p_4_42 & p_5_31;
    assign g_6_42 = (p_4_42 & g_5_31) | g_4_42;
    wire logic p_6_43;
    wire logic g_6_43;
    assign p_6_43 = p_4_43 & p_5_31;
    assign g_6_43 = (p_4_43 & g_5_31) | g_4_43;
    wire logic p_6_44;
    wire logic g_6_44;
    assign p_6_44 = p_4_44 & p_5_31;
    assign g_6_44 = (p_4_44 & g_5_31) | g_4_44;
    wire logic p_6_45;
    wire logic g_6_45;
    assign p_6_45 = p_4_45 & p_5_31;
    assign g_6_45 = (p_4_45 & g_5_31) | g_4_45;
    wire logic p_6_46;
    wire logic g_6_46;
    assign p_6_46 = p_4_46 & p_5_31;
    assign g_6_46 = (p_4_46 & g_5_31) | g_4_46;
    wire logic p_6_47;
    wire logic g_6_47;
    assign p_6_47 = p_4_47 & p_5_31;
    assign g_6_47 = (p_4_47 & g_5_31) | g_4_47;
    wire logic p_6_48;
    wire logic g_6_48;
    assign p_6_48 = p_5_48 & p_5_31;
    assign g_6_48 = (p_5_48 & g_5_31) | g_5_48;
    wire logic p_6_49;
    wire logic g_6_49;
    assign p_6_49 = p_5_49 & p_5_31;
    assign g_6_49 = (p_5_49 & g_5_31) | g_5_49;
    wire logic p_6_50;
    wire logic g_6_50;
    assign p_6_50 = p_5_50 & p_5_31;
    assign g_6_50 = (p_5_50 & g_5_31) | g_5_50;
    wire logic p_6_51;
    wire logic g_6_51;
    assign p_6_51 = p_5_51 & p_5_31;
    assign g_6_51 = (p_5_51 & g_5_31) | g_5_51;
    wire logic p_6_52;
    wire logic g_6_52;
    assign p_6_52 = p_5_52 & p_5_31;
    assign g_6_52 = (p_5_52 & g_5_31) | g_5_52;
    wire logic p_6_53;
    wire logic g_6_53;
    assign p_6_53 = p_5_53 & p_5_31;
    assign g_6_53 = (p_5_53 & g_5_31) | g_5_53;
    wire logic p_6_54;
    wire logic g_6_54;
    assign p_6_54 = p_5_54 & p_5_31;
    assign g_6_54 = (p_5_54 & g_5_31) | g_5_54;
    wire logic p_6_55;
    wire logic g_6_55;
    assign p_6_55 = p_5_55 & p_5_31;
    assign g_6_55 = (p_5_55 & g_5_31) | g_5_55;
    wire logic p_6_56;
    wire logic g_6_56;
    assign p_6_56 = p_5_56 & p_5_31;
    assign g_6_56 = (p_5_56 & g_5_31) | g_5_56;
    wire logic p_6_57;
    wire logic g_6_57;
    assign p_6_57 = p_5_57 & p_5_31;
    assign g_6_57 = (p_5_57 & g_5_31) | g_5_57;
    wire logic p_6_58;
    wire logic g_6_58;
    assign p_6_58 = p_5_58 & p_5_31;
    assign g_6_58 = (p_5_58 & g_5_31) | g_5_58;
    wire logic p_6_59;
    wire logic g_6_59;
    assign p_6_59 = p_5_59 & p_5_31;
    assign g_6_59 = (p_5_59 & g_5_31) | g_5_59;
    wire logic p_6_60;
    wire logic g_6_60;
    assign p_6_60 = p_5_60 & p_5_31;
    assign g_6_60 = (p_5_60 & g_5_31) | g_5_60;
    wire logic p_6_61;
    wire logic g_6_61;
    assign p_6_61 = p_5_61 & p_5_31;
    assign g_6_61 = (p_5_61 & g_5_31) | g_5_61;
    wire logic p_6_62;
    wire logic g_6_62;
    assign p_6_62 = p_5_62 & p_5_31;
    assign g_6_62 = (p_5_62 & g_5_31) | g_5_62;
    wire logic p_6_63;
    wire logic g_6_63;
    assign p_6_63 = p_5_63 & p_5_31;
    assign g_6_63 = (p_5_63 & g_5_31) | g_5_63;
    wire logic p_6_96;
    wire logic g_6_96;
    assign p_6_96 = p_0[96] & p_5_95;
    assign g_6_96 = (p_0[96] & g_5_95) | g_0[96];
    wire logic p_6_97;
    wire logic g_6_97;
    assign p_6_97 = p_1_97 & p_5_95;
    assign g_6_97 = (p_1_97 & g_5_95) | g_1_97;
    wire logic p_6_98;
    wire logic g_6_98;
    assign p_6_98 = p_2_98 & p_5_95;
    assign g_6_98 = (p_2_98 & g_5_95) | g_2_98;
    wire logic p_6_99;
    wire logic g_6_99;
    assign p_6_99 = p_2_99 & p_5_95;
    assign g_6_99 = (p_2_99 & g_5_95) | g_2_99;
    wire logic p_6_100;
    wire logic g_6_100;
    assign p_6_100 = p_3_100 & p_5_95;
    assign g_6_100 = (p_3_100 & g_5_95) | g_3_100;
    wire logic p_6_101;
    wire logic g_6_101;
    assign p_6_101 = p_3_101 & p_5_95;
    assign g_6_101 = (p_3_101 & g_5_95) | g_3_101;
    wire logic p_6_102;
    wire logic g_6_102;
    assign p_6_102 = p_3_102 & p_5_95;
    assign g_6_102 = (p_3_102 & g_5_95) | g_3_102;
    wire logic p_6_103;
    wire logic g_6_103;
    assign p_6_103 = p_3_103 & p_5_95;
    assign g_6_103 = (p_3_103 & g_5_95) | g_3_103;
    wire logic p_6_104;
    wire logic g_6_104;
    assign p_6_104 = p_4_104 & p_5_95;
    assign g_6_104 = (p_4_104 & g_5_95) | g_4_104;
    wire logic p_6_105;
    wire logic g_6_105;
    assign p_6_105 = p_4_105 & p_5_95;
    assign g_6_105 = (p_4_105 & g_5_95) | g_4_105;
    wire logic p_6_106;
    wire logic g_6_106;
    assign p_6_106 = p_4_106 & p_5_95;
    assign g_6_106 = (p_4_106 & g_5_95) | g_4_106;
    wire logic p_6_107;
    wire logic g_6_107;
    assign p_6_107 = p_4_107 & p_5_95;
    assign g_6_107 = (p_4_107 & g_5_95) | g_4_107;
    wire logic p_6_108;
    wire logic g_6_108;
    assign p_6_108 = p_4_108 & p_5_95;
    assign g_6_108 = (p_4_108 & g_5_95) | g_4_108;
    wire logic p_6_109;
    wire logic g_6_109;
    assign p_6_109 = p_4_109 & p_5_95;
    assign g_6_109 = (p_4_109 & g_5_95) | g_4_109;
    wire logic p_6_110;
    wire logic g_6_110;
    assign p_6_110 = p_4_110 & p_5_95;
    assign g_6_110 = (p_4_110 & g_5_95) | g_4_110;
    wire logic p_6_111;
    wire logic g_6_111;
    assign p_6_111 = p_4_111 & p_5_95;
    assign g_6_111 = (p_4_111 & g_5_95) | g_4_111;
    wire logic p_6_112;
    wire logic g_6_112;
    assign p_6_112 = p_5_112 & p_5_95;
    assign g_6_112 = (p_5_112 & g_5_95) | g_5_112;
    wire logic p_6_113;
    wire logic g_6_113;
    assign p_6_113 = p_5_113 & p_5_95;
    assign g_6_113 = (p_5_113 & g_5_95) | g_5_113;
    wire logic p_6_114;
    wire logic g_6_114;
    assign p_6_114 = p_5_114 & p_5_95;
    assign g_6_114 = (p_5_114 & g_5_95) | g_5_114;
    wire logic p_6_115;
    wire logic g_6_115;
    assign p_6_115 = p_5_115 & p_5_95;
    assign g_6_115 = (p_5_115 & g_5_95) | g_5_115;
    wire logic p_6_116;
    wire logic g_6_116;
    assign p_6_116 = p_5_116 & p_5_95;
    assign g_6_116 = (p_5_116 & g_5_95) | g_5_116;
    wire logic p_6_117;
    wire logic g_6_117;
    assign p_6_117 = p_5_117 & p_5_95;
    assign g_6_117 = (p_5_117 & g_5_95) | g_5_117;
    wire logic p_6_118;
    wire logic g_6_118;
    assign p_6_118 = p_5_118 & p_5_95;
    assign g_6_118 = (p_5_118 & g_5_95) | g_5_118;
    wire logic p_6_119;
    wire logic g_6_119;
    assign p_6_119 = p_5_119 & p_5_95;
    assign g_6_119 = (p_5_119 & g_5_95) | g_5_119;
    wire logic p_6_120;
    wire logic g_6_120;
    assign p_6_120 = p_5_120 & p_5_95;
    assign g_6_120 = (p_5_120 & g_5_95) | g_5_120;
    wire logic p_6_121;
    wire logic g_6_121;
    assign p_6_121 = p_5_121 & p_5_95;
    assign g_6_121 = (p_5_121 & g_5_95) | g_5_121;
    wire logic p_6_122;
    wire logic g_6_122;
    assign p_6_122 = p_5_122 & p_5_95;
    assign g_6_122 = (p_5_122 & g_5_95) | g_5_122;
    wire logic p_6_123;
    wire logic g_6_123;
    assign p_6_123 = p_5_123 & p_5_95;
    assign g_6_123 = (p_5_123 & g_5_95) | g_5_123;
    wire logic p_6_124;
    wire logic g_6_124;
    assign p_6_124 = p_5_124 & p_5_95;
    assign g_6_124 = (p_5_124 & g_5_95) | g_5_124;
    wire logic p_6_125;
    wire logic g_6_125;
    assign p_6_125 = p_5_125 & p_5_95;
    assign g_6_125 = (p_5_125 & g_5_95) | g_5_125;
    wire logic p_6_126;
    wire logic g_6_126;
    assign p_6_126 = p_5_126 & p_5_95;
    assign g_6_126 = (p_5_126 & g_5_95) | g_5_126;
    wire logic p_6_127;
    wire logic g_6_127;
    assign p_6_127 = p_5_127 & p_5_95;
    assign g_6_127 = (p_5_127 & g_5_95) | g_5_127;
    
    // LF stage 7
    wire logic p_7_64;
    wire logic g_7_64;
    assign p_7_64 = p_0[64] & p_6_63;
    assign g_7_64 = (p_0[64] & g_6_63) | g_0[64];
    wire logic p_7_65;
    wire logic g_7_65;
    assign p_7_65 = p_1_65 & p_6_63;
    assign g_7_65 = (p_1_65 & g_6_63) | g_1_65;
    wire logic p_7_66;
    wire logic g_7_66;
    assign p_7_66 = p_2_66 & p_6_63;
    assign g_7_66 = (p_2_66 & g_6_63) | g_2_66;
    wire logic p_7_67;
    wire logic g_7_67;
    assign p_7_67 = p_2_67 & p_6_63;
    assign g_7_67 = (p_2_67 & g_6_63) | g_2_67;
    wire logic p_7_68;
    wire logic g_7_68;
    assign p_7_68 = p_3_68 & p_6_63;
    assign g_7_68 = (p_3_68 & g_6_63) | g_3_68;
    wire logic p_7_69;
    wire logic g_7_69;
    assign p_7_69 = p_3_69 & p_6_63;
    assign g_7_69 = (p_3_69 & g_6_63) | g_3_69;
    wire logic p_7_70;
    wire logic g_7_70;
    assign p_7_70 = p_3_70 & p_6_63;
    assign g_7_70 = (p_3_70 & g_6_63) | g_3_70;
    wire logic p_7_71;
    wire logic g_7_71;
    assign p_7_71 = p_3_71 & p_6_63;
    assign g_7_71 = (p_3_71 & g_6_63) | g_3_71;
    wire logic p_7_72;
    wire logic g_7_72;
    assign p_7_72 = p_4_72 & p_6_63;
    assign g_7_72 = (p_4_72 & g_6_63) | g_4_72;
    wire logic p_7_73;
    wire logic g_7_73;
    assign p_7_73 = p_4_73 & p_6_63;
    assign g_7_73 = (p_4_73 & g_6_63) | g_4_73;
    wire logic p_7_74;
    wire logic g_7_74;
    assign p_7_74 = p_4_74 & p_6_63;
    assign g_7_74 = (p_4_74 & g_6_63) | g_4_74;
    wire logic p_7_75;
    wire logic g_7_75;
    assign p_7_75 = p_4_75 & p_6_63;
    assign g_7_75 = (p_4_75 & g_6_63) | g_4_75;
    wire logic p_7_76;
    wire logic g_7_76;
    assign p_7_76 = p_4_76 & p_6_63;
    assign g_7_76 = (p_4_76 & g_6_63) | g_4_76;
    wire logic p_7_77;
    wire logic g_7_77;
    assign p_7_77 = p_4_77 & p_6_63;
    assign g_7_77 = (p_4_77 & g_6_63) | g_4_77;
    wire logic p_7_78;
    wire logic g_7_78;
    assign p_7_78 = p_4_78 & p_6_63;
    assign g_7_78 = (p_4_78 & g_6_63) | g_4_78;
    wire logic p_7_79;
    wire logic g_7_79;
    assign p_7_79 = p_4_79 & p_6_63;
    assign g_7_79 = (p_4_79 & g_6_63) | g_4_79;
    wire logic p_7_80;
    wire logic g_7_80;
    assign p_7_80 = p_5_80 & p_6_63;
    assign g_7_80 = (p_5_80 & g_6_63) | g_5_80;
    wire logic p_7_81;
    wire logic g_7_81;
    assign p_7_81 = p_5_81 & p_6_63;
    assign g_7_81 = (p_5_81 & g_6_63) | g_5_81;
    wire logic p_7_82;
    wire logic g_7_82;
    assign p_7_82 = p_5_82 & p_6_63;
    assign g_7_82 = (p_5_82 & g_6_63) | g_5_82;
    wire logic p_7_83;
    wire logic g_7_83;
    assign p_7_83 = p_5_83 & p_6_63;
    assign g_7_83 = (p_5_83 & g_6_63) | g_5_83;
    wire logic p_7_84;
    wire logic g_7_84;
    assign p_7_84 = p_5_84 & p_6_63;
    assign g_7_84 = (p_5_84 & g_6_63) | g_5_84;
    wire logic p_7_85;
    wire logic g_7_85;
    assign p_7_85 = p_5_85 & p_6_63;
    assign g_7_85 = (p_5_85 & g_6_63) | g_5_85;
    wire logic p_7_86;
    wire logic g_7_86;
    assign p_7_86 = p_5_86 & p_6_63;
    assign g_7_86 = (p_5_86 & g_6_63) | g_5_86;
    wire logic p_7_87;
    wire logic g_7_87;
    assign p_7_87 = p_5_87 & p_6_63;
    assign g_7_87 = (p_5_87 & g_6_63) | g_5_87;
    wire logic p_7_88;
    wire logic g_7_88;
    assign p_7_88 = p_5_88 & p_6_63;
    assign g_7_88 = (p_5_88 & g_6_63) | g_5_88;
    wire logic p_7_89;
    wire logic g_7_89;
    assign p_7_89 = p_5_89 & p_6_63;
    assign g_7_89 = (p_5_89 & g_6_63) | g_5_89;
    wire logic p_7_90;
    wire logic g_7_90;
    assign p_7_90 = p_5_90 & p_6_63;
    assign g_7_90 = (p_5_90 & g_6_63) | g_5_90;
    wire logic p_7_91;
    wire logic g_7_91;
    assign p_7_91 = p_5_91 & p_6_63;
    assign g_7_91 = (p_5_91 & g_6_63) | g_5_91;
    wire logic p_7_92;
    wire logic g_7_92;
    assign p_7_92 = p_5_92 & p_6_63;
    assign g_7_92 = (p_5_92 & g_6_63) | g_5_92;
    wire logic p_7_93;
    wire logic g_7_93;
    assign p_7_93 = p_5_93 & p_6_63;
    assign g_7_93 = (p_5_93 & g_6_63) | g_5_93;
    wire logic p_7_94;
    wire logic g_7_94;
    assign p_7_94 = p_5_94 & p_6_63;
    assign g_7_94 = (p_5_94 & g_6_63) | g_5_94;
    wire logic p_7_95;
    wire logic g_7_95;
    assign p_7_95 = p_5_95 & p_6_63;
    assign g_7_95 = (p_5_95 & g_6_63) | g_5_95;
    wire logic p_7_96;
    wire logic g_7_96;
    assign p_7_96 = p_6_96 & p_6_63;
    assign g_7_96 = (p_6_96 & g_6_63) | g_6_96;
    wire logic p_7_97;
    wire logic g_7_97;
    assign p_7_97 = p_6_97 & p_6_63;
    assign g_7_97 = (p_6_97 & g_6_63) | g_6_97;
    wire logic p_7_98;
    wire logic g_7_98;
    assign p_7_98 = p_6_98 & p_6_63;
    assign g_7_98 = (p_6_98 & g_6_63) | g_6_98;
    wire logic p_7_99;
    wire logic g_7_99;
    assign p_7_99 = p_6_99 & p_6_63;
    assign g_7_99 = (p_6_99 & g_6_63) | g_6_99;
    wire logic p_7_100;
    wire logic g_7_100;
    assign p_7_100 = p_6_100 & p_6_63;
    assign g_7_100 = (p_6_100 & g_6_63) | g_6_100;
    wire logic p_7_101;
    wire logic g_7_101;
    assign p_7_101 = p_6_101 & p_6_63;
    assign g_7_101 = (p_6_101 & g_6_63) | g_6_101;
    wire logic p_7_102;
    wire logic g_7_102;
    assign p_7_102 = p_6_102 & p_6_63;
    assign g_7_102 = (p_6_102 & g_6_63) | g_6_102;
    wire logic p_7_103;
    wire logic g_7_103;
    assign p_7_103 = p_6_103 & p_6_63;
    assign g_7_103 = (p_6_103 & g_6_63) | g_6_103;
    wire logic p_7_104;
    wire logic g_7_104;
    assign p_7_104 = p_6_104 & p_6_63;
    assign g_7_104 = (p_6_104 & g_6_63) | g_6_104;
    wire logic p_7_105;
    wire logic g_7_105;
    assign p_7_105 = p_6_105 & p_6_63;
    assign g_7_105 = (p_6_105 & g_6_63) | g_6_105;
    wire logic p_7_106;
    wire logic g_7_106;
    assign p_7_106 = p_6_106 & p_6_63;
    assign g_7_106 = (p_6_106 & g_6_63) | g_6_106;
    wire logic p_7_107;
    wire logic g_7_107;
    assign p_7_107 = p_6_107 & p_6_63;
    assign g_7_107 = (p_6_107 & g_6_63) | g_6_107;
    wire logic p_7_108;
    wire logic g_7_108;
    assign p_7_108 = p_6_108 & p_6_63;
    assign g_7_108 = (p_6_108 & g_6_63) | g_6_108;
    wire logic p_7_109;
    wire logic g_7_109;
    assign p_7_109 = p_6_109 & p_6_63;
    assign g_7_109 = (p_6_109 & g_6_63) | g_6_109;
    wire logic p_7_110;
    wire logic g_7_110;
    assign p_7_110 = p_6_110 & p_6_63;
    assign g_7_110 = (p_6_110 & g_6_63) | g_6_110;
    wire logic p_7_111;
    wire logic g_7_111;
    assign p_7_111 = p_6_111 & p_6_63;
    assign g_7_111 = (p_6_111 & g_6_63) | g_6_111;
    wire logic p_7_112;
    wire logic g_7_112;
    assign p_7_112 = p_6_112 & p_6_63;
    assign g_7_112 = (p_6_112 & g_6_63) | g_6_112;
    wire logic p_7_113;
    wire logic g_7_113;
    assign p_7_113 = p_6_113 & p_6_63;
    assign g_7_113 = (p_6_113 & g_6_63) | g_6_113;
    wire logic p_7_114;
    wire logic g_7_114;
    assign p_7_114 = p_6_114 & p_6_63;
    assign g_7_114 = (p_6_114 & g_6_63) | g_6_114;
    wire logic p_7_115;
    wire logic g_7_115;
    assign p_7_115 = p_6_115 & p_6_63;
    assign g_7_115 = (p_6_115 & g_6_63) | g_6_115;
    wire logic p_7_116;
    wire logic g_7_116;
    assign p_7_116 = p_6_116 & p_6_63;
    assign g_7_116 = (p_6_116 & g_6_63) | g_6_116;
    wire logic p_7_117;
    wire logic g_7_117;
    assign p_7_117 = p_6_117 & p_6_63;
    assign g_7_117 = (p_6_117 & g_6_63) | g_6_117;
    wire logic p_7_118;
    wire logic g_7_118;
    assign p_7_118 = p_6_118 & p_6_63;
    assign g_7_118 = (p_6_118 & g_6_63) | g_6_118;
    wire logic p_7_119;
    wire logic g_7_119;
    assign p_7_119 = p_6_119 & p_6_63;
    assign g_7_119 = (p_6_119 & g_6_63) | g_6_119;
    wire logic p_7_120;
    wire logic g_7_120;
    assign p_7_120 = p_6_120 & p_6_63;
    assign g_7_120 = (p_6_120 & g_6_63) | g_6_120;
    wire logic p_7_121;
    wire logic g_7_121;
    assign p_7_121 = p_6_121 & p_6_63;
    assign g_7_121 = (p_6_121 & g_6_63) | g_6_121;
    wire logic p_7_122;
    wire logic g_7_122;
    assign p_7_122 = p_6_122 & p_6_63;
    assign g_7_122 = (p_6_122 & g_6_63) | g_6_122;
    wire logic p_7_123;
    wire logic g_7_123;
    assign p_7_123 = p_6_123 & p_6_63;
    assign g_7_123 = (p_6_123 & g_6_63) | g_6_123;
    wire logic p_7_124;
    wire logic g_7_124;
    assign p_7_124 = p_6_124 & p_6_63;
    assign g_7_124 = (p_6_124 & g_6_63) | g_6_124;
    wire logic p_7_125;
    wire logic g_7_125;
    assign p_7_125 = p_6_125 & p_6_63;
    assign g_7_125 = (p_6_125 & g_6_63) | g_6_125;
    wire logic p_7_126;
    wire logic g_7_126;
    assign p_7_126 = p_6_126 & p_6_63;
    assign g_7_126 = (p_6_126 & g_6_63) | g_6_126;
    wire logic p_7_127;
    wire logic g_7_127;
    assign p_7_127 = p_6_127 & p_6_63;
    assign g_7_127 = (p_6_127 & g_6_63) | g_6_127;
    
    // LF stage 8
    wire logic p_8_128;
    wire logic g_8_128;
    assign p_8_128 = p_0[128] & p_7_127;
    assign g_8_128 = (p_0[128] & g_7_127) | g_0[128];
    wire logic p_8_129;
    wire logic g_8_129;
    assign p_8_129 = p_1_129 & p_7_127;
    assign g_8_129 = (p_1_129 & g_7_127) | g_1_129;
    wire logic p_8_130;
    wire logic g_8_130;
    assign p_8_130 = p_2_130 & p_7_127;
    assign g_8_130 = (p_2_130 & g_7_127) | g_2_130;
    
    // LF postprocess 
    assign OUT[0] = p_0[0];
    assign OUT[1] = p_0[1] ^ g_0[0];
    assign OUT[2] = p_0[2] ^ g_1_1;
    assign OUT[3] = p_0[3] ^ g_2_2;
    assign OUT[4] = p_0[4] ^ g_2_3;
    assign OUT[5] = p_0[5] ^ g_3_4;
    assign OUT[6] = p_0[6] ^ g_3_5;
    assign OUT[7] = p_0[7] ^ g_3_6;
    assign OUT[8] = p_0[8] ^ g_3_7;
    assign OUT[9] = p_0[9] ^ g_4_8;
    assign OUT[10] = p_0[10] ^ g_4_9;
    assign OUT[11] = p_0[11] ^ g_4_10;
    assign OUT[12] = p_0[12] ^ g_4_11;
    assign OUT[13] = p_0[13] ^ g_4_12;
    assign OUT[14] = p_0[14] ^ g_4_13;
    assign OUT[15] = p_0[15] ^ g_4_14;
    assign OUT[16] = p_0[16] ^ g_4_15;
    assign OUT[17] = p_0[17] ^ g_5_16;
    assign OUT[18] = p_0[18] ^ g_5_17;
    assign OUT[19] = p_0[19] ^ g_5_18;
    assign OUT[20] = p_0[20] ^ g_5_19;
    assign OUT[21] = p_0[21] ^ g_5_20;
    assign OUT[22] = p_0[22] ^ g_5_21;
    assign OUT[23] = p_0[23] ^ g_5_22;
    assign OUT[24] = p_0[24] ^ g_5_23;
    assign OUT[25] = p_0[25] ^ g_5_24;
    assign OUT[26] = p_0[26] ^ g_5_25;
    assign OUT[27] = p_0[27] ^ g_5_26;
    assign OUT[28] = p_0[28] ^ g_5_27;
    assign OUT[29] = p_0[29] ^ g_5_28;
    assign OUT[30] = p_0[30] ^ g_5_29;
    assign OUT[31] = p_0[31] ^ g_5_30;
    assign OUT[32] = p_0[32] ^ g_5_31;
    assign OUT[33] = p_0[33] ^ g_6_32;
    assign OUT[34] = p_0[34] ^ g_6_33;
    assign OUT[35] = p_0[35] ^ g_6_34;
    assign OUT[36] = p_0[36] ^ g_6_35;
    assign OUT[37] = p_0[37] ^ g_6_36;
    assign OUT[38] = p_0[38] ^ g_6_37;
    assign OUT[39] = p_0[39] ^ g_6_38;
    assign OUT[40] = p_0[40] ^ g_6_39;
    assign OUT[41] = p_0[41] ^ g_6_40;
    assign OUT[42] = p_0[42] ^ g_6_41;
    assign OUT[43] = p_0[43] ^ g_6_42;
    assign OUT[44] = p_0[44] ^ g_6_43;
    assign OUT[45] = p_0[45] ^ g_6_44;
    assign OUT[46] = p_0[46] ^ g_6_45;
    assign OUT[47] = p_0[47] ^ g_6_46;
    assign OUT[48] = p_0[48] ^ g_6_47;
    assign OUT[49] = p_0[49] ^ g_6_48;
    assign OUT[50] = p_0[50] ^ g_6_49;
    assign OUT[51] = p_0[51] ^ g_6_50;
    assign OUT[52] = p_0[52] ^ g_6_51;
    assign OUT[53] = p_0[53] ^ g_6_52;
    assign OUT[54] = p_0[54] ^ g_6_53;
    assign OUT[55] = p_0[55] ^ g_6_54;
    assign OUT[56] = p_0[56] ^ g_6_55;
    assign OUT[57] = p_0[57] ^ g_6_56;
    assign OUT[58] = p_0[58] ^ g_6_57;
    assign OUT[59] = p_0[59] ^ g_6_58;
    assign OUT[60] = p_0[60] ^ g_6_59;
    assign OUT[61] = p_0[61] ^ g_6_60;
    assign OUT[62] = p_0[62] ^ g_6_61;
    assign OUT[63] = p_0[63] ^ g_6_62;
    assign OUT[64] = p_0[64] ^ g_6_63;
    assign OUT[65] = p_0[65] ^ g_7_64;
    assign OUT[66] = p_0[66] ^ g_7_65;
    assign OUT[67] = p_0[67] ^ g_7_66;
    assign OUT[68] = p_0[68] ^ g_7_67;
    assign OUT[69] = p_0[69] ^ g_7_68;
    assign OUT[70] = p_0[70] ^ g_7_69;
    assign OUT[71] = p_0[71] ^ g_7_70;
    assign OUT[72] = p_0[72] ^ g_7_71;
    assign OUT[73] = p_0[73] ^ g_7_72;
    assign OUT[74] = p_0[74] ^ g_7_73;
    assign OUT[75] = p_0[75] ^ g_7_74;
    assign OUT[76] = p_0[76] ^ g_7_75;
    assign OUT[77] = p_0[77] ^ g_7_76;
    assign OUT[78] = p_0[78] ^ g_7_77;
    assign OUT[79] = p_0[79] ^ g_7_78;
    assign OUT[80] = p_0[80] ^ g_7_79;
    assign OUT[81] = p_0[81] ^ g_7_80;
    assign OUT[82] = p_0[82] ^ g_7_81;
    assign OUT[83] = p_0[83] ^ g_7_82;
    assign OUT[84] = p_0[84] ^ g_7_83;
    assign OUT[85] = p_0[85] ^ g_7_84;
    assign OUT[86] = p_0[86] ^ g_7_85;
    assign OUT[87] = p_0[87] ^ g_7_86;
    assign OUT[88] = p_0[88] ^ g_7_87;
    assign OUT[89] = p_0[89] ^ g_7_88;
    assign OUT[90] = p_0[90] ^ g_7_89;
    assign OUT[91] = p_0[91] ^ g_7_90;
    assign OUT[92] = p_0[92] ^ g_7_91;
    assign OUT[93] = p_0[93] ^ g_7_92;
    assign OUT[94] = p_0[94] ^ g_7_93;
    assign OUT[95] = p_0[95] ^ g_7_94;
    assign OUT[96] = p_0[96] ^ g_7_95;
    assign OUT[97] = p_0[97] ^ g_7_96;
    assign OUT[98] = p_0[98] ^ g_7_97;
    assign OUT[99] = p_0[99] ^ g_7_98;
    assign OUT[100] = p_0[100] ^ g_7_99;
    assign OUT[101] = p_0[101] ^ g_7_100;
    assign OUT[102] = p_0[102] ^ g_7_101;
    assign OUT[103] = p_0[103] ^ g_7_102;
    assign OUT[104] = p_0[104] ^ g_7_103;
    assign OUT[105] = p_0[105] ^ g_7_104;
    assign OUT[106] = p_0[106] ^ g_7_105;
    assign OUT[107] = p_0[107] ^ g_7_106;
    assign OUT[108] = p_0[108] ^ g_7_107;
    assign OUT[109] = p_0[109] ^ g_7_108;
    assign OUT[110] = p_0[110] ^ g_7_109;
    assign OUT[111] = p_0[111] ^ g_7_110;
    assign OUT[112] = p_0[112] ^ g_7_111;
    assign OUT[113] = p_0[113] ^ g_7_112;
    assign OUT[114] = p_0[114] ^ g_7_113;
    assign OUT[115] = p_0[115] ^ g_7_114;
    assign OUT[116] = p_0[116] ^ g_7_115;
    assign OUT[117] = p_0[117] ^ g_7_116;
    assign OUT[118] = p_0[118] ^ g_7_117;
    assign OUT[119] = p_0[119] ^ g_7_118;
    assign OUT[120] = p_0[120] ^ g_7_119;
    assign OUT[121] = p_0[121] ^ g_7_120;
    assign OUT[122] = p_0[122] ^ g_7_121;
    assign OUT[123] = p_0[123] ^ g_7_122;
    assign OUT[124] = p_0[124] ^ g_7_123;
    assign OUT[125] = p_0[125] ^ g_7_124;
    assign OUT[126] = p_0[126] ^ g_7_125;
    assign OUT[127] = p_0[127] ^ g_7_126;
    assign OUT[128] = p_0[128] ^ g_7_127;
    assign OUT[129] = p_0[129] ^ g_8_128;
    assign OUT[130] = p_0[130] ^ g_8_129;
    assign OUT[131] = g_8_130;
endmodule


// SUMMATION TREE TO MERGE THE RESULT OF DOT PRODUCT
module DOT_Merger_Signed_4_33x33_129_128(
        input logic [65:0]   m0_0,
        input logic [65:0]   m0_1,
        input logic [65:0]   m1_0,
        input logic [65:0]   m1_1,
        input logic [65:0]   m2_0,
        input logic [65:0]   m2_1,
        input logic [65:0]   m3_0,
        input logic [65:0]   m3_1,
        input logic [128:0]  IN3,
        output logic [127:0] result0,
	output logic [127:0] result1
					 );
    wire logic const1;
    assign const_1 = 1'b1;
    
    // Dadda Summation Stage 1
    logic s0 ,c0; 
    fa fa0 (IN3[0], m0_0[0], m0_1[0], s0, c0);
    logic s1 ,c1;
    ha ha1 (m1_0[0], m1_1[0], s1, c1);
    logic s2 ,c2; 
    fa fa2 (IN3[1], m0_0[1], m0_1[1], s2, c2);
    logic s3 ,c3; 
    fa fa3 (m1_0[1], m1_1[1], m2_0[1], s3, c3);
    logic s4 ,c4;
    ha ha4 (m2_1[1], m3_0[1], s4, c4);
    logic s5 ,c5; 
    fa fa5 (IN3[2], m0_0[2], m0_1[2], s5, c5);
    logic s6 ,c6; 
    fa fa6 (m1_0[2], m1_1[2], m2_0[2], s6, c6);
    logic s7 ,c7; 
    fa fa7 (m2_1[2], m3_0[2], m3_1[2], s7, c7);
    logic s8 ,c8; 
    fa fa8 (IN3[3], m0_0[3], m0_1[3], s8, c8);
    logic s9 ,c9; 
    fa fa9 (m1_0[3], m1_1[3], m2_0[3], s9, c9);
    logic s10 ,c10; 
    fa fa10 (m2_1[3], m3_0[3], m3_1[3], s10, c10);
    logic s11 ,c11; 
    fa fa11 (IN3[4], m0_0[4], m0_1[4], s11, c11);
    logic s12 ,c12; 
    fa fa12 (m1_0[4], m1_1[4], m2_0[4], s12, c12);
    logic s13 ,c13; 
    fa fa13 (m2_1[4], m3_0[4], m3_1[4], s13, c13);
    logic s14 ,c14; 
    fa fa14 (IN3[5], m0_0[5], m0_1[5], s14, c14);
    logic s15 ,c15; 
    fa fa15 (m1_0[5], m1_1[5], m2_0[5], s15, c15);
    logic s16 ,c16; 
    fa fa16 (m2_1[5], m3_0[5], m3_1[5], s16, c16);
    logic s17 ,c17; 
    fa fa17 (IN3[6], m0_0[6], m0_1[6], s17, c17);
    logic s18 ,c18; 
    fa fa18 (m1_0[6], m1_1[6], m2_0[6], s18, c18);
    logic s19 ,c19; 
    fa fa19 (m2_1[6], m3_0[6], m3_1[6], s19, c19);
    logic s20 ,c20; 
    fa fa20 (IN3[7], m0_0[7], m0_1[7], s20, c20);
    logic s21 ,c21; 
    fa fa21 (m1_0[7], m1_1[7], m2_0[7], s21, c21);
    logic s22 ,c22; 
    fa fa22 (m2_1[7], m3_0[7], m3_1[7], s22, c22);
    logic s23 ,c23; 
    fa fa23 (IN3[8], m0_0[8], m0_1[8], s23, c23);
    logic s24 ,c24; 
    fa fa24 (m1_0[8], m1_1[8], m2_0[8], s24, c24);
    logic s25 ,c25; 
    fa fa25 (m2_1[8], m3_0[8], m3_1[8], s25, c25);
    logic s26 ,c26; 
    fa fa26 (IN3[9], m0_0[9], m0_1[9], s26, c26);
    logic s27 ,c27; 
    fa fa27 (m1_0[9], m1_1[9], m2_0[9], s27, c27);
    logic s28 ,c28; 
    fa fa28 (m2_1[9], m3_0[9], m3_1[9], s28, c28);
    logic s29 ,c29; 
    fa fa29 (IN3[10], m0_0[10], m0_1[10], s29, c29);
    logic s30 ,c30; 
    fa fa30 (m1_0[10], m1_1[10], m2_0[10], s30, c30);
    logic s31 ,c31; 
    fa fa31 (m2_1[10], m3_0[10], m3_1[10], s31, c31);
    logic s32 ,c32; 
    fa fa32 (IN3[11], m0_0[11], m0_1[11], s32, c32);
    logic s33 ,c33; 
    fa fa33 (m1_0[11], m1_1[11], m2_0[11], s33, c33);
    logic s34 ,c34; 
    fa fa34 (m2_1[11], m3_0[11], m3_1[11], s34, c34);
    logic s35 ,c35; 
    fa fa35 (IN3[12], m0_0[12], m0_1[12], s35, c35);
    logic s36 ,c36; 
    fa fa36 (m1_0[12], m1_1[12], m2_0[12], s36, c36);
    logic s37 ,c37; 
    fa fa37 (m2_1[12], m3_0[12], m3_1[12], s37, c37);
    logic s38 ,c38; 
    fa fa38 (IN3[13], m0_0[13], m0_1[13], s38, c38);
    logic s39 ,c39; 
    fa fa39 (m1_0[13], m1_1[13], m2_0[13], s39, c39);
    logic s40 ,c40; 
    fa fa40 (m2_1[13], m3_0[13], m3_1[13], s40, c40);
    logic s41 ,c41; 
    fa fa41 (IN3[14], m0_0[14], m0_1[14], s41, c41);
    logic s42 ,c42; 
    fa fa42 (m1_0[14], m1_1[14], m2_0[14], s42, c42);
    logic s43 ,c43; 
    fa fa43 (m2_1[14], m3_0[14], m3_1[14], s43, c43);
    logic s44 ,c44; 
    fa fa44 (IN3[15], m0_0[15], m0_1[15], s44, c44);
    logic s45 ,c45; 
    fa fa45 (m1_0[15], m1_1[15], m2_0[15], s45, c45);
    logic s46 ,c46; 
    fa fa46 (m2_1[15], m3_0[15], m3_1[15], s46, c46);
    logic s47 ,c47; 
    fa fa47 (IN3[16], m0_0[16], m0_1[16], s47, c47);
    logic s48 ,c48; 
    fa fa48 (m1_0[16], m1_1[16], m2_0[16], s48, c48);
    logic s49 ,c49; 
    fa fa49 (m2_1[16], m3_0[16], m3_1[16], s49, c49);
    logic s50 ,c50; 
    fa fa50 (IN3[17], m0_0[17], m0_1[17], s50, c50);
    logic s51 ,c51; 
    fa fa51 (m1_0[17], m1_1[17], m2_0[17], s51, c51);
    logic s52 ,c52; 
    fa fa52 (m2_1[17], m3_0[17], m3_1[17], s52, c52);
    logic s53 ,c53; 
    fa fa53 (IN3[18], m0_0[18], m0_1[18], s53, c53);
    logic s54 ,c54; 
    fa fa54 (m1_0[18], m1_1[18], m2_0[18], s54, c54);
    logic s55 ,c55; 
    fa fa55 (m2_1[18], m3_0[18], m3_1[18], s55, c55);
    logic s56 ,c56; 
    fa fa56 (IN3[19], m0_0[19], m0_1[19], s56, c56);
    logic s57 ,c57; 
    fa fa57 (m1_0[19], m1_1[19], m2_0[19], s57, c57);
    logic s58 ,c58; 
    fa fa58 (m2_1[19], m3_0[19], m3_1[19], s58, c58);
    logic s59 ,c59; 
    fa fa59 (IN3[20], m0_0[20], m0_1[20], s59, c59);
    logic s60 ,c60; 
    fa fa60 (m1_0[20], m1_1[20], m2_0[20], s60, c60);
    logic s61 ,c61; 
    fa fa61 (m2_1[20], m3_0[20], m3_1[20], s61, c61);
    logic s62 ,c62; 
    fa fa62 (IN3[21], m0_0[21], m0_1[21], s62, c62);
    logic s63 ,c63; 
    fa fa63 (m1_0[21], m1_1[21], m2_0[21], s63, c63);
    logic s64 ,c64; 
    fa fa64 (m2_1[21], m3_0[21], m3_1[21], s64, c64);
    logic s65 ,c65; 
    fa fa65 (IN3[22], m0_0[22], m0_1[22], s65, c65);
    logic s66 ,c66; 
    fa fa66 (m1_0[22], m1_1[22], m2_0[22], s66, c66);
    logic s67 ,c67; 
    fa fa67 (m2_1[22], m3_0[22], m3_1[22], s67, c67);
    logic s68 ,c68; 
    fa fa68 (IN3[23], m0_0[23], m0_1[23], s68, c68);
    logic s69 ,c69; 
    fa fa69 (m1_0[23], m1_1[23], m2_0[23], s69, c69);
    logic s70 ,c70; 
    fa fa70 (m2_1[23], m3_0[23], m3_1[23], s70, c70);
    logic s71 ,c71; 
    fa fa71 (IN3[24], m0_0[24], m0_1[24], s71, c71);
    logic s72 ,c72; 
    fa fa72 (m1_0[24], m1_1[24], m2_0[24], s72, c72);
    logic s73 ,c73; 
    fa fa73 (m2_1[24], m3_0[24], m3_1[24], s73, c73);
    logic s74 ,c74; 
    fa fa74 (IN3[25], m0_0[25], m0_1[25], s74, c74);
    logic s75 ,c75; 
    fa fa75 (m1_0[25], m1_1[25], m2_0[25], s75, c75);
    logic s76 ,c76; 
    fa fa76 (m2_1[25], m3_0[25], m3_1[25], s76, c76);
    logic s77 ,c77; 
    fa fa77 (IN3[26], m0_0[26], m0_1[26], s77, c77);
    logic s78 ,c78; 
    fa fa78 (m1_0[26], m1_1[26], m2_0[26], s78, c78);
    logic s79 ,c79; 
    fa fa79 (m2_1[26], m3_0[26], m3_1[26], s79, c79);
    logic s80 ,c80; 
    fa fa80 (IN3[27], m0_0[27], m0_1[27], s80, c80);
    logic s81 ,c81; 
    fa fa81 (m1_0[27], m1_1[27], m2_0[27], s81, c81);
    logic s82 ,c82; 
    fa fa82 (m2_1[27], m3_0[27], m3_1[27], s82, c82);
    logic s83 ,c83; 
    fa fa83 (IN3[28], m0_0[28], m0_1[28], s83, c83);
    logic s84 ,c84; 
    fa fa84 (m1_0[28], m1_1[28], m2_0[28], s84, c84);
    logic s85 ,c85; 
    fa fa85 (m2_1[28], m3_0[28], m3_1[28], s85, c85);
    logic s86 ,c86; 
    fa fa86 (IN3[29], m0_0[29], m0_1[29], s86, c86);
    logic s87 ,c87; 
    fa fa87 (m1_0[29], m1_1[29], m2_0[29], s87, c87);
    logic s88 ,c88; 
    fa fa88 (m2_1[29], m3_0[29], m3_1[29], s88, c88);
    logic s89 ,c89; 
    fa fa89 (IN3[30], m0_0[30], m0_1[30], s89, c89);
    logic s90 ,c90; 
    fa fa90 (m1_0[30], m1_1[30], m2_0[30], s90, c90);
    logic s91 ,c91; 
    fa fa91 (m2_1[30], m3_0[30], m3_1[30], s91, c91);
    logic s92 ,c92; 
    fa fa92 (IN3[31], m0_0[31], m0_1[31], s92, c92);
    logic s93 ,c93; 
    fa fa93 (m1_0[31], m1_1[31], m2_0[31], s93, c93);
    logic s94 ,c94; 
    fa fa94 (m2_1[31], m3_0[31], m3_1[31], s94, c94);
    logic s95 ,c95; 
    fa fa95 (IN3[32], m0_0[32], m0_1[32], s95, c95);
    logic s96 ,c96; 
    fa fa96 (m1_0[32], m1_1[32], m2_0[32], s96, c96);
    logic s97 ,c97; 
    fa fa97 (m2_1[32], m3_0[32], m3_1[32], s97, c97);
    logic s98 ,c98; 
    fa fa98 (IN3[33], m0_0[33], m0_1[33], s98, c98);
    logic s99 ,c99; 
    fa fa99 (m1_0[33], m1_1[33], m2_0[33], s99, c99);
    logic s100 ,c100; 
    fa fa100 (m2_1[33], m3_0[33], m3_1[33], s100, c100);
    logic s101 ,c101; 
    fa fa101 (IN3[34], m0_0[34], m0_1[34], s101, c101);
    logic s102 ,c102; 
    fa fa102 (m1_0[34], m1_1[34], m2_0[34], s102, c102);
    logic s103 ,c103; 
    fa fa103 (m2_1[34], m3_0[34], m3_1[34], s103, c103);
    logic s104 ,c104; 
    fa fa104 (IN3[35], m0_0[35], m0_1[35], s104, c104);
    logic s105 ,c105; 
    fa fa105 (m1_0[35], m1_1[35], m2_0[35], s105, c105);
    logic s106 ,c106; 
    fa fa106 (m2_1[35], m3_0[35], m3_1[35], s106, c106);
    logic s107 ,c107; 
    fa fa107 (IN3[36], m0_0[36], m0_1[36], s107, c107);
    logic s108 ,c108; 
    fa fa108 (m1_0[36], m1_1[36], m2_0[36], s108, c108);
    logic s109 ,c109; 
    fa fa109 (m2_1[36], m3_0[36], m3_1[36], s109, c109);
    logic s110 ,c110; 
    fa fa110 (IN3[37], m0_0[37], m0_1[37], s110, c110);
    logic s111 ,c111; 
    fa fa111 (m1_0[37], m1_1[37], m2_0[37], s111, c111);
    logic s112 ,c112; 
    fa fa112 (m2_1[37], m3_0[37], m3_1[37], s112, c112);
    logic s113 ,c113; 
    fa fa113 (IN3[38], m0_0[38], m0_1[38], s113, c113);
    logic s114 ,c114; 
    fa fa114 (m1_0[38], m1_1[38], m2_0[38], s114, c114);
    logic s115 ,c115; 
    fa fa115 (m2_1[38], m3_0[38], m3_1[38], s115, c115);
    logic s116 ,c116; 
    fa fa116 (IN3[39], m0_0[39], m0_1[39], s116, c116);
    logic s117 ,c117; 
    fa fa117 (m1_0[39], m1_1[39], m2_0[39], s117, c117);
    logic s118 ,c118; 
    fa fa118 (m2_1[39], m3_0[39], m3_1[39], s118, c118);
    logic s119 ,c119; 
    fa fa119 (IN3[40], m0_0[40], m0_1[40], s119, c119);
    logic s120 ,c120; 
    fa fa120 (m1_0[40], m1_1[40], m2_0[40], s120, c120);
    logic s121 ,c121; 
    fa fa121 (m2_1[40], m3_0[40], m3_1[40], s121, c121);
    logic s122 ,c122; 
    fa fa122 (IN3[41], m0_0[41], m0_1[41], s122, c122);
    logic s123 ,c123; 
    fa fa123 (m1_0[41], m1_1[41], m2_0[41], s123, c123);
    logic s124 ,c124; 
    fa fa124 (m2_1[41], m3_0[41], m3_1[41], s124, c124);
    logic s125 ,c125; 
    fa fa125 (IN3[42], m0_0[42], m0_1[42], s125, c125);
    logic s126 ,c126; 
    fa fa126 (m1_0[42], m1_1[42], m2_0[42], s126, c126);
    logic s127 ,c127; 
    fa fa127 (m2_1[42], m3_0[42], m3_1[42], s127, c127);
    logic s128 ,c128; 
    fa fa128 (IN3[43], m0_0[43], m0_1[43], s128, c128);
    logic s129 ,c129; 
    fa fa129 (m1_0[43], m1_1[43], m2_0[43], s129, c129);
    logic s130 ,c130; 
    fa fa130 (m2_1[43], m3_0[43], m3_1[43], s130, c130);
    logic s131 ,c131; 
    fa fa131 (IN3[44], m0_0[44], m0_1[44], s131, c131);
    logic s132 ,c132; 
    fa fa132 (m1_0[44], m1_1[44], m2_0[44], s132, c132);
    logic s133 ,c133; 
    fa fa133 (m2_1[44], m3_0[44], m3_1[44], s133, c133);
    logic s134 ,c134; 
    fa fa134 (IN3[45], m0_0[45], m0_1[45], s134, c134);
    logic s135 ,c135; 
    fa fa135 (m1_0[45], m1_1[45], m2_0[45], s135, c135);
    logic s136 ,c136; 
    fa fa136 (m2_1[45], m3_0[45], m3_1[45], s136, c136);
    logic s137 ,c137; 
    fa fa137 (IN3[46], m0_0[46], m0_1[46], s137, c137);
    logic s138 ,c138; 
    fa fa138 (m1_0[46], m1_1[46], m2_0[46], s138, c138);
    logic s139 ,c139; 
    fa fa139 (m2_1[46], m3_0[46], m3_1[46], s139, c139);
    logic s140 ,c140; 
    fa fa140 (IN3[47], m0_0[47], m0_1[47], s140, c140);
    logic s141 ,c141; 
    fa fa141 (m1_0[47], m1_1[47], m2_0[47], s141, c141);
    logic s142 ,c142; 
    fa fa142 (m2_1[47], m3_0[47], m3_1[47], s142, c142);
    logic s143 ,c143; 
    fa fa143 (IN3[48], m0_0[48], m0_1[48], s143, c143);
    logic s144 ,c144; 
    fa fa144 (m1_0[48], m1_1[48], m2_0[48], s144, c144);
    logic s145 ,c145; 
    fa fa145 (m2_1[48], m3_0[48], m3_1[48], s145, c145);
    logic s146 ,c146; 
    fa fa146 (IN3[49], m0_0[49], m0_1[49], s146, c146);
    logic s147 ,c147; 
    fa fa147 (m1_0[49], m1_1[49], m2_0[49], s147, c147);
    logic s148 ,c148; 
    fa fa148 (m2_1[49], m3_0[49], m3_1[49], s148, c148);
    logic s149 ,c149; 
    fa fa149 (IN3[50], m0_0[50], m0_1[50], s149, c149);
    logic s150 ,c150; 
    fa fa150 (m1_0[50], m1_1[50], m2_0[50], s150, c150);
    logic s151 ,c151; 
    fa fa151 (m2_1[50], m3_0[50], m3_1[50], s151, c151);
    logic s152 ,c152; 
    fa fa152 (IN3[51], m0_0[51], m0_1[51], s152, c152);
    logic s153 ,c153; 
    fa fa153 (m1_0[51], m1_1[51], m2_0[51], s153, c153);
    logic s154 ,c154; 
    fa fa154 (m2_1[51], m3_0[51], m3_1[51], s154, c154);
    logic s155 ,c155; 
    fa fa155 (IN3[52], m0_0[52], m0_1[52], s155, c155);
    logic s156 ,c156; 
    fa fa156 (m1_0[52], m1_1[52], m2_0[52], s156, c156);
    logic s157 ,c157; 
    fa fa157 (m2_1[52], m3_0[52], m3_1[52], s157, c157);
    logic s158 ,c158; 
    fa fa158 (IN3[53], m0_0[53], m0_1[53], s158, c158);
    logic s159 ,c159; 
    fa fa159 (m1_0[53], m1_1[53], m2_0[53], s159, c159);
    logic s160 ,c160; 
    fa fa160 (m2_1[53], m3_0[53], m3_1[53], s160, c160);
    logic s161 ,c161; 
    fa fa161 (IN3[54], m0_0[54], m0_1[54], s161, c161);
    logic s162 ,c162; 
    fa fa162 (m1_0[54], m1_1[54], m2_0[54], s162, c162);
    logic s163 ,c163; 
    fa fa163 (m2_1[54], m3_0[54], m3_1[54], s163, c163);
    logic s164 ,c164; 
    fa fa164 (IN3[55], m0_0[55], m0_1[55], s164, c164);
    logic s165 ,c165; 
    fa fa165 (m1_0[55], m1_1[55], m2_0[55], s165, c165);
    logic s166 ,c166; 
    fa fa166 (m2_1[55], m3_0[55], m3_1[55], s166, c166);
    logic s167 ,c167; 
    fa fa167 (IN3[56], m0_0[56], m0_1[56], s167, c167);
    logic s168 ,c168; 
    fa fa168 (m1_0[56], m1_1[56], m2_0[56], s168, c168);
    logic s169 ,c169; 
    fa fa169 (m2_1[56], m3_0[56], m3_1[56], s169, c169);
    logic s170 ,c170; 
    fa fa170 (IN3[57], m0_0[57], m0_1[57], s170, c170);
    logic s171 ,c171; 
    fa fa171 (m1_0[57], m1_1[57], m2_0[57], s171, c171);
    logic s172 ,c172; 
    fa fa172 (m2_1[57], m3_0[57], m3_1[57], s172, c172);
    logic s173 ,c173; 
    fa fa173 (IN3[58], m0_0[58], m0_1[58], s173, c173);
    logic s174 ,c174; 
    fa fa174 (m1_0[58], m1_1[58], m2_0[58], s174, c174);
    logic s175 ,c175; 
    fa fa175 (m2_1[58], m3_0[58], m3_1[58], s175, c175);
    logic s176 ,c176; 
    fa fa176 (IN3[59], m0_0[59], m0_1[59], s176, c176);
    logic s177 ,c177; 
    fa fa177 (m1_0[59], m1_1[59], m2_0[59], s177, c177);
    logic s178 ,c178; 
    fa fa178 (m2_1[59], m3_0[59], m3_1[59], s178, c178);
    logic s179 ,c179; 
    fa fa179 (IN3[60], m0_0[60], m0_1[60], s179, c179);
    logic s180 ,c180; 
    fa fa180 (m1_0[60], m1_1[60], m2_0[60], s180, c180);
    logic s181 ,c181; 
    fa fa181 (m2_1[60], m3_0[60], m3_1[60], s181, c181);
    logic s182 ,c182; 
    fa fa182 (IN3[61], m0_0[61], m0_1[61], s182, c182);
    logic s183 ,c183; 
    fa fa183 (m1_0[61], m1_1[61], m2_0[61], s183, c183);
    logic s184 ,c184; 
    fa fa184 (m2_1[61], m3_0[61], m3_1[61], s184, c184);
    logic s185 ,c185; 
    fa fa185 (IN3[62], m0_0[62], m0_1[62], s185, c185);
    logic s186 ,c186; 
    fa fa186 (m1_0[62], m1_1[62], m2_0[62], s186, c186);
    logic s187 ,c187; 
    fa fa187 (m2_1[62], m3_0[62], m3_1[62], s187, c187);
    logic s188 ,c188; 
    fa fa188 (IN3[63], m0_0[63], m0_1[63], s188, c188);
    logic s189 ,c189; 
    fa fa189 (m1_0[63], m1_1[63], m2_0[63], s189, c189);
    logic s190 ,c190; 
    fa fa190 (m2_1[63], m3_0[63], m3_1[63], s190, c190);
    logic s191 ,c191; 
    fa fa191 (IN3[64], m0_0[64], m0_1[64], s191, c191);
    logic s192 ,c192; 
    fa fa192 (m1_0[64], m1_1[64], m2_0[64], s192, c192);
    logic s193 ,c193; 
    fa fa193 (m2_1[64], m3_0[64], m3_1[64], s193, c193);
    logic s194 ,c194; 
    fa fa194 (IN3[65], m0_0[65], m0_1[65], s194, c194);
    logic s195 ,c195; 
    fa fa195 (m1_0[65], m1_1[65], m2_0[65], s195, c195);
    logic s196 ,c196; 
    fa fa196 (m2_1[65], m3_0[65], m3_1[65], s196, c196);
    
    // Dadda Summation Stage 2
    logic s197 ,c197; 
    fa fa197 (m2_0[0], m2_1[0], m3_0[0], s197, c197);
    logic s198 ,c198; 
    fa fa198 (m3_1[1], c0, c1, s198, c198);
    logic s199 ,c199;
    ha ha199 (s2, s3, s199, c199);
    logic s200 ,c200; 
    fa fa200 (c2, c3, c4, s200, c200);
    logic s201 ,c201; 
    fa fa201 (s5, s6, s7, s201, c201);
    logic s202 ,c202; 
    fa fa202 (c5, c6, c7, s202, c202);
    logic s203 ,c203; 
    fa fa203 (s8, s9, s10, s203, c203);
    logic s204 ,c204; 
    fa fa204 (c8, c9, c10, s204, c204);
    logic s205 ,c205; 
    fa fa205 (s11, s12, s13, s205, c205);
    logic s206 ,c206; 
    fa fa206 (c11, c12, c13, s206, c206);
    logic s207 ,c207; 
    fa fa207 (s14, s15, s16, s207, c207);
    logic s208 ,c208; 
    fa fa208 (c14, c15, c16, s208, c208);
    logic s209 ,c209; 
    fa fa209 (s17, s18, s19, s209, c209);
    logic s210 ,c210; 
    fa fa210 (c17, c18, c19, s210, c210);
    logic s211 ,c211; 
    fa fa211 (s20, s21, s22, s211, c211);
    logic s212 ,c212; 
    fa fa212 (c20, c21, c22, s212, c212);
    logic s213 ,c213; 
    fa fa213 (s23, s24, s25, s213, c213);
    logic s214 ,c214; 
    fa fa214 (c23, c24, c25, s214, c214);
    logic s215 ,c215; 
    fa fa215 (s26, s27, s28, s215, c215);
    logic s216 ,c216; 
    fa fa216 (c26, c27, c28, s216, c216);
    logic s217 ,c217; 
    fa fa217 (s29, s30, s31, s217, c217);
    logic s218 ,c218; 
    fa fa218 (c29, c30, c31, s218, c218);
    logic s219 ,c219; 
    fa fa219 (s32, s33, s34, s219, c219);
    logic s220 ,c220; 
    fa fa220 (c32, c33, c34, s220, c220);
    logic s221 ,c221; 
    fa fa221 (s35, s36, s37, s221, c221);
    logic s222 ,c222; 
    fa fa222 (c35, c36, c37, s222, c222);
    logic s223 ,c223; 
    fa fa223 (s38, s39, s40, s223, c223);
    logic s224 ,c224; 
    fa fa224 (c38, c39, c40, s224, c224);
    logic s225 ,c225; 
    fa fa225 (s41, s42, s43, s225, c225);
    logic s226 ,c226; 
    fa fa226 (c41, c42, c43, s226, c226);
    logic s227 ,c227; 
    fa fa227 (s44, s45, s46, s227, c227);
    logic s228 ,c228; 
    fa fa228 (c44, c45, c46, s228, c228);
    logic s229 ,c229; 
    fa fa229 (s47, s48, s49, s229, c229);
    logic s230 ,c230; 
    fa fa230 (c47, c48, c49, s230, c230);
    logic s231 ,c231; 
    fa fa231 (s50, s51, s52, s231, c231);
    logic s232 ,c232; 
    fa fa232 (c50, c51, c52, s232, c232);
    logic s233 ,c233; 
    fa fa233 (s53, s54, s55, s233, c233);
    logic s234 ,c234; 
    fa fa234 (c53, c54, c55, s234, c234);
    logic s235 ,c235; 
    fa fa235 (s56, s57, s58, s235, c235);
    logic s236 ,c236; 
    fa fa236 (c56, c57, c58, s236, c236);
    logic s237 ,c237; 
    fa fa237 (s59, s60, s61, s237, c237);
    logic s238 ,c238; 
    fa fa238 (c59, c60, c61, s238, c238);
    logic s239 ,c239; 
    fa fa239 (s62, s63, s64, s239, c239);
    logic s240 ,c240; 
    fa fa240 (c62, c63, c64, s240, c240);
    logic s241 ,c241; 
    fa fa241 (s65, s66, s67, s241, c241);
    logic s242 ,c242; 
    fa fa242 (c65, c66, c67, s242, c242);
    logic s243 ,c243; 
    fa fa243 (s68, s69, s70, s243, c243);
    logic s244 ,c244; 
    fa fa244 (c68, c69, c70, s244, c244);
    logic s245 ,c245; 
    fa fa245 (s71, s72, s73, s245, c245);
    logic s246 ,c246; 
    fa fa246 (c71, c72, c73, s246, c246);
    logic s247 ,c247; 
    fa fa247 (s74, s75, s76, s247, c247);
    logic s248 ,c248; 
    fa fa248 (c74, c75, c76, s248, c248);
    logic s249 ,c249; 
    fa fa249 (s77, s78, s79, s249, c249);
    logic s250 ,c250; 
    fa fa250 (c77, c78, c79, s250, c250);
    logic s251 ,c251; 
    fa fa251 (s80, s81, s82, s251, c251);
    logic s252 ,c252; 
    fa fa252 (c80, c81, c82, s252, c252);
    logic s253 ,c253; 
    fa fa253 (s83, s84, s85, s253, c253);
    logic s254 ,c254; 
    fa fa254 (c83, c84, c85, s254, c254);
    logic s255 ,c255; 
    fa fa255 (s86, s87, s88, s255, c255);
    logic s256 ,c256; 
    fa fa256 (c86, c87, c88, s256, c256);
    logic s257 ,c257; 
    fa fa257 (s89, s90, s91, s257, c257);
    logic s258 ,c258; 
    fa fa258 (c89, c90, c91, s258, c258);
    logic s259 ,c259; 
    fa fa259 (s92, s93, s94, s259, c259);
    logic s260 ,c260; 
    fa fa260 (c92, c93, c94, s260, c260);
    logic s261 ,c261; 
    fa fa261 (s95, s96, s97, s261, c261);
    logic s262 ,c262; 
    fa fa262 (c95, c96, c97, s262, c262);
    logic s263 ,c263; 
    fa fa263 (s98, s99, s100, s263, c263);
    logic s264 ,c264; 
    fa fa264 (c98, c99, c100, s264, c264);
    logic s265 ,c265; 
    fa fa265 (s101, s102, s103, s265, c265);
    logic s266 ,c266; 
    fa fa266 (c101, c102, c103, s266, c266);
    logic s267 ,c267; 
    fa fa267 (s104, s105, s106, s267, c267);
    logic s268 ,c268; 
    fa fa268 (c104, c105, c106, s268, c268);
    logic s269 ,c269; 
    fa fa269 (s107, s108, s109, s269, c269);
    logic s270 ,c270; 
    fa fa270 (c107, c108, c109, s270, c270);
    logic s271 ,c271; 
    fa fa271 (s110, s111, s112, s271, c271);
    logic s272 ,c272; 
    fa fa272 (c110, c111, c112, s272, c272);
    logic s273 ,c273; 
    fa fa273 (s113, s114, s115, s273, c273);
    logic s274 ,c274; 
    fa fa274 (c113, c114, c115, s274, c274);
    logic s275 ,c275; 
    fa fa275 (s116, s117, s118, s275, c275);
    logic s276 ,c276; 
    fa fa276 (c116, c117, c118, s276, c276);
    logic s277 ,c277; 
    fa fa277 (s119, s120, s121, s277, c277);
    logic s278 ,c278; 
    fa fa278 (c119, c120, c121, s278, c278);
    logic s279 ,c279; 
    fa fa279 (s122, s123, s124, s279, c279);
    logic s280 ,c280; 
    fa fa280 (c122, c123, c124, s280, c280);
    logic s281 ,c281; 
    fa fa281 (s125, s126, s127, s281, c281);
    logic s282 ,c282; 
    fa fa282 (c125, c126, c127, s282, c282);
    logic s283 ,c283; 
    fa fa283 (s128, s129, s130, s283, c283);
    logic s284 ,c284; 
    fa fa284 (c128, c129, c130, s284, c284);
    logic s285 ,c285; 
    fa fa285 (s131, s132, s133, s285, c285);
    logic s286 ,c286; 
    fa fa286 (c131, c132, c133, s286, c286);
    logic s287 ,c287; 
    fa fa287 (s134, s135, s136, s287, c287);
    logic s288 ,c288; 
    fa fa288 (c134, c135, c136, s288, c288);
    logic s289 ,c289; 
    fa fa289 (s137, s138, s139, s289, c289);
    logic s290 ,c290; 
    fa fa290 (c137, c138, c139, s290, c290);
    logic s291 ,c291; 
    fa fa291 (s140, s141, s142, s291, c291);
    logic s292 ,c292; 
    fa fa292 (c140, c141, c142, s292, c292);
    logic s293 ,c293; 
    fa fa293 (s143, s144, s145, s293, c293);
    logic s294 ,c294; 
    fa fa294 (c143, c144, c145, s294, c294);
    logic s295 ,c295; 
    fa fa295 (s146, s147, s148, s295, c295);
    logic s296 ,c296; 
    fa fa296 (c146, c147, c148, s296, c296);
    logic s297 ,c297; 
    fa fa297 (s149, s150, s151, s297, c297);
    logic s298 ,c298; 
    fa fa298 (c149, c150, c151, s298, c298);
    logic s299 ,c299; 
    fa fa299 (s152, s153, s154, s299, c299);
    logic s300 ,c300; 
    fa fa300 (c152, c153, c154, s300, c300);
    logic s301 ,c301; 
    fa fa301 (s155, s156, s157, s301, c301);
    logic s302 ,c302; 
    fa fa302 (c155, c156, c157, s302, c302);
    logic s303 ,c303; 
    fa fa303 (s158, s159, s160, s303, c303);
    logic s304 ,c304; 
    fa fa304 (c158, c159, c160, s304, c304);
    logic s305 ,c305; 
    fa fa305 (s161, s162, s163, s305, c305);
    logic s306 ,c306; 
    fa fa306 (c161, c162, c163, s306, c306);
    logic s307 ,c307; 
    fa fa307 (s164, s165, s166, s307, c307);
    logic s308 ,c308; 
    fa fa308 (c164, c165, c166, s308, c308);
    logic s309 ,c309; 
    fa fa309 (s167, s168, s169, s309, c309);
    logic s310 ,c310; 
    fa fa310 (c167, c168, c169, s310, c310);
    logic s311 ,c311; 
    fa fa311 (s170, s171, s172, s311, c311);
    logic s312 ,c312; 
    fa fa312 (c170, c171, c172, s312, c312);
    logic s313 ,c313; 
    fa fa313 (s173, s174, s175, s313, c313);
    logic s314 ,c314; 
    fa fa314 (c173, c174, c175, s314, c314);
    logic s315 ,c315; 
    fa fa315 (s176, s177, s178, s315, c315);
    logic s316 ,c316; 
    fa fa316 (c176, c177, c178, s316, c316);
    logic s317 ,c317; 
    fa fa317 (s179, s180, s181, s317, c317);
    logic s318 ,c318; 
    fa fa318 (c179, c180, c181, s318, c318);
    logic s319 ,c319; 
    fa fa319 (s182, s183, s184, s319, c319);
    logic s320 ,c320; 
    fa fa320 (c182, c183, c184, s320, c320);
    logic s321 ,c321; 
    fa fa321 (s185, s186, s187, s321, c321);
    logic s322 ,c322; 
    fa fa322 (c185, c186, c187, s322, c322);
    logic s323 ,c323; 
    fa fa323 (s188, s189, s190, s323, c323);
    logic s324 ,c324; 
    fa fa324 (c188, c189, c190, s324, c324);
    logic s325 ,c325; 
    fa fa325 (s191, s192, s193, s325, c325);
    logic s326 ,c326; 
    fa fa326 (c191, c192, c193, s326, c326);
    logic s327 ,c327; 
    fa fa327 (s194, s195, s196, s327, c327);
    logic s328 ,c328; 
    fa fa328 (IN3[66], c194, c195, s328, c328);
    
    // Dadda Summation Stage 3
    logic s329 ,c329;
    ha ha329 (m3_1[0], s0, s329, c329);
    logic s330 ,c330; 
    fa fa330 (s4, c197, s198, s330, c330);
    logic s331 ,c331; 
    fa fa331 (c198, c199, s200, s331, c331);
    logic s332 ,c332; 
    fa fa332 (c200, c201, s202, s332, c332);
    logic s333 ,c333; 
    fa fa333 (c202, c203, s204, s333, c333);
    logic s334 ,c334; 
    fa fa334 (c204, c205, s206, s334, c334);
    logic s335 ,c335; 
    fa fa335 (c206, c207, s208, s335, c335);
    logic s336 ,c336; 
    fa fa336 (c208, c209, s210, s336, c336);
    logic s337 ,c337; 
    fa fa337 (c210, c211, s212, s337, c337);
    logic s338 ,c338; 
    fa fa338 (c212, c213, s214, s338, c338);
    logic s339 ,c339; 
    fa fa339 (c214, c215, s216, s339, c339);
    logic s340 ,c340; 
    fa fa340 (c216, c217, s218, s340, c340);
    logic s341 ,c341; 
    fa fa341 (c218, c219, s220, s341, c341);
    logic s342 ,c342; 
    fa fa342 (c220, c221, s222, s342, c342);
    logic s343 ,c343; 
    fa fa343 (c222, c223, s224, s343, c343);
    logic s344 ,c344; 
    fa fa344 (c224, c225, s226, s344, c344);
    logic s345 ,c345; 
    fa fa345 (c226, c227, s228, s345, c345);
    logic s346 ,c346; 
    fa fa346 (c228, c229, s230, s346, c346);
    logic s347 ,c347; 
    fa fa347 (c230, c231, s232, s347, c347);
    logic s348 ,c348; 
    fa fa348 (c232, c233, s234, s348, c348);
    logic s349 ,c349; 
    fa fa349 (c234, c235, s236, s349, c349);
    logic s350 ,c350; 
    fa fa350 (c236, c237, s238, s350, c350);
    logic s351 ,c351; 
    fa fa351 (c238, c239, s240, s351, c351);
    logic s352 ,c352; 
    fa fa352 (c240, c241, s242, s352, c352);
    logic s353 ,c353; 
    fa fa353 (c242, c243, s244, s353, c353);
    logic s354 ,c354; 
    fa fa354 (c244, c245, s246, s354, c354);
    logic s355 ,c355; 
    fa fa355 (c246, c247, s248, s355, c355);
    logic s356 ,c356; 
    fa fa356 (c248, c249, s250, s356, c356);
    logic s357 ,c357; 
    fa fa357 (c250, c251, s252, s357, c357);
    logic s358 ,c358; 
    fa fa358 (c252, c253, s254, s358, c358);
    logic s359 ,c359; 
    fa fa359 (c254, c255, s256, s359, c359);
    logic s360 ,c360; 
    fa fa360 (c256, c257, s258, s360, c360);
    logic s361 ,c361; 
    fa fa361 (c258, c259, s260, s361, c361);
    logic s362 ,c362; 
    fa fa362 (c260, c261, s262, s362, c362);
    logic s363 ,c363; 
    fa fa363 (c262, c263, s264, s363, c363);
    logic s364 ,c364; 
    fa fa364 (c264, c265, s266, s364, c364);
    logic s365 ,c365; 
    fa fa365 (c266, c267, s268, s365, c365);
    logic s366 ,c366; 
    fa fa366 (c268, c269, s270, s366, c366);
    logic s367 ,c367; 
    fa fa367 (c270, c271, s272, s367, c367);
    logic s368 ,c368; 
    fa fa368 (c272, c273, s274, s368, c368);
    logic s369 ,c369; 
    fa fa369 (c274, c275, s276, s369, c369);
    logic s370 ,c370; 
    fa fa370 (c276, c277, s278, s370, c370);
    logic s371 ,c371; 
    fa fa371 (c278, c279, s280, s371, c371);
    logic s372 ,c372; 
    fa fa372 (c280, c281, s282, s372, c372);
    logic s373 ,c373; 
    fa fa373 (c282, c283, s284, s373, c373);
    logic s374 ,c374; 
    fa fa374 (c284, c285, s286, s374, c374);
    logic s375 ,c375; 
    fa fa375 (c286, c287, s288, s375, c375);
    logic s376 ,c376; 
    fa fa376 (c288, c289, s290, s376, c376);
    logic s377 ,c377; 
    fa fa377 (c290, c291, s292, s377, c377);
    logic s378 ,c378; 
    fa fa378 (c292, c293, s294, s378, c378);
    logic s379 ,c379; 
    fa fa379 (c294, c295, s296, s379, c379);
    logic s380 ,c380; 
    fa fa380 (c296, c297, s298, s380, c380);
    logic s381 ,c381; 
    fa fa381 (c298, c299, s300, s381, c381);
    logic s382 ,c382; 
    fa fa382 (c300, c301, s302, s382, c382);
    logic s383 ,c383; 
    fa fa383 (c302, c303, s304, s383, c383);
    logic s384 ,c384; 
    fa fa384 (c304, c305, s306, s384, c384);
    logic s385 ,c385; 
    fa fa385 (c306, c307, s308, s385, c385);
    logic s386 ,c386; 
    fa fa386 (c308, c309, s310, s386, c386);
    logic s387 ,c387; 
    fa fa387 (c310, c311, s312, s387, c387);
    logic s388 ,c388; 
    fa fa388 (c312, c313, s314, s388, c388);
    logic s389 ,c389; 
    fa fa389 (c314, c315, s316, s389, c389);
    logic s390 ,c390; 
    fa fa390 (c316, c317, s318, s390, c390);
    logic s391 ,c391; 
    fa fa391 (c318, c319, s320, s391, c391);
    logic s392 ,c392; 
    fa fa392 (c320, c321, s322, s392, c392);
    logic s393 ,c393; 
    fa fa393 (c322, c323, s324, s393, c393);
    logic s394 ,c394; 
    fa fa394 (c324, c325, s326, s394, c394);
    logic s395 ,c395; 
    fa fa395 (c196, c326, c327, s395, c395);
    
    // Dadda Summation Stage 4
    logic s396 ,c396;
    ha ha396 (s1, s197, s396, c396);
    logic s397 ,c397; 
    fa fa397 (s199, c329, s330, s397, c397);
    logic s398 ,c398; 
    fa fa398 (s201, c330, s331, s398, c398);
    logic s399 ,c399; 
    fa fa399 (s203, c331, s332, s399, c399);
    logic s400 ,c400; 
    fa fa400 (s205, c332, s333, s400, c400);
    logic s401 ,c401; 
    fa fa401 (s207, c333, s334, s401, c401);
    logic s402 ,c402; 
    fa fa402 (s209, c334, s335, s402, c402);
    logic s403 ,c403; 
    fa fa403 (s211, c335, s336, s403, c403);
    logic s404 ,c404; 
    fa fa404 (s213, c336, s337, s404, c404);
    logic s405 ,c405; 
    fa fa405 (s215, c337, s338, s405, c405);
    logic s406 ,c406; 
    fa fa406 (s217, c338, s339, s406, c406);
    logic s407 ,c407; 
    fa fa407 (s219, c339, s340, s407, c407);
    logic s408 ,c408; 
    fa fa408 (s221, c340, s341, s408, c408);
    logic s409 ,c409; 
    fa fa409 (s223, c341, s342, s409, c409);
    logic s410 ,c410; 
    fa fa410 (s225, c342, s343, s410, c410);
    logic s411 ,c411; 
    fa fa411 (s227, c343, s344, s411, c411);
    logic s412 ,c412; 
    fa fa412 (s229, c344, s345, s412, c412);
    logic s413 ,c413; 
    fa fa413 (s231, c345, s346, s413, c413);
    logic s414 ,c414; 
    fa fa414 (s233, c346, s347, s414, c414);
    logic s415 ,c415; 
    fa fa415 (s235, c347, s348, s415, c415);
    logic s416 ,c416; 
    fa fa416 (s237, c348, s349, s416, c416);
    logic s417 ,c417; 
    fa fa417 (s239, c349, s350, s417, c417);
    logic s418 ,c418; 
    fa fa418 (s241, c350, s351, s418, c418);
    logic s419 ,c419; 
    fa fa419 (s243, c351, s352, s419, c419);
    logic s420 ,c420; 
    fa fa420 (s245, c352, s353, s420, c420);
    logic s421 ,c421; 
    fa fa421 (s247, c353, s354, s421, c421);
    logic s422 ,c422; 
    fa fa422 (s249, c354, s355, s422, c422);
    logic s423 ,c423; 
    fa fa423 (s251, c355, s356, s423, c423);
    logic s424 ,c424; 
    fa fa424 (s253, c356, s357, s424, c424);
    logic s425 ,c425; 
    fa fa425 (s255, c357, s358, s425, c425);
    logic s426 ,c426; 
    fa fa426 (s257, c358, s359, s426, c426);
    logic s427 ,c427; 
    fa fa427 (s259, c359, s360, s427, c427);
    logic s428 ,c428; 
    fa fa428 (s261, c360, s361, s428, c428);
    logic s429 ,c429; 
    fa fa429 (s263, c361, s362, s429, c429);
    logic s430 ,c430; 
    fa fa430 (s265, c362, s363, s430, c430);
    logic s431 ,c431; 
    fa fa431 (s267, c363, s364, s431, c431);
    logic s432 ,c432; 
    fa fa432 (s269, c364, s365, s432, c432);
    logic s433 ,c433; 
    fa fa433 (s271, c365, s366, s433, c433);
    logic s434 ,c434; 
    fa fa434 (s273, c366, s367, s434, c434);
    logic s435 ,c435; 
    fa fa435 (s275, c367, s368, s435, c435);
    logic s436 ,c436; 
    fa fa436 (s277, c368, s369, s436, c436);
    logic s437 ,c437; 
    fa fa437 (s279, c369, s370, s437, c437);
    logic s438 ,c438; 
    fa fa438 (s281, c370, s371, s438, c438);
    logic s439 ,c439; 
    fa fa439 (s283, c371, s372, s439, c439);
    logic s440 ,c440; 
    fa fa440 (s285, c372, s373, s440, c440);
    logic s441 ,c441; 
    fa fa441 (s287, c373, s374, s441, c441);
    logic s442 ,c442; 
    fa fa442 (s289, c374, s375, s442, c442);
    logic s443 ,c443; 
    fa fa443 (s291, c375, s376, s443, c443);
    logic s444 ,c444; 
    fa fa444 (s293, c376, s377, s444, c444);
    logic s445 ,c445; 
    fa fa445 (s295, c377, s378, s445, c445);
    logic s446 ,c446; 
    fa fa446 (s297, c378, s379, s446, c446);
    logic s447 ,c447; 
    fa fa447 (s299, c379, s380, s447, c447);
    logic s448 ,c448; 
    fa fa448 (s301, c380, s381, s448, c448);
    logic s449 ,c449; 
    fa fa449 (s303, c381, s382, s449, c449);
    logic s450 ,c450; 
    fa fa450 (s305, c382, s383, s450, c450);
    logic s451 ,c451; 
    fa fa451 (s307, c383, s384, s451, c451);
    logic s452 ,c452; 
    fa fa452 (s309, c384, s385, s452, c452);
    logic s453 ,c453; 
    fa fa453 (s311, c385, s386, s453, c453);
    logic s454 ,c454; 
    fa fa454 (s313, c386, s387, s454, c454);
    logic s455 ,c455; 
    fa fa455 (s315, c387, s388, s455, c455);
    logic s456 ,c456; 
    fa fa456 (s317, c388, s389, s456, c456);
    logic s457 ,c457; 
    fa fa457 (s319, c389, s390, s457, c457);
    logic s458 ,c458; 
    fa fa458 (s321, c390, s391, s458, c458);
    logic s459 ,c459; 
    fa fa459 (s323, c391, s392, s459, c459);
    logic s460 ,c460; 
    fa fa460 (s325, c392, s393, s460, c460);
    logic s461 ,c461; 
    fa fa461 (s327, c393, s394, s461, c461);
    logic s462 ,c462; 
    fa fa462 (s328, c394, s395, s462, c462);
    logic s463 ,c463; 
    fa fa463 (IN3[67], c328, c395, s463, c463);
    logic s464 ,c464;
    ha ha464 (IN3[68], const_1, s464, c464);
    logic s465 ,c465;
    ha ha465 (IN3[69], const_1, s465, c465);
    logic s466 ,c466;
    ha ha466 (IN3[70], const_1, s466, c466);
    logic s467 ,c467;
    ha ha467 (IN3[71], const_1, s467, c467);
    logic s468 ,c468;
    ha ha468 (IN3[72], const_1, s468, c468);
    logic s469 ,c469;
    ha ha469 (IN3[73], const_1, s469, c469);
    logic s470 ,c470;
    ha ha470 (IN3[74], const_1, s470, c470);
    logic s471 ,c471;
    ha ha471 (IN3[75], const_1, s471, c471);
    logic s472 ,c472;
    ha ha472 (IN3[76], const_1, s472, c472);
    logic s473 ,c473;
    ha ha473 (IN3[77], const_1, s473, c473);
    logic s474 ,c474;
    ha ha474 (IN3[78], const_1, s474, c474);
    logic s475 ,c475;
    ha ha475 (IN3[79], const_1, s475, c475);
    logic s476 ,c476;
    ha ha476 (IN3[80], const_1, s476, c476);
    logic s477 ,c477;
    ha ha477 (IN3[81], const_1, s477, c477);
    logic s478 ,c478;
    ha ha478 (IN3[82], const_1, s478, c478);
    logic s479 ,c479;
    ha ha479 (IN3[83], const_1, s479, c479);
    logic s480 ,c480;
    ha ha480 (IN3[84], const_1, s480, c480);
    logic s481 ,c481;
    ha ha481 (IN3[85], const_1, s481, c481);
    logic s482 ,c482;
    ha ha482 (IN3[86], const_1, s482, c482);
    logic s483 ,c483;
    ha ha483 (IN3[87], const_1, s483, c483);
    logic s484 ,c484;
    ha ha484 (IN3[88], const_1, s484, c484);
    logic s485 ,c485;
    ha ha485 (IN3[89], const_1, s485, c485);
    logic s486 ,c486;
    ha ha486 (IN3[90], const_1, s486, c486);
    logic s487 ,c487;
    ha ha487 (IN3[91], const_1, s487, c487);
    logic s488 ,c488;
    ha ha488 (IN3[92], const_1, s488, c488);
    logic s489 ,c489;
    ha ha489 (IN3[93], const_1, s489, c489);
    logic s490 ,c490;
    ha ha490 (IN3[94], const_1, s490, c490);
    logic s491 ,c491;
    ha ha491 (IN3[95], const_1, s491, c491);
    logic s492 ,c492;
    ha ha492 (IN3[96], const_1, s492, c492);
    logic s493 ,c493;
    ha ha493 (IN3[97], const_1, s493, c493);
    logic s494 ,c494;
    ha ha494 (IN3[98], const_1, s494, c494);
    logic s495 ,c495;
    ha ha495 (IN3[99], const_1, s495, c495);
    logic s496 ,c496;
    ha ha496 (IN3[100], const_1, s496, c496);
    logic s497 ,c497;
    ha ha497 (IN3[101], const_1, s497, c497);
    logic s498 ,c498;
    ha ha498 (IN3[102], const_1, s498, c498);
    logic s499 ,c499;
    ha ha499 (IN3[103], const_1, s499, c499);
    logic s500 ,c500;
    ha ha500 (IN3[104], const_1, s500, c500);
    logic s501 ,c501;
    ha ha501 (IN3[105], const_1, s501, c501);
    logic s502 ,c502;
    ha ha502 (IN3[106], const_1, s502, c502);
    logic s503 ,c503;
    ha ha503 (IN3[107], const_1, s503, c503);
    logic s504 ,c504;
    ha ha504 (IN3[108], const_1, s504, c504);
    logic s505 ,c505;
    ha ha505 (IN3[109], const_1, s505, c505);
    logic s506 ,c506;
    ha ha506 (IN3[110], const_1, s506, c506);
    logic s507 ,c507;
    ha ha507 (IN3[111], const_1, s507, c507);
    logic s508 ,c508;
    ha ha508 (IN3[112], const_1, s508, c508);
    logic s509 ,c509;
    ha ha509 (IN3[113], const_1, s509, c509);
    logic s510 ,c510;
    ha ha510 (IN3[114], const_1, s510, c510);
    logic s511 ,c511;
    ha ha511 (IN3[115], const_1, s511, c511);
    logic s512 ,c512;
    ha ha512 (IN3[116], const_1, s512, c512);
    logic s513 ,c513;
    ha ha513 (IN3[117], const_1, s513, c513);
    logic s514 ,c514;
    ha ha514 (IN3[118], const_1, s514, c514);
    logic s515 ,c515;
    ha ha515 (IN3[119], const_1, s515, c515);
    logic s516 ,c516;
    ha ha516 (IN3[120], const_1, s516, c516);
    logic s517 ,c517;
    ha ha517 (IN3[121], const_1, s517, c517);
    logic s518 ,c518;
    ha ha518 (IN3[122], const_1, s518, c518);
    logic s519 ,c519;
    ha ha519 (IN3[123], const_1, s519, c519);
    logic s520 ,c520;
    ha ha520 (IN3[124], const_1, s520, c520);
    logic s521 ,c521;
    ha ha521 (IN3[125], const_1, s521, c521);
    logic s522 ,c522;
    ha ha522 (IN3[126], const_1, s522, c522);
    logic s523 ,c523;
    ha ha523 (IN3[127], const_1, s523, c523);
   
   assign result0 = {c522, c521, c520, c519, c518, c517, c516, c515, c514, c513, c512, c511, c510, c509, c508, c507, c506, c505, c504, c503, c502, c501, c500, c499, c498, c497, c496, c495, c494, c493, c492, c491, c490, c489, c488, c487, c486, c485, c484, c483, c482, c481, c480, c479, c478, c477, c476, c475, c474, c473, c472, c471, c470, c469, c468, c467, c466, c465, c464, c463, c462, c461, c460, c459, c458, c457, c456, c455, c454, c453, c452, c451, c450, c449, c448, c447, c446, c445, c444, c443, c442, c441, c440, c439, c438, c437, c436, c435, c434, c433, c432, c431, c430, c429, c428, c427, c426, c425, c424, c423, c422, c421, c420, c419, c418, c417, c416, c415, c414, c413, c412, c411, c410, c409, c408, c407, c406, c405, c404, c403, c402, c401, c400, c399, c398, c397, c396, s329 };
   
   assign result1 = {s523, s522, s521, s520, s519, s518, s517, s516, s515, s514, s513, s512, s511, s510, s509, s508, s507, s506, s505, s504, s503, s502, s501, s500, s499, s498, s497, s496, s495, s494, s493, s492, s491, s490, s489, s488, s487, s486, s485, s484, s483, s482, s481, s480, s479, s478, s477, s476, s475, s474, s473, s472, s471, s470, s469, s468, s467, s466, s465, s464, s463, s462, s461, s460, s459, s458, s457, s456, s455, s454, s453, s452, s451, s450, s449, s448, s447, s446, s445, s444, s443, s442, s441, s440, s439, s438, s437, s436, s435, s434, s433, s432, s431, s430, s429, s428, s427, s426, s425, s424, s423, s422, s421, s420, s419, s418, s417, s416, s415, s414, s413, s412, s411, s410, s409, s408, s407, s406, s405, s404, s403, s402, s401, s400, s399, s398, s397, s396 };
   

endmodule // DOT_Merger_Signed_4_33x33_129_128


// SUMMATION REDUCTION TREE TO MERGE THE RESULT OF A SINGLE LANGE OF FOUR-LANES MULTIPLICATION
module FMA_Merger_Signed_33x33_64_64(
        input logic [63:0]  m1_0,
        input logic [63:0]  m1_1,
        input logic [63:0]  IN3,
        output logic [63:0] result0,
	output logic [63:0] result1);
    wire logic const1;
    assign const_1 = 1'b1;
    
    // The values to be summed in the summation tree:
     // m1_0[0]  m1_0[1]  m1_0[2]  m1_0[3]  m1_0[4]  m1_0[5]  m1_0[6]  m1_0[7]  m1_0[8]  m1_0[9]  m1_0[10] m1_0[11] m1_0[12] m1_0[13] m1_0[14] m1_0[15] m1_0[16] m1_0[17] m1_0[18] m1_0[19] m1_0[20] m1_0[21] m1_0[22] m1_0[23] m1_0[24] m1_0[25] m1_0[26] m1_0[27] m1_0[28] m1_0[29] m1_0[30] m1_0[31] m1_0[32] m1_0[33] m1_0[34] m1_0[35] m1_0[36] m1_0[37] m1_0[38] m1_0[39] m1_0[40] m1_0[41] m1_0[42] m1_0[43] m1_0[44] m1_0[45] m1_0[46] m1_0[47] m1_0[48] m1_0[49] m1_0[50] m1_0[51] m1_0[52] m1_0[53] m1_0[54] m1_0[55] m1_0[56] m1_0[57] m1_0[58] m1_0[59] m1_0[60] m1_0[61] m1_0[62] m1_0[63] 
     // m1_1[0]  m1_1[1]  m1_1[2]  m1_1[3]  m1_1[4]  m1_1[5]  m1_1[6]  m1_1[7]  m1_1[8]  m1_1[9]  m1_1[10] m1_1[11] m1_1[12] m1_1[13] m1_1[14] m1_1[15] m1_1[16] m1_1[17] m1_1[18] m1_1[19] m1_1[20] m1_1[21] m1_1[22] m1_1[23] m1_1[24] m1_1[25] m1_1[26] m1_1[27] m1_1[28] m1_1[29] m1_1[30] m1_1[31] m1_1[32] m1_1[33] m1_1[34] m1_1[35] m1_1[36] m1_1[37] m1_1[38] m1_1[39] m1_1[40] m1_1[41] m1_1[42] m1_1[43] m1_1[44] m1_1[45] m1_1[46] m1_1[47] m1_1[48] m1_1[49] m1_1[50] m1_1[51] m1_1[52] m1_1[53] m1_1[54] m1_1[55] m1_1[56] m1_1[57] m1_1[58] m1_1[59] m1_1[60] m1_1[61] m1_1[62] m1_1[63] 
     // IN3[0]   IN3[1]   IN3[2]   IN3[3]   IN3[4]   IN3[5]   IN3[6]   IN3[7]   IN3[8]   IN3[9]   IN3[10]  IN3[11]  IN3[12]  IN3[13]  IN3[14]  IN3[15]  IN3[16]  IN3[17]  IN3[18]  IN3[19]  IN3[20]  IN3[21]  IN3[22]  IN3[23]  IN3[24]  IN3[25]  IN3[26]  IN3[27]  IN3[28]  IN3[29]  IN3[30]  IN3[31]  IN3[32]  IN3[33]  IN3[34]  IN3[35]  IN3[36]  IN3[37]  IN3[38]  IN3[39]  IN3[40]  IN3[41]  IN3[42]  IN3[43]  IN3[44]  IN3[45]  IN3[46]  IN3[47]  IN3[48]  IN3[49]  IN3[50]  IN3[51]  IN3[52]  IN3[53]  IN3[54]  IN3[55]  IN3[56]  IN3[57]  IN3[58]  IN3[59]  IN3[60]  IN3[61]  IN3[62]  ~IN3[63] 
     //   --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --       --     const_1  
    
    // Dadda Summation Stage 1
    logic s0 ,c0;
    ha ha0 (m1_0[63], m1_1[63], s0, c0);
    
    // Dadda Summation Stage 2
    logic s1 ,c1;
    ha ha1 (m1_0[0], m1_1[0], s1, c1);
    logic s2 ,c2; 
    fa fa2 (m1_0[1], m1_1[1], IN3[1], s2, c2);
    logic s3 ,c3; 
    fa fa3 (m1_0[2], m1_1[2], IN3[2], s3, c3);
    logic s4 ,c4; 
    fa fa4 (m1_0[3], m1_1[3], IN3[3], s4, c4);
    logic s5 ,c5; 
    fa fa5 (m1_0[4], m1_1[4], IN3[4], s5, c5);
    logic s6 ,c6; 
    fa fa6 (m1_0[5], m1_1[5], IN3[5], s6, c6);
    logic s7 ,c7; 
    fa fa7 (m1_0[6], m1_1[6], IN3[6], s7, c7);
    logic s8 ,c8; 
    fa fa8 (m1_0[7], m1_1[7], IN3[7], s8, c8);
    logic s9 ,c9; 
    fa fa9 (m1_0[8], m1_1[8], IN3[8], s9, c9);
    logic s10 ,c10; 
    fa fa10 (m1_0[9], m1_1[9], IN3[9], s10, c10);
    logic s11 ,c11; 
    fa fa11 (m1_0[10], m1_1[10], IN3[10], s11, c11);
    logic s12 ,c12; 
    fa fa12 (m1_0[11], m1_1[11], IN3[11], s12, c12);
    logic s13 ,c13; 
    fa fa13 (m1_0[12], m1_1[12], IN3[12], s13, c13);
    logic s14 ,c14; 
    fa fa14 (m1_0[13], m1_1[13], IN3[13], s14, c14);
    logic s15 ,c15; 
    fa fa15 (m1_0[14], m1_1[14], IN3[14], s15, c15);
    logic s16 ,c16; 
    fa fa16 (m1_0[15], m1_1[15], IN3[15], s16, c16);
    logic s17 ,c17; 
    fa fa17 (m1_0[16], m1_1[16], IN3[16], s17, c17);
    logic s18 ,c18; 
    fa fa18 (m1_0[17], m1_1[17], IN3[17], s18, c18);
    logic s19 ,c19; 
    fa fa19 (m1_0[18], m1_1[18], IN3[18], s19, c19);
    logic s20 ,c20; 
    fa fa20 (m1_0[19], m1_1[19], IN3[19], s20, c20);
    logic s21 ,c21; 
    fa fa21 (m1_0[20], m1_1[20], IN3[20], s21, c21);
    logic s22 ,c22; 
    fa fa22 (m1_0[21], m1_1[21], IN3[21], s22, c22);
    logic s23 ,c23; 
    fa fa23 (m1_0[22], m1_1[22], IN3[22], s23, c23);
    logic s24 ,c24; 
    fa fa24 (m1_0[23], m1_1[23], IN3[23], s24, c24);
    logic s25 ,c25; 
    fa fa25 (m1_0[24], m1_1[24], IN3[24], s25, c25);
    logic s26 ,c26; 
    fa fa26 (m1_0[25], m1_1[25], IN3[25], s26, c26);
    logic s27 ,c27; 
    fa fa27 (m1_0[26], m1_1[26], IN3[26], s27, c27);
    logic s28 ,c28; 
    fa fa28 (m1_0[27], m1_1[27], IN3[27], s28, c28);
    logic s29 ,c29; 
    fa fa29 (m1_0[28], m1_1[28], IN3[28], s29, c29);
    logic s30 ,c30; 
    fa fa30 (m1_0[29], m1_1[29], IN3[29], s30, c30);
    logic s31 ,c31; 
    fa fa31 (m1_0[30], m1_1[30], IN3[30], s31, c31);
    logic s32 ,c32; 
    fa fa32 (m1_0[31], m1_1[31], IN3[31], s32, c32);
    logic s33 ,c33; 
    fa fa33 (m1_0[32], m1_1[32], IN3[32], s33, c33);
    logic s34 ,c34; 
    fa fa34 (m1_0[33], m1_1[33], IN3[33], s34, c34);
    logic s35 ,c35; 
    fa fa35 (m1_0[34], m1_1[34], IN3[34], s35, c35);
    logic s36 ,c36; 
    fa fa36 (m1_0[35], m1_1[35], IN3[35], s36, c36);
    logic s37 ,c37; 
    fa fa37 (m1_0[36], m1_1[36], IN3[36], s37, c37);
    logic s38 ,c38; 
    fa fa38 (m1_0[37], m1_1[37], IN3[37], s38, c38);
    logic s39 ,c39; 
    fa fa39 (m1_0[38], m1_1[38], IN3[38], s39, c39);
    logic s40 ,c40; 
    fa fa40 (m1_0[39], m1_1[39], IN3[39], s40, c40);
    logic s41 ,c41; 
    fa fa41 (m1_0[40], m1_1[40], IN3[40], s41, c41);
    logic s42 ,c42; 
    fa fa42 (m1_0[41], m1_1[41], IN3[41], s42, c42);
    logic s43 ,c43; 
    fa fa43 (m1_0[42], m1_1[42], IN3[42], s43, c43);
    logic s44 ,c44; 
    fa fa44 (m1_0[43], m1_1[43], IN3[43], s44, c44);
    logic s45 ,c45; 
    fa fa45 (m1_0[44], m1_1[44], IN3[44], s45, c45);
    logic s46 ,c46; 
    fa fa46 (m1_0[45], m1_1[45], IN3[45], s46, c46);
    logic s47 ,c47; 
    fa fa47 (m1_0[46], m1_1[46], IN3[46], s47, c47);
    logic s48 ,c48; 
    fa fa48 (m1_0[47], m1_1[47], IN3[47], s48, c48);
    logic s49 ,c49; 
    fa fa49 (m1_0[48], m1_1[48], IN3[48], s49, c49);
    logic s50 ,c50; 
    fa fa50 (m1_0[49], m1_1[49], IN3[49], s50, c50);
    logic s51 ,c51; 
    fa fa51 (m1_0[50], m1_1[50], IN3[50], s51, c51);
    logic s52 ,c52; 
    fa fa52 (m1_0[51], m1_1[51], IN3[51], s52, c52);
    logic s53 ,c53; 
    fa fa53 (m1_0[52], m1_1[52], IN3[52], s53, c53);
    logic s54 ,c54; 
    fa fa54 (m1_0[53], m1_1[53], IN3[53], s54, c54);
    logic s55 ,c55; 
    fa fa55 (m1_0[54], m1_1[54], IN3[54], s55, c55);
    logic s56 ,c56; 
    fa fa56 (m1_0[55], m1_1[55], IN3[55], s56, c56);
    logic s57 ,c57; 
    fa fa57 (m1_0[56], m1_1[56], IN3[56], s57, c57);
    logic s58 ,c58; 
    fa fa58 (m1_0[57], m1_1[57], IN3[57], s58, c58);
    logic s59 ,c59; 
    fa fa59 (m1_0[58], m1_1[58], IN3[58], s59, c59);
    logic s60 ,c60; 
    fa fa60 (m1_0[59], m1_1[59], IN3[59], s60, c60);
    logic s61 ,c61; 
    fa fa61 (m1_0[60], m1_1[60], IN3[60], s61, c61);
    logic s62 ,c62; 
    fa fa62 (m1_0[61], m1_1[61], IN3[61], s62, c62);
    logic s63 ,c63; 
    fa fa63 (m1_0[62], m1_1[62], IN3[62], s63, c63);
    logic s64 ,c64; 
    fa fa64 (~IN3[63], const_1, s0, s64, c64);
    
   assign result0 = {c63, c62, c61, c60, c59, c58, c57, c56, c55, c54, c53, c52, c51, c50, c49, c48, c47, c46, c45, c44, c43, c42, c41, c40, c39, c38, c37, c36, c35, c34, c33, c32, c31, c30, c29, c28, c27, c26, c25, c24, c23, c22, c21, c20, c19, c18, c17, c16, c15, c14, c13, c12, c11, c10, c9, c8, c7, c6, c5, c4, c3, c2, c1, IN3[0] };
   assign result1 = {s64, s63, s62, s61, s60, s59, s58, s57, s56, s55, s54, s53, s52, s51, s50, s49, s48, s47, s46, s45, s44, s43, s42, s41, s40, s39, s38, s37, s36, s35, s34, s33, s32, s31, s30, s29, s28, s27, s26, s25, s24, s23, s22, s21, s20, s19, s18, s17, s16, s15, s14, s13, s12, s11, s10, s9, s8, s7, s6, s5, s4, s3, s2, s1 };
   

endmodule // FMA_Merger_Signed_33x33_64_64
