; APT (Automated Program Transformations) Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "APT")

(include-book "kestrel/event-macros/xdoc-constructors" :dir :system)
(include-book "utilities/xdoc-constructors")

; (depends-on "design-notes/expdata.pdf")
; (depends-on "kestrel/design-notes/notation.pdf" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst *expdata-design-notes*
  (xdoc::&& "@('expdata') "
            (xdoc::ahref "res/kestrel-apt-design-notes/expdata.pdf"
                         "design notes")))

(defconst *isodata-design-notes*
  (xdoc::&& "@('isodata') "
            (xdoc::ahref "res/kestrel-apt-design-notes/isodata.pdf"
                         "design notes")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc expdata

  :parents (apt)

  :short "APT expanded data transformation:
          change function arguments and results
          into expanded representations."

  :long

  (xdoc::topstring

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   (xdoc::evmac-section-intro

    (xdoc::p
     "This transformation changes the representation of
      one or more of a function's arguments and results
      into an expanded representation
      in the sense that the same element in the old representation
      may correspond to multiple elements in the new representation.
      This is more general than @(tsee isodata),
      where old and new representation are isomorphic:
      in @('expdata'), there is a surjection
      from the new representation to the old representation.
      The @('expdata') transformation is useful
      to carry out certain data type refinements
      (when synthesizing programs),
      or perhaps also to raise the level of abstraction of certain types
      (when analyzing programs);
      however, its usefulness in program analysis
      is not as obvious as @(tsee isodata),
      given that isomorphisms are inherently invertible
      while mere surjections are not.")

    (xdoc::p
     "For a given (user-specified) surjective conversion
      from the new representation to the old representation
      there may be different possible right inverse conversions
      from the old representation to the new representation.
      Different such inverse conversions could be used
      in different parts of the new function generated by @('expdata').
      The current version of this transformation, for simplicity,
      use the same (user-specified) inverse conversion;
      future versions of @('expdata') may support different conversions,
      which may be necessary to make @('expdata') more practically useful.
      Thus, the current version of @('expdata')
      is very similar to @(tsee isodata),
      except that the conversions are not necessarily isomorphisms.")

    (xdoc::p
     "When at least one argument's representation is being changed,
      then by regarding the arguments whose representation is not changed
      as being changed via an indentity surjection,
      we can say that this transformation changes the representation of
      (the tuple of) all the function's arguments
      into a new representation that consists of
      (the tuple of) all the new function's arguments.
      In this case, there are two variants of this transformation:")
    (xdoc::ul
     (xdoc::li
      "When the function operates only on argument tuples
       in the old representation
       (i.e. when the function's guard is a subset of the old representation),
       the function is transformed to operate in the same way on
       exactly the argument tuples in the new representation
       that are in the surjection's preimage of the old guard.")
     (xdoc::li
      "When the function operates on
       at least all the tuples in the old representation (and possibly more)
       (i.e. the function's guard is a superset of the old representation),
       and is used as a predicate to recognize
       a subset of argument tuples all of which are in the old representation,
       the function is transformed to recognize
       exactly the argument tuples in the new representation
       that are in the surjection's preimage of
       the ones recognized by the old function."))
    (xdoc::p
     "These two variants involve slightly different applicability conditions
      and produce slightly different results.
      These two variants are selected
      via the @(':predicate') input (see below).")
    (xdoc::p
     "If only the representation of some results (and of no arguments)
      is changed, then there is a single variant of this transformation,
      namely one that operates on the same tuples as the old function
      but returns results in the new representation
      that are in the surjection's preimage.")

    (xdoc::p
     "The " *expdata-design-notes* ", which use "
     (xdoc::a :href "res/kestrel-design-notes/notation.pdf" "this notation")
     ", provide the mathematical concepts and template proofs
      upon which this transformation is based.
      These notes should be read alongside this reference documentation,
      which refers to the them in numerous places.
      Given that, as noted above,
      the current version of @('expdata') is very similar to @(tsee isodata),
      the @('expdata') design notes mostly refer to
      the " *isodata-design-notes* ";
      thus, the @(tsee isodata) design notes should be read as well."))

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   (xdoc::evmac-section-form
    (xdoc::codeblock
     "(expdata old"
     "         surjmaps"
     "         :predicate           ; default nil"
     "         :new-name            ; default :auto"
     "         :new-enable          ; default :auto"
     "         :old-to-new-name     ; default from table"
     "         :old-to-new-enable   ; default from table"
     "         :new-to-old-name     ; default from table"
     "         :new-to-old-enable   ; default from table"
     "         :newp-of-new-name    ; default :auto"
     "         :newp-of-new-enable  ; default t"
     "         :verify-guards       ; default :auto"
     "         :untranslate         ; default :nice"
     "         :hints               ; default nil"
     "         :print               ; default :result"
     "         :show-only           ; default nil"
     "  )"))

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   (xdoc::evmac-section-inputs

    (xdoc::desc-apt-input-old
     (xdoc::p
      "@('old') must
       be in logic mode,
       be " (xdoc::seetopic "acl2::function-definedness" "defined") ", and
       have no input or output "
      (xdoc::seetopic "acl2::stobj" "stobjs")
      ". If the @(':predicate') input (see below) is @('t'),
       then @('old') must return
       a non-" (xdoc::seetopic "mv" "multiple") " value.
       If @('old') is recursive, it must
       be singly (not mutually) recursive,
       not have a @(':?') measure, and
       not occur in its own "
      (xdoc::seetopic "tthm" "termination theorem")
      " (i.e. not occur in the tests and arguments of its own recursive calls).
       If the @(':verify-guards') input is @('t'),
       @('old') must be guard-verified.")
     (xdoc::p
      "In the rest of this documentation page:")
     (xdoc::ul
      (xdoc::li
       "Let @('x1'), ..., @('xn') be the arguments of @('old'),
        where @('n') &gt; 0.")
      (xdoc::li
       "Let @('m') &gt; 0 be the number of results of @('old').")
      (xdoc::li
       "Let @('old-guard<x1,...,xn>') be the guard term of @('old').")
      (xdoc::li
       "If @('old') is not recursive, let
        @({
          old-body<x1,...,xn>
        })
        be the body of @('old').")
      (xdoc::li
       "If @('old') is recursive, let
        @({
          old-body<x1,...,xn,
                   (old update1-x1<x1,...,xn>
                        ...
                        update1-xn<x1,...,xn>)
                   ...
                   (old updater-x1<x1,...,xn>
                        ...
                        updater-xn<x1,...,xn>)>
        })
        be the body of @('old'),
        where @('r') &gt; 0 is the number of recursive calls
        in the body of @('old')
        and each @('updatek-xi<x1,...,xn>') is
        the @('i')-th actual argument passed to the @('k')-th recursive call.
        Furthermore,
        let @('contextk<x1,...,xn>') be the context (i.e. controlling tests)
        in which the @('k')-th recursive call occurs,
        and let @('measure<x1,...,xn>') be the measure term of @('old')."))
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       @('old') is denoted by
       @($f$) when @(':predicate') is @('nil'),
       and @($p$) when @(':predicate') is @('t')."))

    (xdoc::desc
     "@('surjmaps')"
     (xdoc::p
      "Specifies the arguments and results of @('old') that are transformed
       and the way in which they are transformed.")
     (xdoc::p
      "It must be a non-empty list of doublets
       @('((arg/res-list1 surj) ... (arg/res-listq surjq))'),
       where:")
     (xdoc::ul
      (xdoc::li
       (xdoc::p
        "Each @('arg/res-listk') denotes
         the subset of the arguments and results of @('old')
         whose representation is transformed according to @('surjk').")
       (xdoc::p
        "It must be one of the following:")
       (xdoc::ul
        (xdoc::li
         "A non-empty list without duplicates of elements among
          @('x1'), ... @('xn'), @(':result1'), ..., @(':resultm'),
          in any order.")
        (xdoc::li
         "A single element among
          @('x1'), ... @('xn'), @(':result1'), ..., @(':resultm'),
          abbreviating the singleton list with that element.")
        (xdoc::li
         "The single element @(':result'),
          abbreviating the singleton list @(':result1').
          This is allowed only if @('m') is 1.")))
      (xdoc::li
       (xdoc::p
        "Each @('surjk') denotes the surjective mapping
         to apply to the arguments and results in @('arg/res-listk').
         Each @('surjk') specifies
         old and new representations
         and the surjection between them,
         along with a right inverse of the surjection.")
       (xdoc::p
        "It must be one of the following:")
       (xdoc::ul
        (xdoc::li
         "A symbol that references
          a previous successful call of @(tsee defsurj),
          i.e. the symbol must be the @('name') input of that call.
          The domains and conversions recorded under that name specify:
          the recognizer of the old representation (@('domb')),
          which we call @('oldp') here;
          the recognizer of the new representation (@('doma')),
          which we call @('newp') here;
          the surjective conversion
          from the new to the old representation (@('alpha')),
          which we call @('back') here; and
          the right inverse conversion
          from the old to the new representation (@('beta')),
          which we call @('forth') here.
          Both @('oldp') and @('newp') must be unary.
          If the generated function is guard-verified
          (which is determined by the @(':verify-guards') input; see below),
          the call of @(tsee defsurj)
          must have @(':guard-thms') equal to @('t'),
          i.e. it must have proved and recorded the guard-related theorems.")
        (xdoc::li
         "A list @('(newp oldp back forth :hints hints)')
          such that the call
          @('(defsurj name newp oldp back forth
               :guard-thms guard-thms :thm-names thm-names :hints hints)')
          is successful,
          where @('name') and @('thm-names') consist of suitably fresh names,
          and where @('guard-thms') is
          @('t') if the generated function is guard-verified
          (which is determined by the @(':verify-guards') input; see below)
          and @('nil') otherwise.
          A list @('(newp oldp back forth)')
          abbreviates @('(newp oldp back forth :hints nil)').
          The @('expdata') transformation generates
          this call of @(tsee defsurj),
          and uses it in the same way as it uses a call referenced by @('surj')
          when @('surj') is a symbol;
          however, this generated @(tsee defsurj) call
          is local to the @(tsee encapsulate) generated by @('expdata'),
          and cannot be therefore referenced
          after the call of @('expdata').")))
      (xdoc::li
       (xdoc::p
        "The lists @('arg/res-list1'), ..., @('arg/res-listq')
         are pairwise disjoint,
         i.e. each @('xi') and @(':result') appears
         in at most one of those lists.")))
     (xdoc::p
      "In the rest of this documentation page,
       for each @('i') from 1 to @('n'),
       let @('oldpi'), @('newpi'), @('forthi'), and @('backi') be:")
     (xdoc::ul
      (xdoc::li
       "The @('oldp'), @('newp'), @('forth'), and @('back')
        of the (pre-existing or locally generated) @(tsee defsurj)
        specified by @('surjk'),
        if @('xi') is in @('arg/res-listk').")
      (xdoc::li
       "The functions
        @('(lambda (x) t)'),
        @('(lambda (x) t)'),
        @('(lambda (x) x)'), and
        @('(lambda (x) x)')
        that form the identity surjective mapping over all values,
        if @('xi') is not in any @('arg/res-listk')."))
     (xdoc::p
      "Furthermore,
       for each @('j') from 1 to @('m'),
       let @('oldp_rj'), @('newp_rj'), @('forth_rj'), and @('back_rj') be:")
     (xdoc::ul
      (xdoc::li
       "The @('oldp'), @('newp'), @('forth'), and @('back')
        of the (pre-existing or locally generated) @(tsee defsurj)
        specified by @('surjk'),
        if @(':resultj') is in @('arg/res-listk').")
      (xdoc::li
       "The functions
        @('(lambda (x) t)'),
        @('(lambda (x) t)'),
        @('(lambda (x) x)'), and
        @('(lambda (x) x)')
        that form the identity surjective mapping over all values,
        if @(':resultj') is not in any @('arg/res-listk')."))
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       the section
       `Compositional Establishment of Isomorphic Mappings on Tuples'
       describes the compositional establishment of an isomorphic mapping
       between the inputs of old and new function;
       the same concepts apply to surjective mappings.
       The @('surjmaps') input currently supported by @('expdata')
       amounts to the following partitioning and sub-mappings:")
     (xdoc::ul
      (xdoc::li
       "The new function's arguments are the same (i.e. have the same names)
        as the old function's arguments, i.e. @('x1'), ..., @('xn').")
      (xdoc::li
       "The new function has the same number of results as the old function.")
      (xdoc::li
       "The arguments are partitioned into @('n') singleton partitions.")
      (xdoc::li
       "The results are partitioned into @('m') singleton partitions.")
      (xdoc::li
       "The surjective mapping consisting of
        @('oldpi'), @('newpi'), @('forthi'), and @('backi')
        is used for the partition consisting of argument @('xi').")
      (xdoc::li
       "The surjective mapping consisting of
        @('oldp_rj'), @('newp_rj'), @('forth_rj'), and @('back_rj')
        is used for the partition consisting of result @('j').")
      (xdoc::li
       "The identity surjective mapping
        is used for the partitions of all the results,
        if @(':result') is not in any @('arg/res-listk')."))
     (xdoc::p
      "In the " *expdata-design-notes* ",
       the resulting surjecive mapping over all function arguments
       is denoted as consisting of
       the domains @($A$) and @($A'$) and
       the conversions @($\\alpha$) and @($\\alpha'$),
       and the resulting surjective mapping over all function results
       is denoted as consisting of
       the domains @($B$) and @($B'$) and
       the conversions @($\\beta$) and @($\\beta'$).
       The design notes allow these domains and conversions
       to be specified via (the design notes counterparts of)
       either @(tsee definj) or @(tsee defsurj)
       (in opposite directions);
       for the currently implemented @('expdata'),
       only @(tsee defsurj) may be used."))

    (xdoc::desc
     "@(':predicate') &mdash; default @('nil')"
     (xdoc::p
      "Selects between the two variants of this transformation:")
     (xdoc::ul
      (xdoc::li
       "@('t'), to select the variant in which @('old')
        is treated like a predicate that recognizes
        argument tuples that are all in the old representation.")
      (xdoc::li
       "@('nil'), to select the variant in which @('old')
        is treated as a function that operates
        only on argument tuples that are all in the old representation."))
     (xdoc::p
      "This input may be @('t') only if @('surjmaps')
       does not include @(':result').")
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       the sections with `Function' in their title
       refer to the case in which @(':predicate') is @('nil'),
       while the sections with `Predicate' in their title
       refer to the case in which @(':predicate') is @('t')."))

    (xdoc::desc-apt-input-new-name)

    (xdoc::desc-apt-input-new-enable)

    (xdoc::desc-apt-input-old-to-new-name)

    (xdoc::desc-apt-input-old-to-new-enable)

    (xdoc::desc-apt-input-new-to-old-name)

    (xdoc::desc-apt-input-new-to-old-enable)

    (xdoc::desc
     "@(':newp-of-new-name') &mdash; default @(':auto')"
     (xdoc::p
      "Determines the name of the theorem asserting that @('new') maps
       arguments in the new representation
       to results in the new representation.")
     (xdoc::p
      "It must be one of the following:")
     (xdoc::ul
      (xdoc::li
       "@(':auto'), to use the concatenation of
        the name of @('new') followed by @('-new-representation'),
        in the same package as @('new').")
      (xdoc::li
       "Any other symbol, to use as the name of the theorem."))
     (xdoc::p
      "This input may be present only if
       @('surjmaps') includes some @(':resultj').")
     (xdoc::p
      "In the rest of this documentation page,
       let @('newp-of-new') be the name of this theorem."))

    (xdoc::desc
     "@(':newp-of-new-enable') &mdash; default @('t')"
     (xdoc::p
      "Determines whether @('newp-of-new') is enabled.")
     (xdoc::p
      "It must be one of the following:")
     (xdoc::ul
      (xdoc::li
       "@('t'), to enable the theorem.")
      (xdoc::li
       "@('nil'), to disable the theorem."))
     (xdoc::p
      "This input may be present only if
       @('surjmaps') includes some @(':resultj')."))

    (xdoc::desc-apt-input-verify-guards :plural-functions nil)

    (xdoc::desc-apt-input-untranslate)

    (xdoc::evmac-input-hints)

    (xdoc::evmac-input-print expdata)

    (xdoc::evmac-input-show-only expdata))

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   (xdoc::evmac-section-appconds

    expdata

    (xdoc::p
     "The following conditions must be proved
      in order for the transformation to apply.")

    (xdoc::evmac-appcond
     ":oldp-of-old"
     (xdoc::&&
      (xdoc::p
       "@('old') maps arguments in the old representation
        to results in the old representation:")
      (xdoc::codeblock
       ";; when m = 1:"
       "(implies (and (oldp1 x1)"
       "              ..."
       "              (oldpn xn))"
       "         (oldp_r1 (old x1 ... xn)))"
       ""
       ";; when m > 1:"
       "(implies (and (oldp1 x1)"
       "              ..."
       "              (oldpn xn))"
       "         (mv-let (y1 ... ym)"
       "           (old x1 ... xn)"
       "           (and (oldp_r1 y1)"
       "                ..."
       "                (oldp_rm ym))))"))
     :design-notes (xdoc::&& *isodata-design-notes*
                             ", referenced by the "
                             *expdata-design-notes*
                             ",")
     :design-notes-appcond "@($fAB$)"
     :presence "@('surjmaps') includes some @(':resultj')")

    (xdoc::evmac-appcond
     ":oldp-when-old"
     (xdoc::&&
      (xdoc::p
       "@('old') holds only on argument tuples
        such that @('x1'), ..., @('xn') are all in the old representation:")
      (xdoc::codeblock
       "(implies (old x1 ... xn)"
       "         (and (oldp1 x1)"
       "              ..."
       "              (oldpn xn)))"))
     :design-notes (xdoc::&& *isodata-design-notes*
                             ", referenced by the "
                             *expdata-design-notes*
                             ",")
     :design-notes-appcond "@($pA$)"
     :presence "@(':predicate') is @('t')")

    (xdoc::evmac-appcond
     ":oldp-of-rec-call-args"
     (xdoc::&&
      (xdoc::p
       "The old representation is preserved on
        the arguments of the recursive calls of @('old'):")
      (xdoc::codeblock
       "(implies (and (oldp1 x1)"
       "              ..."
       "              (oldpn xn))"
       "         (and (implies context1<x1,...,xn>"
       "                       (and (oldp1 update1-x1<x1,...,xn>)"
       "                            ..."
       "                            (oldpn update1-xn<x1,...,xn>)))"
       "              ..."
       "              (implies contextr<x1,...,xn>"
       "                       (and (oldp1 updater-x1<x1,...,xn>)"
       "                            ..."
       "                            (oldpn updater-xp<x1,...,xn>)))))"))
     :design-notes (xdoc::&& *isodata-design-notes*
                             ", referenced by the "
                             *expdata-design-notes*
                             ",")
     :design-notes-appcond "@($Ad$)"
     :presence "@('old') is recursive")

    (xdoc::evmac-appcond
     ":old-guard"
     (xdoc::&&
      (xdoc::p
       "@('old') is well-defined (according to its guard)
        only on tuples in the old representation:")
      (xdoc::codeblock
       "(implies old-guard<x1,...,xn>"
       "         (and (oldp1 x1)"
       "              ..."
       "              (oldpn xn)))"))
     :design-notes (xdoc::&& *isodata-design-notes*
                             ", referenced by the "
                             *expdata-design-notes*
                             ",")
     :design-notes-appcond "@($Gf$)"
     :presence "the generated function is guard-verified
                (which is determined by the @(':verify-guards') input;
                see above)
                and @(':predicate') is @('nil')")

    (xdoc::evmac-appcond
     ":old-guard-pred"
     (xdoc::&&
      (xdoc::p
       "@('old') is well-defined (according to its guard)
        on all tuples in the old representation:")
      (xdoc::codeblock
       "(implies (and (oldp1 x1)"
       "              ..."
       "              (oldpn xn))"
       "         old-guard<x1,...,xn>)"))
     :design-notes (xdoc::&& *isodata-design-notes*
                             ", referenced by the "
                             *expdata-design-notes*
                             ",")
     :design-notes-appcond "@($Gp$)"
     :presence "the generated function is guard-verified
                (which is determined by the @(':verify-guards') input;
                see above)
                and @(':predicate') is @('t')")

    (xdoc::p
     "Unless @('surj1'), ..., @('surjq')
      are all names of pre-existing @(tsee defsurj)s,
      there are additional applicability conditions
      that pertain to the locally generated @(tsee defsurj)s.
      These additional applicability conditions are described
      in the documentation of @(tsee defsurj)."))

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   (xdoc::evmac-section-generated

    (xdoc::desc
     "@('new')"
     (xdoc::p
      "Expanded version of @('old'):")
     (xdoc::codeblock
      ";; when old is not recursive"
      ";; and :predicate is t:"
      "(defun new (x1 ... xn)"
      "  (and (mbt$ (and (newp1 x1)"
      "                  ..."
      "                  (newpn xn)))"
      "       old-body<(back1 x1),...,(backn xn)>))"
      ""
      ";; when old is not recursive,"
      ";; :predicate is nil:"
      ";; and surjmaps includes no :resultj:"
      "(defun new (x1 ... xn)"
      "  (if (mbt$ (and (newp1 x1)"
      "                 ..."
      "                 (newpn xn)))"
      "      old-body<(back1 x1),...,(backn xn)>"
      "    nil)) ; or (mv nil ... nil)"
      ""
      ";; when old is not recursive,"
      ";; :predicate is nil,"
      ";; m = 1,"
      ";; and surjmaps includes :result1 (or :result):"
      ""
      "(defun new (x1 ... xn)"
      "  (if (mbt$ (and (newp1 x1)"
      "                 ..."
      "                 (newpn xn)))"
      "      (forth_r1 old-body<(back1 x1),...,(backn xn)>)"
      "    nil))"
      ""
      ";; when old is not recursive,"
      ";; :predicate is nil,"
      ";; m > 1,"
      ";; and surjmaps includes some :resultj:"
      "(defun new (x1 ... xn)"
      "  (if (mbt$ (and (newp1 x1)"
      "                 ..."
      "                 (newpn xn)))"
      "      (mv-let (y1 ... ym)"
      "        old-body<(back1 x1),...,(backn xn)>"
      "        (mv (forth_r1 y1) ... (forth_rm ym)))"
      "    (mv nil ... nil)))"
      ""
      ";; when old is recursive"
      ";; and :predicate is t:"
      "(defun new (x1 ... xn)"
      "  (and (mbt$ (and (newp1 x1)"
      "                  ..."
      "                  (newpn xn)))"
      "       old-body<(back1 x1),...,(backn xn),"
      "                (new (forth1 update1-x1<(back1 x1),"
      "                                        ...,"
      "                                        (backn xn)>)"
      "                     ..."
      "                     (forthn update1-xn<(back1 x1),"
      "                                        ...,"
      "                                        (backn xn)>)),"
      "                ..."
      "                (new (forth1 updater-x1<(back1 x1),"
      "                                        ...,"
      "                                        (backn xn)>)"
      "                     ..."
      "                     (forthn updater-xn<(back1 x1),"
      "                                        ...,"
      "                                        (backn xn)>))>))"
      ""
      ";; when old is recursive,"
      ";; :predicate is nil,"
      ";; and surjmaps includes no :resultj:"
      "(defun new (x1 ... xn)"
      "  (if (mbt$ (and (newp1 x1)"
      "                 ..."
      "                 (newpn xn)))"
      "      old-body<(back1 x1),...,(backn xn),"
      "               (new (forth1 update1-x1<(back1 x1),"
      "                                       ...,"
      "                                       (backn xn)>)"
      "                    ..."
      "                    (forthn update1-xn<(back1 x1),"
      "                                       ...,"
      "                                       (backn xn)>)),"
      "               ..."
      "               (new (forth1 updater-x1<(back1 x1),"
      "                                       ...,"
      "                                       (backn xn)>)"
      "                    ..."
      "                    (forthn updater-xn<(back1 x1),"
      "                                       ...,"
      "                                       (backn xn)>))>"
      "    nil)) ; or (mv nil ... nil)"
      ""
      ";; when old is recursive,"
      ";; :predicate is nil,"
      ";; m = 1,"
      ";; and surjmaps include :result1 (or :result):"
      "(defun new (x1 ... xn)"
      "  (if (mbt$ (and (newp1 x1)"
      "                 ..."
      "                 (newpn xn)))"
      "      (forth_r1 old-body<(back1 x1),...,(backn xn),"
      "                         (back_r1"
      "                          (new (forth1 update1-x1<(back1 x1),"
      "                                                  ...,"
      "                                                  (backn xn)>)"
      "                               ..."
      "                               (forthn update1-xn<(back1 x1),"
      "                                                  ...,"
      "                                                  (backn xn)>))),"
      "                         ..."
      "                         (back_r1"
      "                          (new (forth1 updater-x1<(back1 x1),"
      "                                                  ...,"
      "                                                  (backn xn)>)"
      "                               ..."
      "                               (forthn updater-xn<(back1 x1),"
      "                                                  ...,"
      "                                                  (backn xn)>)))>)"
      "    nil))"
      ""
      ";; when old is recursive,"
      ";; :predicate is nil,"
      ";; m > 1,"
      ";; and surjmaps includes some :resultj:"
      "(defun new (x1 ... xn)"
      "  (if (mbt$ (and (newp1 x1)"
      "                 ..."
      "                 (newpn xn)))"
      "      (mv-let (y1 ... ym)"
      "        old-body<(back1 x1),...,(backn xn),"
      "                 (mv-let (y1 ... ym)"
      "                   (new (forth1 update1-x1<(back1 x1),"
      "                                           ...,"
      "                                           (backn xn)>)"
      "                        ..."
      "                        (forthn update1-xn<(back1 x1),"
      "                                           ...,"
      "                                           (backn xn)>))"
      "                   (mv (forth_r1 y1) ... (forth_rm ym))),"
      "                 ..."
      "                 (mv-let (y1 ... ym)"
      "                   (new (forth1 updater-x1<(back1 x1),"
      "                                           ...,"
      "                                           (backn xn)>)"
      "                        ..."
      "                        (forthn updater-xn<(back1 x1),"
      "                                           ...,"
      "                                           (backn xn)>))"
      "                   (mv (forth_r1 y1) ... (forth_rm ym)))>"
      "        (mv (forth_r1 y1) ... (forth_rm ym)))"
      "    (mv nil ... nil)))")
     (xdoc::p
      "If @('old') is recursive,
       the measure term of @('new') is
       @('measure<(back1 x1),...,(backn xn)>')
       and the well-founded relation of @('new') is
       the same as @('old').")
     (xdoc::p
      "The guard of @('new') is:")
     (xdoc::codeblock
      ";; when :predicate is nil:"
      "(and (newp1 x1)"
      "     ..."
      "     (newpn xn)"
      "     old-guard<(back1 x1),...,(backn xn)>)"
      ""
      ";; when :predicate is t:"
      "(and (newp1 x1)"
      "     ..."
      "     (newpn xn))")
     (xdoc::p
      "That is, when @(':predicate') is @('t')
       the guard consists of the new representation;
       when @(':predicate') is @('nil'),
       the guard consists of the argument tuples
       that are in the surjection's preimage of the guard of @('old').")
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       @('new') is denoted by
       @($f'$) when @(':predicate') is @('nil'),
       and @($p'$) when @(':predicate') is @('t').")
     (xdoc::p
      "Note that:")
     (xdoc::ul
      (xdoc::li
       "When @(':predicate') is @('t'),
        @('new') is defined to hold exactly
        on the argument tuples in the new representation
        that are in the surjection's preimage of
        the argument tuples in the old representation
        on which @('old') holds.")
      (xdoc::li
       "When @(':predicate') is @('nil'),
        @('new') is defined to map
        each argument tuple in the new representation
        to the same or corresponding value
        (where `corresponding' here means in the surjection's preimage)
        that @('old') maps
        the corresponding argument tuple in the old representation
        (where `corresponding' here means in the surjection's image).")))

    (xdoc::desc
     "@('new-to-old')"
     (xdoc::p
      "Theorem that relates @('new') to @('old'):")
     (xdoc::codeblock
      ";; when :predicate is t:"
      "(defthm new-to-old"
      "  (implies (and (newp1 x1)"
      "                ..."
      "                (newpn xn))"
      "           (equal (new x1 ... xn)"
      "                  (old (back1 x1) ... (backn xn)))))"
      ""
      ";; when :predicate is nil"
      ";; and surjmaps includes no :resultj:"
      "(implies (and (newp1 x1)"
      "              ..."
      "              (newpn xn))"
      "         (equal (new x1 ... xn)"
      "                (old (back1 x1) ... (backn xn))))"
      ""
      ";; when :predicate is nil,"
      ";; m = 1,"
      ";; and surjmaps includes :result1 (or :result):"
      "(implies (and (newp1 x1)"
      "              ..."
      "              (newp1 xn))"
      "         (equal (new x1 ... xn)"
      "                (forth_r1 (old (back1 x1) ... (backn xn)))))"
      ""
      ";; when :predicate is nil,"
      ";; m > 1,"
      ";; and surjmaps includes some :resultj:"
      "(implies (and (newp1 x1)"
      "              ..."
      "              (newp1 xn))"
      "         (and (equal (mv-nth 0 (new x1 ... xn))"
      "                     (forth_r1 (mv-nth 0 (old (back1 x1)"
      "                                              ..."
      "                                              (backn xn)))))"
      "              ..."
      "              (equal (mv-nth m-1 (new x1 ... xn))"
      "                     (forth_rm (mv-nth m-1 (old (back1 x1)"
      "                                                ..."
      "                                                (backn xn)))))))")
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       @('new-to-old') is denoted by
       @($f'f$) when @(':predicate') is @('nil'),
       and @($'pp$) when @(':predicate') is @('t')."))

    (xdoc::desc
     "@('old-to-new')"
     (xdoc::p
      "Theorem that relates @('old') to @('new'):")
     (xdoc::codeblock
      ";; when :predicate is t:"
      "(defthm old-to-new"
      "  (implies (and (oldp1 x1)"
      "                ..."
      "                (oldpn xn))"
      "           (equal (old x1 ... xn)"
      "                  (new (forth1 x1) ... (forthn xn)))))"
      ""
      ";; when :predicate is nil"
      ";; and surjmaps includes no :resultj:"
      "(defthm old-to-new"
      "  (implies (and (oldp1 x1)"
      "                ..."
      "                (oldpn xn))"
      "           (equal (old x1 ... xn)"
      "                  (new (forth1 x1) ... (forthn xn)))))"
      ""
      ";; when :predicate is nil,"
      ";; m = 1,"
      ";; and surjmaps includes :result1 (or :result):"
      "(defthm old-to-new"
      "  (implies (and (oldp1 x1)"
      "                ..."
      "                (oldpn xn))"
      "           (equal (old x1 ... xn)"
      "                  (back_r1 (new (forth1 x1) ... (forthn xn))))))"
      ""
      ";; when :predicate is nil,"
      ";; m > 1,"
      ";; and surjmaps includes some :resultj:"
      "(defthm old-to-new"
      "  (implies (and (oldp1 x1)"
      "                ..."
      "                (oldpn xn))"
      "           (and (equal (mv-nth 0 (old x1 ... xn))"
      "                       (back_r1 (mv-nth 0 (new (forth1 x1)"
      "                                               ..."
      "                                               (forthn xn)))))"
      "                ..."
      "                (equal (mv-nth m-1 (old x1 ... xn))"
      "                       (back_rm (mv-nth m-1 (new (forth1 x1)"
      "                                                 ..."
      "                                                 (forthn xn))))))))")
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       @('old-to-new') is denoted by
       @($ff'$) when @(':predicate') is @('nil'),
       and @($pp'$) when @(':predicate') is @('t')."))

    (xdoc::desc
     "@('newp-of-new')"
     (xdoc::p
      "Theorem asserting that @('new') maps
       arguments in the new representation
       to results in the new representation:")
     (xdoc::codeblock
      ";; when m = 1:"
      "(defthm newp-of-new"
      "  (implies (and (newp1 x1)"
      "                ..."
      "                (newpn xn))"
      "           (newp_r1 (new x1 ... xn))))"
      ""
      ";; when m > 1:"
      "(defthm newp-of-new"
      "  (implies (and (newp1 x1)"
      "                ..."
      "                (newpn xn))"
      "           (mv-let (y1 ... ym)"
      "             (new x1 ... xn)"
      "             (and (newp_r1 y1)"
      "                  ..."
      "                  (newp_rm ym)))))")
     (xdoc::p
      "In the " *isodata-design-notes* ",
       referenced by the " *expdata-design-notes* ",
       @('newp-of-new') is denoted by @($f'A'B'$).")
     (xdoc::p
      "This is generated if and only if
       @('surjmaps') includes some @(':resultj')."))

    (xdoc::p
     "A theory invariant is also generated to prevent
       both @('new-to-old') and @('old-to-new')
       from being enabled at the same time."))

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   (xdoc::evmac-section-redundancy expdata)))
