(in-package "ACL2")

(push :raw *features*)

(pushnew :rec-dcls *features*)
(pushnew :rec-stobj *features*)
(pushnew :stv-dcls *features*)
(pushnew :stv-stobj *features*)

(setq *read-default-float-format*

; By binding *read-default-float-format* to DOUBLE-FLOAT, we are telling Lisp
; that we intend that floating point numbers are read by default as doubles.
; So for example, in CCL, we will print 3.0000001 as "3.0000001", and if
; *read-default-float-format* is DOUBLE-FLOAT then it would be read back in the
; same way: with enough precision to see the final "1", which isn't the case by
; default.  The danger is that we write such a number with
; *read-default-float-format* set (or bound) to DOUBLE-FLOAT, but we read it
; back in with the default *read-default-float-format* of SINGLE-FLOAT -- then
; we'd lose that precision.

      'DOUBLE-FLOAT)

(defmacro defstobj (&whole event name &rest args)

; This macro is a replacement for raw-Lisp ACL2 defstobj.  The only changes
; below are in code involving *installing-raw-or-logic*.

; Warning: If you change this definition, consider the possibility of making
; corresponding changes to the #-acl2-loop-only definition of defabsstobj.

; This function is run when we evaluate (defstobj name . args) in raw lisp.
; A typical such form is

; (defstobj $st
;   (flag :type t :initially run)
;   (pc   :type (integer 0 255) :initially 128)
;   (mem  :type (array (integer 0 255) (256)) :initially 0))

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

; This function must contend with a problem analogous to the one addressed by
; acl2::defconst in acl2.lisp: the need to avoid re-declaration of the same
; stobj.  We use redundant-raw-lisp-discriminator in much the same way as in
; the raw lisp defmacro of acl2::defconst.

  (let* ((template (defstobj-template name args nil))
         (congruent-to (access defstobj-template template :congruent-to))
         (congruent-stobj-rep (if congruent-to
                                  (congruent-stobj-rep-raw congruent-to)
                                name))
         (creator (access defstobj-template template :creator))
         (non-memoizable (access defstobj-template template :non-memoizable))
         (non-executable (access defstobj-template template :non-executable))
         (init-form (defstobj-raw-init template))
         (the-live-name (the-live-var name)))
    `(progn
       ,@(and (null congruent-to)

; It has occurred to us that this defg form might be avoidable when
; non-memoizable is true, since the purpose of st-lst is probably only to
; support memoize-flush.  However, it seems harmless enough to lay down this
; form even when non-memoizable is true, so we go ahead and do so rather than
; think carefully about avoiding it.

              `((defg ,(st-lst name) nil)))

; Now we lay down the defuns of the recognizers, accessors and updaters as
; generated by defstobj-raw-defs.  The boilerplate below just adds the DEFUN to
; the front of each def generated, preserving the order of the defs as
; generated.  We deal here with the :inline case; note that
; *stobj-inline-declare* was added in defstobj-field-fns-raw-defs.

       ,@(mapcar (function (lambda (def)
                             (if (member-equal *stobj-inline-declare* def)
                                 (cons 'DEFABBREV
                                       (remove-stobj-inline-declare def))
                               (cons 'DEFUN def))))
                 (defstobj-raw-defs name template congruent-stobj-rep nil))
       ,@(defstobj-defconsts
           (strip-accessor-names (access defstobj-template template
                                         :field-templates))
           0)
       (let* ((event ',event)
              (creator ',creator)
              (congruent-stobj-rep ',congruent-stobj-rep)
              (non-memoizable ',non-memoizable)
              (non-executable ',non-executable)
              (old-pair (assoc-eq ',name *user-stobj-alist*))
              (d (and (or old-pair non-executable) ; optimization
                      (get ',the-live-name
                           'redundant-raw-lisp-discriminator)))
              (ok-p (and (consp d)
                         (eq (car d) 'defstobj)
                         #-raw ; to be skipped here
                         (equal (access
                                 defstobj-redundant-raw-lisp-discriminator-value
                                 (cdr d)
                                 :event)
                                event))))
         (cond
          (ok-p
           #+raw ; included here
           (progn
             (assert old-pair)
             (setf (cdr old-pair) ,init-form))
           ',name)
          ((and old-pair (not (raw-mode-p *the-live-state*)))

; If the old stobj was non-executable, then it should be OK to redeclare it
; since there is no issue about what to do with the old value (there isn't
; one).

           (interface-er
            "Illegal attempt to redeclare the single-threaded object ~s0."
            ',name))
          (t
           (setf (get ',the-live-name 'redundant-raw-lisp-discriminator)
                 (cons 'defstobj
                       (make defstobj-redundant-raw-lisp-discriminator-value
                             :event event
                             :creator creator
                             :congruent-stobj-rep congruent-stobj-rep
                             :non-memoizable non-memoizable
                             :non-executable non-executable)))
           (cond (old-pair ; hence raw-mode
                  (fms "Note:  Redefining and ~@0 stobj ~x1 in ~
                        raw mode.~%"
                       (list (cons #\0 (if non-executable
                                           "removing executability of"
                                         "reinitializing"))
                             (cons #\1 ',name))
                       (standard-co *the-live-state*) *the-live-state* nil)
                  (if non-executable
                      (assert$
                       (not (member-eq ',name *non-executable-user-stobj-lst*))
                       (setq *user-stobj-alist*
                             (remove1-assoc-eq ',name *user-stobj-alist*)))
                    (setf (cdr old-pair) ,init-form)))
                 (non-executable
                  (pushnew ',name *non-executable-user-stobj-lst*))
                 (t
                  (setq *user-stobj-alist* ; update-user-stobj-alist
                        (cons (cons ',name ,init-form)
                              *user-stobj-alist*))))
           ',name))))))

(defun princ$ (x channel state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

; The ACL2 princ$ does not handle conses because we are unsure what
; the specification of the real Common Lisp princ is concerning the
; insertion of spaces and newlines into the resulting text.

  (declare (xargs :guard (and (or (acl2-numberp x)
                                  (characterp x)
                                  (stringp x)
                                  (symbolp x))
                              (state-p1 state-state)
                              (symbolp channel)
                              (open-output-channel-p1
                               channel :character state-state))))
;;; Matt K. mod: This file is loaded in raw-mode, inside the ACL2
;;; read-eval-print loop.  Before 6/6/2023, feature :acl2-loop-only was false
;;; inside that loop; now it's true.  Since we are redefining princ$ in raw
;;; Lisp here, we want the "raw" code evaluated, so we no longer can afford the
;;; restriction to #-acl2-loop-only.
;;;  #-acl2-loop-only
  (cond ((live-state-p state-state)
         (cond ((and *wormholep*
                     (not (eq channel *standard-co*)))

; If the live state is protected, then we allow output only to the
; *standard-co* channel.  This is a little unexpected.  The intuitive
; arrangement would be to allow output only to a channel whose actual
; stream was pouring into the wormhole window.  Unfortunately, we do not
; know a good way to determine the ultimate stream to which a synonym
; stream is directed and hence cannot implement the intuitive
; arrangement.  Instead we must assume that if *the-live-state-
; protected* is non-nil, then the standard channels have all been
; directed to acceptable streams and that doing i/o on them will not
; affect the streams to which they are normally directed.

                (wormhole-er 'princ$ (list x channel))))
         (let ((stream (get-output-stream-from-channel channel)))
           (cond
            ((stringp x)

; We get a potentially significant efficiency boost by using write-string when
; x is a string.  A few experiments suggest that write-string may be slightly
; more efficient than write-sequence (which isn't available in non-ANSI GCL
; anyhow), which in turn may be much more efficient than princ.  It appears
; that the various print-controls don't affect the printing of strings, except
; for *print-escape* and *print-readably*; and the binding of *print-escape* to
; nil by princ seems to give the behavior of write-string, which is specified
; simply to print the characters of the string.

             (write-string x stream))
            ((characterp x) ; faster write as for strings; see comment above
             (write-char x stream)
             (when (eql x #\Newline)
               (force-output stream)))
            #+raw
            ((typep x 'float)
             (princ x stream))
            (t
             (with-print-controls

; We use :defaults here, binding only *print-escape* and *print-readably* (to
; avoid |..| on symbols), to ensure that raw Lisp agrees with the logical
; definition.

              :defaults
              ((*print-escape* nil)
               (*print-readably* nil) ; unnecessary if we keep current default
               (*print-base* (f-get-global 'print-base state))
               (*print-radix* (f-get-global 'print-radix state))
               (*print-case* (f-get-global 'print-case state)))
              #+acl2-print-number-base-16-upcase-digits
              (cond ((and (acl2-numberp x)
                          (> *print-base* 10))
                     (print-number-base-16-upcase-digits x stream))
                    (t (princ x stream)))
              #-acl2-print-number-base-16-upcase-digits
              (princ x stream))))
           (return-from princ$ *the-live-state*))))
  (let ((entry (cdr (assoc-eq channel (open-output-channels state-state)))))
    (update-open-output-channels
     (add-pair channel
               (cons (car entry)
                     (revappend
                      (if (and (symbolp x)

; The form (cdr (assoc-eq ...)) below is closely related to a call of
; print-case where state is replaced by state-state.  However, the problem
; explained in the essay "On STATE-STATE" hits us here.  That is, print-case
; generates a call of get-global, which, by the time this form is processed in
; the logic during boot-strap, expects state as an argument.  We do not have
; state available here.  We could modify print-case to take an optional
; argument and supply state-state for that argument, but that would not work
; either because get-global expects state.

                               (eq (cdr (assoc-eq 'print-case
                                                  (global-table state-state)))
                                   :downcase))
                          (coerce (string-downcase (symbol-name x))
                                  'list)
                        (explode-atom+ x
                                       (cdr (assoc-eq 'print-base
                                                      (global-table
                                                       state-state)))
                                       (cdr (assoc-eq 'print-radix
                                                      (global-table
                                                       state-state)))))
                      (cdr entry)))
               (open-output-channels state-state))
     state-state)))
