# Copyright (c) 2022 Fortinet
# GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type


# BEGIN STATIC DATA / MESSAGES
class FMGRMethods:
    GET = "get"
    SET = "set"
    EXEC = "exec"
    EXECUTE = "exec"
    UPDATE = "update"
    ADD = "add"
    DELETE = "delete"
    REPLACE = "replace"
    CLONE = "clone"
    MOVE = "move"


BASE_HEADERS = {
    'Content-Type': 'application/json',
    'Accept': 'application/json'
}


# FMGR RETURN CODES
FMGR_RC = {
    "fmgr_return_codes": {
        0: {
            "msg": "OK",
            "changed": True,
            "stop_on_success": True
        },
        -100000: {
            "msg": "Module returned without actually running anything. "
            "Check parameters, and please contact the authors if needed.",
            "failed": True
        },
        -2: {
            "msg": "Object already exists.",
            "skipped": True,
            "changed": False,
            "good_codes": [0, -2]
        },
        -6: {
            "msg": "Invalid Url. Sometimes this can happen because the path is mapped to a hostname or object that"
            " doesn't exist. Double check your input object parameters."
        },
        -3: {
            "msg": "Object doesn't exist.",
            "skipped": True,
            "changed": False,
            "good_codes": [0, -3]
        },
        -10131: {
            "msg": "Object dependency failed. Do all named objects in parameters exist?",
            "changed": False,
            "skipped": True
        },
        -9998: {
            "msg": "Duplicate object. Try using mode='set', if using add. STOPPING. Use 'ignore_errors=yes' in playbook"
            "to override and mark successful.",
        },
        -20042: {
            "msg": "Device Unreachable.",
            "skipped": True
        },
        -10033: {
            "msg": "Duplicate object. Try using mode='set', if using add.",
            "changed": False,
            "skipped": True
        },
        -10000: {
            "msg": "Duplicate object. Try using mode='set', if using add.",
            "changed": False,
            "skipped": True
        },
        -20010: {
            "msg": "Device already added to FortiManager. Serial number already in use.",
            "good_codes": [0, -20010],
            "changed": False,
            "stop_on_success": True
        },
        -20002: {
            "msg": "Invalid Argument -- Does this Device exist on FortiManager?",
            "changed": False,
            "skipped": True,
        }
    }
}

DEFAULT_RESULT_OBJ = (-100000, {"msg": "Nothing Happened. Check that handle_response is being called!"})
FAIL_SOCKET_MSG = {"msg": "Socket Path Empty! The persistent connection manager is messed up. "
                   "Try again in a few moments."}


# BEGIN ERROR EXCEPTIONS
class FMGBaseException(Exception):
    """Wrapper to catch the unexpected"""

    def __init__(self, msg=None, *args, **kwargs):
        if msg is None:
            msg = "An exception occurred within the fortimanager.py httpapi connection plugin."
        super(FMGBaseException, self).__init__(msg, *args)

# END ERROR CLASSES


# BEGIN CLASSES
class FMGRCommon(object):

    @staticmethod
    def format_request(method, url, *args, **kwargs):
        """
        Formats the payload from the module, into a payload the API handler can use.

        :param url: Connection URL to access
        :type url: string
        :param method: The preferred API Request method (GET, ADD, POST, etc....)
        :type method: basestring
        :param kwargs: The payload dictionary from the module to be converted.

        :return: Properly formatted dictionary payload for API Request via Connection Plugin.
        :rtype: dict
        """

        params = [{"url": url}]
        if args:
            for arg in args:
                params[0].update(arg)
        if kwargs:
            keylist = list(kwargs)
            for k in keylist:
                kwargs[k.replace("__", "-")] = kwargs.pop(k)
            if method == "get" or method == "clone":
                params[0].update(kwargs)
            else:
                if kwargs.get("data", False):
                    params[0]["data"] = kwargs["data"]
                else:
                    params[0]["data"] = kwargs
        return params

    @staticmethod
    def split_comma_strings_into_lists(obj):
        """
        Splits a CSV String into a list.  Also takes a dictionary, and converts any CSV strings in any key, to a list.

        :param obj: object in CSV format to be parsed.
        :type obj: str or dict

        :return: A list containing the CSV items.
        :rtype: list
        """
        return_obj = ()
        if isinstance(obj, dict):
            if len(obj) > 0:
                for k, v in obj.items():
                    if isinstance(v, str):
                        new_list = list()
                        if "," in v:
                            new_items = v.split(",")
                            for item in new_items:
                                new_list.append(item.strip())
                            obj[k] = new_list
                return_obj = obj
        elif isinstance(obj, str):
            return_obj = obj.replace(" ", "").split(",")

        return return_obj

    @staticmethod
    def cidr_to_netmask(cidr):
        """
        Converts a CIDR Network string to full blown IP/Subnet format in decimal format.
        Decided not use IP Address module to keep includes to a minimum.

        :param cidr: String object in CIDR format to be processed
        :type cidr: str

        :return: A string object that looks like this "x.x.x.x/y.y.y.y"
        :rtype: str
        """
        if isinstance(cidr, str):
            cidr = int(cidr)
            mask = (0xffffffff >> (32 - cidr)) << (32 - cidr)
            return (str((0xff000000 & mask) >> 24) + '.'
                    + str((0x00ff0000 & mask) >> 16) + '.'
                    + str((0x0000ff00 & mask) >> 8) + '.'
                    + str((0x000000ff & mask)))

    @staticmethod
    def paramgram_child_list_override(list_overrides, paramgram, module):
        """
        If a list of items was provided to a "parent" paramgram attribute, the paramgram needs to be rewritten.
        The child keys of the desired attribute need to be deleted, and then that "parent" keys' contents is replaced
        With the list of items that was provided.

        :param list_overrides: Contains the response from the FortiManager.
        :type list_overrides: list
        :param paramgram: Contains the paramgram passed to the modules' local modify function.
        :type paramgram: dict
        :param module: Contains the Ansible Module Object being used by the module.
        :type module: classObject

        :return: A new "paramgram" refactored to allow for multiple entries being added.
        :rtype: dict
        """
        if len(list_overrides) > 0:
            for list_variable in list_overrides:
                try:
                    list_variable = list_variable.replace("-", "_")
                    override_data = module.params[list_variable]
                    if override_data:
                        del paramgram[list_variable]
                        paramgram[list_variable] = override_data
                except BaseException as e:
                    raise FMGBaseException("Error occurred merging custom lists for the paramgram parent: " + str(e))
        return paramgram

    @staticmethod
    def syslog(module, msg):
        try:
            module.log(msg=msg)
        except BaseException:
            pass


# RECURSIVE FUNCTIONS START
def prepare_dict(obj):
    """
    Removes any keys from a dictionary that are only specific to our use in the module. FortiManager will reject
    requests with these empty/None keys in it.

    :param obj: Dictionary object to be processed.
    :type obj: dict

    :return: Processed dictionary.
    :rtype: dict
    """

    list_of_elems = ["mode", "adom", "host", "username", "password"]

    if isinstance(obj, dict):
        obj = dict((key, prepare_dict(value)) for (key, value) in obj.items() if key not in list_of_elems)
    return obj


def scrub_dict(obj):
    """
    Removes any keys from a dictionary that are EMPTY -- this includes parent keys. FortiManager doesn't
    like empty keys in dictionaries

    :param obj: Dictionary object to be processed.
    :type obj: dict

    :return: Processed dictionary.
    :rtype: dict
    """

    if isinstance(obj, dict):
        return dict((k, scrub_dict(v)) for k, v in obj.items() if v and scrub_dict(v))
    else:
        return obj
