/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2019 the original author or authors.
 */
package org.assertj.core.api.assumptions;

import static org.assertj.core.api.Assertions.assertThatCode;
import static org.assertj.core.api.Assertions.assertThatExceptionOfType;
import static org.assertj.core.api.Assumptions.assumeThat;
import static org.assertj.core.util.Arrays.array;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicLongArray;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.concurrent.atomic.AtomicMarkableReference;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import java.util.concurrent.atomic.AtomicStampedReference;
import java.util.stream.Stream;

import org.junit.AssumptionViolatedException;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class Assumptions_assumeThat_Atomics_Test {

  private static final VolatileFieldsHolder VOLATILE_FIELDS_HOLDER = new VolatileFieldsHolder();

  static Stream<AssumptionRunner<?>> provideAssumptionsRunners() {
    return Stream.of(
        new AssumptionRunner<AtomicBoolean>(new AtomicBoolean(true)) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).isFalse();
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).isTrue();
          }
        },
        new AssumptionRunner<AtomicInteger>(new AtomicInteger(42)) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasNegativeValue();
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasPositiveValue();
          }
        },
        new AssumptionRunner<AtomicIntegerArray>(new AtomicIntegerArray(new int[] { 2, 5, 7 })) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).contains(20);
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).contains(7);
          }
        },
        new AssumptionRunner<AtomicIntegerFieldUpdater<VolatileFieldsHolder>>(AtomicIntegerFieldUpdater.newUpdater(VolatileFieldsHolder.class,
                                                                                                                     "intValue")) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasValue(10, VOLATILE_FIELDS_HOLDER);
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasValue(0, VOLATILE_FIELDS_HOLDER);
          }
        },
        new AssumptionRunner<AtomicLong>(new AtomicLong(42)) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasNegativeValue();
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasPositiveValue();
          }
        },
        new AssumptionRunner<AtomicLongArray>(new AtomicLongArray(new long[] { 2, 5, 7 })) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).contains(20);
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).contains(7);
          }
        },
        new AssumptionRunner<AtomicLongFieldUpdater<VolatileFieldsHolder>>(AtomicLongFieldUpdater.newUpdater(VolatileFieldsHolder.class,
                                                                                                               "longValue")) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasValue(10L, VOLATILE_FIELDS_HOLDER);
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasValue(0L, VOLATILE_FIELDS_HOLDER);
          }
        },
        new AssumptionRunner<AtomicReference<String>>(new AtomicReference<>("test")) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasValue("other");
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasValue("test");
          }
        },
        new AssumptionRunner<AtomicReferenceArray<String>>(new AtomicReferenceArray<>(array("2", "5", "7"))) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).contains("20");
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).contains("7");
          }
        },
        new AssumptionRunner<AtomicReferenceFieldUpdater<VolatileFieldsHolder, String>>(AtomicReferenceFieldUpdater.newUpdater(VolatileFieldsHolder.class,
                                                                                                                                 String.class,
                                                                                                                                 "stringValue")) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasValue("other", VOLATILE_FIELDS_HOLDER);
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasValue("test", VOLATILE_FIELDS_HOLDER);
          }
        },
        new AssumptionRunner<AtomicMarkableReference<String>>(new AtomicMarkableReference<>("test", true)) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasReference("other");
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasReference("test");
          }
        },
        new AssumptionRunner<AtomicStampedReference<String>>(new AtomicStampedReference<>("test", 1)) {
          @Override
          public void runFailingAssumption() {
            assumeThat(actual).hasStamp(0);
          }

          @Override
          public void runPassingAssumption() {
            assumeThat(actual).hasStamp(1);
          }
        });
  }

  @ParameterizedTest
  @MethodSource("provideAssumptionsRunners")
  public void should_ignore_test_when_assumption_fails(AssumptionRunner<?> assumptionRunner) {
    assertThatExceptionOfType(AssumptionViolatedException.class).isThrownBy(() -> assumptionRunner.runFailingAssumption());
  }

  @ParameterizedTest
  @MethodSource("provideAssumptionsRunners")
  public void should_run_test_when_assumption_passes(AssumptionRunner<?> assumptionRunner) {
    assertThatCode(() -> assumptionRunner.runPassingAssumption()).doesNotThrowAnyException();
  }

  @SuppressWarnings("unused")
  private static class VolatileFieldsHolder {
    volatile int intValue;
    volatile long longValue;
    volatile String stringValue = "test";
  }
}
