/* This file is part of the 'atomes' software

'atomes' is free software: you can redistribute it and/or modify it under the terms
of the GNU Affero General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.

'atomes' is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU Affero General Public License along with 'atomes'.
If not, see <https://www.gnu.org/licenses/>

Copyright (C) 2022-2024 by CNRS and University of Strasbourg */

/*!
* @file cvff.c
* @short CVFF force field, from 'CVFF.frc'
* @author Sébastien Le Roux <sebastien.leroux@ipcms.unistra.fr>
*/

/*
* This file: 'cvff.c'
*
* Contains:
*

  CVFF force field, data from taken from:

       - the 'CVFF.frc' file

  This file contains several tables:

    Atoms            : CVFF_atoms[133][5]
    Equivalence      : CVFF_equivalence_auto[123][11]
    Equivalence      : CVFF_equivalence[129][7]
    Quadractic bonds : CVFF_bonds_auto[776][5]
    Morse bonds      : CVFF_morse_bonds[775][6]
    Quadratic angles : CVFF_angles_auto[563][6]
    Torsions         : CVFF_torsions_auto[295][8]
    Impropers        : CVFF_impropers[41][8]
    Non bonded       : CVFF_vdw[45][4]
    Bond increments  : CVFF_bond_increments[683][5]

*/

#include "global.h"

int CVFF_objects[14] = {133, 123, 129, 776, 0, 775, 563, 0, 295, 0, 0, 41, 45, 683};
int CVFF_dim[14] = {5, 11, 7, 5, 0, 6, 6, 0, 8, 0, 0, 8, 4, 5};

/*
 Atoms
  0= Element, 1= Mass, 2= Key, 3= Description, 4= CVFF info
*/
char * CVFF_atoms[133][5]= {{"H", "1.007970", "h", "Hydrogen bonded to C. Masses from CRC 1973/74 pages B-250.", "Ver. 1.0, Ref. 1"},
                            {"D", "2.014000", "d", "General Deuterium Atom", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hn", "Hydrogen bonded to N", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "ho", "Hydrogen bonded to O", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hp", "Hydrogen bonded to P", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hs", "Hydrogen bonded to S", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "h*", "Hydrogen in water molecule", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hspc", "Hydrogen in SPC water molecule", "Ver. 2.3, Ref. 25"},
                            {"H", "1.007970", "htip", "Hydrogen in TIP water molecule", "Ver. 2.3, Ref. 25"},
                            {"H", "1.007970", "h$", "Hydrogen atom for automatic parameter assignment", "Ver. 1.0, Ref. 1"},
                            {"L", "0.001097", "lp", "Lone Pair (massless)", "Ver. 1.0, Ref. 1"},
                            {"L", "1.000000", "lp", "Lone Pair", "Ver. 1.1, Ref. 2"},
                            {"H", "1.007970", "h+", "Charged hydrogen in cations", "Ver. 2.0, Ref. 18"},
                            {"H", "1.007970", "hc", "Hydrogen bonded to carbon", "Ver. 2.0, Ref. 18"},
                            {"H", "1.007970", "hi", "Hydrogen in charged imidazole ring", "Ver. 2.0, Ref. 18"},
                            {"H", "1.007970", "hw", "Hydrogen in water molecule", "Ver. 2.0, Ref. 18"},
                            {"D", "2.014000", "dw", "Deuterium in heavy water", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "c", "Sp3 aliphatic carbon", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "cg", "Sp3 alpha carbon in glycine", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c'", "Sp2 carbon in carbonyl (C=O) group", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c*", "Carbon in carbonyl group, non_amides", "Ver. 2.0, Ref. 21"},
                            {"C", "12.011150", "c\"", "Carbon in carbonyl group, non_amides", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "cp", "Sp2 aromatic carbon (partial double bonds)", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "cr", "Carbon in guanidinium group (HN=C(NH2)2)", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c+", "C in guanidinium group", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "c-", "Carbon in charged carboxylate (COO-) group", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "ca", "General amino acid alpha carbon (sp3)", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c3", "Sp3 carbon in methyl (CH3) group", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "cn", "Sp3 Carbon bonded to N", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c2", "Sp3 carbon bonded to 2 H's, 2 heavy atoms", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c1", "Sp3 carbon bonded to 1 H, 3 Heavy atoms", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c5", "Sp2 aromatic carbon in five membered ring", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "cs", "Sp2 carbon involved in thiophene", "Ver. 1.3, Ref. 6"},
                            {"C", "12.011150", "c=", "Non-aromatic end doubly bonded carbon", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c=1", "Non-aromatic, next to end doubly bonded carbon", "Ver. 2.0, Ref. 19"},
                            {"C", "12.011150", "c=2", "Non-aromatic doubly bonded carbon", "Ver. 2.0, Ref. 19"},
                            {"C", "12.011150", "ct", "Sp carbon involved in triple bond", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "ci", "Aromatic carbon in a charged imidazole ring (HIS+)", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "c$", "Carbon atom for automatic parameter assignment", "Ver. 1.0, Ref. 1"},
                            {"C", "12.011150", "co", "Sp3 carbon in acetals", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "c3m", "Sp3 carbon in 3-membered ring", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "c4m", "Sp3 carbon in 4-membered ring", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "coh", "Sp3 carbon in acetals with hydrogen", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "c3h", "Sp3 carbon in 3-membered ring with hydrogens", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "c4h", "Sp3 carbon in 4-membered ring with hydrogens", "Ver. 2.0, Ref. 18"},
                            {"C", "12.011150", "ci", "Sp2 aromatic carbon in charged imidazole ring (His+)", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "n", "Sp2 nitrogen with 1 H, 2 heavy atoms (amide group)", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "no", "Sp2 nitrogen in nitro group", "Ver. 2.3, Ref. 23"},
                            {"N", "14.006700", "n2", "Sp2 nitrogen (NH2 in the guanidinium group (HN=C(NH2)2))", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "np", "Sp2 aromatic nitrogen (partial double bonds)", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "n3", "Sp3 nitrogen with three substituents", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "n4", "Sp3 nitrogen with four substituents", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "n=", "Non-aromatic end double bonded nitrogen", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "n=1", "Non-aromatic, next to end doubly bonded carbon", "Ver. 2.0, Ref. 19"},
                            {"N", "14.006700", "n=2", "Non-aromatic doubly bonded nitrogen", "Ver. 2.0, Ref. 19"},
                            {"N", "14.006700", "nt", "Sp nitrogen involved in triple bond", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "nz", "Sp nitrogen in N2", "Ver. 1.3, Ref. 4"},
                            {"N", "14.006700", "n1", "Sp2 nitrogen in charged arginine", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "ni", "Sp2 nitrogen in a charged imidazole ring (HIS+)", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "n$", "Nitrogen atom for automatic parameter assignment", "Ver. 1.0, Ref. 1"},
                            {"N", "14.006700", "na", "Sp3 nitrogen in amines", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "n3m", "Sp3 nitrogen in 3- membered ring", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "n4m", "Sp3 nitrogen in 4- membered ring", "Ver. 2.0, Ref. 18"},
                            {"N", "14.00670", "n3n", "Sp2 nitrogen in 3- membered ring", "Ver. 2.0, Ref. 18"},
                            {"N", "14.00670", "n4n", "Sp2 nitrogen in 4- membered ring", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "nb", "Sp2 nitrogen in aromatic amines", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "nn", "Sp2 nitrogen in aromatic amines", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "npc", "Sp2 nitrogen in 5- or 6- membered ring bonded to a heavy atom", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "nh", "Sp2 nitrogen in 5-or 6- membered ring with hydrogen attached", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "nho", "Sp2 nitrogen in 6- membered ring next to a carbonyl group and with a hydrogen", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "nh+", "Protonated nitrogen in 6- membered ring with hydrogen attached", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "n+", "Sp3 nitrogen in protonated amines", "Ver. 2.0, Ref. 18"},
                            {"N", "14.006700", "nr", "Sp2 nitrogen (NH2) in guanidinium group (HN=C(NH2)2)", "Ver. 2.0, Ref. 18"},
                            {"O", "15.999400", "o'", "Oxygen in carbonyl (C=O) group", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "o", "Sp3 oxygen in ether or ester groups", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "oz", "Oxygen in Zeolite", "Ver. 1.8, Ref. 14"},
                            {"O", "15.999400", "o-", "Oxygen in charged carboxylate (COO-) group", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "oh", "Oxygen in hydroxyl (OH) group", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "o*", "Oxygen in water molecule", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "ospc", "Oxygen in SPC water molecule", "Ver. 2.3, Ref. 25"},
                            {"O", "15.999400", "otip", "Oxygen in TIP3P water molecule", "Ver. 2.3, Ref. 25"},
                            {"O", "15.999400", "op", "Oxygen in aromatic rings. e.g. furan", "Ver. 1.3, Ref. 8"},
                            {"O", "15.999400", "of", "Oxygen ", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "o$", "Oxygen atom for automatic parameter assignment", "Ver. 1.0, Ref. 1"},
                            {"O", "15.999400", "oc", "Sp3 oxygen in ether or acetals", "Ver. 2.0, Ref. 18"},
                            {"O", "15.999400", "oe", "Sp3 oxygen in ester", "Ver. 2.0, Ref. 18"},
                            {"O", "15.999400", "o3e", "Sp3 oxygen in three membered ring", "Ver. 2.0, Ref. 18"},
                            {"O", "15.999400", "o4e", "Sp3 oxygen in four membered ring", "Ver. 2.0, Ref. 18"},
                            {"S", "32.064000", "s", "Sulfur in methionine (C-S-C) group", "Ver. 1.0, Ref. 1"},
                            {"S", "32.064000", "s1", "Sulfur involved in S-S disulfide bond", "Ver. 1.0, Ref. 1"},
                            {"S", "32.064000", "sh", "Sulfur in sulfhydryl (-SH) group", "Ver. 1.0, Ref. 1"},
                            {"S", "32.064000", "sp", "Sulfur in thiophene", "Ver. 1.3, Ref. 6"},
                            {"S", "32.064000", "s'", "Sulfur in thioketone (>C=S) group", "Ver. 1.3, Ref. 7"},
                            {"S", "32.064000", "s$", "Sulfur atom for automatic parameter assignment", "Ver. 1.0, Ref. 1"},
                            {"S", "32.064000", "sc", "Sp3 sulfur in methionines (C-S-C) group", "Ver. 2.0, Ref. 18"},
                            {"S", "32.06400", "s3e", "Sulfur in three membered ring", "Ver. 2.0, Ref. 18"},
                            {"S", "32.06400", "s4e", "Sulfur in four membered ring", "Ver. 2.0, Ref. 18"},
                            {"S", "32.064000", "s-", "Partial double sulfur bonded to something then bonded to another partial double oxygen or sulfur", "Ver. 2.0, Ref. 18"},
                            {"P", "30.973800", "p", "General phosphorous atom", "Ver. 1.0, Ref. 1"},
                            {"P", "30.973800", "pz", "Phosphorous atom in ALPO type structure", "Ver. 1.8, Ref. 14"},
                            {"P", "30.973800", "p$", "Phosphorous atom for automatic parameter assignment", "Ver. 1.0, Ref. 1"},
                            {"Ca", "40.079800", "ca+", "ion Ca2+, mass = mass of Ca - 2*electron mass.", "Ver. 2.0, Ref. 18"},
                            {"F", "18.998400", "f", "Fluorine bonded to a carbon", "Ver. 1.0, Ref. 1"},
                            {"Cl", "35.453000", "cl", "Chlorine bonded to a carbon", "Ver. 1.0, Ref. 1"},
                            {"Br", "79.909000", "br", "Bromine bonded to a carbon", "Ver. 1.0, Ref. 1"},
                            {"I", "126.9045", "i", "Covalently bound Iodine", "Ver. 1.3, Ref. 4"},
                            {"Si", "28.086000", "si", "Silicon ", "Ver. 1.0, Ref. 1"},
                            {"Si", "28.086000", "sz", "Silicon atom in a Zeolite or Silicate", "Ver. 1.8, Ref. 14"},
                            {"D", "12.000000", "nu", "atom for relative free energy", "Ver. 1.0, Ref. 1"},
                            {"Cl", "35.453000", "Cl", "Chloride ion Cl-", "Ver. 1.0, Ref. 1"},
                            {"Br", "79.904000", "Br", "Bromide ion Br-", "Ver. 1.0, Ref. 1"},
                            {"Na", "22.989800", "Na", "Sodium ion", "Ver. 1.0, Ref. 1"},
                            {"Ar", "39.948", "ar", "Argon atom", "Ver. 1.3, Ref. 9"},
                            {"He", "4.002600", "he", "Helium atom", "Ver. 2.4, Ref. 30"},
                            {"Al", "26.982000", "al", "Aluminium in alumino-silicate", "Ver. 2.1, Ref. 26"},
                            {"Al", "26.982000", "az", "Aluminium in alumino-silicate", "Ver. 2.1, Ref. 26"},
                            {"Al", "26.982000", "Al", "Aluminium metal", "Ver. 2.1, Ref. 26"},
                            {"Na", "22.990000", "Na", "Sodium metal", "Ver. 2.1, Ref. 26"},
                            {"Pt", "195.090000", "Pt", "Platinum metal", "Ver. 2.1, Ref. 26"},
                            {"Pd", "106.400000", "Pd", "Palladium metal", "Ver. 2.1, Ref. 26"},
                            {"Au", "196.967000", "Au", "Gold metal", "Ver. 2.1, Ref. 26"},
                            {"Ag", "107.868000", "Ag", "Silver metal", "Ver. 2.1, Ref. 26"},
                            {"Sn", "118.690000", "Sn", "Tin metal", "Ver. 2.1, Ref. 26"},
                            {"K", "39.102000", "K", "Potassium metal", "Ver. 2.1, Ref. 26"},
                            {"Li", "6.940000", "Li", "Lithium metal", "Ver. 2.1, Ref. 26"},
                            {"Mo", "95.940000", "Mo", "Molybdenum metal", "Ver. 2.1, Ref. 26"},
                            {"Fe", "55.847000", "Fe", "Iron metal", "Ver. 2.1, Ref. 26"},
                            {"W", "183.850000", "W", "Tungsten metal", "Ver. 2.1, Ref. 26"},
                            {"Ni", "58.710000", "Ni", "Nickel metal", "Ver. 2.1, Ref. 26"},
                            {"Cr", "51.996000", "Cr", "Chromium metal", "Ver. 2.1, Ref. 26"},
                            {"Cu", "63.546000", "Cu", "Copper metal", "Ver. 2.1, Ref. 26"},
                            {"Pb", "207.200000", "Pb", "Lead metal", "Ver. 2.1, Ref. 26"},
                            {"Mg", "100.000000", "mg", "Magnesium ", "Ver. 2.1, Ref. 27"}};

/*
 Equivalence table for key symbols CVFF auto with:
   0 = Key in atom list
   1 = Non-bonded
   2 = Bond increments
   3 = Bonds
   4 = Angles end atom
   5 = Angles appex atom
   6 = Torsion end atom
   7 = Torsion center atom
   8 = Inversion end atom
   9 = Inversion center atom
  10 = CVFF info
*/
char * CVFF_equivalence_auto[123][11]= {{"h", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"d", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"hc", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"hp", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"hs", "h", "hs", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"hn", "hn", "hn", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"hi", "hn", "hn", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"h+", "hn", "hn", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"ho", "hn", "ho", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"h*", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"hspc", "hspc", "hspc", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.3, Ref. 25"},
                                        {"htip", "htip", "htip", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.3, Ref. 25"},
                                        {"hw", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"dw", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"lp", "h", "lp", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 18"},
                                        {"c", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 24"},
                                        {"co", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 24"},
                                        {"c3m", "cg", "c", "c3m_", "c3m_", "c3m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 24"},
                                        {"c4m", "cg", "c", "c4m_", "c4m_", "c4m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 24"},
                                        {"coh", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"c3h", "cg", "c", "c3m_", "c3m_", "c3m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"c4h", "cg", "c", "c4m_", "c4m_", "c4m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"cg", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"ca", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"cn", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"c3", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"c2", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"c1", "cg", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 18"},
                                        {"ci", "c'", "ci", "ci_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 18"},
                                        {"c*", "c'", "c'", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 21"},
                                        {"c\"", "c'", "c'", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 18"},
                                        {"c'", "c'", "c'", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 18"},
                                        {"cp", "c'", "cp", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 18"},
                                        {"c5", "c'", "c5", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 18"},
                                        {"cs", "c'", "cs", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 18"},
                                        {"cr", "c'", "cr", "cr_", "c_", "c'_", "c_", "c=_3", "c_", "c'_", "Ver. 2.0, Ref. 18"},
                                        {"c+", "c'", "cr", "cr_", "c_", "c'_", "c_", "c+_", "c_", "c'_", "Ver. 2.0, Ref. 18"},
                                        {"c-", "c'", "c'", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 18"},
                                        {"c=", "c'", "c=", "c=_3", "c_", "c=_", "c_", "c=_3", "c_", "c=_", "Ver. 2.0, Ref. 18"},
                                        {"c=1", "c'", "c=", "c=_1", "c_", "c=_", "c_", "c=_1", "c_", "c=_", "Ver. 2.0, Ref. 19"},
                                        {"c=2", "c'", "c=", "c=_2", "c_", "c=_", "c_", "c=_2", "c_", "c=_", "Ver. 2.0, Ref. 19"},
                                        {"ct", "c'", "ct", "ct_", "c_", "ct_", "c_", "ct_", "c_", "ct_", "Ver. 2.0, Ref. 18"},
                                        {"na", "n", "n3", "na_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 18"},
                                        {"n3", "n", "n3", "na_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 18"},
                                        {"n3m", "n", "n3", "n3m_", "n3m_", "n3m_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 18"},
                                        {"n4m", "n", "n3", "n4m_", "n4m_", "n4m_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 18"},
                                        {"np", "n", "np", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 18"},
                                        {"npc", "n", "np", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 18"},
                                        {"nh", "n", "np", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 18"},
                                        {"nho", "n", "np", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 18"},
                                        {"nh+", "n", "nh+", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 18"},
                                        {"ni", "n", "ni", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 18"},
                                        {"nn", "n", "n3", "na_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"nb", "n", "n3", "np_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"n+", "n", "n4", "n+_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 18"},
                                        {"n4", "n", "n4", "n+_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 18"},
                                        {"n", "n", "n", "n_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"no", "n", "n", "n_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.3, Ref. 23"},
                                        {"n3n", "n", "n", "n3m_", "n3m_", "n3m_", "n_", "n3n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"n4n", "n", "n", "n4m_", "n4m_", "n4m_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"nr", "n", "n2", "n_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"n2", "n", "n2", "n_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"n1", "n", "n1", "n_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 18"},
                                        {"n=", "n", "n=", "n=_3", "n_", "n_", "n_", "n=_3", "n_", "n=_", "Ver. 2.0, Ref. 18"},
                                        {"n=1", "n", "n=", "n=_1", "n_", "n_", "n_", "n=_1", "n_", "n=_", "Ver. 2.0, Ref. 19"},
                                        {"n=2", "n", "n=", "n=_2", "n_", "n_", "n_", "n=_2", "n_", "n=_", "Ver. 2.0, Ref. 19"},
                                        {"nt", "n", "nt", "nt_", "n_", "nt_", "n_", "nt_", "n_", "nt_", "Ver. 2.0, Ref. 18"},
                                        {"nz", "n", "nz", "nz_", "n_", "nz_", "n_", "nz_", "n_", "nz_", "Ver. 2.0, Ref. 18"},
                                        {"o", "o'", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"oz", "oz", "oz", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"o*", "o*", "o*", "o_", "o_", "o*_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"otip", "otip", "otip", "o_", "o_", "o*_", "o_", "o_", "o_", "o_", "Ver. 2.3, Ref. 25"},
                                        {"ospc", "ospc", "ospc", "o_", "o_", "o*_", "o_", "o_", "o_", "o_", "Ver. 2.3, Ref. 25"},
                                        {"oh", "o'", "oh", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"oc", "o'", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"oe", "o'", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"o3e", "o'", "o", "o3e_", "o3e_", "o3e_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"o4e", "o'", "o", "o4e_", "o4e_", "o4e_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"op", "o'", "op", "op_", "o_", "op_", "o_", "o_", "o_", "op_", "Ver. 2.0, Ref. 18"},
                                        {"o'", "o'", "o'", "o'_", "o'_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"of", "o'", "oh", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"o-", "o'", "o-", "o-_", "o'_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 18"},
                                        {"s", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"s'", "s'", "s'", "s'_", "s'_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"s-", "s'", "s-", "s-_", "s'_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"sc", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"s3e", "s", "s", "s3e_", "s3e_", "s3e_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"s4e", "s", "s", "s4e_", "s4e_", "s4e_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"s1", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"sh", "s", "sh", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 18"},
                                        {"sp", "s'", "sp", "sp_", "s_", "sp_", "s_", "sp_", "s_", "sp_", "Ver. 2.0, Ref. 18"},
                                        {"p", "p", "p", "p_", "p_", "p_", "p_", "p_", "p_", "p_", "Ver. 2.0, Ref. 18"},
                                        {"pz", "p", "p", "p_", "p_", "p_", "p_", "p_", "p_", "p_", "Ver. 2.0, Ref. 18"},
                                        {"ca+", "ca+", "ca+", "ca+_", "ca+_", "ca+_", "ca+_", "ca+_", "ca+_", "ca+_", "Ver. 2.0, Ref. 18"},
                                        {"f", "f", "f", "f_", "f_", "f_", "f_", "f_", "f_", "f_", "Ver. 2.0, Ref. 18"},
                                        {"i", "i", "i", "i_", "f_", "i_", "i_", "i_", "i_", "i_", "Ver. 2.0, Ref. 18"},
                                        {"cl", "cl", "cl", "cl_", "f_", "cl_", "c_", "cl_", "cl_", "cl_", "Ver. 2.0, Ref. 18"},
                                        {"br", "br", "br", "br_", "f_", "br_", "br_", "br_", "br_", "br_", "Ver. 2.0, Ref. 18"},
                                        {"si", "si", "si", "si_", "si_", "si_", "si_", "si_", "si_", "si_", "Ver. 2.0, Ref. 18"},
                                        {"sz", "sz", "sz", "sz", "si_", "si_", "si_", "si_", "si_", "si_", "Ver. 2.0, Ref. 18"},
                                        {"nu", "nu", "nu", "nu_", "nu_", "nu_", "nu_", "nu_", "nu_", "nu_", "Ver. 2.0, Ref. 18"},
                                        {"Cl", "Cl", "Cl", "Cl_", "Cl_", "Cl_", "Cl_", "Cl_", "Cl_", "Cl_", "Ver. 2.0, Ref. 18"},
                                        {"Br", "Br", "Br", "Br_", "Br_", "Br_", "Br_", "Br_", "Br_", "Br_", "Ver. 2.0, Ref. 18"},
                                        {"Na", "Na", "Na", "Na_", "Na_", "Na_", "Na_", "Na_", "Na_", "Na_", "Ver. 2.0, Ref. 18"},
                                        {"ar", "ar", "ar", "ar_", "ar_", "ar_", "ar_", "ar_", "ar_", "ar_", "Ver. 2.0, Ref. 18"},
                                        {"he", "he", "he", "he_", "he_", "he_", "he_", "he_", "he_", "he_", "Ver. 2.4, Ref. 30"},
                                        {"az", "si", "az", "az_", "si_", "si_", "si_", "si_", "si_", "si_", "Ver. 2.1, Ref. 26"},
                                        {"al", "si", "az", "az_", "si_", "si_", "si_", "si_", "si_", "si_", "Ver. 2.1, Ref. 26"},
                                        {"mg", "mg", "si", "si_", "si_", "si_", "si_", "si_", "si_", "si_", "Ver. 2.1, Ref. 26"},
                                        {"Al", "Al", "Al", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Pb", "Pb", "Pb", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Pt", "Pt", "Pt", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Pd", "Pd", "Pd", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Au", "Au", "Au", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Ag", "Ag", "Ag", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Sn", "Sn", "Sn", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Li", "Li", "Li", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Mo", "Mo", "Mo", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Fe", "Fe", "Fe", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"W", "W", "W", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Ni", "Ni", "Ni", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Cr", "Cr", "Cr", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"},
                                        {"Cu", "Cu", "Cu", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.1, Ref. 27"}};

/*
 Equivalence table for key symbols CVFF with:
   0 = Key in atom list
   1 = Non-bonded
   2 = Bonds
   3 = Angles
   4 = Torsion
   5 = Inversion
   6 = CVFF info
*/
char * CVFF_equivalence[129][7]= {{"h", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                  {"d", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                  {"hp", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                  {"hc", "h", "h", "h", "h", "h", "Ver. 2.0, Ref. 18"},
                                  {"hs", "h", "hs", "hs", "hs", "hs", "Ver. 1.0, Ref. 1"},
                                  {"hn", "hn", "hn", "hn", "hn", "hn", "Ver. 1.0, Ref. 1"},
                                  {"ho", "hn", "ho", "ho", "ho", "ho", "Ver. 1.0, Ref. 1"},
                                  {"hi", "hn", "hn", "hn", "hn", "hn", "Ver. 2.0, Ref. 18"},
                                  {"h+", "hn", "hn", "hn", "hn", "hn", "Ver. 2.0, Ref. 18"},
                                  {"h*", "h*", "h*", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                  {"hspc", "hspc", "hspc", "hspc", "hspc", "hspc", "Ver. 2.3, Ref. 25"},
                                  {"htip", "htip", "htip", "htip", "htip", "htip", "Ver. 2.3, Ref. 25"},
                                  {"hw", "h*", "h*", "h*", "h*", "h*", "Ver. 2.0, Ref. 18"},
                                  {"dw", "h*", "h*", "h*", "h*", "h*", "Ver. 2.0, Ref. 18"},
                                  {"h$", "h$", "h$", "h$", "h$", "h$", "Ver. 1.0, Ref. 1"},
                                  {"lp", "h", "lp", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                  {"c", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 24"},
                                  {"co", "cg", "c", "c", "c", "c", "Ver. 2.0, Ref. 24"},
                                  {"c3m", "cg", "c", "c", "c", "c", "Ver. 2.0, Ref. 24"},
                                  {"c4m", "cg", "c", "c", "c", "c", "Ver. 2.0, Ref. 24"},
                                  {"coh", "cg", "c", "c", "c", "c", "Ver. 2.0, Ref. 18"},
                                  {"c3h", "cg", "c", "c", "c", "c", "Ver. 2.0, Ref. 18"},
                                  {"c4h", "cg", "c", "c", "c", "c", "Ver. 2.0, Ref. 18"},
                                  {"cg", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                  {"ca", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                  {"cn", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                  {"c3", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                  {"c2", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                  {"c1", "cg", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                  {"c'", "c'", "c'", "c'", "c'", "c'", "Ver. 1.0, Ref. 1"},
                                  {"c\"", "c'", "c'", "c'", "c'", "c'", "Ver. 2.0, Ref. 18"},
                                  {"c*", "c'", "c'", "c'", "c'", "c'", "Ver. 2.0, Ref. 21"},
                                  {"cp", "c'", "cp", "cp", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                  {"c5", "c'", "c5", "c5", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                  {"cs", "c'", "cs", "cs", "cs", "cs", "Ver. 1.3, Ref. 6"},
                                  {"cr", "c'", "cr", "c'", "cr", "c'", "Ver. 1.0, Ref. 1"},
                                  {"c+", "c'", "cr", "c'", "cr", "c'", "Ver. 2.0, Ref. 18"},
                                  {"c-", "c'", "c'", "c'", "c'", "c'", "Ver. 1.0, Ref. 1"},
                                  {"c=", "c'", "c=", "c=", "c=", "c=", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c'", "c=1", "c=", "c=1", "c=", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c'", "c=2", "c=", "c=2", "c=", "Ver. 2.0, Ref. 19"},
                                  {"ct", "c'", "ct", "ct", "ct", "ct", "Ver. 1.0, Ref. 1"},
                                  {"ci", "c'", "ci", "c5", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                  {"c$", "c$", "c$", "c$", "c$", "c$", "Ver. 1.0, Ref. 1"},
                                  {"n", "n", "n", "n", "n", "n", "Ver. 1.0, Ref. 1"},
                                  {"no", "n", "no", "no", "no", "no", "Ver. 2.3, Ref. 23"},
                                  {"n3n", "n", "n", "n", "n", "n", "Ver. 2.0, Ref. 18"},
                                  {"n4n", "n", "n", "n", "n", "n", "Ver. 2.0, Ref. 18"},
                                  {"n2", "n", "n2", "n", "n2", "n2", "Ver. 1.0, Ref. 1"},
                                  {"n3", "n", "n3", "n3", "n3", "n3", "Ver. 1.0, Ref. 1"},
                                  {"na", "n", "n3", "n3", "n3", "n3", "Ver. 2.0, Ref. 18"},
                                  {"n3m", "n", "n3", "n3", "n3", "n3", "Ver. 2.0, Ref. 18"},
                                  {"n4m", "n", "n3", "n3", "n3", "n3", "Ver. 2.0, Ref. 18"},
                                  {"n4", "n", "n4", "n3", "n3", "n3", "Ver. 1.0, Ref. 1"},
                                  {"n+", "n", "n4", "n3", "n3", "n3", "Ver. 2.0, Ref. 18"},
                                  {"nn", "n", "n3", "n3", "n3", "n3", "Ver. 2.0, Ref. 18"},
                                  {"nb", "n", "n3", "n3", "n3", "n3", "Ver. 2.0, Ref. 18"},
                                  {"np", "n", "np", "np", "np", "np", "Ver. 1.0, Ref. 1"},
                                  {"n=", "n", "n=", "np", "np", "np", "Ver. 1.0, Ref. 1"},
                                  {"n=1", "n", "n=1", "np", "np", "np", "Ver. 2.0, Ref. 19"},
                                  {"n=2", "n", "n=2", "np", "np", "np", "Ver. 2.0, Ref. 19"},
                                  {"nt", "n", "nt", "nt", "nt", "nt", "Ver. 1.0, Ref. 1"},
                                  {"nz", "n", "nz", "nz", "nz", "nz", "Ver. 1.3, Ref. 4"},
                                  {"n1", "n", "n1", "n", "n", "n", "Ver. 1.0, Ref. 1"},
                                  {"ni", "n", "ni", "np", "np", "np", "Ver. 1.0, Ref. 1"},
                                  {"nh", "n", "np", "np", "np", "np", "Ver. 2.0, Ref. 18"},
                                  {"npc", "n", "np", "np", "np", "np", "Ver. 2.0, Ref. 18"},
                                  {"nho", "n", "np", "np", "np", "np", "Ver. 2.0, Ref. 18"},
                                  {"nh+", "n", "nh+", "np", "np", "np", "Ver. 2.0, Ref. 18"},
                                  {"nr", "n", "n2", "n2", "n2", "n2", "Ver. 2.0, Ref. 18"},
                                  {"n$", "n$", "n$", "n$", "n$", "n$", "Ver. 1.0, Ref. 1"},
                                  {"o'", "o'", "o'", "o'", "o'", "o'", "Ver. 1.0, Ref. 1"},
                                  {"o", "o'", "o", "o", "o", "o'", "Ver. 1.0, Ref. 1"},
                                  {"oz", "oz", "oz", "oz", "oz", "oz", "Ver. 1.8, Ref. 14"},
                                  {"o-", "o'", "o-", "o-", "o'", "o'", "Ver. 1.0, Ref. 1"},
                                  {"oh", "o'", "oh", "o", "o", "o'", "Ver. 1.0, Ref. 1"},
                                  {"o*", "o*", "o*", "o*", "o*", "o*", "Ver. 1.0, Ref. 1"},
                                  {"ospc", "ospc", "ospc", "o*", "o*", "o*", "Ver. 2.3, Ref. 25"},
                                  {"otip", "otip", "otip", "o*", "o*", "o*", "Ver. 2.3, Ref. 25"},
                                  {"op", "o'", "op", "op", "op", "op", "Ver. 1.3, Ref. 8"},
                                  {"of", "o'", "oh", "o", "of", "o'", "Ver. 1.0, Ref. 1"},
                                  {"o$", "o$", "o$", "o$", "o$", "o$", "Ver. 1.0, Ref. 1"},
                                  {"oc", "o'", "o", "o", "o", "o'", "Ver. 2.0, Ref. 18"},
                                  {"oe", "o'", "o", "o", "o", "o'", "Ver. 2.0, Ref. 18"},
                                  {"o3e", "o'", "o", "o", "o", "o'", "Ver. 2.0, Ref. 18"},
                                  {"o4e", "o'", "o", "o", "o", "o'", "Ver. 2.0, Ref. 18"},
                                  {"s", "s", "s", "s", "s", "s", "Ver. 1.0, Ref. 1"},
                                  {"s1", "s", "s", "s", "s", "s", "Ver. 1.0, Ref. 1"},
                                  {"sh", "s", "sh", "sh", "sh", "sh", "Ver. 1.0, Ref. 1"},
                                  {"s'", "s'", "s'", "s'", "s'", "s'", "Ver. 1.3, Ref. 7"},
                                  {"sp", "s'", "sp", "sp", "sp", "sp", "Ver. 1.3, Ref. 6"},
                                  {"s$", "s$", "s$", "s$", "s$", "s$", "Ver. 1.0, Ref. 1"},
                                  {"sc", "s", "s", "s", "s", "s", "Ver. 2.0, Ref. 18"},
                                  {"s3e", "s", "s", "s", "s", "s", "Ver. 2.0, Ref. 18"},
                                  {"s4e", "s", "s", "s", "s", "s", "Ver. 2.0, Ref. 18"},
                                  {"p", "p", "p", "p", "p", "p", "Ver. 1.0, Ref. 1"},
                                  {"pz", "p", "p", "p", "p", "p", "Ver. 1.8, Ref. 14"},
                                  {"p$", "p$", "p$", "p$", "p$", "p$", "Ver. 1.0, Ref. 1"},
                                  {"ca+", "ca+", "ca+", "ca+", "ca+", "ca+", "Ver. 1.0, Ref. 1"},
                                  {"f", "f", "f", "f", "f", "f", "Ver. 1.0, Ref. 1"},
                                  {"i", "i", "i", "i", "i", "i", "Ver. 1.3, Ref. 4"},
                                  {"cl", "cl", "cl", "cl", "cl", "cl", "Ver. 1.0, Ref. 1"},
                                  {"br", "br", "br", "br", "br", "br", "Ver. 1.0, Ref. 1"},
                                  {"si", "si", "si", "si", "si", "si", "Ver. 1.0, Ref. 1"},
                                  {"sz", "sz", "sz", "sz", "sz", "sz", "Ver. 1.8, Ref. 14"},
                                  {"nu", "nu", "nu", "nu", "nu", "nu", "Ver. 1.0, Ref. 1"},
                                  {"Cl", "Cl", "Cl", "Cl", "Cl", "Cl", "Ver. 1.0, Ref. 1"},
                                  {"Br", "Br", "Br", "Br", "Br", "Br", "Ver. 1.0, Ref. 1"},
                                  {"Na", "Na", "Na", "Na", "Na", "Na", "Ver. 1.0, Ref. 1"},
                                  {"ar", "ar", "ar", "ar", "ar", "ar", "Ver. 1.3, Ref. 9"},
                                  {"he", "he", "he", "he", "he", "he", "Ver. 2.4, Ref. 30"},
                                  {"al", "sz", "al", "si", "si", "si", "Ver. 2.1, Ref. 26"},
                                  {"az", "sz", "az", "sz", "sz", "sz", "Ver. 2.1, Ref. 26"},
                                  {"Al", "Al", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Pt", "Pt", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Pd", "Pd", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Au", "Au", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Ag", "Ag", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Sn", "Na", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"K", "K", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Li", "Li", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Mo", "Mo", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Fe", "Fe", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"W", "W", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Ni", "Ni", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Cr", "Cr", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Cu", "Cu", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"Pb", "Pb", "Br", "c", "c", "c", "Ver. 2.1, Ref. 26"},
                                  {"mg", "mg", "sz", "sz", "sz", "sz", "Ver. 2.1, Ref. 27"}};

/*
 Quadratic bonds
  0= Key_a, 1= Key_b, 2= R0 (A), 3= K2 (kcal mol-1), 4= CVFF info

  E = K2 * (R - R0)^2
*/
char * CVFF_bonds_auto[776][5]= {{"no", "o-", "1.2178", "560.9942", "Ver. 2.3, Ref. 23"},
                                 {"no", "cp", "1.4720", "351.2527", "Ver. 2.3, Ref. 23"},
                                 {"c'", "cp", "1.4720", "351.2527", "Ver. 2.3, Ref. 23"},
                                 {"c", "o", "1.4250", "273.2000", "Ver. 1.0, Ref. 1"},
                                 {"c", "h", "1.1050", "340.6175", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "1.5260", "322.7158", "Ver. 1.0, Ref. 1"},
                                 {"c", "c'", "1.5200", "283.0924", "Ver. 1.0, Ref. 1"},
                                 {"c'", "o'", "1.2300", "615.3220", "Ver. 1.0, Ref. 1"},
                                 {"n", "hn", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"n", "lp", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"n", "c'", "1.3200", "388.0000", "Ver. 1.0, Ref. 1"},
                                 {"n", "cr", "1.3200", "388.0000", "Ver. 1.0, Ref. 1"},
                                 {"c", "n", "1.4600", "377.5752", "Ver. 1.0, Ref. 1"},
                                 {"n1", "cr", "1.3200", "388.0000", "Ver. 1.0, Ref. 1"},
                                 {"n1", "hn", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"n1", "lp", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"c", "n1", "1.4600", "377.5752", "Ver. 1.0, Ref. 1"},
                                 {"oh", "ho", "0.9600", "540.6336", "Ver. 1.0, Ref. 1"},
                                 {"oh", "lp", "0.9600", "540.6336", "Ver. 1.0, Ref. 1"},
                                 {"oh", "c", "1.4200", "384.0000", "Ver. 1.0, Ref. 1"},
                                 {"o", "ho", "0.9600", "493.8480", "Ver. 1.0, Ref. 1"},
                                 {"o", "lp", "0.9600", "493.8480", "Ver. 1.0, Ref. 1"},
                                 {"c'", "h", "1.1050", "340.6175", "Ver. 1.0, Ref. 1"},
                                 {"c'", "o", "1.3700", "400.0000", "Ver. 1.0, Ref. 1"},
                                 {"c'", "oh", "1.3700", "400.0000", "Ver. 1.0, Ref. 1"},
                                 {"c'", "o-", "1.2500", "540.0000", "Ver. 1.0, Ref. 1"},
                                 {"cp", "h", "1.0800", "363.4164", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "1.3400", "480.0000", "Ver. 1.0, Ref. 1"},
                                 {"cp", "c", "1.5100", "283.0924", "Ver. 1.0, Ref. 1"},
                                 {"cp", "oh", "1.3700", "384.0000", "Ver. 1.0, Ref. 1"},
                                 {"sh", "hs", "1.3300", "274.1288", "Ver. 1.0, Ref. 1"},
                                 {"sh", "lp", "1.3300", "274.1288", "Ver. 1.0, Ref. 1"},
                                 {"c", "sh", "1.8200", "228.0000", "Ver. 1.0, Ref. 1"},
                                 {"c", "s", "1.8000", "228.0000", "Ver. 1.0, Ref. 1"},
                                 {"c", "s1", "1.8000", "228.0000", "Ver. 1.0, Ref. 1"},
                                 {"s", "s", "2.0000", "180.0000", "Ver. 1.0, Ref. 1"},
                                 {"s1", "s1", "2.0000", "180.0000", "Ver. 1.0, Ref. 1"},
                                 {"n2", "hn", "1.0260", "457.4592", "Ver. 1.0, Ref. 1"},
                                 {"n2", "lp", "1.0260", "457.4592", "Ver. 1.0, Ref. 1"},
                                 {"c", "n2", "1.4700", "356.5988", "Ver. 1.0, Ref. 1"},
                                 {"n2", "c'", "1.3200", "388.0000", "Ver. 1.0, Ref. 1"},
                                 {"n2", "cr", "1.3200", "388.0000", "Ver. 1.0, Ref. 1"},
                                 {"n2", "cp", "1.3200", "388.0000", "Ver. 1.0, Ref. 1"},
                                 {"n3", "hn", "1.0260", "457.4592", "Ver. 1.0, Ref. 1"},
                                 {"n3", "lp", "1.0260", "457.4592", "Ver. 1.0, Ref. 1"},
                                 {"c", "n3", "1.4700", "356.5988", "Ver. 1.0, Ref. 1"},
                                 {"n4", "hn", "1.0260", "457.4592", "Ver. 1.0, Ref. 1"},
                                 {"n4", "lp", "1.0260", "457.4592", "Ver. 1.0, Ref. 1"},
                                 {"c", "n4", "1.4700", "356.5988", "Ver. 1.0, Ref. 1"},
                                 {"cp", "np", "1.2600", "560.0000", "Ver. 1.0, Ref. 1"},
                                 {"np", "hn", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"np", "lp", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"cp", "c5", "1.3400", "280.0000", "Ver. 1.0, Ref. 1"},
                                 {"c5", "h", "1.0800", "363.4164", "Ver. 1.0, Ref. 1"},
                                 {"c5", "c5", "1.3900", "280.0000", "Ver. 1.0, Ref. 1"},
                                 {"c5", "c", "1.5100", "283.0924", "Ver. 1.0, Ref. 1"},
                                 {"c5", "np", "1.3800", "320.0000", "Ver. 1.0, Ref. 1"},
                                 {"c5", "o", "1.3700", "420.0000", "Ver. 1.2, Ref. 3"},
                                 {"c5", "op", "1.3700", "420.0000", "Ver. 1.3, Ref. 8"},
                                 {"c5", "s", "1.7300", "228.0000", "Ver. 1.2, Ref. 3"},
                                 {"cs", "h", "1.0800", "379.0809", "Ver. 1.3, Ref. 6"},
                                 {"cs", "c5", "1.3630", "400.0000", "Ver. 1.3, Ref. 6"},
                                 {"sp", "cs", "1.7106", "320.0000", "Ver. 1.3, Ref. 6"},
                                 {"ci", "h", "1.0800", "363.4164", "Ver. 1.0, Ref. 1"},
                                 {"ci", "ci", "1.3900", "280.0000", "Ver. 1.0, Ref. 1"},
                                 {"ci", "c", "1.5100", "283.0924", "Ver. 1.0, Ref. 1"},
                                 {"ci", "ni", "1.3800", "320.0000", "Ver. 1.0, Ref. 1"},
                                 {"ni", "hn", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"ni", "lp", "1.0260", "483.4512", "Ver. 1.0, Ref. 1"},
                                 {"cp", "n", "1.4200", "280.0000", "Ver. 1.0, Ref. 1"},
                                 {"o*", "h*", "0.9600", "540.6336", "Ver. 1.0, Ref. 1"},
                                 {"ospc", "hspc", "1.0000", "540.6336", "Ver. 2.3, Ref. 25"},
                                 {"otip", "htip", "0.9570", "540.6336", "Ver. 2.3, Ref. 25"},
                                 {"o*", "lp", "0.9600", "540.6336", "Ver. 1.0, Ref. 1"},
                                 {"p", "oh", "1.5700", "300.0000", "Ver. 1.0, Ref. 1"},
                                 {"p", "o'", "1.5300", "560.0000", "Ver. 1.0, Ref. 1"},
                                 {"p", "o", "1.6100", "245.2000", "Ver. 1.0, Ref. 1"},
                                 {"p", "o-", "1.5300", "480.0000", "Ver. 1.0, Ref. 1"},
                                 {"p", "h", "1.5000", "224.0000", "Ver. 1.0, Ref. 1"},
                                 {"p", "lp", "1.5000", "224.0000", "Ver. 1.0, Ref. 1"},
                                 {"np", "c", "1.4750", "336.8000", "Ver. 1.0, Ref. 1"},
                                 {"n=", "c", "1.4750", "336.8000", "Ver. 1.0, Ref. 1"},
                                 {"n=1", "c", "1.4750", "336.8000", "Ver. 2.0, Ref. 19"},
                                 {"n=2", "c", "1.4750", "336.8000", "Ver. 2.0, Ref. 19"},
                                 {"c=", "n=", "1.2600", "560.0000", "Ver. 1.0, Ref. 1"},
                                 {"c=", "n=1", "1.2600", "560.0000", "Ver. 2.0, Ref. 19"},
                                 {"c=1", "n=", "1.2600", "560.0000", "Ver. 2.0, Ref. 19"},
                                 {"cr", "n=", "1.2600", "560.0000", "Ver. 1.0, Ref. 1"},
                                 {"cr", "n=1", "1.2600", "560.0000", "Ver. 2.0, Ref. 19"},
                                 {"cr", "n=2", "1.2600", "560.0000", "Ver. 2.0, Ref. 19"},
                                 {"c=", "c'", "1.5000", "322.8000", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "c'", "1.5000", "322.8000", "Ver. 2.0, Ref. 19"},
                                 {"c=2", "c'", "1.5000", "322.8000", "Ver. 2.0, Ref. 19"},
                                 {"c=", "c", "1.5000", "322.8000", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "c", "1.5000", "322.8000", "Ver. 2.0, Ref. 19"},
                                 {"c=2", "c", "1.5000", "322.8000", "Ver. 2.0, Ref. 19"},
                                 {"c=", "c=", "1.3300", "655.2000", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "c=", "1.3300", "655.2000", "Ver. 2.0, Ref. 19"},
                                 {"c=", "h", "1.0900", "361.6000", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "h", "1.0900", "361.6000", "Ver. 2.0, Ref. 19"},
                                 {"c=2", "h", "1.0900", "361.6000", "Ver. 2.0, Ref. 19"},
                                 {"f", "c", "1.3630", "496.0000", "Ver. 1.0, Ref. 1"},
                                 {"f", "lp", "1.3630", "496.0000", "Ver. 1.0, Ref. 1"},
                                 {"cl", "c", "1.7610", "314.0000", "Ver. 1.0, Ref. 1"},
                                 {"cl", "lp", "1.7610", "314.0000", "Ver. 1.0, Ref. 1"},
                                 {"br", "c", "1.9200", "223.6000", "Ver. 1.0, Ref. 1"},
                                 {"br", "lp", "1.9200", "223.6000", "Ver. 1.0, Ref. 1"},
                                 {"f", "cp", "1.3630", "496.0000", "Ver. 1.0, Ref. 1"},
                                 {"cl", "cp", "1.7610", "314.0000", "Ver. 1.0, Ref. 1"},
                                 {"br", "cp", "1.9200", "223.6000", "Ver. 1.0, Ref. 1"},
                                 {"si", "c", "1.8090", "238.0000", "Ver. 1.0, Ref. 1"},
                                 {"si", "h", "1.3820", "222.4000", "Ver. 1.0, Ref. 1"},
                                 {"si", "o", "1.6650", "392.8000", "Ver. 1.0, Ref. 1"},
                                 {"h", "h", "0.74611", "398.7500", "Ver. 1.3, Ref. 4"},
                                 {"d", "d", "0.74164", "398.2400", "Ver. 1.3, Ref. 4"},
                                 {"f", "f", "1.417", "259.0700", "Ver. 1.3, Ref. 4"},
                                 {"cl", "cl", "1.988", "236.5500", "Ver. 1.3, Ref. 4"},
                                 {"br", "br", "2.290", "175.6400", "Ver. 1.3, Ref. 4"},
                                 {"i", "i", "2.662", "123.2100", "Ver. 1.3, Ref. 4"},
                                 {"o", "o", "1.208", "833.7200", "Ver. 1.3, Ref. 4"},
                                 {"nz", "nz", "1.09758", "1632.5100", "Ver. 1.3, Ref. 4"},
                                 {"nt", "nt", "1.09758", "1632.5100", "Ver. 1.3, Ref. 5"},
                                 {"c'", "s'", "1.6110", "510.2775", "Ver. 1.3, Ref. 7"},
                                 {"sz", "c", "1.8090", "238.0000", "Ver. 1.0, Ref. 1"},
                                 {"sz", "o", "1.6150", "392.8000", "Ver. 1.8, Ref. 14"},
                                 {"sz", "oz", "1.6150", "392.8000", "Ver. 1.8, Ref. 14"},
                                 {"sz", "oh", "1.6350", "392.8000", "Ver. 1.8, Ref. 14"},
                                 {"sz", "h", "1.3820", "222.4000", "Ver. 1.8, Ref. 14"},
                                 {"oz", "ho", "0.9600", "540.6336", "Ver. 1.8, Ref. 14"},
                                 {"al", "c", "1.8090", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"al", "h", "1.3820", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"al", "o", "1.7750", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"al", "oz", "1.7750", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"az", "c", "1.8090", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"az", "o", "1.7750", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"az", "oz", "1.7750", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"az", "oh", "1.7750", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"az", "h", "1.3820", "392.8000", "Ver. 2.1, Ref. 26"},
                                 {"sz", "f", "1.6650", "392.8000", "Ver. 2.1, Ref. 28"},
                                 {"sz", "sz", "3.0900", "392.8000", "Ver. 2.1, Ref. 28"},
                                 {"sz", "az", "3.0900", "392.8000", "Ver. 2.1, Ref. 28"},
                                 {"az", "az", "3.0900", "392.8000", "Ver. 2.1, Ref. 28"},
                                 {"sz", "al", "3.0900", "392.8000", "Ver. 2.1, Ref. 28"},
                                 {"si", "al", "3.0900", "392.8000", "Ver. 2.1, Ref. 28"},
                                 {"c3m_", "c3m_", "1.5100", "322.7158", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c4m_", "1.5260", "322.7158", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c_", "1.5260", "322.7158", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c'_", "1.5200", "283.0924", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "cp_", "1.5100", "283.0924", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "ct_", "1.4000", "340.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "na_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n3m_", "1.4850", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n4m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n_", "1.4600", "377.5752", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "np_", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n=", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n=_1", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n=_2", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n=_3", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "n+_", "1.4620", "270.8836", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "o_", "1.4250", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "o'_", "1.3800", "318.9484", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "o3e_", "1.4340", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "o4e_", "1.4250", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "op_", "1.3800", "346.5484", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "s_", "1.8000", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "sp_", "1.7700", "242.5324", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "s'_", "1.7700", "257.3324", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "s3e_", "1.8000", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "s4e_", "1.8000", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "p_", "1.7500", "249.1344", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "si_", "1.8090", "238.0000", "Ver. 2.0, Ref. 18"},
                                 {"c3m_", "i_", "2.1200", "200.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c4m_", "1.5520", "322.7158", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c_", "1.5260", "322.7158", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c'_", "1.5200", "283.0924", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "cp_", "1.5100", "283.0924", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "ct_", "1.4000", "340.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "na_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n3m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n4m_", "1.4670", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n_", "1.4600", "377.5752", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "np_", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n=", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n=_1", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n=_2", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n=_3", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "n+_", "1.4620", "270.8836", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "o_", "1.4250", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "o'_", "1.3800", "318.9484", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "o3e_", "1.4250", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "o4e_", "1.4462", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "op_", "1.3800", "346.5484", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "s_", "1.8000", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "sp_", "1.7700", "242.5324", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "s'_", "1.7700", "257.3324", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "s3e_", "1.8000", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "s4e_", "1.8470", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "p_", "1.7500", "249.1344", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "si_", "1.8090", "238.0000", "Ver. 2.0, Ref. 18"},
                                 {"c4m_", "i_", "2.1200", "200.0000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n3m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n4m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n3m_", "1.4460", "272.0000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n4m_", "1.4000", "332.0000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "s3e_", "1.7700", "175.0035", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "s4e_", "1.7700", "175.0035", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n3m_", "1.4200", "280.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n4m_", "1.4200", "280.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "s3e", "1.7300", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "s4e", "1.7300", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n3m_", "1.3820", "286.8096", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n4m_", "1.3820", "286.8096", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "s3e_", "1.7200", "271.4328", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "s4e_", "1.7200", "271.4328", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n3m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n4m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 18"},
                                 {"na_", "s3e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"na_", "s4e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n3m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n4m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n_", "1.3670", "221.6968", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "np_", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n=", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n=_1", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n=_2", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n=_3", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "n+_", "1.3940", "211.1592", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "o_", "1.3250", "301.3500", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "op_", "1.3120", "274.1760", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "o'_", "1.3120", "246.5760", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "s_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "sp_", "1.7020", "190.9752", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "s'_", "1.7020", "205.7752", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "s3e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "s4e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "p_", "1.6820", "210.5400", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "h_", "1.0260", "457.4592", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "f_", "1.3520", "200.9852", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "cl_", "1.6890", "226.4260", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "br_", "1.8370", "203.8340", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "i_", "2.0230", "184.0104", "Ver. 2.0, Ref. 18"},
                                 {"n3m_", "si_", "1.7920", "204.4236", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n4m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n_", "1.3670", "221.6968", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "np_", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n=", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n=_1", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n=_2", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n=_3", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "n+_", "1.3940", "211.1592", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "o_", "1.3250", "301.3500", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "op_", "1.3120", "274.1760", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "o'_", "1.3120", "246.5760", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "s_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "sp_", "1.7020", "190.9752", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "s'_", "1.7020", "205.7752", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "s3e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "s4e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "p_", "1.6820", "210.5400", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "h_", "1.0260", "457.4592", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "f_", "1.3520", "200.9852", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "cl_", "1.6890", "226.4260", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "br_", "1.8370", "203.8340", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "i_", "2.0230", "184.0104", "Ver. 2.0, Ref. 18"},
                                 {"n4m_", "si_", "1.7920", "204.4236", "Ver. 2.0, Ref. 18"},
                                 {"n_", "s3e_", "1.7050", "210.6208", "Ver. 2.0, Ref. 18"},
                                 {"n_", "s4e_", "1.7050", "210.6208", "Ver. 2.0, Ref. 18"},
                                 {"np_", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"np_", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"o_", "s3e_", "1.6930", "288.0848", "Ver. 2.0, Ref. 18"},
                                 {"o_", "s4e_", "1.6930", "288.0848", "Ver. 2.0, Ref. 18"},
                                 {"op_", "s3e_", "1.6800", "266.7748", "Ver. 2.0, Ref. 18"},
                                 {"op_", "s4e_", "1.6800", "266.7748", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "s3e_", "1.6500", "239.5744", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "s4e_", "1.6500", "239.5744", "Ver. 2.0, Ref. 18"},
                                 {"s_", "s3e_", "2.1000", "160.0000", "Ver. 2.0, Ref. 18"},
                                 {"s_", "s4e_", "2.1000", "160.0000", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "s3e_", "2.0400", "175.1260", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "s4e_", "2.0400", "175.1260", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "s3e_", "2.0400", "189.9260", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "s4e_", "2.0400", "189.9260", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "s3e_", "2.0000", "180.0000", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "s4e_", "2.0000", "180.0000", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "p_", "2.0200", "186.8792", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "h_", "1.3300", "274.1288", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "f_", "1.6900", "204.8184", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "cl_", "2.0270", "212.0812", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "br_", "2.1750", "187.8836", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "i_", "2.3610", "167.7624", "Ver. 2.0, Ref. 18"},
                                 {"s3e_", "si_", "2.1300", "177.2928", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "s4e_", "2.0000", "180.0000", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "p_", "2.0200", "186.8792", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "h_", "1.3300", "274.1288", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "f_", "1.6900", "204.8184", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "cl_", "2.0270", "212.0812", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "br_", "2.1750", "187.8836", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "i_", "2.3610", "167.7624", "Ver. 2.0, Ref. 18"},
                                 {"s4e_", "si_", "2.1300", "177.2928", "Ver. 2.0, Ref. 18"},
                                 {"c_", "c_", "1.5260", "322.7158", "Ver. 2.0, Ref. 18"},
                                 {"c_", "c'_", "1.5200", "283.0924", "Ver. 2.0, Ref. 18"},
                                 {"c_", "cp_", "1.5100", "283.0924", "Ver. 2.0, Ref. 18"},
                                 {"c_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "ct_", "1.4000", "340.0000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "na_", "1.4700", "356.5988", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n_", "1.4600", "377.5752", "Ver. 2.0, Ref. 18"},
                                 {"c_", "np_", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n=", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n=_1", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n=_2", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n=_3", "1.4750", "336.8000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "n+_", "1.4620", "270.8836", "Ver. 2.0, Ref. 18"},
                                 {"c_", "o_", "1.4250", "273.2000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "op_", "1.3800", "346.5484", "Ver. 2.0, Ref. 18"},
                                 {"c_", "o'_", "1.3800", "318.9484", "Ver. 2.0, Ref. 18"},
                                 {"c_", "s_", "1.8000", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "sp_", "1.7700", "242.5324", "Ver. 2.0, Ref. 18"},
                                 {"c_", "s'_", "1.7700", "257.3324", "Ver. 2.0, Ref. 18"},
                                 {"c_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 18"},
                                 {"c_", "p_", "1.7500", "249.1344", "Ver. 2.0, Ref. 18"},
                                 {"c_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "si_", "1.8090", "238.0000", "Ver. 2.0, Ref. 18"},
                                 {"c_", "i_", "2.1200", "200.0000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "c'_", "1.5000", "266.4000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "cp_", "1.5000", "284.3316", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "ct_", "1.4200", "311.1492", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n_", "1.3600", "388.0000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n=", "1.4050", "296.2996", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n=_1", "1.4050", "296.2996", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n=_2", "1.4050", "296.2996", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "n=_3", "1.4050", "296.2996", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "np_", "1.4050", "296.2996", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "o_", "1.3400", "400.0000", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "o'_", "1.2200", "615.3220", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "o-_", "1.2500", "572.8860", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "op_", "1.3500", "294.1008", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "s_", "1.7700", "175.0035", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "s'_", "1.6110", "510.2775", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "s-_", "1.6800", "280.3060", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "sp_", "1.7400", "215.3532", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "p_", "1.7200", "241.3820", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "f_", "1.3900", "217.7092", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "cl_", "1.7270", "249.8588", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "br_", "1.8750", "228.2808", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "si_", "1.8300", "239.3552", "Ver. 2.0, Ref. 18"},
                                 {"c'_", "i_", "2.0610", "208.6024", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "cp_", "1.3900", "480.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "ct_", "1.4000", "321.6716", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "na_", "1.4120", "257.7752", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n_", "1.4200", "280.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n=", "1.3850", "316.0380", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n=_1", "1.3850", "316.0380", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n=_2", "1.3850", "316.0380", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n=_3", "1.3850", "316.0380", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "np_", "1.3500", "440.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "n+_", "1.4120", "251.3344", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "o_", "1.3700", "384.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "o'_", "1.3300", "297.4852", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "op_", "1.3700", "420.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "s_", "1.7300", "228.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "s'_", "1.7200", "239.8024", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "sp_", "1.7106", "320.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "h_", "1.0800", "363.4164", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "p_", "1.7000", "235.0428", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "i_", "2.0410", "217.4512", "Ver. 2.0, Ref. 18"},
                                 {"cp_", "si_", "1.8100", "225.1676", "Ver. 2.0, Ref. 18"},
                                 {"ci", "ci", "1.3900", "280.0000", "Ver. 2.0, Ref. 18"},
                                 {"ci", "ni", "1.3800", "320.0000", "Ver. 2.0, Ref. 18"},
                                 {"ci", "h_", "1.0800", "363.4164", "Ver. 2.0, Ref. 18"},
                                 {"c=", "c=", "1.3300", "655.2000", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "c=_3", "1.3300", "655.2000", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "c=_3", "1.3300", "655.2000", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "c=_2", "1.4100", "480.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "c=_2", "1.4800", "320.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "c=_1", "1.4800", "320.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "c=_3", "1.4800", "320.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 18"},
                                 {"c=", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 18"},
                                 {"c=", "n=", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "n=_1", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "n=_2", "1.3430", "493.5268", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "n=_2", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "n=_1", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "n=_1", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "n=_2", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "n=_3", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 18"},
                                 {"c=", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 18"},
                                 {"c=", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 18"},
                                 {"c=", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 18"},
                                 {"c=", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 18"},
                                 {"c=", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 18"},
                                 {"c=", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 18"},
                                 {"c=", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 18"},
                                 {"c=", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 18"},
                                 {"c=", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 18"},
                                 {"c=", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 18"},
                                 {"c=", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 18"},
                                 {"c=_1", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 18"},
                                 {"c=_2", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 18"},
                                 {"c=_3", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 18"},
                                 {"cr_", "n=", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"cr_", "n=_1", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"cr_", "n=_2", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"cr_", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 18"},
                                 {"cr_", "n_", "1.3200", "388.0000", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "ct_", "1.2040", "800.0000", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "nt_", "1.1580", "880.0000", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "na_", "1.3820", "286.8096", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n_", "1.3550", "289.4448", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n=", "1.3550", "342.2448", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n=_1", "1.3550", "342.2448", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n=_2", "1.3550", "342.2448", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n=_3", "1.3550", "342.2448", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "n+_", "1.3820", "278.3768", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "o_", "1.3130", "367.8164", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "s_", "1.7200", "271.4328", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "op_", "1.3000", "346.4680", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "o'_", "1.3000", "318.8680", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "sp_", "1.6900", "256.0344", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "s'_", "1.6900", "270.8344", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "p_", "1.6700", "273.4668", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "h_", "1.0530", "316.9016", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "f_", "1.3400", "278.3932", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "cl_", "1.6770", "292.1952", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "br_", "1.8250", "268.8528", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "i_", "2.0110", "248.9268", "Ver. 2.0, Ref. 18"},
                                 {"ct_", "si_", "1.7800", "267.2964", "Ver. 2.0, Ref. 18"},
                                 {"na_", "na_", "1.3940", "220.8000", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n_", "1.3670", "221.6968", "Ver. 2.0, Ref. 18"},
                                 {"na_", "np_", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n=", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n=_1", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n=_2", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n=_3", "1.3670", "274.4968", "Ver. 2.0, Ref. 18"},
                                 {"na_", "n+_", "1.3940", "211.1592", "Ver. 2.0, Ref. 18"},
                                 {"na_", "o_", "1.3250", "301.3500", "Ver. 2.0, Ref. 18"},
                                 {"na_", "s_", "1.7320", "206.9404", "Ver. 2.0, Ref. 18"},
                                 {"na_", "op_", "1.3120", "274.1760", "Ver. 2.0, Ref. 18"},
                                 {"na_", "o'_", "1.3120", "246.5760", "Ver. 2.0, Ref. 18"},
                                 {"na_", "sp_", "1.7020", "190.9752", "Ver. 2.0, Ref. 18"},
                                 {"na_", "s'_", "1.7020", "205.7752", "Ver. 2.0, Ref. 18"},
                                 {"na_", "p_", "1.6820", "210.5400", "Ver. 2.0, Ref. 18"},
                                 {"na_", "h_", "1.0260", "457.4592", "Ver. 2.0, Ref. 18"},
                                 {"na_", "f_", "1.3520", "200.9852", "Ver. 2.0, Ref. 18"},
                                 {"na_", "cl_", "1.6890", "226.4260", "Ver. 2.0, Ref. 18"},
                                 {"na_", "br_", "1.8370", "203.8340", "Ver. 2.0, Ref. 18"},
                                 {"na_", "i_", "2.0230", "184.0104", "Ver. 2.0, Ref. 18"},
                                 {"na_", "si_", "1.7920", "204.4236", "Ver. 2.0, Ref. 18"},
                                 {"n_", "n_", "1.3400", "220.0000", "Ver. 2.0, Ref. 18"},
                                 {"n_", "np_", "1.3400", "272.8000", "Ver. 2.0, Ref. 18"},
                                 {"n_", "n=", "1.3400", "272.8000", "Ver. 2.0, Ref. 18"},
                                 {"n_", "n=_1", "1.3400", "272.8000", "Ver. 2.0, Ref. 18"},
                                 {"n_", "n=_2", "1.3400", "272.8000", "Ver. 2.0, Ref. 18"},
                                 {"n_", "n=_3", "1.3400", "272.8000", "Ver. 2.0, Ref. 18"},
                                 {"n_", "n+_", "1.3670", "210.6360", "Ver. 2.0, Ref. 18"},
                                 {"n_", "o_", "1.2980", "301.2632", "Ver. 2.0, Ref. 18"},
                                 {"n_", "s_", "1.7050", "210.6208", "Ver. 2.0, Ref. 18"},
                                 {"n_", "op_", "1.2850", "267.1080", "Ver. 2.0, Ref. 18"},
                                 {"n_", "o'_", "1.2850", "239.5080", "Ver. 2.0, Ref. 18"},
                                 {"n_", "sp_", "1.6750", "193.4844", "Ver. 2.0, Ref. 18"},
                                 {"n_", "s'_", "1.6750", "208.2844", "Ver. 2.0, Ref. 18"},
                                 {"n_", "p_", "1.6550", "219.9524", "Ver. 2.0, Ref. 18"},
                                 {"n_", "h_", "1.0260", "483.4512", "Ver. 2.0, Ref. 18"},
                                 {"n_", "f_", "1.3250", "189.3856", "Ver. 2.0, Ref. 18"},
                                 {"n_", "cl_", "1.6620", "227.5604", "Ver. 2.0, Ref. 18"},
                                 {"n_", "br_", "1.8100", "206.3980", "Ver. 2.0, Ref. 18"},
                                 {"n_", "i_", "1.9960", "186.6972", "Ver. 2.0, Ref. 18"},
                                 {"n_", "si_", "1.7650", "216.8064", "Ver. 2.0, Ref. 18"},
                                 {"np_", "np_", "1.3400", "408.0000", "Ver. 2.0, Ref. 18"},
                                 {"np_", "n=", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"np_", "n=_1", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"np_", "n=_2", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"np_", "n=_3", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"np_", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 18"},
                                 {"np_", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 18"},
                                 {"np_", "o'_", "1.2850", "292.3080", "Ver. 2.0, Ref. 18"},
                                 {"np_", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"np_", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"np_", "s'_", "1.6750", "261.0844", "Ver. 2.0, Ref. 18"},
                                 {"np_", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 18"},
                                 {"np_", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 18"},
                                 {"np_", "h_", "1.0260", "483.4512", "Ver. 2.0, Ref. 18"},
                                 {"np_", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 18"},
                                 {"np_", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 18"},
                                 {"np_", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 18"},
                                 {"np_", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 18"},
                                 {"np_", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 18"},
                                 {"n=", "n=", "1.2100", "651.2000", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "n=_3", "1.2100", "651.2000", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "n=_3", "1.2100", "651.2000", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "n=_2", "1.2760", "488.0000", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "n=_2", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "n=_1", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "n=_3", "1.3400", "325.6000", "Ver. 2.0, Ref. 18"},
                                 {"n=", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 18"},
                                 {"n=", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 18"},
                                 {"n=", "o'_", "1.1600", "575.8720", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "o'_", "1.2850", "292.3080", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "o'_", "1.2225", "434.0900", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "o'_", "1.1600", "575.8720", "Ver. 2.0, Ref. 18"},
                                 {"n=", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"np_", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 18"},
                                 {"n=", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 18"},
                                 {"n=", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 18"},
                                 {"n=", "s'_", "1.5900", "489.2400", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "s'_", "1.6750", "261.0844", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "s'_", "1.6325", "375.1624", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "s'_", "1.5900", "489.2400", "Ver. 2.0, Ref. 18"},
                                 {"n=", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 18"},
                                 {"n=", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 18"},
                                 {"n=", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 18"},
                                 {"n=", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 18"},
                                 {"n=", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 18"},
                                 {"n=", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 18"},
                                 {"n=", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 18"},
                                 {"n=_1", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 18"},
                                 {"n=_2", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 18"},
                                 {"n=_3", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "n+_", "1.3940", "193.5604", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "o_", "1.3250", "291.0020", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "s_", "1.7320", "200.1168", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "op_", "1.3120", "261.7060", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "o'_", "1.3120", "234.1060", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "sp_", "1.7020", "183.2952", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "s'_", "1.7020", "198.0952", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "p_", "1.6820", "209.1536", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "h_", "1.0650", "461.1848", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "f_", "1.3520", "178.0372", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "cl_", "1.6890", "217.7248", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "br_", "1.8370", "196.5940", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "i_", "2.0230", "177.2764", "Ver. 2.0, Ref. 18"},
                                 {"n+_", "si_", "1.7920", "206.9412", "Ver. 2.0, Ref. 18"},
                                 {"nz_", "nz_", "1.0976", "1632.4955", "Ver. 2.0, Ref. 18"},
                                 {"nt_", "nt_", "1.0976", "1632.4955", "Ver. 2.0, Ref. 18"},
                                 {"o_", "o_", "1.2080", "833.6868", "Ver. 2.0, Ref. 18"},
                                 {"o_", "s_", "1.6930", "288.0848", "Ver. 2.0, Ref. 18"},
                                 {"o_", "op_", "1.2430", "350.7720", "Ver. 2.0, Ref. 18"},
                                 {"o_", "o'_", "1.2430", "323.1720", "Ver. 2.0, Ref. 18"},
                                 {"o_", "sp_", "1.6330", "271.9584", "Ver. 2.0, Ref. 18"},
                                 {"o_", "s'_", "1.6330", "286.7584", "Ver. 2.0, Ref. 18"},
                                 {"o_", "p_", "1.6100", "245.2000", "Ver. 2.0, Ref. 18"},
                                 {"o_", "si_", "1.6650", "392.8000", "Ver. 2.0, Ref. 18"},
                                 {"o_", "h_", "0.9600", "493.8480", "Ver. 2.0, Ref. 18"},
                                 {"o_", "f_", "1.4180", "224.0000", "Ver. 2.0, Ref. 18"},
                                 {"o_", "cl_", "1.6500", "307.0632", "Ver. 2.0, Ref. 18"},
                                 {"o_", "br_", "1.7980", "284.7988", "Ver. 2.0, Ref. 18"},
                                 {"o_", "i_", "1.9840", "264.9868", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "o'_", "1.1100", "484.8000", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "op_", "1.2300", "272.4000", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "s_", "1.5900", "360.4188", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "sp_", "1.5600", "341.2736", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "s'_", "1.5400", "421.5188", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "p_", "1.4800", "524.0000", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "h_", "0.9830", "458.4610", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "f_", "1.2700", "204.0505", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "cl_", "1.6070", "251.7939", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "br_", "1.7550", "233.2954", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "i_", "1.9410", "213.2317", "Ver. 2.0, Ref. 18"},
                                 {"o'_", "si_", "1.6500", "454.7464", "Ver. 2.0, Ref. 18"},
                                 {"op_", "op_", "1.2300", "300.0000", "Ver. 2.0, Ref. 18"},
                                 {"op_", "s_", "1.6800", "266.7748", "Ver. 2.0, Ref. 18"},
                                 {"op_", "sp_", "1.6500", "247.8440", "Ver. 2.0, Ref. 18"},
                                 {"op_", "s'_", "1.6200", "262.9744", "Ver. 2.0, Ref. 18"},
                                 {"op_", "p_", "1.6300", "286.0904", "Ver. 2.0, Ref. 18"},
                                 {"op_", "h_", "0.9830", "486.0610", "Ver. 2.0, Ref. 18"},
                                 {"op_", "f_", "1.2700", "231.6505", "Ver. 2.0, Ref. 18"},
                                 {"op_", "cl_", "1.6070", "279.3939", "Ver. 2.0, Ref. 18"},
                                 {"op_", "br_", "1.7550", "260.8954", "Ver. 2.0, Ref. 18"},
                                 {"op_", "i_", "1.9410", "240.8317", "Ver. 2.0, Ref. 18"},
                                 {"op_", "si", "1.5870", "292.2400", "Ver. 2.0, Ref. 18"},
                                 {"o-_", "p_", "1.4800", "428.0000", "Ver. 2.0, Ref. 18"},
                                 {"s_", "s_", "2.0547", "180.0000", "Ver. 2.0, Ref. 18"},
                                 {"s_", "sp_", "2.0400", "175.1260", "Ver. 2.0, Ref. 18"},
                                 {"s_", "s'_", "2.0400", "189.9260", "Ver. 2.0, Ref. 18"},
                                 {"s_", "p_", "2.0200", "186.8792", "Ver. 2.0, Ref. 18"},
                                 {"s_", "h_", "1.3300", "274.1288", "Ver. 2.0, Ref. 18"},
                                 {"s_", "f_", "1.6900", "204.8184", "Ver. 2.0, Ref. 18"},
                                 {"s_", "cl_", "2.0270", "212.0812", "Ver. 2.0, Ref. 18"},
                                 {"s_", "br_", "2.1750", "187.8836", "Ver. 2.0, Ref. 18"},
                                 {"s_", "i_", "2.3610", "167.7624", "Ver. 2.0, Ref. 18"},
                                 {"s_", "si_", "2.1300", "177.2928", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "s'_", "1.9700", "320.0000", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "p_", "1.9700", "255.2524", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "h_", "1.3730", "316.8138", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "f_", "1.6600", "195.3021", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "cl_", "1.9970", "211.1368", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "br_", "2.1450", "187.6017", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "i_", "2.3310", "167.6272", "Ver. 2.0, Ref. 18"},
                                 {"s'_", "si_", "2.0800", "247.5744", "Ver. 2.0, Ref. 18"},
                                 {"s-_", "p_", "1.9800", "210.9800", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "sp_", "2.0100", "160.0000", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "s'_", "1.9900", "240.0000", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "p_", "1.9900", "175.1796", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "h_", "1.3730", "236.5449", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "f_", "1.6600", "180.5021", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "cl_", "1.9970", "196.3368", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "br_", "2.1450", "172.8017", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "i_", "2.3310", "152.8272", "Ver. 2.0, Ref. 18"},
                                 {"sp_", "si_", "2.1000", "167.4260", "Ver. 2.0, Ref. 18"},
                                 {"p_", "p_", "1.9700", "176.0000", "Ver. 2.0, Ref. 18"},
                                 {"p_", "h_", "1.4300", "224.0000", "Ver. 2.0, Ref. 18"},
                                 {"p_", "f_", "1.5400", "230.3664", "Ver. 2.0, Ref. 18"},
                                 {"p_", "cl_", "2.0430", "208.8228", "Ver. 2.0, Ref. 18"},
                                 {"p_", "br_", "2.1800", "183.1472", "Ver. 2.0, Ref. 18"},
                                 {"p_", "i_", "2.3110", "162.7080", "Ver. 2.0, Ref. 18"},
                                 {"p_", "si_", "1.9170", "168.2072", "Ver. 2.0, Ref. 18"},
                                 {"h_", "h_", "0.7461", "398.7301", "Ver. 2.0, Ref. 18"},
                                 {"h_", "f_", "1.0230", "520.7304", "Ver. 2.0, Ref. 18"},
                                 {"h_", "cl_", "1.3600", "345.9024", "Ver. 2.0, Ref. 18"},
                                 {"h_", "br_", "1.5080", "314.1728", "Ver. 2.0, Ref. 18"},
                                 {"h_", "i_", "1.6940", "292.0432", "Ver. 2.0, Ref. 18"},
                                 {"h_", "si_", "1.4630", "288.3168", "Ver. 2.0, Ref. 18"},
                                 {"dw", "dw", "0.7416", "398.2392", "Ver. 2.0, Ref. 18"},
                                 {"f_", "f_", "1.4170", "259.0683", "Ver. 2.0, Ref. 18"},
                                 {"f_", "cl_", "1.6470", "207.1180", "Ver. 2.0, Ref. 18"},
                                 {"f_", "br_", "1.7950", "193.5000", "Ver. 2.0, Ref. 18"},
                                 {"f_", "i_", "1.9810", "174.1664", "Ver. 2.0, Ref. 18"},
                                 {"f_", "si_", "1.5870", "297.3400", "Ver. 2.0, Ref. 18"},
                                 {"cl_", "cl_", "1.9880", "236.5339", "Ver. 2.0, Ref. 18"},
                                 {"cl_", "br_", "2.1320", "209.1876", "Ver. 2.0, Ref. 18"},
                                 {"cl_", "i_", "2.3180", "189.3396", "Ver. 2.0, Ref. 18"},
                                 {"cl_", "si_", "2.0870", "207.9748", "Ver. 2.0, Ref. 18"},
                                 {"br_", "br_", "2.2900", "175.6329", "Ver. 2.0, Ref. 18"},
                                 {"br_", "i_", "2.4660", "165.6156", "Ver. 2.0, Ref. 18"},
                                 {"br_", "si_", "2.2350", "180.2320", "Ver. 2.0, Ref. 18"},
                                 {"i_", "i_", "2.6620", "123.2110", "Ver. 2.0, Ref. 18"},
                                 {"i_", "si_", "2.4210", "158.9664", "Ver. 2.0, Ref. 18"},
                                 {"si_", "si_", "2.4200", "144.0000", "Ver. 2.0, Ref. 18"}};

/*
 Morse bonds
  0= Key_a, 1= Key_b, 2= R0 (A), 3= D, 4= ALPHA, 5= CVFF info

  E = D * (1 - exp(-ALPHA*(R - R0)))^2
*/
char * CVFF_morse_bonds[775][6]= {{"c3m_", "c3m_", "1.5100", "88.0000", "1.9150", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c4m_", "1.5260", "88.0000", "1.9150", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c_", "1.5260", "88.0000", "1.9150", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c'_", "1.5200", "76.0000", "1.9300", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "cp_", "1.5100", "76.0000", "1.9300", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c=", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c=_1", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c=_2", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c=_3", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "ct_", "1.4000", "85.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "na_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n3m_", "1.4850", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n4m_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n_", "1.4600", "72.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "np_", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n=", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n=_1", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n=_2", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n=_3", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n+_", "1.4620", "67.7209", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "o_", "1.4250", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "o'_", "1.3800", "79.7371", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "o3e_", "1.4340", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "o4e_", "1.4250", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "op_", "1.3800", "86.6371", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s_", "1.8000", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "sp_", "1.7700", "60.6331", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s'_", "1.7700", "64.3331", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s3e_", "1.8000", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s4e_", "1.8000", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "h_", "1.1050", "108.6000", "1.7710", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "p_", "1.7500", "62.2836", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "f_", "1.3630", "124.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "cl_", "1.7610", "78.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "br_", "1.9200", "55.9000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "si_", "1.8090", "59.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "i_", "2.1200", "50.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "1.5520", "88.0000", "1.9150", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c_", "1.5260", "88.0000", "1.9150", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c'_", "1.5200", "76.0000", "1.9300", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "cp_", "1.5100", "76.0000", "1.9300", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c=", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c=_1", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c=_2", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c=_3", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "ct_", "1.4000", "85.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "na_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n3m_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n4m_", "1.4670", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n_", "1.4600", "72.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "np_", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n=", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n=_1", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n=_2", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n=_3", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n+_", "1.4620", "67.7209", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "o_", "1.4250", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "o'_", "1.3800", "79.7371", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "o3e_", "1.4250", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "o4e_", "1.4462", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "op_", "1.3800", "86.6371", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s_", "1.8000", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "sp_", "1.7700", "60.6331", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s'_", "1.7700", "64.3331", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s3e_", "1.8000", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s4e_", "1.8470", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "h_", "1.1050", "108.6000", "1.7710", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "p_", "1.7500", "62.2836", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "f_", "1.3630", "124.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "cl_", "1.7610", "78.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "br_", "1.9200", "55.9000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "si_", "1.8090", "59.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "i_", "2.1200", "50.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n3m_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n4m_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n3m_", "1.4460", "68.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n4m_", "1.4000", "83.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "s3e_", "1.7700", "58.0627", "1.7361", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "s4e_", "1.7700", "58.0627", "1.7361", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n3m_", "1.4200", "70.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n4m_", "1.4200", "70.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "s3e", "1.7300", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "s4e", "1.7300", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "n3m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "n4m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "n3m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "n4m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "n3m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "n4m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "n3m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "n4m_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "s3e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "s4e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "s3e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "s4e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "s3e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "s4e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "s3e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "s4e_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n3m_", "1.3820", "71.7024", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n4m_", "1.3820", "71.7024", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "s3e_", "1.7200", "67.8582", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "s4e_", "1.7200", "67.8582", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n3m_", "1.3940", "55.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n4m_", "1.3940", "55.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "s3e_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "s4e_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n3m_", "1.3940", "55.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n4m_", "1.3940", "55.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n_", "1.3670", "55.4242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "np_", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n=", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n=_1", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n=_2", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n=_3", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n+_", "1.3940", "52.7898", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "o_", "1.3250", "75.3375", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "op_", "1.3120", "68.5440", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "o'_", "1.3120", "61.6440", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "sp_", "1.7020", "47.7438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s'_", "1.7020", "51.4438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s3e_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s4e_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "p_", "1.6820", "52.6350", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "h_", "1.0260", "88.0000", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "f_", "1.3520", "50.2463", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "cl_", "1.6890", "56.6065", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "br_", "1.8370", "50.9585", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "i_", "2.0230", "46.0026", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "si_", "1.7920", "51.1059", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n4m_", "1.3940", "55.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n_", "1.3670", "55.4242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "np_", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n=", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n=_1", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n=_2", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n=_3", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n+_", "1.3940", "52.7898", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "o_", "1.3250", "75.3375", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "op_", "1.3120", "68.5440", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "o'_", "1.3120", "61.6440", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "sp_", "1.7020", "47.7438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s'_", "1.7020", "51.4438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s3e_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s4e_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "p_", "1.6820", "52.6350", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "h_", "1.0260", "88.0000", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "f_", "1.3520", "50.2463", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "cl_", "1.6890", "56.6065", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "br_", "1.8370", "50.9585", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "i_", "2.0230", "46.0026", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "si_", "1.7920", "51.1059", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s3e_", "1.7050", "52.6552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s4e_", "1.7050", "52.6552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "s3e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "s4e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "s3e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "s4e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "s3e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "s4e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "s3e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "s4e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "s3e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "s4e_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s3e_", "1.6930", "72.0212", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s4e_", "1.6930", "72.0212", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "s3e_", "1.6800", "66.6937", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "s4e_", "1.6800", "66.6937", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s3e_", "1.6500", "59.8936", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s4e_", "1.6500", "59.8936", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s3e_", "2.1000", "40.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s4e_", "2.1000", "40.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "s3e_", "2.0400", "43.7815", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "s4e_", "2.0400", "43.7815", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "s3e_", "2.0400", "47.4815", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "s4e_", "2.0400", "47.4815", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "s3e_", "2.0000", "45.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "s4e_", "2.0000", "45.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "p_", "2.0200", "46.7198", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "h_", "1.3300", "87.5000", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "f_", "1.6900", "51.2046", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "cl_", "2.0270", "53.0203", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "br_", "2.1750", "46.9709", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "i_", "2.3610", "41.9406", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "si_", "2.1300", "44.3232", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "s4e_", "2.0000", "45.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "p_", "2.0200", "46.7198", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "h_", "1.3300", "87.5000", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "f_", "1.6900", "51.2046", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "cl_", "2.0270", "53.0203", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "br_", "2.1750", "46.9709", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "i_", "2.3610", "41.9406", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "si_", "2.1300", "44.3232", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c_", "1.5260", "88.0000", "1.9150", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c'_", "1.5200", "76.0000", "1.9300", "Ver. 2.0, Ref. 18"},
                                  {"c_", "cp_", "1.5100", "76.0000", "1.9300", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c=", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c=_1", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c=_2", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c=_3", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "ct_", "1.4000", "85.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "na_", "1.4700", "68.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n_", "1.4600", "72.0000", "2.2900", "Ver. 2.0, Ref. 18"},
                                  {"c_", "np_", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n=", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n=_1", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n=_2", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n=_3", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n+_", "1.4620", "67.7209", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "o_", "1.4250", "68.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "op_", "1.3800", "86.6371", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "o'_", "1.3800", "79.7371", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "s_", "1.8000", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "sp_", "1.7700", "60.6331", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "s'_", "1.7700", "64.3331", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "h_", "1.1050", "108.6000", "1.7710", "Ver. 2.0, Ref. 18"},
                                  {"c_", "p_", "1.7500", "62.2836", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "f_", "1.3630", "124.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "cl_", "1.7610", "78.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "br_", "1.9200", "55.9000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "si_", "1.8090", "59.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "i_", "2.1200", "50.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "c'_", "1.5000", "66.6000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "cp_", "1.5000", "71.0829", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "c=", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "c=_1", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "c=_2", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "c=_3", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "ct_", "1.4200", "77.7873", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n_", "1.3600", "97.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n=", "1.4050", "74.0749", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n=_1", "1.4050", "74.0749", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n=_2", "1.4050", "74.0749", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "n=_3", "1.4050", "74.0749", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "np_", "1.4050", "74.0749", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "o_", "1.3400", "100.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "o'_", "1.2200", "145.0000", "2.0600", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "o-_", "1.2500", "135.0000", "2.0600", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "op_", "1.3500", "73.5252", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "s_", "1.7700", "58.0627", "1.7361", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "s'_", "1.6110", "169.3000", "1.7361", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "s-_", "1.6800", "93.0000", "1.7361", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "sp_", "1.7400", "53.8383", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "h_", "1.1050", "108.6000", "1.7710", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "p_", "1.7200", "60.3455", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "f_", "1.3900", "54.4273", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "cl_", "1.7270", "62.4647", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "br_", "1.8750", "57.0702", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "si_", "1.8300", "59.8388", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c'_", "i_", "2.0610", "52.1506", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "cp_", "1.3900", "120.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "c=", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "c=_1", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "c=_2", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "c=_3", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "ct_", "1.4000", "80.4179", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "na_", "1.4120", "64.4438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n_", "1.4200", "70.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n=", "1.3850", "79.0095", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n=_1", "1.3850", "79.0095", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n=_2", "1.3850", "79.0095", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n=_3", "1.3850", "79.0095", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "np_", "1.3500", "110.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "n+_", "1.4120", "62.8336", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "o_", "1.3700", "96.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "o'_", "1.3300", "74.3713", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "op_", "1.3700", "105.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "s_", "1.7300", "57.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "s'_", "1.7200", "59.9506", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "sp_", "1.7106", "80.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "h_", "1.0800", "116.0000", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "p_", "1.7000", "58.7607", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "f_", "1.3630", "124.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "cl_", "1.7610", "78.5000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "br_", "1.9200", "55.9000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "i_", "2.0410", "54.3628", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cp_", "si_", "1.8100", "56.2919", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ci_", "ci_", "1.3900", "70.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ci_", "ni", "1.3800", "80.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ci_", "h_", "1.0800", "116.0000", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"c=", "c=", "1.3300", "163.8000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "c=_3", "1.3300", "163.8000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "c=_3", "1.3300", "163.8000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "c=_2", "1.4100", "120.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "c=_2", "1.4800", "80.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "c=_1", "1.4800", "80.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "c=_3", "1.4800", "80.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "ct_", "1.4250", "84.4106", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "na_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "n_", "1.4100", "69.7685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "ct_", "1.4250", "84.4106", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "na_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "n_", "1.4100", "69.7685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "ct_", "1.4250", "84.4106", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "na_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "n_", "1.4100", "69.7685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "ct_", "1.4250", "84.4106", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "na_", "1.4370", "68.4292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "n_", "1.4100", "69.7685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "n=", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "n=_3", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "n=_3", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "n=_1", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "n=_2", "1.3430", "123.3817", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "n=_2", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "n=_1", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "n=_1", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "n=_2", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "n=_3", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "np_", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "o_", "1.3680", "88.7997", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "op_", "1.3550", "85.1279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "o'_", "1.3550", "78.2279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "h_", "1.0900", "90.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "p_", "1.7250", "62.7497", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "s_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "sp_", "1.7450", "60.2470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "s'_", "1.7450", "63.9470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "f_", "1.3950", "71.2830", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "cl_", "1.7320", "69.6283", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "br_", "1.8800", "63.4252", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "i_", "2.0660", "58.3608", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=", "si_", "1.8350", "60.2606", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "np_", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "o_", "1.3680", "88.7997", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "op_", "1.3550", "85.1279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "o'_", "1.3550", "78.2279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "h_", "1.0900", "90.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "p_", "1.7250", "62.7497", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "s_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "sp_", "1.7450", "60.2470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "s'_", "1.7450", "63.9470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "f_", "1.3950", "71.2830", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "cl_", "1.7320", "69.6283", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "br_", "1.8800", "63.4252", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "i_", "2.0660", "58.3608", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_1", "si_", "1.8350", "60.2606", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "np_", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "o_", "1.3680", "88.7997", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "op_", "1.3550", "85.1279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "o'_", "1.3550", "78.2279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "h_", "1.0900", "90.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "p_", "1.7250", "62.7497", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "s_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "sp_", "1.7450", "60.2470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "s'_", "1.7450", "63.9470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "f_", "1.3950", "71.2830", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "cl_", "1.7320", "69.6283", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "br_", "1.8800", "63.4252", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "i_", "2.0660", "58.3608", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_2", "si_", "1.8350", "60.2606", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "np_", "1.4100", "82.9685", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "o_", "1.3680", "88.7997", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "op_", "1.3550", "85.1279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "o'_", "1.3550", "78.2279", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "h_", "1.0900", "90.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "p_", "1.7250", "62.7497", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "s_", "1.7750", "63.7360", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "sp_", "1.7450", "60.2470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "s'_", "1.7450", "63.9470", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "f_", "1.3950", "71.2830", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "cl_", "1.7320", "69.6283", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "br_", "1.8800", "63.4252", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "i_", "2.0660", "58.3608", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"c=_3", "si_", "1.8350", "60.2606", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cr_", "n=", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cr_", "n=_1", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cr_", "n=_2", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cr_", "n=_3", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cr_", "n_", "1.3200", "97.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "ct_", "1.2040", "200.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "nt_", "1.1580", "220.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "na_", "1.3820", "71.7024", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n_", "1.3550", "72.3612", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n=", "1.3550", "85.5612", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n=_1", "1.3550", "85.5612", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n=_2", "1.3550", "85.5612", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n=_3", "1.3550", "85.5612", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "n+_", "1.3820", "69.5942", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "o_", "1.3130", "91.9541", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "s_", "1.7200", "67.8582", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "op_", "1.3000", "86.6170", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "o'_", "1.3000", "79.7170", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "sp_", "1.6900", "64.0086", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "s'_", "1.6900", "67.7086", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "p_", "1.6700", "68.3667", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "h_", "1.0530", "101.1528", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "f_", "1.3400", "69.5983", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "cl_", "1.6770", "73.0488", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "br_", "1.8250", "67.2132", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "i_", "2.0110", "62.2317", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"ct_", "si_", "1.7800", "66.8241", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "na_", "1.3940", "55.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n_", "1.3670", "55.4242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "np_", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n=", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n=_1", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n=_2", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n=_3", "1.3670", "68.6242", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "n+_", "1.3940", "52.7898", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "o_", "1.3250", "75.3375", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "s_", "1.7320", "51.7351", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "op_", "1.3120", "68.5440", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "o'_", "1.3120", "61.6440", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "sp_", "1.7020", "47.7438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "s'_", "1.7020", "51.4438", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "p_", "1.6820", "52.6350", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "h_", "1.0260", "88.0000", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"na_", "f_", "1.3520", "50.2463", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "cl_", "1.6890", "56.6065", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "br_", "1.8370", "50.9585", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "i_", "2.0230", "46.0026", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"na_", "si_", "1.7920", "51.1059", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n_", "1.3400", "55.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "np_", "1.3400", "68.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n=", "1.3400", "68.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n=_1", "1.3400", "68.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n=_2", "1.3400", "68.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n=_3", "1.3400", "68.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n+_", "1.3670", "52.6590", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "o_", "1.2980", "75.3158", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s_", "1.7050", "52.6552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "op_", "1.2850", "66.7770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "o'_", "1.2850", "59.8770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "sp_", "1.6750", "48.3711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s'_", "1.6750", "52.0711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "p_", "1.6550", "54.9881", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "h_", "1.0260", "93.0000", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n_", "f_", "1.3250", "47.3464", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "cl_", "1.6620", "56.8901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "br_", "1.8100", "51.5995", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "i_", "1.9960", "46.6743", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "si_", "1.7650", "54.2016", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "np_", "1.3400", "102.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "n=", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "n=_1", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "n=_2", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "n=_3", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "n+_", "1.3670", "65.8590", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "o_", "1.2980", "88.5158", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "o'_", "1.2850", "73.0770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "o-_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "op_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "s_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "s'_", "1.6750", "65.2711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "sp_", "1.6750", "61.5711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "p_", "1.6550", "68.1881", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "h_", "1.0260", "93.0000", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"np_", "f_", "1.3250", "60.5464", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "cl_", "1.6620", "70.0901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "br_", "1.8100", "64.7995", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "i_", "1.9960", "59.8743", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"np_", "si_", "1.7650", "67.4016", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "n=", "1.2100", "162.8000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "n=_3", "1.2100", "162.8000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "n=_3", "1.2100", "162.8000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "n=_2", "1.2760", "122.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "n=_2", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "n=_1", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "n=_3", "1.3400", "81.4000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "n+_", "1.3670", "65.8590", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "o_", "1.2980", "88.5158", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "n+_", "1.3670", "65.8590", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "o_", "1.2980", "88.5158", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "n+_", "1.3670", "65.8590", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "o_", "1.2980", "88.5158", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "n+_", "1.3670", "65.8590", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "o_", "1.2980", "88.5158", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "o'_", "1.1600", "143.9680", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "o'_", "1.2850", "73.0770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "o'_", "1.2225", "108.5225", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "o'_", "1.1600", "143.9680", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "o-_", "1.2225", "108.5225", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "o-_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "o-_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "o-_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "o-_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "op_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "op_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "op_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "op_", "1.2850", "79.9770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "s_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "s_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "s_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "s_", "1.7050", "65.8552", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "sp_", "1.6750", "61.5711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "sp_", "1.6750", "61.5711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "sp_", "1.6750", "61.5711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "sp_", "1.6750", "61.5711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "s'_", "1.5900", "122.3100", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "s'_", "1.6750", "65.2711", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "s'_", "1.6325", "93.7906", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "s'_", "1.5900", "122.3100", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "p_", "1.6550", "68.1881", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "h_", "1.0380", "106.0338", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n=", "f_", "1.3250", "60.5464", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "cl_", "1.6620", "70.0901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "br_", "1.8100", "64.7995", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "i_", "1.9960", "59.8743", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=", "si_", "1.7650", "67.4016", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "p_", "1.6550", "68.1881", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "h_", "1.0380", "106.0338", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "f_", "1.3250", "60.5464", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "cl_", "1.6620", "70.0901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "br_", "1.8100", "64.7995", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "i_", "1.9960", "59.8743", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_1", "si_", "1.7650", "67.4016", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "p_", "1.6550", "68.1881", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "h_", "1.0380", "106.0338", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "f_", "1.3250", "60.5464", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "cl_", "1.6620", "70.0901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "br_", "1.8100", "64.7995", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "i_", "1.9960", "59.8743", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_2", "si_", "1.7650", "67.4016", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "p_", "1.6550", "68.1881", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "h_", "1.0380", "106.0338", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "f_", "1.3250", "60.5464", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "cl_", "1.6620", "70.0901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "br_", "1.8100", "64.7995", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "i_", "1.9960", "59.8743", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n=_3", "si_", "1.7650", "67.4016", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "n+_", "1.3940", "48.3901", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "o_", "1.3250", "72.7505", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "s_", "1.7320", "50.0292", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "op_", "1.3120", "65.4265", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "o'_", "1.3120", "58.5265", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "sp_", "1.7020", "45.8238", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "s'_", "1.7020", "49.5238", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "p_", "1.6820", "52.2884", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "h_", "1.0650", "88.7167", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "f_", "1.3520", "44.5093", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "cl_", "1.6890", "54.4312", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "br_", "1.8370", "49.1485", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "i_", "2.0230", "44.3191", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"n+_", "si_", "1.7920", "51.7353", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"nz_", "nz_", "1.09758", "226.8", "2.6829", "Ver. 2.0, Ref. 18"},
                                  {"nt_", "nt_", "1.09758", "226.8000", "2.6829", "Ver. 2.0, Ref. 18"},
                                  {"o_", "o_", "1.20800", "118.86", "2.6484", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s_", "1.6930", "72.0212", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "op_", "1.2430", "87.6930", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "o'_", "1.2430", "80.7930", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "sp_", "1.6330", "67.9896", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s'_", "1.6330", "71.6896", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "p_", "1.6100", "61.3000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "si_", "1.6650", "98.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "h_", "0.9600", "95.0000", "2.2800", "Ver. 2.0, Ref. 18"},
                                  {"o_", "f_", "1.4180", "56.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "cl_", "1.6500", "76.7658", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "br_", "1.7980", "71.1997", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "i_", "1.9840", "66.2467", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "o'_", "1.1100", "121.2000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "op_", "1.2300", "68.1000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s_", "1.5900", "90.1047", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "sp_", "1.5600", "85.3184", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s'_", "1.5400", "105.3797", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "p_", "1.4800", "131.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "h_", "0.9830", "114.6152", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "f_", "1.2700", "51.0126", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "cl_", "1.6070", "62.9485", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "br_", "1.7550", "58.3239", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "i_", "1.9410", "53.3079", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "si_", "1.6500", "113.6866", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "op_", "1.2300", "75.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "s_", "1.6800", "66.6937", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "sp_", "1.6500", "61.9610", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "s'_", "1.6200", "65.7436", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "p_", "1.6300", "71.5226", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "h_", "0.9830", "121.5152", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "f_", "1.2700", "57.9126", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "cl_", "1.6070", "69.8485", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "br_", "1.7550", "65.2239", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "i_", "1.9410", "60.2079", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"op_", "si", "1.5870", "73.0600", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"o-_", "p_", "1.4800", "107.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s_", "2.0547", "45.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "sp_", "2.0400", "43.7815", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s'_", "2.0400", "47.4815", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "p_", "2.0200", "46.7198", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "h_", "1.3300", "87.5000", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"s_", "f_", "1.6900", "51.2046", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "cl_", "2.0270", "53.0203", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "br_", "2.1750", "46.9709", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "i_", "2.3610", "41.9406", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "si_", "2.1300", "44.3232", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "s'_", "1.9700", "80.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "p_", "1.9700", "63.8131", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "h_", "1.3730", "79.2035", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "f_", "1.6600", "48.8255", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "cl_", "1.9970", "52.7842", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "br_", "2.1450", "46.9004", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "i_", "2.3310", "41.9068", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "si_", "2.0800", "61.8936", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"s-_", "p_", "1.9800", "52.7450", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "sp_", "2.0100", "40.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "s'_", "1.9900", "60.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "p_", "1.9900", "43.7949", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "h_", "1.3730", "75.5035", "1.7700", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "f_", "1.6600", "45.1255", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "cl_", "1.9970", "49.0842", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "br_", "2.1450", "43.2004", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "i_", "2.3310", "38.2068", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"sp_", "si_", "2.1000", "41.8565", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "p_", "1.9700", "44.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "h_", "1.4300", "56.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "f_", "1.5400", "57.5916", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "cl_", "2.0430", "52.2057", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "br_", "2.1800", "45.7868", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "i_", "2.3110", "40.6770", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"p_", "si_", "1.9170", "42.0518", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "h_", "0.74611", "104.207", "1.9561", "Ver. 2.0, Ref. 18"},
                                  {"h_", "f_", "1.0230", "130.1826", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "cl_", "1.3600", "86.4756", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "br_", "1.5080", "78.5432", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "i_", "1.6940", "73.0108", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "si_", "1.4630", "72.0792", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"dw", "dw", "0.74164", "106.010", "1.9382", "Ver. 2.0, Ref. 18"},
                                  {"f_", "f_", "1.4170", "37.5", "2.6284", "Ver. 2.0, Ref. 18"},
                                  {"f_", "cl_", "1.6470", "51.7795", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "br_", "1.7950", "48.3750", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "i_", "1.9810", "43.5416", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "si_", "1.5870", "74.3350", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cl_", "cl_", "1.988", "58.066", "2.0183", "Ver. 2.0, Ref. 18"},
                                  {"cl_", "br_", "2.1320", "52.2969", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cl_", "i_", "2.3180", "47.3349", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"cl_", "si_", "2.0870", "51.9937", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"br_", "br_", "2.290", "46.336", "1.9469", "Ver. 2.0, Ref. 18"},
                                  {"br_", "i_", "2.4660", "41.4039", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"br_", "si_", "2.2350", "45.0580", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"i_", "i_", "2.662", "36.46", "1.8383", "Ver. 2.0, Ref. 18"},
                                  {"i_", "si_", "2.4210", "39.7416", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "si_", "3.0900", "36.0000", "2.0000", "Ver. 2.0, Ref. 18"},
                                  {"no", "o-", "1.2178", "140.2486", "2.0000", "Ver. 2.3, Ref. 23"},
                                  {"no", "cp", "1.4720", "87.8132", "2.0000", "Ver. 2.3, Ref. 23"},
                                  {"c'", "cp", "1.4720", "87.8132", "2.0000", "Ver. 2.3, Ref. 23"},
                                  {"c", "o", "1.4250", "68.3000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "h", "1.1050", "108.6000", "1.7710", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "1.5260", "88.0000", "1.9150", "Ver. 1.0, Ref. 1"},
                                  {"c", "c'", "1.5200", "76.0000", "1.9300", "Ver. 1.0, Ref. 1"},
                                  {"c'", "o'", "1.2300", "145.0000", "2.0600", "Ver. 1.0, Ref. 1"},
                                  {"n", "hn", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"n", "lp", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"n", "c'", "1.3200", "97.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n", "cr", "1.3200", "97.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "n", "1.4600", "72.0000", "2.2900", "Ver. 1.0, Ref. 1"},
                                  {"n1", "cr", "1.3200", "97.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n1", "hn", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"n1", "lp", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"c", "n1", "1.4600", "72.0000", "2.2900", "Ver. 1.0, Ref. 1"},
                                  {"oh", "ho", "0.9600", "104.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"oh", "lp", "0.9600", "104.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"oh", "c", "1.4200", "96.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"o", "ho", "0.9600", "95.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"o", "lp", "0.9600", "95.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"c'", "h", "1.1050", "108.6000", "1.7710", "Ver. 1.0, Ref. 1"},
                                  {"c'", "o", "1.3700", "100.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "oh", "1.3700", "100.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "o-", "1.2500", "135.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "h", "1.0800", "116.0000", "1.7700", "Ver. 1.0, Ref. 1"},
                                  {"cp", "cp", "1.3400", "120.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "c", "1.5100", "76.0000", "1.9300", "Ver. 1.0, Ref. 1"},
                                  {"cp", "oh", "1.3700", "96.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"sh", "hs", "1.3300", "87.5000", "1.7700", "Ver. 1.0, Ref. 1"},
                                  {"sh", "lp", "1.3300", "87.5000", "1.7700", "Ver. 1.0, Ref. 1"},
                                  {"c", "sh", "1.8200", "57.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "s", "1.8000", "57.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "s1", "1.8000", "57.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"s", "s", "2.0000", "45.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"s1", "s1", "2.0000", "45.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n2", "hn", "1.0260", "88.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"n2", "lp", "1.0260", "88.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"c", "n2", "1.4700", "68.0000", "2.2900", "Ver. 1.0, Ref. 1"},
                                  {"n2", "c'", "1.3200", "97.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n2", "cr", "1.3200", "97.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n2", "cp", "1.3200", "97.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n3", "hn", "1.0260", "88.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"n3", "lp", "1.0260", "88.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"c", "n3", "1.4700", "68.0000", "2.2900", "Ver. 1.0, Ref. 1"},
                                  {"n4", "hn", "1.0260", "88.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"n4", "lp", "1.0260", "88.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"c", "n4", "1.4700", "68.0000", "2.2900", "Ver. 1.0, Ref. 1"},
                                  {"cp", "np", "1.2600", "140.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "hn", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"np", "lp", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"cp", "c5", "1.3400", "70.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "h", "1.0800", "116.0000", "1.7700", "Ver. 1.0, Ref. 1"},
                                  {"c5", "c5", "1.3900", "70.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "c", "1.5100", "76.0000", "1.9300", "Ver. 1.0, Ref. 1"},
                                  {"c5", "np", "1.3800", "80.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "o", "1.3700", "105.0000", "2.0000", "Ver. 1.2, Ref. 3"},
                                  {"c5", "op", "1.3700", "105.0000", "2.0000", "Ver. 1.3, Ref. 8"},
                                  {"c5", "s", "1.7300", "57.0000", "2.0000", "Ver. 1.2, Ref. 3"},
                                  {"cs", "h", "1.0800", "121.0000", "1.7700", "Ver. 1.3, Ref. 6"},
                                  {"cs", "c5", "1.3630", "100.0000", "2.0000", "Ver. 1.3, Ref. 6"},
                                  {"sp", "cs", "1.7106", "80.0000", "2.0000", "Ver. 1.3, Ref. 6"},
                                  {"ci", "h", "1.0800", "116.0000", "1.7700", "Ver. 1.0, Ref. 1"},
                                  {"ci", "ci", "1.3900", "70.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"ci", "c", "1.5100", "76.0000", "1.9300", "Ver. 1.0, Ref. 1"},
                                  {"ci", "ni", "1.3800", "80.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"ni", "hn", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"ni", "lp", "1.0260", "93.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"cp", "n", "1.4200", "70.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"o*", "h*", "0.9600", "104.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"ospc", "hspc", "1.0000", "104.0000", "2.2800", "Ver. 2.3, Ref. 25"},
                                  {"otip", "htip", "0.9570", "104.0000", "2.2800", "Ver. 2.3, Ref. 25"},
                                  {"o*", "lp", "0.9600", "104.0000", "2.2800", "Ver. 1.0, Ref. 1"},
                                  {"p", "oh", "1.5700", "75.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"p", "o'", "1.5300", "140.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"p", "o", "1.6100", "61.3000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"p", "o-", "1.5300", "120.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"p", "h", "1.5000", "56.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"p", "lp", "1.5000", "56.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "c", "1.4750", "84.2000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n=", "c", "1.4750", "84.2000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"n=1", "c", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"n=2", "c", "1.4750", "84.2000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "n=", "1.2600", "140.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=", "n=1", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "n=", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"cr", "n=", "1.2600", "140.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"cr", "n=1", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"cr", "n=2", "1.2600", "140.0000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c'", "1.5000", "80.7000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c'", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c'", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c", "1.5000", "80.7000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c", "1.5000", "80.7000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=", "1.3300", "163.8000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=", "c=1", "1.3300", "163.8000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "h", "1.0900", "90.4000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "h", "1.0900", "90.4000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "h", "1.0900", "90.4000", "2.0000", "Ver. 2.0, Ref. 19"},
                                  {"f", "c", "1.3630", "124.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"f", "lp", "1.3630", "124.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"cl", "c", "1.7610", "78.5000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"cl", "lp", "1.7610", "78.5000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"br", "c", "1.9200", "55.9000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"br", "lp", "1.9200", "55.9000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"f", "cp", "1.3630", "124.0000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"cl", "cp", "1.7610", "78.5000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"br", "cp", "1.9200", "55.9000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"si", "c", "1.8090", "59.5000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"si", "h", "1.3820", "55.6000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"si", "o", "1.6650", "98.2000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "h", "0.74611", "104.207", "1.9561", "Ver. 1.3, Ref. 4"},
                                  {"d", "d", "0.74164", "106.010", "1.9382", "Ver. 1.3, Ref. 4"},
                                  {"f", "f", "1.417", "37.5", "2.6284", "Ver. 1.3, Ref. 4"},
                                  {"cl", "cl", "1.988", "58.066", "2.0183", "Ver. 1.3, Ref. 4"},
                                  {"br", "br", "2.290", "46.336", "1.9469", "Ver. 1.3, Ref. 4"},
                                  {"i", "i", "2.662", "36.46", "1.8383", "Ver. 1.3, Ref. 4"},
                                  {"o", "o", "1.208", "118.86", "2.6484", "Ver. 1.3, Ref. 4"},
                                  {"nz", "nz", "1.09758", "226.8", "2.6829", "Ver. 1.3, Ref. 4"},
                                  {"nt", "nt", "1.09758", "226.8", "2.6829", "Ver. 1.3, Ref. 5"},
                                  {"c'", "s'", "1.6110", "169.3000", "1.7361", "Ver. 1.3, Ref. 7"},
                                  {"oz", "ho", "0.9600", "104.0000", "2.2800", "Ver. 1.8, Ref. 14"},
                                  {"sz", "o", "1.6150", "98.2000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"sz", "oz", "1.6150", "98.2000", "2.0000", "Ver. 1.8, Ref. 14"},
                                  {"sz", "oh", "1.6350", "98.2000", "2.0000", "Ver. 1.8, Ref. 14"},
                                  {"sz", "h", "1.3820", "55.6000", "2.0000", "Ver. 1.8, Ref. 14"},
                                  {"sz", "c", "1.8090", "59.5000", "2.0000", "Ver. 1.0, Ref. 1"},
                                  {"al", "c", "1.8090", "59.5000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"al", "h", "1.3820", "55.6000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"al", "o", "1.7750", "98.2000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"al", "oz", "1.7750", "98.2000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"az", "oz", "1.7750", "98.2000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"az", "oh", "1.7750", "98.2000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"az", "h", "1.3820", "55.6000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"az", "c", "1.8090", "59.5000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"az", "o", "1.7750", "98.2000", "2.0000", "Ver. 2.1, Ref. 26"},
                                  {"sz", "sz", "3.0900", "392.8000", "2.0000", "Ver. 2.1, Ref. 28"},
                                  {"sz", "az", "3.0900", "392.8000", "2.0000", "Ver. 2.1, Ref. 28"},
                                  {"si", "al", "3.0900", "392.8000", "2.0000", "Ver. 2.1, Ref. 28"},
                                  {"az", "az", "3.0900", "392.8000", "2.0000", "Ver. 2.1, Ref. 28"}};

/*
 Quadratic angles
  0= Key_a, 1= Key_b, 2= Key_c, 3= Theta0, 4= K2, 5= CVFF info

  E = K2 * (Theta - Theta0)^2
  With:
   - Theta = Angle
   - Theta0 = Equilibrium angle
   - K2 = Spring constant
*/
char * CVFF_angles_auto[563][6]= {{"cp", "cp", "c'", "120.0000", "34.6799", "Ver. 2.3, Ref. 23"},
                                  {"cp", "c'", "o", "120.0000", "54.4949", "Ver. 2.3, Ref. 23"},
                                  {"cp", "c'", "o'", "120.0000", "54.4949", "Ver. 2.3, Ref. 23"},
                                  {"cp", "cp", "no", "120.0000", "34.6799", "Ver. 2.3, Ref. 23"},
                                  {"cp", "no", "o-", "120.0000", "54.4949", "Ver. 2.3, Ref. 23"},
                                  {"o-", "no", "o-", "120.0000", "113.5731", "Ver. 2.3, Ref. 23"},
                                  {"c", "o", "c'", "109.5000", "60.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "h", "106.4000", "39.5000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "c", "110.0000", "44.4000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "c", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c'", "o'", "120.0000", "68.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "c'", "109.5000", "45.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "c'", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "n", "hn", "115.0000", "37.5000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "n", "c", "118.0000", "111.0000", "Ver. 1.0, Ref. 1"},
                                  {"hn", "n", "c", "122.0000", "35.0000", "Ver. 1.0, Ref. 1"},
                                  {"n", "c", "c", "109.5000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"n", "c", "h", "109.5000", "51.5000", "Ver. 1.0, Ref. 1"},
                                  {"n", "c", "c'", "109.5000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c'", "n", "114.1000", "53.5000", "Ver. 1.0, Ref. 1"},
                                  {"o'", "c'", "n", "120.0000", "68.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "o", "ho", "106.0000", "58.5000", "Ver. 1.0, Ref. 1"},
                                  {"o", "c", "h", "109.5000", "57.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "o", "109.5000", "70.0000", "Ver. 1.0, Ref. 1"},
                                  {"o", "c", "o", "109.5000", "70.0000", "Ver. 1.0, Ref. 1"},
                                  {"hn", "n", "hn", "125.0000", "33.0000", "Ver. 1.0, Ref. 1"},
                                  {"o'", "c'", "o", "123.0000", "145.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c'", "o", "110.0000", "122.8000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "o", "ho", "112.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c'", "o", "110.0000", "55.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c'", "o'", "120.0000", "55.0000", "Ver. 1.0, Ref. 1"},
                                  {"o-", "c'", "o-", "123.0000", "145.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c'", "o-", "120.0000", "68.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c'", "o-", "120.0000", "55.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "n", "c", "120.0000", "37.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c'", "n", "120.0000", "45.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "cp", "h", "120.0000", "37.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "cp", "cp", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "cp", "110.0000", "44.4000", "Ver. 1.0, Ref. 1"},
                                  {"c", "cp", "cp", "120.0000", "44.2000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "cp", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "o", "ho", "109.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"o", "cp", "cp", "120.0000", "60.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "sh", "hs", "96.0000", "48.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "sh", "109.0000", "45.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "s", "109.0000", "45.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "sh", "109.0000", "62.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "s", "109.0000", "62.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "s", "c", "99.0000", "58.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "s", "s", "109.5000", "75.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "n3", "109.5000", "57.3000", "Ver. 1.0, Ref. 1"},
                                  {"hn", "n3", "hn", "105.5000", "36.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "n3", "c", "112.0000", "86.3000", "Ver. 1.0, Ref. 1"},
                                  {"c", "n3", "hn", "110.0000", "41.6000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "n3", "109.5000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "cp", "np", "134.0000", "102.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "cp", "np", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "cp", "np", "120.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "np", "cp", "114.0000", "75.1000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "np", "hn", "120.0000", "27.5000", "Ver. 1.0, Ref. 1"},
                                  {"h", "cp", "c5", "120.0000", "37.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "cp", "c5", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "c5", "c5", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "c5", "h", "120.0000", "37.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "c5", "c5", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "c5", "110.0000", "44.4000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c5", "c5", "120.0000", "44.2000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c", "c5", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"np", "c5", "np", "134.0000", "102.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "c5", "np", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "c5", "np", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c5", "np", "120.0000", "40.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "np", "c5", "114.0000", "75.1000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "np", "hn", "120.0000", "27.5000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c5", "np", "120.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"n", "c'", "n", "120.0000", "102.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "o", "c", "109.5000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "cp", "n", "120.0000", "102.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "cp", "n", "120.0000", "102.0000", "Ver. 1.0, Ref. 1"},
                                  {"hn", "n", "cp", "115.0000", "37.5000", "Ver. 1.0, Ref. 1"},
                                  {"n3", "c", "c'", "109.5000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"cp", "c", "cp", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"c", "o", "c", "109.5000", "60.0000", "Ver. 1.0, Ref. 1"},
                                  {"h*", "o*", "h*", "104.5000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"hspc", "ospc", "hspc", "109.4700", "50.0000", "Ver. 2.3, Ref. 25"},
                                  {"htip", "otip", "htip", "104.5200", "50.0000", "Ver. 2.3, Ref. 25"},
                                  {"p", "o", "c", "120.0000", "72.0000", "Ver. 1.0, Ref. 1"},
                                  {"p", "o", "p", "120.0000", "72.0000", "Ver. 1.0, Ref. 1"},
                                  {"o", "p", "o", "109.5000", "110.0000", "Ver. 1.0, Ref. 1"},
                                  {"o'", "p", "o", "109.5000", "110.0000", "Ver. 1.0, Ref. 1"},
                                  {"o'", "p", "o'", "109.5000", "110.0000", "Ver. 1.0, Ref. 1"},
                                  {"o-", "p", "o-", "109.5000", "110.0000", "Ver. 1.0, Ref. 1"},
                                  {"o", "p", "o-", "109.5000", "110.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "p", "o-", "109.5000", "80.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "np", "c", "128.8000", "70.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "c", "o", "109.5000", "80.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "c", "h", "109.5000", "80.0000", "Ver. 1.0, Ref. 1"},
                                  {"np", "c", "c", "109.5000", "80.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=", "c'", "n", "114.1000", "53.5000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c'", "n", "114.1000", "53.5000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c'", "n", "114.1000", "53.5000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=", "c=", "122.3000", "36.2000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "c=1", "c=", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=2", "c=", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=", "c=1", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=1", "c=1", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=2", "c=1", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=", "c=2", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=1", "c=2", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c=2", "c=2", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=", "c=", "122.3000", "36.2000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c=1", "c=", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=2", "c=", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=", "c=1", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=1", "c=1", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=2", "c=1", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=", "c=2", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=1", "c=2", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=2", "c=2", "122.3000", "36.2000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=", "h", "120.0000", "37.5000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c=1", "h", "120.0000", "37.5000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=2", "h", "120.0000", "37.5000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c", "c=", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c", "c=", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c", "c=", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "c", "c=1", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "c", "c=2", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c", "c=2", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c", "c=", "110.5000", "46.6000", "Ver. 1.0, Ref. 1"},
                                  {"c'", "c", "c=1", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"c'", "c", "c=2", "110.5000", "46.6000", "Ver. 2.0, Ref. 19"},
                                  {"h", "c", "c=", "110.0000", "44.4000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c", "c=1", "110.0000", "44.4000", "Ver. 2.0, Ref. 19"},
                                  {"h", "c", "c=2", "110.0000", "44.4000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=", "h", "121.2000", "33.8000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c=", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c=", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=1", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "c=1", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c=1", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=2", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "c=2", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c=2", "h", "121.2000", "33.8000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=", "np", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "c=", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c=", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=1", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "c=1", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c=1", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "c=2", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "c=2", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "c=2", "np", "120.0000", "90.0000", "Ver. 2.0, Ref. 19"},
                                  {"c=", "np", "c=", "114.0000", "75.1000", "Ver. 1.0, Ref. 1"},
                                  {"c=1", "np", "c=", "114.0000", "75.1000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "np", "c=", "114.0000", "75.1000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "np", "c=1", "114.0000", "75.1000", "Ver. 2.0, Ref. 19"},
                                  {"c=1", "np", "c=2", "114.0000", "75.1000", "Ver. 2.0, Ref. 19"},
                                  {"c=2", "np", "c=2", "114.0000", "75.1000", "Ver. 2.0, Ref. 19"},
                                  {"o'", "c'", "c=", "120.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"o'", "c'", "c=1", "120.0000", "50.0000", "Ver. 2.0, Ref. 19"},
                                  {"o'", "c'", "c=2", "120.0000", "50.0000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=", "c'", "120.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "c=1", "c'", "120.0000", "50.0000", "Ver. 2.0, Ref. 19"},
                                  {"c", "c=2", "c'", "120.0000", "50.0000", "Ver. 2.0, Ref. 19"},
                                  {"h", "c=", "np", "120.0000", "40.0000", "Ver. 1.0, Ref. 1"},
                                  {"h", "c=1", "np", "120.0000", "40.0000", "Ver. 2.0, Ref. 19"},
                                  {"h", "c=2", "np", "120.0000", "40.0000", "Ver. 2.0, Ref. 19"},
                                  {"c", "np", "c=", "120.0000", "50.0000", "Ver. 1.0, Ref. 1"},
                                  {"c", "np", "c=1", "120.0000", "50.0000", "Ver. 2.0, Ref. 19"},
                                  {"c", "np", "c=2", "120.0000", "50.0000", "Ver. 2.0, Ref. 19"},
                                  {"cp", "np", "c5", "114.0000", "75.1000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "cp", "np", "120.0000", "90.0000", "Ver. 1.0, Ref. 1"},
                                  {"c5", "cp", "n", "120.0000", "102.0000", "Ver. 1.0, Ref. 1"},
                                  {"f", "c", "c", "107.8000", "99.0000", "Ver. 1.0, Ref. 1"},
                                  {"f", "c", "h", "107.1000", "62.0000", "Ver. 1.0, Ref. 1"},
                                  {"cl", "c", "c", "107.8000", "85.0000", "Ver. 1.0, Ref. 1"},
                                  {"cl", "c", "h", "107.1000", "67.0000", "Ver. 1.0, Ref. 1"},
                                  {"br", "c", "c", "107.8000", "96.0000", "Ver. 1.0, Ref. 1"},
                                  {"br", "c", "h", "107.1000", "56.0000", "Ver. 1.0, Ref. 1"},
                                  {"f", "cp", "cp", "120.0000", "99.0000", "Ver. 1.0, Ref. 1"},
                                  {"cl", "cp", "cp", "120.0000", "85.0000", "Ver. 1.0, Ref. 1"},
                                  {"br", "cp", "cp", "120.0000", "96.0000", "Ver. 1.0, Ref. 1"},
                                  {"si", "c", "h", "112.3000", "34.6000", "Ver. 1.0, Ref. 1"},
                                  {"c", "si", "c", "113.5000", "44.4000", "Ver. 1.0, Ref. 1"},
                                  {"c", "si", "h", "109.1000", "32.7000", "Ver. 1.0, Ref. 1"},
                                  {"h", "si", "h", "112.0000", "31.8000", "Ver. 1.0, Ref. 1"},
                                  {"si", "c", "si", "122.5000", "42.2000", "Ver. 1.0, Ref. 1"},
                                  {"c", "si", "o", "117.3000", "44.1000", "Ver. 1.0, Ref. 1"},
                                  {"o", "si", "h", "113.4000", "33.3000", "Ver. 1.0, Ref. 1"},
                                  {"si", "o", "si", "149.8000", "31.1000", "Ver. 1.0, Ref. 1"},
                                  {"o", "si", "o", "113.1000", "42.3000", "Ver. 1.0, Ref. 1"},
                                  {"si", "o", "c", "124.1000", "56.4000", "Ver. 1.0, Ref. 1"},
                                  {"nt", "nt", "cp", "180.0", "50.0", "Ver. 1.3, Ref. 5"},
                                  {"nt", "nt", "c", "180.0", "50.0", "Ver. 1.3, Ref. 5"},
                                  {"cs", "c5", "c5", "112.7000", "70.0000", "Ver. 1.3, Ref. 6"},
                                  {"cs", "c5", "h", "124.0000", "30.0000", "Ver. 1.3, Ref. 6"},
                                  {"c5", "cs", "h", "124.0000", "30.0000", "Ver. 1.3, Ref. 6"},
                                  {"cs", "sp", "cs", "92.5670", "126.5060", "Ver. 1.3, Ref. 6"},
                                  {"sp", "cs", "c5", "111.0390", "88.7595", "Ver. 1.3, Ref. 6"},
                                  {"sp", "cs", "h", "119.9000", "30.2723", "Ver. 1.3, Ref. 6"},
                                  {"s'", "c'", "h", "121.6300", "40.9360", "Ver. 1.3, Ref. 7"},
                                  {"h", "c'", "h", "117.0200", "26.3900", "Ver. 1.3, Ref. 7"},
                                  {"c", "c'", "c", "120.0000", "40.0000", "Ver. 1.3, Ref. 7"},
                                  {"s'", "c'", "c", "120.0000", "40.0000", "Ver. 1.3, Ref. 7"},
                                  {"c5", "op", "c5", "120.0000", "75.0000", "Ver. 1.3, Ref. 8"},
                                  {"sz", "oz", "sz", "149.8000", "31.1000", "Ver. 1.8, Ref. 14"},
                                  {"oz", "sz", "oz", "109.4700", "100.3000", "Ver. 1.8, Ref. 14"},
                                  {"oz", "sz", "h", "113.4000", "33.3000", "Ver. 1.8, Ref. 14"},
                                  {"sz", "oz", "ho", "113.4000", "33.3000", "Ver. 1.8, Ref. 14"},
                                  {"az", "oz", "ho", "113.4000", "33.3000", "Ver. 1.8, Ref. 14"},
                                  {"h", "sz", "h", "112.0000", "31.8000", "Ver. 1.8, Ref. 14"},
                                  {"al", "oz", "al", "149.8000", "31.1000", "Ver. 2.1, Ref. 26"},
                                  {"oz", "al", "oz", "109.4700", "100.3000", "Ver. 2.1, Ref. 26"},
                                  {"oz", "al", "h", "113.4000", "33.3000", "Ver. 2.1, Ref. 26"},
                                  {"h", "al", "h", "112.0000", "31.8000", "Ver. 2.1, Ref. 26"},
                                  {"az", "oz", "az", "149.8000", "31.1000", "Ver. 2.1, Ref. 26"},
                                  {"oz", "az", "oz", "109.4700", "100.3000", "Ver. 2.1, Ref. 26"},
                                  {"oz", "az", "h", "113.4000", "33.3000", "Ver. 2.1, Ref. 26"},
                                  {"h", "az", "h", "112.0000", "31.8000", "Ver. 2.1, Ref. 26"},
                                  {"o", "al", "o", "113.1000", "42.3000", "Ver. 2.1, Ref. 26"},
                                  {"o", "al", "h", "113.4000", "33.3000", "Ver. 2.1, Ref. 26"},
                                  {"al", "oz", "sz", "149.8000", "31.1000", "Ver. 2.1, Ref. 26"},
                                  {"o", "az", "o", "109.4700", "42.3000", "Ver. 2.1, Ref. 26"},
                                  {"o", "az", "h", "113.4000", "33.3000", "Ver. 2.1, Ref. 26"},
                                  {"f", "sz", "oz", "113.1000", "42.3000", "Ver. 2.1, Ref. 28"},
                                  {"sz", "f", "sz", "113.1000", "42.3000", "Ver. 2.1, Ref. 28"},
                                  {"f", "sz", "sz", "113.1000", "42.3000", "Ver. 2.1, Ref. 28"},
                                  {"f", "sz", "f", "113.1000", "42.3000", "Ver. 2.1, Ref. 28"},
                                  {"sz", "sz", "sz", "109.4700", "5.0000", "Ver. 2.1, Ref. 28"},
                                  {"sz", "az", "sz", "109.4700", "5.0000", "Ver. 2.1, Ref. 28"},
                                  {"az", "sz", "sz", "109.4700", "5.0000", "Ver. 2.1, Ref. 28"},
                                  {"si", "si", "si", "109.4700", "5.0000", "Ver. 2.1, Ref. 28"},
                                  {"si", "al", "si", "109.4700", "5.0000", "Ver. 2.1, Ref. 28"},
                                  {"al", "si", "si", "109.4700", "5.0000", "Ver. 2.1, Ref. 28"},
                                  {"c3m_", "c3m_", "c3m_", "60.0000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c3m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c3m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c3m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c3m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c3m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c3m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c3m_", "c3m_", "60.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c3m_", "n3m_", "60.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c3m_", "c_", "112.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c3m_", "c3m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c3m_", "c4m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o3e_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o3e_", "c3m_", "c3m_", "60.0000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o4e_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c3m_", "c3m_", "60.0000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c3m_", "s3e_", "60.0000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "c3m_", "*", "109.5000", "44.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c3m_", "*", "107.8000", "95.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c3m_", "h_", "107.1000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "c3m_", "*", "112.3000", "34.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "c3m_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "c4m_", "95.0000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "n4m_", "88.3400", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "o4e_", "91.8400", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "s4e_", "94.5900", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c4m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c4m_", "o_", "121.0000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c4m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c4m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c4m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c4m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c4m_", "n4m_", "88.4000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c4m_", "o4e_", "90.0000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c4m_", "s4e_", "89.0000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c4m_", "c_", "112.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c4m_", "c3m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c4m_", "c4m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o3e_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o4e_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o4e_", "c4m_", "o4e_", "90.0000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"o4e_", "c4m_", "s4e_", "89.0000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "c4m_", "s4e_", "91.0000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "c4m_", "*", "109.5000", "44.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c4m_", "*", "107.8000", "95.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c4m_", "h_", "107.1000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "c4m_", "*", "112.3000", "34.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "c4m_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c_", "c_", "114.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c_", "c_", "114.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c_", "c3m_", "114.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c_", "c4m_", "114.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c'_", "*", "120.0000", "53.5000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c'_", "*", "120.0000", "53.5000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c'_", "c_", "114.0000", "82.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c'_", "c_", "114.0000", "82.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c'_", "n3m_", "125.0000", "145.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c'_", "n4m_", "123.0000", "145.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "cp_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "cp_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "cp_", "*", "120.0000", "102.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "cp_", "*", "120.0000", "102.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "cp_", "*", "120.0000", "89.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "cp_", "*", "120.0000", "89.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "c=_", "*", "120.0000", "36.2000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "c=_", "*", "120.0000", "36.2000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "c=_", "*", "120.0000", "90.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "c=_", "*", "120.0000", "90.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n3m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n3m_", "c3m_", "113.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n3m_", "c3m_", "60.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "n3m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "n3m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "n3m_", "*", "110.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "n3m_", "c3m_", "110.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "n3m_", "*", "109.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n4m_", "*", "110.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n4m_", "c4m_", "91.3800", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "n4m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "n4m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "n4m_", "*", "110.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "n4m_", "c4m_", "110.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "n4m_", "*", "109.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "o3e_", "c3m_", "58.9580", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "o4e_", "c4m_", "91.7370", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "o_", "*", "120.0000", "72.0000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "o_", "*", "120.0010", "72.0000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "op_", "*", "108.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "op_", "*", "106.0000", "27.5000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "s_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "s_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s3e_", "c3m_", "44.5000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s3e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s3e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "s3e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "s3e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "s3e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "s3e_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "s3e_", "*", "109.5000", "48.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s4e_", "c4m_", "85.9200", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "s4e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "s4e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "s4e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"n3m_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"n4m_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"s3e_", "s4e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s4e_", "s4e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "s4e_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "s4e_", "*", "109.5000", "48.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"c3m_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 18"},
                                  {"c4m_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c_", "c_", "114.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "c_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c_", "o_", "110.5000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c_", "s_", "115.0000", "46.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "c_", "*", "109.5000", "44.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c_", "*", "107.8000", "95.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c_", "h_", "107.1000", "62.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "c_", "*", "112.3000", "34.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "c_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c'_", "c_", "115.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c'_", "*", "120.0000", "53.5000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "c'_", "*", "110.0000", "122.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c'_", "*", "120.0000", "68.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "c'_", "*", "123.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "c'_", "*", "110.0000", "55.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c'_", "n_", "120.0000", "102.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c'_", "c_", "114.0000", "82.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c'_", "o_", "123.0000", "145.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c'_", "h_", "120.0000", "55.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c'_", "n_", "123.0000", "145.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "c'_", "h_", "117.0200", "26.3900", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c'_", "*", "120.0000", "99.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "c'_", "*", "120.0000", "34.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "c'_", "*", "120.0000", "65.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "cp_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "cp_", "*", "120.0000", "102.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "cp_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "cp_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "cp_", "*", "120.0000", "89.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "cp_", "c_", "114.0000", "89.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "cp_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "cp_", "*", "120.0000", "37.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "cp_", "*", "120.0000", "99.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "cp_", "*", "120.0000", "34.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "cp_", "*", "120.0000", "65.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "c=_", "*", "120.0000", "36.2000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "c=_", "*", "120.0000", "90.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "c=_", "*", "120.0000", "68.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "c=_", "*", "120.0000", "68.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "c=_", "*", "120.0000", "37.5000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "c=_", "*", "120.0000", "96.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "c=_", "*", "120.0000", "34.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "c=_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "ct_", "*", "180.0000", "200.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "na_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "na_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "na_", "*", "110.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "na_", "*", "109.0000", "41.6000", "Ver. 2.0, Ref. 18"},
                                  {"*", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "n_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "n_", "*", "120.0000", "70.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "n_", "*", "122.0000", "35.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "n_", "*", "120.0000", "35.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "np_", "*", "120.0000", "27.5000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "np_", "*", "120.0000", "27.5000", "Ver. 2.0, Ref. 18"},
                                  {"*", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "nt_", "*", "180.0", "50.0", "Ver. 2.0, Ref. 18"},
                                  {"c_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "o_", "*", "120.0000", "72.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "o_", "*", "109.0000", "58.5000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "o*_", "h_", "104.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "o_", "*", "124.1000", "56.4000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "o_", "si", "149.8000", "31.1000", "Ver. 2.0, Ref. 18"},
                                  {"*", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "s_", "*", "102.0000", "58.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "s_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "s_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "s_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "s_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "s_", "*", "109.5000", "48.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "s_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"n_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"o_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"s_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"h_", "sp_", "*", "96.0000", "48.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 18"},
                                  {"si_", "sp_", "*", "96.0000", "48.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "sp_", "*", "92.5670", "120.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"o'_", "p_", "*", "120.0000", "110.0000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"s'_", "p_", "*", "120.0000", "100.0000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "p_", "*", "109.5000", "30.0000", "Ver. 2.0, Ref. 18"},
                                  {"*", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 18"},
                                  {"c_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 18"},
                                  {"n_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 18"},
                                  {"o_", "si_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"s_", "si_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 18"},
                                  {"h_", "si_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 18"},
                                  {"f_", "si_", "*", "117.3000", "44.1000", "Ver. 2.0, Ref. 18"},
                                  {"si_", "si_", "*", "113.4000", "33.3000", "Ver. 2.0, Ref. 18"},
                                  {"*", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 18"}};

/*
 Dihedrals / torsions
 0, 1, 2, 3 = Keys, 4= Kphi, 5= n, 6= Phi0, 7= CVFF info
 E = Kphi * [ 1 + cos(n*Phi - Phi0) ]
 With:
  - Kphi = Rotation barrier
  - n = Periodicity / multiplicity
  - Phi0 = Equilibrium angle
*/
char * CVFF_torsions_auto[295][8]= {{"*", "c_", "n3n_", "*", "0.0500", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n3n_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "n3n_", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "n3n_", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "n3n_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "n3n_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "n3n_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "n3n_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "n3n_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n3n_", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_2", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_3", "n3n_", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "o_", "n3n_", "*", "0.3000", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "s_", "n3n_", "*", "0.3000", "2", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "si_", "n3n_", "*", "0.0500", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "c_", "*", "0.1580", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "c'_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "cp_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "c=", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "c=_1", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "c=_2", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "c=_3", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "na_", "*", "0.0500", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "n_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "np_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "n=", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "n=_1", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "n=_2", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "n=_3", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "o_", "*", "0.1300", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "s_", "*", "0.1367", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "p_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c_", "si_", "*", "0.1111", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "c'_", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "cp_", "*", "2.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "c=", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "c=_1", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "c=_2", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "c=_3", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n_", "*", "3.2000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n_", "h_", "1.2000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n=", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n=_1", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n=_2", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "n=_3", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "np_", "*", "5.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "np_", "h", "1.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "o_", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "op", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "s_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "sp_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c'_", "si_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "cp_", "*", "3.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "c=", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "c=_1", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "c=_2", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "c=_3", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "na_", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "n_", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "np_", "*", "2.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "np_", "h_", "1.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "n=", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "n=_1", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "n=_2", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "n=_3", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "o_", "*", "1.8000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "o_", "h_", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "op", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "s_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "sp_", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "si_", "*", "0.1667", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp_", "p_", "*", "0.2500", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "c=", "*", "4.0750", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "c=_3", "*", "4.0750", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "c=_3", "*", "4.0750", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "c=_2", "*", "3.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "c=_1", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "c=_2", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "c=_3", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "na_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "n_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "np_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "np_", "h_", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "na_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "n_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "np_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "np_", "h_", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "na_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "n_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "np_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "np_", "h_", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "na_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "n_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "np_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "np_", "h_", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "n=", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "n=_3", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "n=_3", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "n=_1", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "n=_2", "*", "2.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "n=_1", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "n=_2", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "n=_1", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "n=_3", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "n=_2", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "o_", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "s_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "sp_", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "si_", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=", "p_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "o_", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "s_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "sp_", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "si_", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_1", "p_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "o_", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "s_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "sp_", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "si_", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_2", "p_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "o_", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "s_", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "sp_", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "si_", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c=_3", "p_", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "c+_", "n_", "*", "3.4000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "ct_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "na_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "n_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "np_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "o_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "s_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "ct_", "si_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "na_", "*", "0.2500", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "n_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "np_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "n=", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "n=_1", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "n=_2", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "n=_3", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "o_", "*", "0.0975", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "s_", "*", "0.0975", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "na_", "si_", "*", "0.0667", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "n_", "*", "0.3750", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "np_", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "np_", "h_", "0.3750", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "n=", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "n=_1", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "n=_2", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "n=_3", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "o_", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "s_", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n_", "si_", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "n=", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "n=_1", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "n=_2", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "n=_3", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "np_", "*", "11.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "o_", "*", "1.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "op", "*", "11.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "s_", "*", "1.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "sp_", "*", "10.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "np_", "si_", "*", "0.2500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "n=", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "n=_1", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "n=_2", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "n=_3", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "np_", "*", "5.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "o_", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "op", "*", "5.50000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "s_", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "sp_", "*", "5.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"h_", "np_", "si_", "*", "0.1250", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=", "n=", "*", "15.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_3", "n=_3", "*", "15.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "n=_3", "*", "15.0000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_2", "n=_2", "*", "7.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "n=_1", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "n=_2", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_2", "n=_3", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=", "o_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=", "s_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=", "si_", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "o_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "s_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_1", "si_", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_2", "o_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_2", "s_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_2", "si_", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_3", "o_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_3", "s_", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "n=_3", "si_", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 18"},
                                    {"*", "o_", "o_", "*", "5.0000", "3", "0.", "Ver. 2.0, Ref. 22"},
                                    {"*", "o_", "s_", "*", "5.0000", "3", "0.", "Ver. 2.0, Ref. 22"},
                                    {"*", "o_", "si_", "*", "0.3333", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "o_", "p_", "*", "0.3750", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "s_", "s_", "*", "5.5000", "2", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "s_", "si_", "*", "0.2333", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "s_", "p_", "*", "0.3750", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "si_", "si_", "*", "0.1667", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "si_", "p_", "*", "0.0000", "3", "0.", "Ver. 2.0, Ref. 18"},
                                    {"*", "cp", "no", "*", "10.0000", "2", "180.0000", "Ver. 2.3, Ref. 23"},
                                    {"*", "cp", "c'", "*", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "c", "*", "1.4225", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "c'", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "n", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "o", "*", "0.3900", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "of", "*", "0.3900", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c'", "n2", "*", "6.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cr", "n2", "*", "13.6000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "n", "cr", "*", "13.6000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c'", "o", "*", "4.5000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cp", "cp", "*", "12.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cp", "c", "*", "0.0000", "2", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"cp", "cp", "c", "cp", "0.6750", "4", "0.0000", "Ver. 1.9, Ref. 17"},
                                    {"*", "ct", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c'", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "o", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "n", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "s", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cp", "ct", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "ct", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.3, Ref. 5"},
                                    {"*", "c'", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.3, Ref. 5"},
                                    {"*", "o", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.3, Ref. 5"},
                                    {"*", "n", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.3, Ref. 5"},
                                    {"*", "s", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.3, Ref. 5"},
                                    {"*", "cp", "nt", "*", "0.0000", "0", "0.0000", "Ver. 1.3, Ref. 5"},
                                    {"*", "cp", "o", "*", "1.5000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"cp", "cp", "o", "c", "1.8000", "2", "180.0000", "Ver. 1.9, Ref. 16"},
                                    {"*", "cp", "of", "*", "3.5000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "sh", "*", "0.4200", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c", "s", "*", "0.4200", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "s", "s", "*", "5.5000", "2", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "n3", "c", "*", "0.8000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "n2", "c", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cp", "np", "*", "4.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cp", "n2", "*", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "cp", "n", "*", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "p", "o", "*", "0.7500", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c=", "np", "*", "4.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c=1", "np", "*", "4.0000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=2", "np", "*", "4.0000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=", "c=", "*", "16.3000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c=1", "c=", "*", "16.3000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=2", "c=", "*", "16.3000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=1", "c=1", "*", "16.3000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=1", "c=2", "*", "16.3000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=2", "c=2", "*", "16.3000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=", "c", "*", "1.2660", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c=1", "c", "*", "1.2660", "3", "0.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c=2", "c", "*", "1.2660", "3", "0.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "np", "c", "*", "0.0000", "0", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c'", "c=", "*", "1.8000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"*", "c'", "c=1", "*", "1.8000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "c'", "c=2", "*", "1.8000", "2", "180.0000", "Ver. 2.0, Ref. 19"},
                                    {"*", "cs", "cp", "*", "6.0000", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                    {"cs", "sp", "cs", "cp", "12.0000", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                    {"cs", "sp", "cs", "h", "3.9316", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                    {"sp", "cs", "cp", "cp", "6.4517", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                    {"sp", "cs", "cp", "h", "2.0000", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                    {"c", "c'", "n", "c", "3.2000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"c", "c'", "n", "hn", "1.2000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"o'", "c'", "n", "c", "3.8000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"o'", "c'", "n", "hn", "1.8000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "c'", "n", "c", "3.2000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "c", "si", "c", "0.4000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "c", "si", "h", "0.2800", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"c", "si", "c", "si", "-1.7000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "si", "c", "si", "1.0000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "c", "si", "o", "-0.1000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"c", "si", "o", "si", "0.4000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "si", "o", "si", "1.0000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"si", "o", "si", "o", "0.3000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"o", "si", "o", "c", "0.1000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"h", "si", "o", "c", "-0.1000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"si", "o", "c", "h", "0.7000", "3", "0.0000", "Ver. 1.0, Ref. 1"},
                                    {"sz", "oz", "sz", "oz", "0.3000", "3", "0.0000", "Ver. 1.8, Ref. 14"},
                                    {"h", "sz", "oz", "sz", "1.0000", "3", "0.0000", "Ver. 1.8, Ref. 14"}};

/*
 Out of plane / impropers / improper
  0, 1, 2, 3 = Keys, 4= K, 5= Chi0, 6= CVFF info
  E = K * (Chi - Chi0)^2
*/
char * CVFF_impropers[41][8]= {{"c'", "cp", "cp", "cp", "10.0000", "2", "180.0000", "Ver. 2.3, Ref. 23"},
                                {"cp", "c'", "o'", "o'", "10.0000", "2", "180.0000", "Ver. 2.3, Ref. 23"},
                                {"cp", "no", "o'", "o'", "10.0000", "2", "180.0000", "Ver. 2.3, Ref. 23"},
                                {"cp", "cp", "cp", "no", "10.0000", "2", "180.0000", "Ver. 2.3, Ref. 23"},
                                {"c", "c'", "n", "o'", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c'", "n", "c", "hn", "0.0500", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c", "c'", "n2", "o'", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"h", "c'", "n2", "o'", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"hn", "n2", "c'", "hn", "0.0500", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c", "c'", "o'", "o'", "11.6000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"h", "c'", "o'", "o'", "11.6000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c'", "n", "c", "c", "0.0500", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"h", "c'", "o'", "n", "0.0500", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "cp", "h", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "cp", "c", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "cp", "cp", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "cp", "o'", "0.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "h", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"h", "cp", "np", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "cp", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"n2", "cp", "np", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "n2", "hn", "hn", "0.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"n", "c'", "n2", "n2", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"hn", "np", "cp", "cp", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c", "cp", "cp", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"n2", "cp", "cp", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c=", "c'", "n2", "o'", "10.0000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c", "c=", "c'", "c=", "11.1000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"h", "c=", "np", "c=", "11.1000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"h", "c=", "c", "c=", "11.1000", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"c", "np", "cp", "cp", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"cp", "cp", "np", "np", "0.3700", "2", "180.0000", "Ver. 1.0, Ref. 1"},
                                {"sp", "cs", "cp", "h", "2.9998", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                {"cp", "cp", "cs", "h", "0.3700", "2", "180.0000", "Ver. 1.3, Ref. 6"},
                                {"h", "c'", "s'", "h", "7.5300", "2", "180.0000", "Ver. 1.3, Ref. 7"},
                                {"c", "c'", "c", "s'", "7.5300", "2", "180.0000", "Ver. 1.3, Ref. 7"},
                                {"*", "c'_", "*", "*", "10.0000", "2", "180.0000", "Ver. 2.0, Ref. 18"},
                                {"*", "cp_", "*", "*", "0.3700", "2", "180.0000", "Ver. 2.0, Ref. 18"},
                                {"*", "c=_", "*", "*", "11.1000", "2", "180.0000", "Ver. 2.0, Ref. 18"},
                                {"*", "n_", "*", "*", "0.0500", "2", "180.0000", "Ver. 2.0, Ref. 18"},
                                {"*", "np_", "*", "*", "0.3700", "2", "180.0000", "Ver. 2.0, Ref. 18"}};

/*
  Non-bonded (12-6)
 0= Key, 1= A, 2= B,  3= CVFF info
 E = Aij/r^12 - Bij/r^6
  where  Aij = sqrt( Ai * Aj )
         Bij = sqrt( Bi * Bj )0
*/
char * CVFF_vdw[45][4]= {{"h", "7108.4660", "32.87076", "Ver. 1.0, Ref. 1"},
                         {"cg", "1790340.7240", "528.48190", "Ver. 1.0, Ref. 1"},
                         {"o'", "272894.7846", "498.87880", "Ver. 1.0, Ref. 1"},
                         {"oz", "272894.7846", "498.87880", "Ver. 1.8, Ref. 14"},
                         {"n", "2266872.4000", "1230.55700", "Ver. 1.0, Ref. 1"},
                         {"c'", "2968753.3590", "1325.70810", "Ver. 1.0, Ref. 1"},
                         {"c", "1981049.2250", "1125.99800", "Ver. 1.0, Ref. 1"},
                         {"hn", "0.00000001", "0.00000", "Ver. 1.0, Ref. 1"},
                         {"s", "365906.4000", "250.80000", "Ver. 1.0, Ref. 1"},
                         {"s'", "1395550.1000", "956.80800", "Ver. 1.3, Ref. 6"},
                         {"o*", "629358.0000", "625.50000", "Ver. 1.0, Ref. 1"},
                         {"ospc", "629358.0000", "625.50000", "Ver. 2.3, Ref. 25"},
                         {"otip", "582000.0000", "595.00000", "Ver. 2.3, Ref. 25"},
                         {"h*", "0.00000001", "0.00000", "Ver. 1.0, Ref. 1"},
                         {"hspc", "0.00000001", "0.00000", "Ver. 2.3, Ref. 25"},
                         {"htip", "0.00000001", "0.00000", "Ver. 2.3, Ref. 25"},
                         {"p", "6025894.0000", "2195.60000", "Ver. 1.0, Ref. 1"},
                         {"ca+", "119025.0000", "240.25000", "Ver. 2.0, Ref. 20"},
                         {"si", "3149175.0000", "710.00000", "Ver. 1.0, Ref. 1"},
                         {"sz", "3149175.0000", "710.00000", "Ver. 1.8, Ref. 14"},
                         {"f", "201106.0000", "235.20000", "Ver. 1.0, Ref. 1"},
                         {"cl", "1059166.0000", "541.00000", "Ver. 1.0, Ref. 1"},
                         {"br", "3572030.0000", "1195.00000", "Ver. 1.0, Ref. 1"},
                         {"Na", "14000.0000", "300.00000", "Ver. 1.0, Ref. 1"},
                         {"Cl", "25552052.0000", "3307.00450", "Ver. 1.0, Ref. 1"},
                         {"Br", "34375640.0000", "3517.84460", "Ver. 1.0, Ref. 1"},
                         {"ar", "2312930.0000", "1484.09200", "Ver. 1.3, Ref. 9"},
                         {"he", "6315.5582", "22.64953", "Ver. 2.4, Ref. 30"},
                         {"nu", "0.00000001", "0.00000", "Ver. 1.0, Ref. 1"},
                         {"Al", "3784321.4254", "11699.84934", "Ver. 2.1, Ref. 27"},
                         {"Au", "4603936.5046", "13692.05223", "Ver. 2.1, Ref. 27"},
                         {"Pb", "24856948.1942", "23280.48320", "Ver. 2.1, Ref. 27"},
                         {"Ni", "955901.6916", "6768.92014", "Ver. 2.1, Ref. 27"},
                         {"Pd", "2581174.9390", "10078.92459", "Ver. 2.1, Ref. 27"},
                         {"Pt", "4576819.9618", "16963.30818", "Ver. 2.1, Ref. 27"},
                         {"Ag", "3712095.6064", "10865.51833", "Ver. 2.1, Ref. 27"},
                         {"Cu", "1007210.0670", "6166.70278", "Ver. 2.1, Ref. 27"},
                         {"Cr", "1222517.4049", "7523.46700", "Ver. 2.1, Ref. 27"},
                         {"Fe", "1186612.1982", "7590.28296", "Ver. 2.1, Ref. 27"},
                         {"Li", "5192358.6600", "9916.81768", "Ver. 2.1, Ref. 27"},
                         {"Mo", "5869689.0344", "21298.66304", "Ver. 2.1, Ref. 27"},
                         {"W", "7876811.6340", "27853.23915", "Ver. 2.1, Ref. 27"},
                         {"al", "11422865.0000", "2282.96606", "Ver. 2.2, Ref. 29"},
                         {"mg", "3149175.0000", "710.00000", "Ver. 2.1, Ref. 27"},
                         {"K", "24856948.1942", "23280.48320", "Ver. 2.1, Ref. 27"}};

/*
 Bond increments
 To implement atomic charges
*/
char * CVFF_bond_increments[683][5]= {{"no", "o-", "0.1684", "-0.1684", "Ver. 2.3, Ref. 23"},
                                      {"no", "cp", "-0.1792", "0.1792", "Ver. 2.3, Ref. 23"},
                                      {"c'", "cp", "-0.1792", "0.1792", "Ver. 2.0, Ref. 18"},
                                      {"c", "cr", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "ci", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "n1", "0.1000", "-0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c", "s1", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "o-", "0.0700", "-0.5700", "Ver. 1.0, Ref. 1"},
                                      {"ci", "h", "-0.2300", "0.2300", "Ver. 1.0, Ref. 1"},
                                      {"ci", "ci", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"ci", "ni", "0.3200", "-0.0700", "Ver. 1.0, Ref. 1"},
                                      {"cr", "n", "0.3800", "-0.3800", "Ver. 1.0, Ref. 1"},
                                      {"cr", "n1", "0.5000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"cr", "n2", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"cr", "n=", "0.4000", "-0.4000", "Ver. 1.0, Ref. 1"},
                                      {"n", "hn", "-0.2800", "0.2800", "Ver. 1.0, Ref. 1"},
                                      {"n", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"n1", "hn", "0.0000", "0.5000", "Ver. 1.0, Ref. 1"},
                                      {"n1", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"n2", "hn", "-0.2800", "0.2800", "Ver. 1.0, Ref. 1"},
                                      {"n2", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"n3", "hn", "-0.1400", "0.1400", "Ver. 1.0, Ref. 1"},
                                      {"n3", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"n4", "hn", "-0.1100", "0.3600", "Ver. 1.0, Ref. 1"},
                                      {"n4", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"np", "hn", "-0.2800", "0.2800", "Ver. 1.0, Ref. 1"},
                                      {"np", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"ni", "hn", "-0.3600", "0.3600", "Ver. 1.0, Ref. 1"},
                                      {"ni", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"o", "ho", "-0.2233", "0.2233", "Ver. 1.0, Ref. 1"},
                                      {"o", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"oh", "ho", "-0.3500", "0.3500", "Ver. 1.0, Ref. 1"},
                                      {"oh", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"o*", "h*", "-0.4100", "0.4100", "Ver. 1.0, Ref. 1"},
                                      {"ospc", "hspc", "-0.4100", "0.4100", "Ver. 2.3, Ref. 25"},
                                      {"otip", "htip", "-0.4170", "0.4170", "Ver. 2.3, Ref. 25"},
                                      {"o*", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"o-", "p", "-0.8500", "0.3500", "Ver. 1.0, Ref. 1"},
                                      {"s-", "p", "-0.6824", "0.1824", "Ver. 2.0, Ref. 18"},
                                      {"oz", "sz", "-0.1500", "0.1500", "Ver. 1.8, Ref. 14"},
                                      {"sh", "hs", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"sh", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"s1", "s1", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"h", "p", "0.1000", "-0.1000", "Ver. 1.0, Ref. 1"},
                                      {"h", "sz", "0.0200", "-0.0200", "Ver. 1.8, Ref. 14"},
                                      {"d", "d", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"p", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"f", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"cl", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"br", "lp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c'", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "cp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c5", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "cs", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c=", "0.1000", "-0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c=1", "0.1000", "-0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c=2", "0.1000", "-0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c", "c-", "0.0865", "-0.0865", "Ver. 2.0, Ref. 18"},
                                      {"c", "ct", "0.0400", "-0.0400", "Ver. 1.2, Ref. 3"},
                                      {"c", "n3", "0.2200", "-0.2200", "Ver. 1.0, Ref. 1"},
                                      {"c", "n", "0.2200", "-0.2200", "Ver. 1.0, Ref. 1"},
                                      {"c", "n2", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c", "n=", "0.1100", "-0.1100", "Ver. 1.0, Ref. 1"},
                                      {"c", "n=1", "0.1100", "-0.1100", "Ver. 1.0, Ref. 1"},
                                      {"c", "n=2", "0.1100", "-0.1100", "Ver. 1.0, Ref. 1"},
                                      {"c", "np", "0.1100", "-0.1100", "Ver. 1.0, Ref. 1"},
                                      {"c", "n4", "0.4200", "-0.1700", "Ver. 1.0, Ref. 1"},
                                      {"c", "nt", "0.1000", "-0.1000", "Ver. 1.3, Ref. 5"},
                                      {"c", "nz", "0.3640", "-0.3640", "Ver. 2.0, Ref. 18"},
                                      {"c", "o", "0.1500", "-0.1500", "Ver. 1.0, Ref. 1"},
                                      {"c", "oh", "0.0300", "-0.0300", "Ver. 1.0, Ref. 1"},
                                      {"c", "oz", "0.1742", "-0.1742", "Ver. 2.0, Ref. 18"},
                                      {"c", "op", "0.3957", "-0.3957", "Ver. 2.0, Ref. 18"},
                                      {"c", "s", "-0.0500", "0.0500", "Ver. 1.0, Ref. 1"},
                                      {"c", "sh", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c", "sp", "0.1180", "-0.1180", "Ver. 2.0, Ref. 18"},
                                      {"c", "s'", "0.1180", "-0.1180", "Ver. 2.0, Ref. 18"},
                                      {"c", "p", "-0.0785", "0.0785", "Ver. 2.0, Ref. 18"},
                                      {"c", "h", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c", "f", "0.2750", "-0.2750", "Ver. 1.0, Ref. 1"},
                                      {"c", "cl", "0.2260", "-0.2260", "Ver. 1.0, Ref. 1"},
                                      {"c", "br", "0.1920", "-0.1920", "Ver. 1.0, Ref. 1"},
                                      {"c", "i", "0.1120", "-0.1120", "Ver. 2.0, Ref. 18"},
                                      {"c", "si", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c'", "c5", "0.0", "0.0", "Ver. 1.5, Ref. 11"},
                                      {"c'", "cs", "0.0", "0.0", "Ver. 1.5, Ref. 11"},
                                      {"c'", "c=", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "c=1", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "c=2", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "c-", "-0.1368", "0.1368", "Ver. 2.0, Ref. 18"},
                                      {"c'", "ct", "-0.0927", "0.0927", "Ver. 2.0, Ref. 18"},
                                      {"c'", "n3", "-0.0442", "0.0442", "Ver. 2.0, Ref. 18"},
                                      {"c'", "n", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "n2", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c'", "n=", "0.0362", "-0.0362", "Ver. 2.0, Ref. 18"},
                                      {"c'", "n=1", "0.0362", "-0.0362", "Ver. 2.0, Ref. 18"},
                                      {"c'", "n=2", "0.0362", "-0.0362", "Ver. 2.0, Ref. 18"},
                                      {"c'", "np", "0.0362", "-0.0362", "Ver. 2.0, Ref. 18"},
                                      {"c'", "n4", "0.1331", "0.1169", "Ver. 2.0, Ref. 18"},
                                      {"c'", "nt", "0.1641", "-0.1641", "Ver. 2.0, Ref. 18"},
                                      {"c'", "o", "0.0300", "-0.0300", "Ver. 1.0, Ref. 1"},
                                      {"c'", "oh", "0.0300", "-0.0300", "Ver. 1.0, Ref. 1"},
                                      {"c'", "oz", "-0.0135", "0.0135", "Ver. 2.0, Ref. 18"},
                                      {"c'", "op", "0.0300", "-0.0300", "Ver. 1.4, Ref. 10"},
                                      {"c'", "o'", "0.3800", "-0.3800", "Ver. 1.0, Ref. 1"},
                                      {"c'", "s", "-0.1528", "0.1528", "Ver. 2.0, Ref. 18"},
                                      {"c'", "sh", "-0.2033", "0.2033", "Ver. 2.0, Ref. 18"},
                                      {"c'", "sp", "-0.1079", "0.1079", "Ver. 2.0, Ref. 18"},
                                      {"c'", "s'", "0.0", "0.0", "Ver. 1.3, Ref. 7"},
                                      {"c'", "p", "-0.3283", "0.3283", "Ver. 2.0, Ref. 18"},
                                      {"c'", "h", "-0.2132", "0.2132", "Ver. 1.0, Ref. 1"},
                                      {"c'", "f", "0.1116", "-0.1116", "Ver. 2.0, Ref. 18"},
                                      {"c'", "cl", "-0.0594", "0.0594", "Ver. 2.0, Ref. 18"},
                                      {"c'", "br", "-0.1152", "0.1152", "Ver. 2.0, Ref. 18"},
                                      {"c'", "i", "-0.1291", "0.1291", "Ver. 2.0, Ref. 18"},
                                      {"c'", "si", "-0.4405", "0.4405", "Ver. 2.0, Ref. 18"},
                                      {"cp", "cp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"cp", "c5", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"cp", "cs", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cp", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cp", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cp", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cp", "c-", "0.0424", "-0.0424", "Ver. 2.0, Ref. 18"},
                                      {"cp", "ct", "0.0852", "-0.0852", "Ver. 2.0, Ref. 18"},
                                      {"cp", "n3", "0.1216", "-0.1216", "Ver. 2.0, Ref. 18"},
                                      {"cp", "n", "0.1100", "-0.1100", "Ver. 1.0, Ref. 1"},
                                      {"cp", "n2", "0.1050", "-0.1050", "Ver. 1.1, Ref. 2"},
                                      {"cp", "n=", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cp", "n=1", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cp", "n=2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cp", "np", "0.1100", "-0.1100", "Ver. 1.0, Ref. 1"},
                                      {"cp", "n4", "0.2989", "-0.0489", "Ver. 2.0, Ref. 18"},
                                      {"cp", "nt", "0.3230", "-0.3230", "Ver. 2.0, Ref. 18"},
                                      {"cp", "o", "0.0282", "-0.0282", "Ver. 1.9, Ref. 16"},
                                      {"cp", "oh", "0.0300", "-0.0300", "Ver. 1.0, Ref. 1"},
                                      {"cp", "oz", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"cp", "op", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"cp", "o'", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"cp", "s", "0.0282", "-0.0282", "Ver. 2.0, Ref. 18"},
                                      {"cp", "sh", "-0.0222", "0.0222", "Ver. 2.0, Ref. 18"},
                                      {"cp", "sp", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"cp", "s'", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"cp", "p", "-0.1267", "0.1267", "Ver. 2.0, Ref. 18"},
                                      {"cp", "h", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"cp", "f", "0.1300", "-0.1300", "Ver. 1.0, Ref. 1"},
                                      {"cp", "cl", "0.1020", "-0.1020", "Ver. 1.0, Ref. 1"},
                                      {"cp", "br", "0.0800", "-0.0800", "Ver. 1.0, Ref. 1"},
                                      {"cp", "i", "0.0642", "-0.0642", "Ver. 2.0, Ref. 18"},
                                      {"cp", "si", "-0.2270", "0.2270", "Ver. 2.0, Ref. 18"},
                                      {"c5", "c5", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c5", "cs", "0.0000", "0.0000", "Ver. 1.3, Ref. 6"},
                                      {"c5", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c5", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c5", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c5", "c-", "0.0424", "-0.0424", "Ver. 2.0, Ref. 18"},
                                      {"c5", "ct", "0.0852", "-0.0852", "Ver. 2.0, Ref. 18"},
                                      {"c5", "n3", "0.1216", "-0.1216", "Ver. 2.0, Ref. 18"},
                                      {"c5", "n", "0.1100", "-0.1100", "Ver. 1.4, Ref. 10"},
                                      {"c5", "n2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c5", "n=", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c5", "n=1", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c5", "n=2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c5", "np", "0.1400", "-0.1400", "Ver. 1.0, Ref. 1"},
                                      {"c5", "n4", "0.2989", "-0.0489", "Ver. 2.0, Ref. 18"},
                                      {"c5", "nt", "0.3230", "-0.3230", "Ver. 2.0, Ref. 18"},
                                      {"c5", "o", "0.1100", "-0.1100", "Ver. 1.2, Ref. 3"},
                                      {"c5", "oh", "0.0297", "-0.0297", "Ver. 2.0, Ref. 18"},
                                      {"c5", "oz", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c5", "op", "0.1100", "-0.1100", "Ver. 1.3, Ref. 8"},
                                      {"c5", "o'", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c5", "s", "-0.1500", "0.1500", "Ver. 1.2, Ref. 3"},
                                      {"c5", "sh", "-0.0222", "0.0222", "Ver. 2.0, Ref. 18"},
                                      {"c5", "sp", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c5", "s'", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c5", "p", "-0.1267", "0.1267", "Ver. 2.0, Ref. 18"},
                                      {"c5", "h", "-0.1300", "0.1300", "Ver. 1.0, Ref. 1"},
                                      {"c5", "f", "0.2589", "-0.2589", "Ver. 2.0, Ref. 18"},
                                      {"c5", "cl", "0.1163", "-0.1163", "Ver. 2.0, Ref. 18"},
                                      {"c5", "br", "0.0725", "-0.0725", "Ver. 2.0, Ref. 18"},
                                      {"c5", "i", "0.0642", "-0.0642", "Ver. 2.0, Ref. 18"},
                                      {"c5", "si", "-0.2270", "0.2270", "Ver. 2.0, Ref. 18"},
                                      {"cs", "cs", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cs", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cs", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cs", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"cs", "c-", "0.0424", "-0.0424", "Ver. 2.0, Ref. 18"},
                                      {"cs", "ct", "0.0852", "-0.0852", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n3", "0.1216", "-0.1216", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n=", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n=1", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n=2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cs", "np", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"cs", "n4", "0.2989", "-0.0489", "Ver. 2.0, Ref. 18"},
                                      {"cs", "nt", "0.3230", "-0.3230", "Ver. 2.0, Ref. 18"},
                                      {"cs", "o", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"cs", "oh", "0.0297", "-0.0297", "Ver. 2.0, Ref. 18"},
                                      {"cs", "oz", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"cs", "op", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"cs", "o'", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"cs", "s", "0.0282", "-0.0282", "Ver. 2.0, Ref. 18"},
                                      {"cs", "sh", "-0.0222", "0.0222", "Ver. 2.0, Ref. 18"},
                                      {"cs", "sp", "-0.1500", "0.1500", "Ver. 1.3, Ref. 6"},
                                      {"cs", "s'", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"cs", "p", "-0.1267", "0.1267", "Ver. 2.0, Ref. 18"},
                                      {"cs", "h", "-0.1300", "0.1300", "Ver. 1.3, Ref. 6"},
                                      {"cs", "f", "0.2589", "-0.2589", "Ver. 2.0, Ref. 18"},
                                      {"cs", "cl", "0.1163", "-0.1163", "Ver. 2.0, Ref. 18"},
                                      {"cs", "br", "0.0725", "-0.0725", "Ver. 2.0, Ref. 18"},
                                      {"cs", "i", "0.0642", "-0.0642", "Ver. 2.0, Ref. 18"},
                                      {"cs", "si", "-0.2270", "0.2270", "Ver. 2.0, Ref. 18"},
                                      {"c=", "c=", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c=", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c=", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c=", "c-", "0.0424", "-0.0424", "Ver. 2.0, Ref. 18"},
                                      {"c=", "ct", "0.0852", "-0.0852", "Ver. 2.0, Ref. 18"},
                                      {"c=", "n3", "0.1216", "-0.1216", "Ver. 2.0, Ref. 18"},
                                      {"c=", "n", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=", "n2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=", "n=", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=", "n=1", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=", "n=2", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=", "np", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=", "n4", "0.2989", "-0.0489", "Ver. 2.0, Ref. 18"},
                                      {"c=", "nt", "0.3230", "-0.3230", "Ver. 2.0, Ref. 18"},
                                      {"c=", "o", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c=", "oh", "0.0297", "-0.0297", "Ver. 2.0, Ref. 18"},
                                      {"c=", "oz", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c=", "op", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c=", "o'", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c=", "s", "0.0282", "-0.0282", "Ver. 2.0, Ref. 18"},
                                      {"c=", "sh", "-0.0222", "0.0222", "Ver. 2.0, Ref. 18"},
                                      {"c=", "sp", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c=", "s'", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c=", "p", "-0.1267", "0.1267", "Ver. 2.0, Ref. 18"},
                                      {"c=", "h", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c=", "f", "0.2589", "-0.2589", "Ver. 2.0, Ref. 18"},
                                      {"c=", "cl", "0.1163", "-0.1163", "Ver. 2.0, Ref. 18"},
                                      {"c=", "br", "0.0725", "-0.0725", "Ver. 2.0, Ref. 18"},
                                      {"c=", "i", "0.0642", "-0.0642", "Ver. 2.0, Ref. 18"},
                                      {"c=", "si", "-0.2270", "0.2270", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "c-", "0.0424", "-0.0424", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "ct", "0.0852", "-0.0852", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "n3", "0.1216", "-0.1216", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "n", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "n2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "n=", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=1", "n=1", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=1", "n=2", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=1", "np", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "n4", "0.2989", "-0.0489", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "nt", "0.3230", "-0.3230", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "o", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "oh", "0.0297", "-0.0297", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "oz", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "op", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "o'", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "s", "0.0282", "-0.0282", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "sh", "-0.0222", "0.0222", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "sp", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "s'", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "p", "-0.1267", "0.1267", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "h", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c=1", "f", "0.2589", "-0.2589", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "cl", "0.1163", "-0.1163", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "br", "0.0725", "-0.0725", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "i", "0.0642", "-0.0642", "Ver. 2.0, Ref. 18"},
                                      {"c=1", "si", "-0.2270", "0.2270", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "c-", "0.0424", "-0.0424", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "ct", "0.0852", "-0.0852", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "n3", "0.1216", "-0.1216", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "n", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "n2", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "n=", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=2", "n=1", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=2", "n=2", "0.3000", "-0.3000", "Ver. 1.0, Ref. 1"},
                                      {"c=2", "np", "0.1993", "-0.1993", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "n4", "0.2989", "-0.0489", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "nt", "0.3230", "-0.3230", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "o", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "oh", "0.0297", "-0.0297", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "oz", "0.1367", "-0.1367", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "op", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "o'", "0.3583", "-0.3583", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "s", "0.0282", "-0.0282", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "sh", "-0.0222", "0.0222", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "sp", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "s'", "0.0732", "-0.0732", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "p", "-0.1267", "0.1267", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "h", "-0.1000", "0.1000", "Ver. 1.0, Ref. 1"},
                                      {"c=2", "f", "0.2589", "-0.2589", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "cl", "0.1163", "-0.1163", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "br", "0.0725", "-0.0725", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "i", "0.0642", "-0.0642", "Ver. 2.0, Ref. 18"},
                                      {"c=2", "si", "-0.2270", "0.2270", "Ver. 2.0, Ref. 18"},
                                      {"c-", "c-", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"c-", "ct", "0.0432", "-0.0432", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n3", "0.0824", "-0.0824", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n", "0.1607", "-0.1607", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n2", "0.1607", "-0.1607", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n=", "0.1607", "-0.1607", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n=1", "0.1607", "-0.1607", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n=2", "0.1607", "-0.1607", "Ver. 2.0, Ref. 18"},
                                      {"c-", "np", "0.1607", "-0.1607", "Ver. 2.0, Ref. 18"},
                                      {"c-", "n4", "0.2597", "-0.0097", "Ver. 2.0, Ref. 18"},
                                      {"c-", "nt", "0.2854", "-0.2854", "Ver. 2.0, Ref. 18"},
                                      {"c-", "o", "0.1012", "-0.1012", "Ver. 2.0, Ref. 18"},
                                      {"c-", "oh", "-0.0058", "0.0058", "Ver. 2.0, Ref. 18"},
                                      {"c-", "oz", "0.1012", "-0.1012", "Ver. 2.0, Ref. 18"},
                                      {"c-", "op", "0.3241", "-0.3241", "Ver. 2.0, Ref. 18"},
                                      {"c-", "o'", "0.3241", "-0.3241", "Ver. 2.0, Ref. 18"},
                                      {"c-", "s", "-0.0146", "0.0146", "Ver. 2.0, Ref. 18"},
                                      {"c-", "sh", "-0.0650", "0.0650", "Ver. 2.0, Ref. 18"},
                                      {"c-", "sp", "0.0304", "-0.0304", "Ver. 2.0, Ref. 18"},
                                      {"c-", "s'", "0.0304", "-0.0304", "Ver. 2.0, Ref. 18"},
                                      {"c-", "s-", "-0.1223", "-0.3777", "Ver. 2.0, Ref. 18"},
                                      {"c-", "p", "-0.1744", "0.1744", "Ver. 2.0, Ref. 18"},
                                      {"c-", "h", "-0.1549", "0.1549", "Ver. 2.0, Ref. 18"},
                                      {"c-", "f", "0.2241", "-0.2241", "Ver. 2.0, Ref. 18"},
                                      {"c-", "cl", "0.0747", "-0.0747", "Ver. 2.0, Ref. 18"},
                                      {"c-", "br", "0.0281", "-0.0281", "Ver. 2.0, Ref. 18"},
                                      {"c-", "i", "0.0185", "-0.0185", "Ver. 2.0, Ref. 18"},
                                      {"c-", "si", "-0.2775", "0.2775", "Ver. 2.0, Ref. 18"},
                                      {"ct", "ct", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n3", "0.0419", "-0.0419", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n", "0.1204", "-0.1204", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n2", "0.1204", "-0.1204", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n=", "0.1204", "-0.1204", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n=1", "0.1204", "-0.1204", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n=2", "0.1204", "-0.1204", "Ver. 2.0, Ref. 18"},
                                      {"ct", "np", "0.1204", "-0.1204", "Ver. 2.0, Ref. 18"},
                                      {"ct", "n4", "0.2181", "0.0319", "Ver. 2.0, Ref. 18"},
                                      {"ct", "nt", "0.2454", "-0.2454", "Ver. 2.0, Ref. 18"},
                                      {"ct", "o", "0.0644", "-0.0644", "Ver. 2.0, Ref. 18"},
                                      {"ct", "oh", "-0.0420", "0.0420", "Ver. 2.0, Ref. 18"},
                                      {"ct", "oz", "0.0644", "-0.0644", "Ver. 2.0, Ref. 18"},
                                      {"ct", "op", "0.2874", "-0.2874", "Ver. 2.0, Ref. 18"},
                                      {"ct", "o'", "0.2874", "-0.2874", "Ver. 2.0, Ref. 18"},
                                      {"ct", "s", "-0.0581", "0.0581", "Ver. 2.0, Ref. 18"},
                                      {"ct", "sh", "-0.1082", "0.1082", "Ver. 2.0, Ref. 18"},
                                      {"ct", "sp", "-0.0135", "0.0135", "Ver. 2.0, Ref. 18"},
                                      {"ct", "s'", "-0.0135", "0.0135", "Ver. 2.0, Ref. 18"},
                                      {"ct", "p", "-0.2216", "0.2216", "Ver. 2.0, Ref. 18"},
                                      {"ct", "h", "-0.2000", "0.2000", "Ver. 1.2, Ref. 3"},
                                      {"ct", "f", "0.1873", "-0.1873", "Ver. 2.0, Ref. 18"},
                                      {"ct", "cl", "0.0319", "-0.0319", "Ver. 2.0, Ref. 18"},
                                      {"ct", "br", "-0.0173", "0.0173", "Ver. 2.0, Ref. 18"},
                                      {"ct", "i", "-0.0281", "0.0281", "Ver. 2.0, Ref. 18"},
                                      {"ct", "si", "-0.3266", "0.3266", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n3", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n", "0.0742", "-0.0742", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n2", "0.0742", "-0.0742", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n=", "0.0742", "-0.0742", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n=1", "0.0742", "-0.0742", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n=2", "0.0742", "-0.0742", "Ver. 2.0, Ref. 18"},
                                      {"n3", "np", "0.0742", "-0.0742", "Ver. 2.0, Ref. 18"},
                                      {"n3", "n4", "0.1650", "0.0850", "Ver. 2.0, Ref. 18"},
                                      {"n3", "nt", "0.1927", "-0.1927", "Ver. 2.0, Ref. 18"},
                                      {"n3", "o", "0.0249", "-0.0249", "Ver. 2.0, Ref. 18"},
                                      {"n3", "oh", "-0.0754", "0.0754", "Ver. 2.0, Ref. 18"},
                                      {"n3", "oz", "0.0249", "-0.0249", "Ver. 2.0, Ref. 18"},
                                      {"n3", "op", "0.2369", "-0.2369", "Ver. 2.0, Ref. 18"},
                                      {"n3", "o'", "0.2369", "-0.2369", "Ver. 2.0, Ref. 18"},
                                      {"n3", "s", "-0.0967", "0.0967", "Ver. 2.0, Ref. 18"},
                                      {"n3", "sh", "-0.1434", "0.1434", "Ver. 2.0, Ref. 18"},
                                      {"n3", "sp", "-0.0551", "0.0551", "Ver. 2.0, Ref. 18"},
                                      {"n3", "s'", "-0.0551", "0.0551", "Ver. 2.0, Ref. 18"},
                                      {"n3", "p", "-0.2518", "0.2518", "Ver. 2.0, Ref. 18"},
                                      {"n3", "h", "-0.2386", "0.2386", "Ver. 2.0, Ref. 18"},
                                      {"n3", "f", "0.1415", "-0.1415", "Ver. 2.0, Ref. 18"},
                                      {"n3", "cl", "-0.0117", "0.0117", "Ver. 2.0, Ref. 18"},
                                      {"n3", "br", "-0.0601", "0.0601", "Ver. 2.0, Ref. 18"},
                                      {"n3", "i", "-0.0714", "0.0714", "Ver. 2.0, Ref. 18"},
                                      {"n3", "si", "-0.3501", "0.3501", "Ver. 2.0, Ref. 18"},
                                      {"n", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n", "n2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n", "n=", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n", "n=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n", "n4", "0.0883", "0.1617", "Ver. 2.0, Ref. 18"},
                                      {"n", "nt", "0.1186", "-0.1186", "Ver. 2.0, Ref. 18"},
                                      {"n", "o", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n", "oh", "-0.1421", "0.1421", "Ver. 2.0, Ref. 18"},
                                      {"n", "oz", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n", "op", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n", "o'", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n", "s", "-0.1755", "0.1755", "Ver. 2.0, Ref. 18"},
                                      {"n", "sh", "-0.2214", "0.2214", "Ver. 2.0, Ref. 18"},
                                      {"n", "sp", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n", "s'", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n", "p", "-0.3359", "0.3359", "Ver. 2.0, Ref. 18"},
                                      {"n", "h", "-0.3278", "0.3278", "Ver. 2.0, Ref. 18"},
                                      {"n", "f", "0.0731", "-0.0731", "Ver. 2.0, Ref. 18"},
                                      {"n", "cl", "-0.0897", "0.0897", "Ver. 2.0, Ref. 18"},
                                      {"n", "br", "-0.1422", "0.1422", "Ver. 2.0, Ref. 18"},
                                      {"n", "i", "-0.1554", "0.1554", "Ver. 2.0, Ref. 18"},
                                      {"n", "si", "-0.4367", "0.4367", "Ver. 2.0, Ref. 18"},
                                      {"n2", "n2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n2", "n=", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n2", "n=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n2", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n2", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n2", "n4", "0.0883", "0.1617", "Ver. 2.0, Ref. 18"},
                                      {"n2", "nt", "0.1186", "-0.1186", "Ver. 2.0, Ref. 18"},
                                      {"n2", "o", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n2", "oh", "-0.1421", "0.1421", "Ver. 2.0, Ref. 18"},
                                      {"n2", "oz", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n2", "op", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n2", "o'", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n2", "s", "-0.1755", "0.1755", "Ver. 2.0, Ref. 18"},
                                      {"n2", "sh", "-0.2214", "0.2214", "Ver. 2.0, Ref. 18"},
                                      {"n2", "sp", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n2", "s'", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n2", "p", "-0.3359", "0.3359", "Ver. 2.0, Ref. 18"},
                                      {"n2", "h", "-0.3278", "0.3278", "Ver. 2.0, Ref. 18"},
                                      {"n2", "f", "0.0731", "-0.0731", "Ver. 2.0, Ref. 18"},
                                      {"n2", "cl", "-0.0897", "0.0897", "Ver. 2.0, Ref. 18"},
                                      {"n2", "br", "-0.1422", "0.1422", "Ver. 2.0, Ref. 18"},
                                      {"n2", "i", "-0.1554", "0.1554", "Ver. 2.0, Ref. 18"},
                                      {"n2", "si", "-0.4367", "0.4367", "Ver. 2.0, Ref. 18"},
                                      {"n=", "n=", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=", "n=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=", "n4", "0.0883", "0.1617", "Ver. 2.0, Ref. 18"},
                                      {"n=", "nt", "0.1186", "-0.1186", "Ver. 2.0, Ref. 18"},
                                      {"n=", "o", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n=", "oh", "-0.1421", "0.1421", "Ver. 2.0, Ref. 18"},
                                      {"n=", "oz", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n=", "op", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=", "o'", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=", "o-", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=", "s", "-0.1755", "0.1755", "Ver. 2.0, Ref. 18"},
                                      {"n=", "sh", "-0.2214", "0.2214", "Ver. 2.0, Ref. 18"},
                                      {"n=", "sp", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n=", "s'", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n=", "p", "-0.3359", "0.3359", "Ver. 2.0, Ref. 18"},
                                      {"n=", "h", "-0.3278", "0.3278", "Ver. 2.0, Ref. 18"},
                                      {"n=", "f", "0.0731", "-0.0731", "Ver. 2.0, Ref. 18"},
                                      {"n=", "cl", "-0.0897", "0.0897", "Ver. 2.0, Ref. 18"},
                                      {"n=", "br", "-0.1422", "0.1422", "Ver. 2.0, Ref. 18"},
                                      {"n=", "i", "-0.1554", "0.1554", "Ver. 2.0, Ref. 18"},
                                      {"n=", "si", "-0.4367", "0.4367", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "n=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "n4", "0.0883", "0.1617", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "nt", "0.1186", "-0.1186", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "o", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "oh", "-0.1421", "0.1421", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "oz", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "op", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "o'", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "s", "-0.1755", "0.1755", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "sh", "-0.2214", "0.2214", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "sp", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "s'", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "p", "-0.3359", "0.3359", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "h", "-0.3278", "0.3278", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "f", "0.0731", "-0.0731", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "cl", "-0.0897", "0.0897", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "br", "-0.1422", "0.1422", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "i", "-0.1554", "0.1554", "Ver. 2.0, Ref. 18"},
                                      {"n=1", "si", "-0.4367", "0.4367", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "n4", "0.0883", "0.1617", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "nt", "0.1186", "-0.1186", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "o", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "oh", "-0.1421", "0.1421", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "oz", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "op", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "o'", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "s", "-0.1755", "0.1755", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "sh", "-0.2214", "0.2214", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "sp", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "s'", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "p", "-0.3359", "0.3359", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "h", "-0.3278", "0.3278", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "f", "0.0731", "-0.0731", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "cl", "-0.0897", "0.0897", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "br", "-0.1422", "0.1422", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "i", "-0.1554", "0.1554", "Ver. 2.0, Ref. 18"},
                                      {"n=2", "si", "-0.4367", "0.4367", "Ver. 2.0, Ref. 18"},
                                      {"np", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"np", "n4", "0.0883", "0.1617", "Ver. 2.0, Ref. 18"},
                                      {"np", "nt", "0.1186", "-0.1186", "Ver. 2.0, Ref. 18"},
                                      {"np", "o", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"np", "oh", "-0.1421", "0.1421", "Ver. 2.0, Ref. 18"},
                                      {"np", "oz", "-0.0432", "0.0432", "Ver. 2.0, Ref. 18"},
                                      {"np", "op", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"np", "o'", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"np", "o-", "0.1684", "-0.1684", "Ver. 2.0, Ref. 18"},
                                      {"np", "s", "-0.1755", "0.1755", "Ver. 2.0, Ref. 18"},
                                      {"np", "sh", "-0.2214", "0.2214", "Ver. 2.0, Ref. 18"},
                                      {"np", "sp", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"np", "s'", "-0.1346", "0.1346", "Ver. 2.0, Ref. 18"},
                                      {"np", "p", "-0.3359", "0.3359", "Ver. 2.0, Ref. 18"},
                                      {"np", "h", "-0.3278", "0.3278", "Ver. 2.0, Ref. 18"},
                                      {"np", "f", "0.0731", "-0.0731", "Ver. 2.0, Ref. 18"},
                                      {"np", "cl", "-0.0897", "0.0897", "Ver. 2.0, Ref. 18"},
                                      {"np", "br", "-0.1422", "0.1422", "Ver. 2.0, Ref. 18"},
                                      {"np", "i", "-0.1554", "0.1554", "Ver. 2.0, Ref. 18"},
                                      {"np", "si", "-0.4367", "0.4367", "Ver. 2.0, Ref. 18"},
                                      {"n4", "n4", "0.2500", "0.2500", "Ver. 2.0, Ref. 18"},
                                      {"n4", "nt", "0.2842", "-0.0342", "Ver. 2.0, Ref. 18"},
                                      {"n4", "o", "0.1245", "0.1255", "Ver. 2.0, Ref. 18"},
                                      {"n4", "oh", "0.0242", "0.2258", "Ver. 2.0, Ref. 18"},
                                      {"n4", "oz", "0.1245", "0.1255", "Ver. 2.0, Ref. 18"},
                                      {"n4", "op", "0.3418", "-0.0918", "Ver. 2.0, Ref. 18"},
                                      {"n4", "o'", "0.3418", "-0.0918", "Ver. 2.0, Ref. 18"},
                                      {"n4", "s", "-0.0257", "0.2757", "Ver. 2.0, Ref. 18"},
                                      {"n4", "sh", "-0.0723", "0.3223", "Ver. 2.0, Ref. 18"},
                                      {"n4", "sp", "0.0159", "0.2341", "Ver. 2.0, Ref. 18"},
                                      {"n4", "s'", "0.0159", "0.2341", "Ver. 2.0, Ref. 18"},
                                      {"n4", "p", "-0.1994", "0.4494", "Ver. 2.0, Ref. 18"},
                                      {"n4", "h", "-0.1978", "0.4478", "Ver. 2.0, Ref. 18"},
                                      {"n4", "f", "0.2438", "0.0062", "Ver. 2.0, Ref. 18"},
                                      {"n4", "cl", "0.0642", "0.1858", "Ver. 2.0, Ref. 18"},
                                      {"n4", "br", "0.0048", "0.2452", "Ver. 2.0, Ref. 18"},
                                      {"n4", "i", "-0.0114", "0.2614", "Ver. 2.0, Ref. 18"},
                                      {"n4", "si", "-0.3083", "0.5583", "Ver. 2.0, Ref. 18"},
                                      {"nt", "nt", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"nt", "o", "-0.1523", "0.1523", "Ver. 2.0, Ref. 18"},
                                      {"nt", "oh", "-0.2490", "0.2490", "Ver. 2.0, Ref. 18"},
                                      {"nt", "oz", "-0.1523", "0.1523", "Ver. 2.0, Ref. 18"},
                                      {"nt", "op", "0.0585", "-0.0585", "Ver. 2.0, Ref. 18"},
                                      {"nt", "o'", "0.0585", "-0.0585", "Ver. 2.0, Ref. 18"},
                                      {"nt", "s", "-0.3010", "0.3010", "Ver. 2.0, Ref. 18"},
                                      {"nt", "sh", "-0.3457", "0.3457", "Ver. 2.0, Ref. 18"},
                                      {"nt", "sp", "-0.2612", "0.2612", "Ver. 2.0, Ref. 18"},
                                      {"nt", "s'", "-0.2612", "0.2612", "Ver. 2.0, Ref. 18"},
                                      {"nt", "p", "-0.4691", "0.4691", "Ver. 2.0, Ref. 18"},
                                      {"nt", "h", "-0.4688", "0.4688", "Ver. 2.0, Ref. 18"},
                                      {"nt", "f", "-0.0367", "0.0367", "Ver. 2.0, Ref. 18"},
                                      {"nt", "cl", "-0.2141", "0.2141", "Ver. 2.0, Ref. 18"},
                                      {"nt", "br", "-0.2727", "0.2727", "Ver. 2.0, Ref. 18"},
                                      {"nt", "i", "-0.2889", "0.2889", "Ver. 2.0, Ref. 18"},
                                      {"nt", "si", "-0.5738", "0.5738", "Ver. 2.0, Ref. 18"},
                                      {"o", "o", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"o", "oh", "-0.0921", "0.0921", "Ver. 2.0, Ref. 18"},
                                      {"o", "oz", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"o", "op", "0.1962", "-0.1962", "Ver. 2.0, Ref. 18"},
                                      {"o", "s", "-0.1143", "0.1143", "Ver. 2.0, Ref. 18"},
                                      {"o", "sh", "-0.1565", "0.1565", "Ver. 2.0, Ref. 18"},
                                      {"o", "sp", "-0.0766", "0.0766", "Ver. 2.0, Ref. 18"},
                                      {"o", "s'", "-0.0766", "0.0766", "Ver. 2.0, Ref. 18"},
                                      {"o", "p", "-0.3500", "0.3500", "Ver. 1.0, Ref. 1"},
                                      {"o", "h", "-0.2432", "0.2432", "Ver. 2.0, Ref. 18"},
                                      {"o", "f", "0.1077", "-0.1077", "Ver. 2.0, Ref. 18"},
                                      {"o", "cl", "-0.0367", "0.0367", "Ver. 2.0, Ref. 18"},
                                      {"o", "br", "-0.0818", "0.0818", "Ver. 2.0, Ref. 18"},
                                      {"o", "i", "-0.0924", "0.0924", "Ver. 2.0, Ref. 18"},
                                      {"o", "si", "-0.1500", "0.1500", "Ver. 1.0, Ref. 1"},
                                      {"oh", "oh", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"oh", "oz", "0.0921", "-0.0921", "Ver. 2.0, Ref. 18"},
                                      {"oh", "op", "0.2853", "-0.2853", "Ver. 2.0, Ref. 18"},
                                      {"oh", "s", "-0.0063", "0.0063", "Ver. 2.0, Ref. 18"},
                                      {"oh", "sh", "-0.0485", "0.0485", "Ver. 2.0, Ref. 18"},
                                      {"oh", "sp", "0.0313", "-0.0313", "Ver. 2.0, Ref. 18"},
                                      {"oh", "s'", "0.0313", "-0.0313", "Ver. 2.0, Ref. 18"},
                                      {"oh", "p", "-0.1500", "0.1500", "Ver. 1.0, Ref. 1"},
                                      {"oh", "h", "-0.1190", "0.1190", "Ver. 2.0, Ref. 18"},
                                      {"oh", "f", "0.1983", "-0.1983", "Ver. 2.0, Ref. 18"},
                                      {"oh", "cl", "0.0686", "-0.0686", "Ver. 2.0, Ref. 18"},
                                      {"oh", "br", "0.0295", "-0.0295", "Ver. 2.0, Ref. 18"},
                                      {"oh", "i", "0.0216", "-0.0216", "Ver. 2.0, Ref. 18"},
                                      {"oh", "si", "-0.2188", "0.2188", "Ver. 2.0, Ref. 18"},
                                      {"oh", "sz", "-0.2188", "0.2188", "Ver. 2.0, Ref. 18"},
                                      {"oh", "az", "-0.2188", "0.2188", "Ver. 2.0, Ref. 18"},
                                      {"oz", "oz", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"oz", "op", "0.1962", "-0.1962", "Ver. 2.0, Ref. 18"},
                                      {"oz", "s", "-0.1143", "0.1143", "Ver. 2.0, Ref. 18"},
                                      {"oz", "sh", "-0.1565", "0.1565", "Ver. 2.0, Ref. 18"},
                                      {"oz", "sp", "-0.0766", "0.0766", "Ver. 2.0, Ref. 18"},
                                      {"oz", "s'", "-0.0766", "0.0766", "Ver. 2.0, Ref. 18"},
                                      {"oz", "p", "-0.2548", "0.2548", "Ver. 2.0, Ref. 18"},
                                      {"oz", "h", "-0.2432", "0.2432", "Ver. 2.0, Ref. 18"},
                                      {"oz", "ho", "-0.1", "0.1", "Ver. 2.0, Ref. 18"},
                                      {"oz", "f", "0.1077", "-0.1077", "Ver. 2.0, Ref. 18"},
                                      {"oz", "cl", "-0.0367", "0.0367", "Ver. 2.0, Ref. 18"},
                                      {"oz", "br", "-0.0818", "0.0818", "Ver. 2.0, Ref. 18"},
                                      {"oz", "i", "-0.0924", "0.0924", "Ver. 2.0, Ref. 18"},
                                      {"oz", "si", "-0.1500", "0.1500", "Ver. 2.0, Ref. 18"},
                                      {"oz", "sz", "-0.1500", "0.1500", "Ver. 2.0, Ref. 18"},
                                      {"oz", "az", "-0.0", "0.0", "Ver. 2.0, Ref. 18"},
                                      {"op", "op", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"op", "s", "-0.3386", "0.3386", "Ver. 2.0, Ref. 18"},
                                      {"op", "sh", "-0.3791", "0.3791", "Ver. 2.0, Ref. 18"},
                                      {"op", "sp", "-0.3024", "0.3024", "Ver. 2.0, Ref. 18"},
                                      {"op", "s'", "-0.3024", "0.3024", "Ver. 2.0, Ref. 18"},
                                      {"op", "p", "-0.4933", "0.4933", "Ver. 2.0, Ref. 18"},
                                      {"op", "h", "-0.4943", "0.4943", "Ver. 2.0, Ref. 18"},
                                      {"op", "f", "-0.0888", "0.0888", "Ver. 2.0, Ref. 18"},
                                      {"op", "cl", "-0.2585", "0.2585", "Ver. 2.0, Ref. 18"},
                                      {"op", "br", "-0.3140", "0.3140", "Ver. 2.0, Ref. 18"},
                                      {"op", "i", "-0.3297", "0.3297", "Ver. 2.0, Ref. 18"},
                                      {"op", "si", "-0.5883", "0.5883", "Ver. 2.0, Ref. 18"},
                                      {"o'", "o'", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"o'", "s", "-0.3386", "0.3386", "Ver. 2.0, Ref. 18"},
                                      {"o'", "sh", "-0.3791", "0.3791", "Ver. 2.0, Ref. 18"},
                                      {"o'", "sp", "-0.3024", "0.3024", "Ver. 2.0, Ref. 18"},
                                      {"o'", "s'", "-0.3024", "0.3024", "Ver. 2.0, Ref. 18"},
                                      {"o'", "p", "-0.8500", "0.3500", "Ver. 1.0, Ref. 1"},
                                      {"o'", "h", "-0.4943", "0.4943", "Ver. 2.0, Ref. 18"},
                                      {"o'", "f", "-0.0888", "0.0888", "Ver. 2.0, Ref. 18"},
                                      {"o'", "cl", "-0.2585", "0.2585", "Ver. 2.0, Ref. 18"},
                                      {"o'", "br", "-0.3140", "0.3140", "Ver. 2.0, Ref. 18"},
                                      {"o'", "i", "-0.3297", "0.3297", "Ver. 2.0, Ref. 18"},
                                      {"o'", "si", "-0.5883", "0.5883", "Ver. 2.0, Ref. 18"},
                                      {"s", "s", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"s", "sh", "-0.0509", "0.0509", "Ver. 2.0, Ref. 18"},
                                      {"s", "sp", "0.0455", "-0.0455", "Ver. 2.0, Ref. 18"},
                                      {"s", "s'", "0.0455", "-0.0455", "Ver. 2.0, Ref. 18"},
                                      {"s", "p", "-0.1600", "0.1600", "Ver. 2.0, Ref. 18"},
                                      {"s", "h", "-0.1392", "0.1392", "Ver. 2.0, Ref. 18"},
                                      {"s", "f", "0.2380", "-0.2380", "Ver. 2.0, Ref. 18"},
                                      {"s", "cl", "0.0898", "-0.0898", "Ver. 2.0, Ref. 18"},
                                      {"s", "br", "0.0437", "-0.0437", "Ver. 2.0, Ref. 18"},
                                      {"s", "i", "0.0345", "-0.0345", "Ver. 2.0, Ref. 18"},
                                      {"s", "si", "-0.2634", "0.2634", "Ver. 2.0, Ref. 18"},
                                      {"sh", "sh", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"sh", "sp", "0.0964", "-0.0964", "Ver. 2.0, Ref. 18"},
                                      {"sh", "s'", "0.0964", "-0.0964", "Ver. 2.0, Ref. 18"},
                                      {"sh", "p", "-0.1032", "0.1032", "Ver. 2.0, Ref. 18"},
                                      {"sh", "h", "-0.0787", "0.0787", "Ver. 2.0, Ref. 18"},
                                      {"sh", "f", "0.2794", "-0.2794", "Ver. 2.0, Ref. 18"},
                                      {"sh", "cl", "0.1392", "-0.1392", "Ver. 2.0, Ref. 18"},
                                      {"sh", "br", "0.0966", "-0.0966", "Ver. 2.0, Ref. 18"},
                                      {"sh", "i", "0.0889", "-0.0889", "Ver. 2.0, Ref. 18"},
                                      {"sh", "si", "-0.2032", "0.2032", "Ver. 2.0, Ref. 18"},
                                      {"sp", "sp", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"sp", "s'", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"sp", "p", "-0.2106", "0.2106", "Ver. 2.0, Ref. 18"},
                                      {"sp", "h", "-0.1932", "0.1932", "Ver. 2.0, Ref. 18"},
                                      {"sp", "f", "0.2011", "-0.2011", "Ver. 2.0, Ref. 18"},
                                      {"sp", "cl", "0.0457", "-0.0457", "Ver. 2.0, Ref. 18"},
                                      {"sp", "br", "-0.0034", "0.0034", "Ver. 2.0, Ref. 18"},
                                      {"sp", "i", "-0.0140", "0.0140", "Ver. 2.0, Ref. 18"},
                                      {"sp", "si", "-0.3172", "0.3172", "Ver. 2.0, Ref. 18"},
                                      {"s'", "s'", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"s'", "p", "-0.2106", "0.2106", "Ver. 2.0, Ref. 18"},
                                      {"s'", "h", "-0.1932", "0.1932", "Ver. 2.0, Ref. 18"},
                                      {"s'", "f", "0.2011", "-0.2011", "Ver. 2.0, Ref. 18"},
                                      {"s'", "cl", "0.0457", "-0.0457", "Ver. 2.0, Ref. 18"},
                                      {"s'", "br", "-0.0034", "0.0034", "Ver. 2.0, Ref. 18"},
                                      {"s'", "i", "-0.0140", "0.0140", "Ver. 2.0, Ref. 18"},
                                      {"s'", "si", "-0.3172", "0.3172", "Ver. 2.0, Ref. 18"},
                                      {"p", "p", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"},
                                      {"p", "h", "0.0356", "-0.0356", "Ver. 2.0, Ref. 18"},
                                      {"p", "f", "0.3869", "-0.3869", "Ver. 2.0, Ref. 18"},
                                      {"p", "cl", "0.2544", "-0.2544", "Ver. 2.0, Ref. 18"},
                                      {"p", "br", "0.2156", "-0.2156", "Ver. 2.0, Ref. 18"},
                                      {"p", "i", "0.2110", "-0.2110", "Ver. 2.0, Ref. 18"},
                                      {"p", "si", "-0.1069", "0.1069", "Ver. 2.0, Ref. 18"},
                                      {"h", "h", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"h", "f", "0.3823", "-0.3823", "Ver. 2.0, Ref. 18"},
                                      {"h", "cl", "0.2404", "-0.2404", "Ver. 2.0, Ref. 18"},
                                      {"h", "br", "0.1978", "-0.1978", "Ver. 2.0, Ref. 18"},
                                      {"h", "i", "0.1923", "-0.1923", "Ver. 2.0, Ref. 18"},
                                      {"h", "si", "0.0200", "-0.0200", "Ver. 1.0, Ref. 1"},
                                      {"f", "f", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"f", "cl", "-0.1589", "0.1589", "Ver. 2.0, Ref. 18"},
                                      {"f", "br", "-0.2099", "0.2099", "Ver. 2.0, Ref. 18"},
                                      {"f", "i", "-0.2234", "0.2234", "Ver. 2.0, Ref. 18"},
                                      {"f", "si", "-0.4789", "0.4789", "Ver. 2.0, Ref. 18"},
                                      {"cl", "cl", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"cl", "br", "-0.0507", "0.0507", "Ver. 2.0, Ref. 18"},
                                      {"cl", "i", "-0.0623", "0.0623", "Ver. 2.0, Ref. 18"},
                                      {"cl", "si", "-0.3598", "0.3598", "Ver. 2.0, Ref. 18"},
                                      {"br", "br", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"br", "i", "-0.0110", "0.0110", "Ver. 2.0, Ref. 18"},
                                      {"br", "si", "-0.3272", "0.3272", "Ver. 2.0, Ref. 18"},
                                      {"i", "i", "0.0", "0.0", "Ver. 1.3, Ref. 4"},
                                      {"i", "si", "-0.3263", "0.3263", "Ver. 2.0, Ref. 18"},
                                      {"si", "si", "0.0000", "0.0000", "Ver. 2.0, Ref. 18"}};
