#ifndef AVIFILE_FORMATS_H
#define AVIFILE_FORMATS_H

#ifdef WIN32
#include <windows.h>
#include <wingdi.h>
#include <mmsystem.h>
#include <mmreg.h>

#else
/* !WIN32 */
#include "avm_default.h"

#ifndef GUID_TYPE
#define GUID_TYPE
/**
 * GUID is unique 16-byte indentifier
 *
 * Usually being shown as: AAAAAAAA-BBBB-CCCC-DDDD-EEEEEEEEEEEE
 *
 * DDDD however is not uint16_t but two uint8_t
 * This representation allows to use it in the natural
 * human readable way
 * Note: f1, f2, f3 are little endian
 */
typedef struct AVMPACKED
{
    uint32_t	f1;
    uint16_t	f2;
    uint16_t	f3;
    uint8_t	f4[8];
} GUID;
#endif /* !GUID_TYPE */

#ifndef _BITMAPINFOHEADER_
#define _BITMAPINFOHEADER_

/**
 *  The BITMAPINFOHEADER contains all the details about the video stream such
 * as the actual image dimensions and their pixel depth. A source filter may
 * also request that the sink take only a section of the video by providing a
 * clipping rectangle in rcSource. In the worst case where the sink filter
 * forgets to check this on connection it will simply render the whole thing
 * which isn't a disaster. Ideally a sink filter will check the rcSource and
 * if it doesn't support image extraction and the rectangle is not empty then
 * it will reject the connection. A filter should use SetRectEmpty to reset a
 * rectangle to all zeroes (and IsRectEmpty to later check the rectangle).
 * The rcTarget specifies the destination rectangle for the video, for most
 * source filters they will set this to all zeroes, a downstream filter may
 * request that the video be placed in a particular area of the buffers it
 * supplies.
 */
typedef struct AVMPACKED
{
    /** Specifies the number of bytes required by the structure. */
    uint32_t	biSize;
    /** Specifies the width of bitmap, in pixels. */
    int32_t	biWidth;
    /**
     * Specifies the height of bitmap, in pixels.
     *
     * For RGB surfaces - negative orientation means top-down image
     * for all other codecs the sign is meangless and should be ignored
     * the FOURCC code uniquely identifies the compression and orientation
     * It is not valid to describe orientation with the sign of biHeight
     *
     * Common YUV format such as YV12, YUY2 are top-down oriented.
     * The sign of biHeight for such formats must always be set positive
     * by driver producing such formats and the sign must be ignored by any
     * driver receiving such formats. For proprietary compression formats with
     * an associated FOURCC, any orientation is acceptable, but must be
     * always the same for all bitmaps of the FOURCC.
     */
    int32_t	biHeight;
    /** Specifies the number of planes for target device. (Must be 1) */
    uint16_t	biPlanes;
    /**
     * Specifies the number of bits-per-pixel.
     *
     * - 0
     *		The number is specified or is implied by the JPEG or PNG
     *
     * - 1
     *		The bitmap is monochrome and the bmiColors member contains
     *		two entries. Each bit in the bitmap array represents a pixel.
     *		If the bit is clear, the pixel is displayed with the color
     *		of the first entry in the bmiColors table; if the bit is set,
     *		the pixel has the color of the second in the table.
     *
     * - 4
     *		The bitmap has a maximum of 16 colors and the bmiColors member
     *		contains up to 16 entries. Each pixel in the bitmap is represented
     *		by a 4-bit index into the color table. For example, if the
     *		first pixel contains the color in the second table entry, and
     *		the second pixel constains the color in the sixteenth table entry
     *
     * - 8
     *		The bitmap has a maximum of 256 color, and the bmiColors member
     *		constains up to 256 entries. In this case, each byte in the
     *		array represents a single pixel.
     *
     * - 16
     *		The bitmap has a maximum of 2^16 color. If the biCompression member
     *		of the BITMPAINFOHEADER is BI_RGB, the bmiColors member is NULL.
     *		Each WORD in the bitmap array represents a single pixel.
     *		The relative intensities of red, green and blue are represented
     *		with five bits for each color component 5-5-5.
     *		When the biCompression member is BI_BITFIELDS, the system supports
     *		only the following 16bpp color masks: A 5-5-5 16bit image,
     *		(b:0x001f, g:0x03e0, r:0x7c00) and 5-6-5 16bit image
     *		(b:0x001f  g:0x07e0  r:0xf800)
     *
     * - 24
     *		Bitmap has a maximum of 2^24 colors, and the bmiColors member
     *		is NULL, Each 3-byte triplet in the bitmap array represents
     *		the realtive intensities of blue, green, and red, respecitively,
     *		for each a pix. The bmiColors color table is used for optimizing
     *		colors used on palette-based devices, and must contain the number
     *		of entries specified by the biClrUsed member of BITMAPINFOHEADER.
     *
     * - 32
     *		The bitmap has a maximum of 2^32. If the biCompression member
     *		of the BITMPAINFOHEADER is BI_RGB, the bmiColors member is NULL.
     *		Each DWORD in the bitmap array represents the relative intensities
     *		of the blue, green and read, respectively, for a pixel. The high
     *		byte in each DWORD is not used. The bmiColors color table is used
     *		for optimizing colors used on palette-based devices,
     *		and must constain the number of entries specified by the biClrUsed
     *		member of the BITMAPINFOHEADER.
     *
     * \todo append remaining info here
     */
    uint16_t	biBitCount;
    /**
     * Specifies the compression format
     *
     * - BI_RGB
     *		An uncompressed format.
     *
     * - BI_RLE8
     *		A run-length encoded (RLE) format for bitmaps with 8bpp.
     *		The compression format is a 2-byte format consisting of
     *		a count byte followed by a byte containg a color index.
     *
     * - BI_RLE4
     *		An RLE format for bitmaps with 4bpp. The compression format
     *		is a 2-byte format consisting of a count byte followed
     *		by two word-length color indexes.
     *
     * - BI_BITFIELDS
     *		Specifies that the bitmap is not compressed and that the
     *		color table consists of three DWORD color masks that
     *		specify the red, green, and blue components, respectively,
     *		of each pixel. This is valid when used with 16- and
     *		32-bpp bitmaps.
     *
     * - BI_JPEG
     *		Image is a JPEG image
     *
     * - BI_PNG
     *		Image is PNG image
     */
    uint32_t	biCompression;
    /**
     * Specifies the size, in bytes, of the image. This may be set to zero
     * for BI_RGB bitmaps.
     */
    uint32_t	biSizeImage;
    /** Specifies the horizontal resolution, in pixels-pre-meter */
    int32_t	biXPelsPerMeter;
    /** Specifies the vertical resolution, in pixels-pre-meter */
    int32_t	biYPelsPerMeter;
    /**
     * Specifies the number of color indexes in the color table that
     * are actually used by the bitmap. If this value is zero, the bitmap
     * uses the maximum number of colors corresponding to the value of
     * the biBitCount member for the compression mode specified by
     * biCompression.
     *
     * If biClrUsed is nonzero and the biBitCount member is less then 16,
     * the biClrUsed member specifies the actual number of colors the graphics
     * engine or device driver accesses. If the biBitCount is 16 or greater,
     * the biClrUsed member specifies the size of the color table used to
     * optimize performance of the system color palettes. If biBitCount
     * equals 16 or 32, the optimal color palette starts immediately
     * following the free DWORD mask.
     *
     * When the bitmap array immediatelly follos the BITMAPINFO structure,
     * it is a packed bitmap. Packed bitmaps are referenced by a single
     * pointer. Packed bitmaps require that the biClrUsed member
     * must be either zero or the actual size of the color table.
     */
    uint32_t	biClrUsed;
    /**
     * Specifies the number of color indexes that are required for the displaying
     * the bitmap. If this value is zero, all colors are required.
     */
    uint32_t	biClrImportant;
} BITMAPINFOHEADER, *PBITMAPINFOHEADER, *LPBITMAPINFOHEADER;

typedef struct AVMPACKED
{
	BITMAPINFOHEADER bmiHeader;
	uint32_t bmiColors[1];
} BITMAPINFO, *LPBITMAPINFO;

#endif /* _BITMAPINFOHEADER */


#ifndef _WAVEFORMATEX_
#define _WAVEFORMATEX_
typedef struct AVMPACKED _WAVEFORMATEX
{
    /**
     * Waveform-audio format type. Format tags are registred with
     * Microsoft Corporation for many compression algorithms.
     */
    uint16_t	wFormatTag;
    /** Number of channels in the waveformat audio data. */
    uint16_t	nChannels;
    /**
     * Sample rate, in samples per seconds (Hertz), that each channel
     * should be played or recorded.
     */
    uint32_t	nSamplesPerSec;
    /**
     * Required average data-transfer rate, in bytes per second, for
     * format tag. For PCM it should be nAvgBytesPerSec * nBlockAlign
     */
    uint32_t	nAvgBytesPerSec;
    /**
     * Block alignment in bytes. The block alignment is the minimum
     * atomic using of the data for the wFormatTag format type.
     * For PCM it should be nChannels * wBitsPerPixel / 8.
     * For Mp3 this is 1 because of the Windows (there is no logic in this)
     */
    uint16_t	nBlockAlign;
    /**
     * Bits per sample for the wFormatTag format type. If wFormatTag
     * is WAVE_FORMAT_PCM then wBitsPerSample should be equal to 8 or 16.
     * Note that some compression schemes cannot define a value for
     * wBitsPerSample, so this member can be zero.
     * For WAVEFORMATEXTENSIBLE it must be byte-aligned
     */
    uint16_t	wBitsPerSample;
    /**
     * Size, in bytes, of extra format information appended to the end
     * of the WAVEFORMATEX structure. This information can be used by
     * non-PCM formats to store extra attributes for the wFormatTag.
     * If no extra information od required by the wFormatTag, this member
     * must be set to zero. For PCM format this member is ignored.
     */
    uint16_t	cbSize;
} WAVEFORMATEX;

typedef WAVEFORMATEX *PWAVEFORMATEX, *NPWAVEFORMATEX, *LPWAVEFORMATEX;
#endif /* !_WAVEFORMATEX_ */

#ifndef _WAVEFORMATEXTENSIBLE_
#define _WAVEFORMATEXTENSIBLE_
/**
 * The WAVEFORMATEXTENSIBLE structure defines the format of waveform-audio
 * data for formats having more than two channels or higher sample
 * resolutions than allowed by WAVEFORMATEX. It can also be used to
 * define any format that can be defined by WAVEFORMATEX.
 *
 * cbSize member must be at least 22
 */
typedef struct AVMPACKED
{
    WAVEFORMATEX Format;
    union {
	uint16_t wValidBitsPerSample;	///< bits of precision
	uint16_t wSamplesPerBlock;	///< valid if wBitsPerSample==0
	int16_t wReserved;		///< If neither applies, set to zero
    } Samples;
    int32_t dwChannelMask;		///< which channels are present in stream
#define SPEAKER_FRONT_LEFT		0x1
#define SPEAKER_FRONT_RIGHT		0x2
#define SPEAKER_FRONT_CENTER		0x4
#define SPEAKER_LOW_FREQUENCY		0x8
#define SPEAKER_BACK_LEFT		0x10
#define SPEAKER_BACK_RIGHT		0x20
#define SPEAKER_FRONT_LEFT_OF_CENTER	0x40
#define SPEAKER_FRONT_RIGHT_OF_CENTER	0x80
#define SPEAKER_BACK_CENTER		0x100
#define SPEAKER_SIDE_LEFT		0x200
#define SPEAKER_SIDE_RIGHT		0x400
#define SPEAKER_TOP_CENTER		0x800
#define SPEAKER_TOP_FRONT_LEFT		0x1000
#define SPEAKER_TOP_FRONT_CENTER	0x2000
#define SPEAKER_TOP_FRONT_RIGHT		0x4000
#define SPEAKER_TOP_BACK_LEFT		0x8000
#define SPEAKER_TOP_BACK_CENTER		0x10000
#define SPEAKER_TOP_BACK_RIGHT		0x20000

    GUID SubFormat;			///< similar to wFormatTag number
} WAVEFORMATEXTENSIBLE, *PWAVEFORMATEXTENSIBLE;
#endif /* !_WAVEFORMATEXTENSIBLE_ */

typedef WAVEFORMATEXTENSIBLE	WAVEFORMATPCMEX; /* Format.cbSize = 22 */

/**
 * WAVE format with extended structure which describes mp3 track
 */
typedef struct AVMPACKED
{
    WAVEFORMATEX wfx;
    uint16_t wID;			///< Set this structure member to MPEGLAYER3_ID_MPEG
#define MPEGLAYER3_ID_UNKNOWN		0
#define MPEGLAYER3_ID_MPEG		1
#define MPEGLAYER3_ID_CONSTANTFRAMESIZE 2

    int32_t fdwFlags;			///< Indicates whether padding is used to adjust the average bitrate to the sampling rate
#define MPEGLAYER3_FLAG_PADDING_ISO 0	///< Insert padding as needed to achieve the stated average bitrate.
#define MPEGLAYER3_FLAG_PADDING_ON  1	///< Always insert padding. The average bit rate may be higher than stated.
#define MPEGLAYER3_FLAG_PADDING_OFF 2	///< Never insert padding. The average bit rate may be lower than stated.

    uint16_t nBlockSize;		///< Block size in bytes. This value equals the frame length in bytes x nFramesPerBlock. For MP3 audio, the frame length is calculated as follows: 144 x (bitrate / sample rate) + padding
    uint16_t nFramesPerBlock;		///< Number of audio frames per block
    uint16_t nCodecDelay;		///< Encoder delay in samples. If you do not know this value, set this structure member to zero
} MPEGLAYER3WAVEFORMAT;

typedef struct AVMPACKED
{
    WAVEFORMATEXTENSIBLE wfex;
    uint32_t nInitSize;      		///< Size of initial ident header
    uint32_t nCommentSize;		///< Size of comment header
    uint32_t nCodebookSize;		///< Size of codebook header
} VORBISWAVEFORMATEX;

#ifndef WAVE_FORMAT_VORBIS1

#define WAVE_FORMAT_VORBIS1	0x674f	///< ("Og") ... Original stream compatible
#define WAVE_FORMAT_VORBIS2	0x6750	///< ("Pg") ... Have independent header
#define WAVE_FORMAT_VORBIS3	0x6751	///< ("Qg") ... Have no codebook header
#define WAVE_FORMAT_VORBIS1P	0x676f	///< ("og") ... Original stream compatible
#define WAVE_FORMAT_VORBIS2P	0x6770	///< ("pg") ... Have independent header
#define WAVE_FORMAT_VORBIS3P	('q' | ('g'<<8)) ///< 0x6771("qg") ... Have no codebook header

typedef struct AVMPACKED
{
    WAVEFORMATEX wfx;
    uint32_t dwVorbisACMVersion;	///< vorbis.acm
    uint32_t dwLibVorbisVersion;	///< libVorbis
    uint8_t byExtra[0];
} OGGWAVEWAVEFORMAT;

#endif /* WAVE_FORMAT_VORBIS1 */

/**
 * this extended WAVEFORMAT struct is used to pass around info about codecs currently unknown to other
 * parts of avifile (and usually to any Microsoft specs)
 * So this format will NOT be playable by anything else except avifile!!!
 */
typedef struct AVMPACKED
{
    WAVEFORMATEXTENSIBLE wfex;
    uint32_t dwCodecID;			///< CodecID from ffmpeg's audio codec
} FFMPEGWAVEFORMATEX;

/* http://msdn.microsoft.com/en-us/library/ms867195.aspx */
/** WAVE form wFormatTag IDs */

#define WAVE_FORMAT_UNKNOWN	0x0000	///< Microsoft Corporation
#if !defined(WAVE_FORMAT_PCM)
#define  WAVE_FORMAT_PCM	0x0001	///< Microsoft Corporation: PCM audio in integer format
#endif /* !WAVE_FORMAT_PCM */
#define WAVE_FORMAT_ADPCM	0x0002	///< Microsoft Corporation: adaptive PCM
#define WAVE_FORMAT_IEEE_FLOAT	0x0003	///< Microsoft Corporation: PCM audio in IEEE floating-point format
#define WAVE_FORMAT_VSELP	0x0004	///< Compaq Computer Corp.: PCM audio in IEEE floating-point format
#define WAVE_FORMAT_IBM_CVSD	0x0005	///< IBM Corporation
#define WAVE_FORMAT_ALAW	0x0006	///< Microsoft Corporation
#define WAVE_FORMAT_MULAW	0x0007	///< Microsoft Corporation
#define WAVE_FORMAT_DTS		0x0008	///< Microsoft Corporation
#define WAVE_FORMAT_DRM		0x0009	///< Microsoft Corporation: Microsoft Digital Rights Managed encrypted audio
#define WAVE_FORMAT_VMPS1      	0x000A	///< Microsoft Corporation: Speech audio codec
#define WAVE_FORMAT_WMSP2	0x000B	///< Microsoft Corporation
#define WAVE_FORMAT_OKI_ADPCM	0x0010	///< OKI
#define WAVE_FORMAT_DVI_ADPCM	0x0011	///< Intel Corporation
#define WAVE_FORMAT_IMA_ADPCM (WAVE_FORMAT_DVI_ADPCM)
#define WAVE_FORMAT_MEDIASPACE_ADPCM 0x0012 ///< Videologic
#define WAVE_FORMAT_SIERRA_ADPCM 0x0013	///< Sierra Semiconductor Corp
#define WAVE_FORMAT_G723_ADPCM	0x0014	///< Antex Electronics Corporation
#define WAVE_FORMAT_DIGISTD	0x0015	///< DSP Solutions, Inc.
#define WAVE_FORMAT_DIGIFIX	0x0016	///< DSP Solutions, Inc.
#define WAVE_FORMAT_DIALOGIC_OKI_ADPCM 0x0017 ///< Dialogic Corporation: OKI ADPCM chips or firmware
#define WAVE_FORMAT_MEDIAVISION_ADPCM 0x0018 ///< Media Vision, Inc.: ADPCM for Jazz 16 chip set
#define WAVE_FORMAT_CU_CODEC	0x0019	///< Hewlett-Packard Company
#define WAVE_FORMAT_HP_DYNAMIC_VOICE 0x001A ///< Hewlett-Packard Company
#define WAVE_FORMAT_YAMAHA_ADPCM 0x0020	///< Yamaha Corporation of America
#define WAVE_FORMAT_SONARC	0x0021	///< Speech Compression
#define WAVE_FORMAT_DSPGROUP_TRUESPEECH 0x0022	///< DSP Group, Inc
#define WAVE_FORMAT_ECHOSC1	0x0023	///< Echo Speech Corporation
#define WAVE_FORMAT_AUDIOFILE_AF36 0x0024 ///< Ahead Inc.
#define WAVE_FORMAT_APTX	0x0025	///< Audio Processing Technology
#define WAVE_FORMAT_AUDIOFILE_AF10 0x0026 ///< Ahead, Inc.
#define WAVE_FORMAT_PROSODY_1612 0x0027	///< Aculab plc: Prosody CTI speech card
#define WAVE_FORMAT_LRC		0x0028	///< Merging Technologies S.A.
#define WAVE_FORMAT_DOLBY_AC2	0x0030	///< Dolby Laboratories
#define WAVE_FORMAT_GSM610	0x0031	///< Microsoft Corporation
#define WAVE_FORMAT_MSNAUDIO	0x0032	///< Microsoft Corporation: Microsoft MSN audio code
#define WAVE_FORMAT_ANTEX_ADPCME 0x0033	///< Antex Electronics Corporation
#define WAVE_FORMAT_CONTROL_RES_VQLPC 0x0034 ///< Control Resources Limited
#define WAVE_FORMAT_DIGIREAL	0x0035	///< DSP Solutions, Inc.
#define WAVE_FORMAT_DIGIADPCM	0x0036	///< DSP Solutions, Inc.
#define WAVE_FORMAT_CONTROL_RES_CR10 0x0037 ///< Control Resources Limited
#define WAVE_FORMAT_NMS_VBXADPCM 0x0038	///< Natural MicroSystems
#define WAVE_FORMAT_ROLAND_RDAC	0x0039	///< Roland Corporation: Roland RDAC proprietary format (IMA ADPCM)
#define WAVE_FORMAT_ECHOSC3 0x003A	///< Echo Speech Corporation: Proprietary compression format
#define WAVE_FORMAT_ROCKWELL_ADPCM 0x003B ///< Rockwell International: Rockwell ADPCM
#define WAVE_FORMAT_ROCKWELL_DIGITALK 0x003C ///< Rockwell International: Rockwell DIGITALK
#define WAVE_FORMAT_XEBEC	0x003D	///< Xebec Multimedia Solutions Ltd.: Proprietary compression format
#define WAVE_FORMAT_G721_ADPCM	0x0040	///< Antex Electronics Corporation
#define WAVE_FORMAT_G728_CELP	0x0041	///< Antex Electronics Corporation
#define WAVE_FORMAT_MSG723	0x0042	///< Microsoft Corporation
#define WAVE_FORMAT_MSG723_1	0x0043	///< Microsoft Corporation
#define WAVE_FORMAT_MSG729	0x0044	///< Microsoft Corporation
#define WAVE_FORMAT_SPG726	0x0045	///< Microsoft Corporation
#define WAVE_FORMAT_MPEG	0x0050	///< Microsoft Corporation
#define WAVE_FORMAT_RT24	0x0052	///< InSoft, Inc.
#define WAVE_FORMAT_PAC		0x0053	///< InSoft, Inc.
#define WAVE_FORMAT_MPEGLAYER3	0x0055	///< ISO/MPEG Layer3 format tag
#define WAVE_FORMAT_LUCENT_G723	0x0059	///< Lucent Technologies
#define WAVE_FORMAT_CIRRUS	0x0060	///< Cirrus Logic
#define WAVE_FORMAT_ESPCM	0x0061	///< ESS Technology
#define WAVE_FORMAT_VOXWARE	0x0062	///< Voxware Inc
#define WAVE_FORMAT_CANOPUS_ATRAC 0x0063 ///< Canopus, co., Ltd.: ATRACWAVEFORMAT
#define WAVE_FORMAT_G726_ADPCM	0x0064	///< APICOM
#define WAVE_FORMAT_G722_ADPCM	0x0065	///< APICOM
#define WAVE_FORMAT_DSAT_DISPLAY 0x0067	///< Microsoft Corporation
#define WAVE_FORMAT_VOXWARE_BYTE_ALIGNED 0x0069	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_AC8	0x0070	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_AC10 0x0071	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_AC16 0x0072	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_AC20 0x0073	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_RT24 0x0074	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_RT29 0x0075	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_RT29HW 0x0076 ///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_VR12 0x0077	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_VR18 0x0078	///< Voxware, Inc.
#define WAVE_FORMAT_VOXWARE_TQ40 0x0079	///< Voxware, Inc.
#define WAVE_FORMAT_SOFTSOUND	0x0080	///< Softsound, Ltd.
#define WAVE_FORMAT_VOXWARE_TQ60 0x0081	///< Voxware, Inc.
#define WAVE_FORMAT_MSRT24	0x0082	///< Microsoft Corporation
#define WAVE_FORMAT_G729A	0x0083	///< AT&T Labs, Inc.
#define WAVE_FORMAT_MVI_MVI2	0x0084	///< Motion Pixels
#define WAVE_FORMAT_DF_G726	0x0085	///< DataFusion Systems (Pty), Ltd.
#define WAVE_FORMAT_DF_GSM610	0x0086	///< DataFusion Systems (Pty), Ltd.
#define WAVE_FORMAT_ISIAUDIO	0x0088	///< Iterated Systems, Inc.
#define WAVE_FORMAT_ONLIVE	0x0089	///< OnLive! Technologies, Inc.
#define WAVE_FORMAT_MULTITUDE_FT_SX20 0x008A ///< Multitude, Inc.
#define WAVE_FORMAT_MM_INFOCOM_ITS_ACM_G721 0x008B ///< Infocom ITS A/S: G.721 ADPCM
#define WAVE_FORMAT_CONVEDIA_G729 0x008C ///< Convedia Corporation
#define WAVE_FORMAT_CONGRUENCY_AUDIO_CODEC 0x008D ///< Congruency, Inc.
#define WAVE_FORMAT_SBC24	0x0091	///< Siemens Business Communications Sys
#define WAVE_FORMAT_DOLBY_AC3_SPDIF 0x0092 ///< Sonic Foundry
#define WAVE_FORMAT_MEDIASONIC_G723 0x0093 ///< MediaSonic
#define WAVE_FORMAT_PROSODY_8KBPS 0x0094 ///< Aculab plc: Prosody CTI speech card
#define WAVE_FORMAT_ZYXEL_ADPCM	0x0097	///< ZyXEL Communications, Inc.: ZyXEL ADPCM
#define WAVE_FORMAT_PHILIPS_LPCBB 0x0098 ///< Philips Speech Processing
#define WAVE_FORMAT_PACKED	0x0099	///< Studer Professional Audio AG
#define WAVE_FORMAT_MALDEN_PHONYTALK 0x00A0 ///< Malden Electronics Ltd.
#define WAVE_FORMAT_RRLGSM	0x00A1	///< Racal Recorders: Racal Recorder GSM
#define WAVE_FORMAT_RRLG729A	0x00A3	///< Racal Recorders: Racal Recorder G720.a
#define WAVE_FORMAT_RRLG7231	0x00A3	///< Racal Recorders: Racal G723.1
#define WAVE_FORMAT_RRLTETRA_ACLEP 0x00A4 ///< Racal Recorders: Racal Tetra ACELP
#define WAVE_FORMAT_NEC_AAC	0x00B0	///< NEC Corporation: NEC AAC
#define WAVE_FORMAT_RHETOREX_ADPCM 0x0100 ///< Rhetorex Inc.: Rhetorex ADPCM wave format type
#define WAVE_FORMAT_IRAT	0x0101	///< BeCubed Software, Inc.
#define WAVE_FORMAT_VIVO_G723	0x0111	///< Vivo Software
#define WAVE_FORMAT_VIVO_SIREN	0x0112	///< Vivo Software
#define WAVE_FORMAT_PHILIPS_CELP 0x0120	///< Philips Speech Processing
#define WAVE_FORMAT_PHILIPS_GRUNDIG 0x0121 ///< Philips Speech Processing
#define WAVE_FORMAT_DIGITAL_G723 0x0123	///< Digital Equipment Corporation
#define WAVE_FORMAT_SANYO_LD_ADPCM 0x0125 ///< Sanyo Electric Co., Ltd.: Sanyo LD-ADPCM wave type
#define WAVE_FORMAT_SIPROLAB_ACEPLNET 0x0130 ///< Sipro Lab Telecom, Inc.
#define WAVE_FORMAT_SIPROLAB_ACELP4800 0x0131 ///< Sipro Lab Telecom, Inc.
#define WAVE_FORMAT_SIPROLAB_ACELP8V3 0x0132 ///< Sipro Lab Telecom, Inc.
#define WAVE_FORMAT_SIPROLAB_G729 0x0133 ///< Sipro Lab Telecom, Inc.
#define WAVE_FORMAT_SIPROLAB_G729A 0x0134 ///< Sipro Lab Telecom, Inc.
#define WAVE_FORMAT_SIPROLAB_KELVIN 0x0135 ///< Sipro Lab Telecom, Inc.
#define WAVE_FORMAT_G726ADPCM	0x0140	///< Dictaphone Corporation
#define WAVE_FORMAT_CELP68	0x0141	///< Dictaphone Corporation
#define WAVE_FORMAT_CELP54	0x0142	///< Dictaphone Corporation
#define WAVE_FORMAT_QUALCOMM_PUREVOICE 0x0150 ///< Qualcomm, Inc.
#define WAVE_FORMAT_QUALCOMM_HALFRATE 0x0151 ///< Qualcomm, Inc.
#define WAVE_FORMAT_TUBGSM	0x0155	///< Ring Zero Systems, Inc.: Related to GSM 6.10
#define WAVE_FORMAT_MSAUDIO1	0x0160	///< Microsoft Corporation
#define WAVE_FORMAT_WMAUDIO2	0x0161	///< Microsoft Corporation: Multichannel WMA
#define WAVE_FORMAT_MSAUDIO2 (WAVE_FORMAT_WMAUDIO2)
#define WAVE_FORMAT_WMAUDIO3	0x0162	///< Microsoft Corporation
#define WAVE_FORMAT_WMAUDIO_LOSSLESS 0x0163 ///< Microsoft Corporation: WMA lossless
#define WAVE_FORMAT_WMASPDIF	0x0164	///< Microsoft Corporation: WMA Pro over S/PDIF
#define WAVE_FORMAT_UNISYS_NAP_ADPCM 0x0170 ///< Unisys Corporation
#define WAVE_FORMAT_UNISYS_NAP_ULAW 0x0171 ///< Unisys Corporation
#define WAVE_FORMAT_UNISYS_NAP_ALAW 0x0172 ///< Unisys Corporation
#define WAVE_FORMAT_UNISYS_NAP_16K 0x0173 ///< Unisys Corporation
#define WAVE_FORMAT_MM_SYCOM_ACM_SYC008 0x0174 ///< SyCom Technologies
#define WAVE_FORMAT_MM_SYCOM_ACM_SYC701_G726L 0x0175 ///< SyCom Technologies
#define WAVE_FORMAT_MM_SYCOM_ACM_SYC701_CELP54 0x0176 ///< SyCom Technologies
#define WAVE_FORMAT_MM_SYCOM_ACM_SYC701_CELP68 0x0177 ///< SyCom Technologies
#define WAVE_FORMAT_KNOWLEDGE_ADVENTURE_ADPCM 0x0177 ///< Knowledge Adventure, Inc.
#define WAVE_FORMAT_MPEG2AAC	0x0180	///< Fraunhofer IIS
#define WAVE_FORMAT_DTS_DS	0x0190	///< Digital Theater Systems, Inc.
#define WAVE_FORMAT_INNINGS_ADPCM 0x0197 ///< Innings Telecom Inc.
#define WAVE_FORMAT_CREATIVE_ADPCM 0x0200 ///< Creative Labs, Inc.
#define WAVE_FORMAT_CREATIVE_FASTSPEECH8 0x0202	///< Creative Labs, Inc.
#define WAVE_FORMAT_CREATIVE_FASTSPEECH10 0x0203 ///< Creative Labs, Inc.
#define WAVE_FORMAT_UHER_ADPCM	0x0210	///< UHER informatic GmbH
#define WAVE_FORMAT_QUARTERDECK	0x0220	///< Quarterdeck Corporation
#define WAVE_FORMAT_ILINK_VC	0x0230	///< I-Link Worldwide
#define WAVE_FORMAT_RAW_SPORT	0x0240	///< Aureal Semiconductor
#define WAVE_FORMAT_IPI_HSX	0x0250	///< Interactive Products, Inc.
#define WAVE_FORMAT_IPI_RPELP	0x0251	///< Interactive Products, Inc.
#define WAVE_FORMAT_CS2		0x0260	///< Consistent Software
#define WAVE_FORMAT_SONY_SCX	0x0270	///< Sony Corporation
#define WAVE_FORMAT_SONY_SCY	0x0271	///< Sony Corporation
#define WAVE_FORMAT_SONY_ATRAC3	0x0272	///< Sony Corporation
#define WAVE_FORMAT_SONY_SPC	0x0273	///< Sony Corporation
#define WAVE_FORMAT_TELUM	0x0280	///< Telum, Inc.
#define WAVE_FORMAT_TELUMIA	0x0281	///< Telum, Inc.
#define WAVE_FORMAT_NCVS_ADPCM	0x0285	///< Norcom Electronics Corporation: Norcom Voice Systems ADPCM
#define WAVE_FORMAT_FM_TOWNS_SND 0x0300	///< Fujitsu Corporation
// 0x0301 .. 0x0308 reserved for  Fujitsu Corporation
#define WAVE_FORMAT_DEVELOPMENT_MICRONAS 0x0350 ///< Micronas Semiconductors, Inc.
#define WAVE_FORMAT_CELP833	0x0351	///< Micronas Semiconductors, Inc.
#define WAVE_FORMAT_BTV_DIGITAL	0x0400	///< Brooktree Corporation: Brooktree digital audio format
#define WAVE_FORMAT_QDESIGN_MUSIC 0x0450 ///< QDesign Corporation
#define WAVE_FORMAT_VME_VMPCM	0x0680	///< AT&T Labs, Inc.
#define WAVE_FORMAT_TPC		0x0681	///< AT&T Labs, Inc.
#define WAVE_FORMAT_OLIGSM	0x1000	///< Ing C. Olivetti & C., S.p.A.
#define WAVE_FORMAT_OLIADPCM	0x1001	///< Ing C. Olivetti & C., S.p.A.
#define WAVE_FORMAT_OLICELP	0x1002	///< Ing C. Olivetti & C., S.p.A.
#define WAVE_FORMAT_OLISBC	0x1003	///< Ing C. Olivetti & C., S.p.A.
#define WAVE_FORMAT_OLIOPR	0x1004	///< Ing C. Olivetti & C., S.p.A.
#define WAVE_FORMAT_LH_CODEC	0x1100	///< Lernout & Hauspie
#define WAVE_FORMAT_LH_CELP	0x1101	///< Lernout & Hauspie
#define WAVE_FORMAT_LH_SB8	0x1102	///< Lernout & Hauspie
#define WAVE_FORMAT_LH_SB12	0x1103	///< Lernout & Hauspie
#define WAVE_FORMAT_LH_SB16	0x1104	///< Lernout & Hauspie
#define WAVE_FORMAT_NORRIS	0x1400	///< Norris Communications, Inc.
#define WAVE_FORMAT_SOUNDSPACE_MUSICOMPRESS 0x1500 ///< AT&T Labs, Inc.
#define WAVE_FORMAT_SONICFOUNDRY_LOSSLESS 0x1971 ///< Sonic Foundry
#define WAVE_FORMAT_DVM		0x2000	///< FAST Multimedia AG
#define WAVE_FORMAT_MSCE	0x2500	///< Microsoft Corporation: reserved rangle to 0x2600
#define WAVE_FORMAT_DIVIO_AAC	0x4143	///< Divio, Inc.: Divio's AAC
#define WAVE_FORMAT_NOKIA_AMR	0x4201	///< Nokia Mobile Phones: Nokia adaptive multirate
#define WAVE_FORMAT_DIVIO_G726	0x4243	///< Divio, Inc.: Divio's G726
#define WAVE_FORMAT_3COM_NBX	0x7000	///< 3Com Corporation Not specified
#define WAVE_FORMAT_NTT_DOCOMO_AMR_NO_SID 0x7A21 ///< Microsoft Corporation Adaptive multirate
#define WAVE_FORMAT_NTT_DOCOMO_AMR_WITH_SID 0x7A22 ///< Microsoft Corporation: AMR with silence detection
#define WAVE_FORMAT_COMVERSEINFOSYS_G723_1 0xA100 ///< Comverse Infosys Ltd.
#define WAVE_FORMAT_COMVERSEINFOSYS_AVQSBC 0xA101 ///< Comverse Infosys Ltd.
#define WAVE_FORMAT_COMVERSEINFOSYS_OLDSBC 0xA102 ///< Comverse Infosys Ltd.
#define WAVE_FORMAT_SYMBOLTECH_G729A 0xA103 ///< Symbol Technologies Canada Symbol Technology's G729A
#define WAVE_FORMAT_VOICEAGE_AMR_WB 0xA104 ///< VoiceAge Corporation
#define WAVE_FORMAT_ITI_G726	0xA105	///< Ingenient Technologies, Inc.: Ingenient's G726
#define WAVE_FORMAT_AAC		0xA106	///< ISO/MPEG-4 advanced audio Coding
#define WAVE_FORMAT_ESLG726	0xA107	///< Encore Software Ltd.: Encore Software Ltd's G726


// WAVEFORMATIEEEFLOATEX KSDATAFORMAT_SUBTYPE_IEEE_FLOAT
// WAVEFORMATPCMEX KSDATAFORMAT_SUBTYPE_PCM

#if !defined(WAVE_FORMAT_EXTENSIBLE)
#define  WAVE_FORMAT_EXTENSIBLE 0xFFFE	///< Microsoft extensible
#endif /* !WAVE_FORMAT_EXTENSIBLE */

/**
 * the WAVE_FORMAT_DEVELOPMENT format tag can be used during the
 * development phase of a new wave format.  Before shipping, you MUST
 * acquire an official format tag from Microsoft.
 */
#define WAVE_FORMAT_DEVELOPMENT 0xFFFF

#endif /* !WIN32 */

/**
 * Image format tag:
 * For more details see: http://www.webartz.com/fourcc/fccyuv.htm
 */

/**
 * Basic Windows bitmap format. 1, 4 and 8 bpp versions are palettised.
 * 16, 24 and 32bpp contain raw BGR samples.
 */
#define BI_RGB		0
/** Run length encoded 8bpp BGR image. */
#define BI_RLE8		1
/** Alias for BI_RLE8 */
#define IMG_FMT_RLE8 0x38454C52
/** Run length encoded 4bpp BGR image. */
#define BI_RLE4		2
/** Alias for BI_RLE4 */
#define IMG_FMT_RLE 0x34454C52
/**
 * Raw BGR with alpha. Sample precision and packing is arbitrary and
 * determined using bit masks for each component, as for BI_BITFIELDS.
 */
#define IMG_FMT_RGBA 0x34454C52
/**
 * Raw BGR with a transparency field. Layout is as for BI_RGB
 * at 16 and 32 bits per pixel but the msb in each pixel
 * indicates whether the pixel is transparent or not.
 */
#define IMG_FMT_RGBT 0x34454C52
/**
 * Raw BGR with arbitrary sample packing within a pixel.
 * Packing and precision of R, G and B components is determined
 * by bit masks for each.
 * bps: 16, 24, 32
 */
#define BI_BITFIELDS	3
/** Alias for BI_RLE4 */
#define IMG_FMT_RAW 0x32776173



/* RGB/BGR Formats */

#define IMG_FMT_RGB	(('R'<<24)|('G'<<16)|('B'<<8))
#define IMG_FMT_RGB8	(IMG_FMT_RGB|8)
#define IMG_FMT_RGB15	(IMG_FMT_RGB|15)
#define IMG_FMT_RGB16	(IMG_FMT_RGB|16)
#define IMG_FMT_RGB24	(IMG_FMT_RGB|24)
#define IMG_FMT_RGB32	(IMG_FMT_RGB|32)

#define IMG_FMT_BGR	BI_RGB // so normal BGR image has just number
#define IMG_FMT_BGR8	(IMG_FMT_BGR|8)
#define IMG_FMT_BGR15	(IMG_FMT_BGR|15)
#define IMG_FMT_BGR16	(IMG_FMT_BGR|16)
#define IMG_FMT_BGR24	(IMG_FMT_BGR|24)
#define IMG_FMT_BGR32	(IMG_FMT_BGR|32)

/**
 * Planar YUV Formats	 U (Cb)  V (Cr)
 */

/** Similar to YV12 but including a level of indirection. */
#define IMG_FMT_CLPL 0x4C504C43
/** 8 bit Y plane followed by 8 bit 4x4 subsampled V and U planes. */
#define IMG_FMT_YVU9 0x39555659
/** As YVU9 but an additional 4x4 subsampled plane is appended and
 *  constains delta information relative to the last frame - bpp is 9.
 *  Intel Intermediate YUV9 */
#define IMG_FMT_IF09 0x39304649
/** 8 bit Y plane followed 8 bit 2x2 subsampled U and V planes */
#define IMG_FMT_I420 0x30323449
/** Duplicate of I420. */
#define IMG_FMT_IYUV 0x56555949
/** 8 bit Y plane follwed 8 bit 2x2 subsampled V and U planes.
 *  similar to I420 except U, V planes swaped */
#define IMG_FMT_YV12 0x32315659
/** 8 bit Y plane followed 8 bit 4x4 subsampled U and V planes
 *  so there is 1/4 width and 1/4 height of Y pixels for each line
 *  this is UNREGISTERED extension used by avifile (YUV411P) */
#define IMG_FMT_I410 0x30313449
/** 8 bit Y plane followed 8 bit 4x1 subsampled U and V planes
 *  so there is 1/4 width of Y pixels for each line
 *  this is UNREGISTERED extension used by avifile (YUV411P) */
#define IMG_FMT_I411 0x31313449
/** 8 bit Y plane followed 8 bit 2x1 subsampled U and V planes
 *  this is UNREGISTERED extension used by avifile (YUV422P) */
#define IMG_FMT_I422 0x32323449
/** 8 bit Y U V planes
 *  this is UNREGISTERED extension used by avifile (YUV444P) */
#define IMG_FMT_I444 0x34343449
/** Single Y plane for monochrome images. */
#define IMG_FMT_Y800 0x30303859
/** Duplicate of Y800. */
#define IMG_FMT_Y8 0x20203859

/**
 * Packed YUV Formats
 */

/** Cirrus Logic format wit 4 pixels packed into uint32
 *  A form of YUV 4:1:1 with less than 8 bits per Y, U and V sample. */
#define IMG_FMT_CLJR 0x524A4C43
/** Interlaced version of UYVY (line order 0,2,4,... 1,3,5,...). */
#define IMG_FMT_IUYV 0x56595549
/** Interlaced version of Y41P (line order 0,2,4,... 1,3,5,...). */
#define IMG_FMT_IY41 0x31435949
/** 12 bit format used in Digital Camera IEEE 1394
 *  lsb: U0,Y0,Y1,V0,Y2,Y3 */
#define IMG_FMT_IYU1 0x31555949
/** YUV 4:2:2 as for UYVY but with different component ordering
 *  within the uint32 macropixel identical to YUNV & V422
 *  lsb: Y0,U0,Y1,V0; Y2,U2,Y3,V2; Y4,U4,Y5,V4; */
#define IMG_FMT_YUY2 0x32595559
/** Duplicate of YUY2. */
#define IMG_FMT_YUYV 0x49595559
/** Duplicate of YUY2. */
#define IMG_FMT_V422 0x32323456
/** Duplicate of YUY2. */
#define IMG_FMT_YUNV 0x564E5559
/** 24 bit format
 *  YUY 4:4:4  lsb: U0,Y0,V0; U1,Y1,V1; */
#define IMG_FMT_YUV  0x20565559
/** duplicate of YUV used in Digital Camera IEEE 1394. */
#define IMG_FMT_IYU2 0x32555949
/** YUV 4:2:2 (Y sample at every pixel, U and V sampled every second
 *  pixel horizontaly on each line). A macropixel constains 2 pixels
 *  in one uint32 - also known as UYNV and UYVY
 *  lsb: U0,Y0,V0,Y1; U2,Y2,V2,Y3; U4,Y4,V4,Y5 */
#define IMG_FMT_Y422 0x32323459
/** Duplicate of Y422. */
#define IMG_FMT_UYVY 0x59565955
/** Duplicate of Y422. */
#define IMG_FMT_CPLA 0x414C5543
/** Duplicate of UYVY - used by some Nvidia */
#define IMG_FMT_UYNV 0x564E5955
/** Format as UYVY but lsb of each Y component is used
 *  to signal pixel transparency. */
#define IMG_FMT_Y42T 0x54323459
/** YCbCr 4:2:2 extended precision 10bits per component - lsb: U0,Y0,V0,Y1 */
#define IMG_FMT_UYVP 0x50565955
/** YCbCr 4:2:2 extended precision 10-bits per component in Y0U0Y1V0 order. */
#define IMG_FMT_YUVP 0x50565559
/** 16 but YUV 4:2:2 - registered by Vitec Multimedia. */
#define IMG_FMT_V655 0x35353656
/** Packed YUV format with Y sampled at every second pixel across each
 *  line and U and V sampled at every fourth pixel. */
#define IMG_FMT_Y211 0x31313259
/** YUV 4:1:1 similar to Y41P
 *  within the 3x uint32 macropixels there are 8 image pixels
 *  lsb: U0,Y0,V1,Y1; U4,Y2,V4,Y3; Y4,Y5,Y6,Y7; */
#define IMG_FMT_Y411 0x31315559
/** Y41P is basically the same as IY41 with the exception that
 *  the data is interlaced. Lines are ordered 0,2,4,....,1,3,5....
 *  instead of 0,1,2,3,4,5,.... */
#define IMG_FMT_Y41P 0x50313459
/** Identical to Y41P except that the least significant bit of each Y
 *  component forms a chromakey channel. (1 means show pixel) */
#define IMG_FMT_Y41T 0x54313459
/** Reodered YUY2  lsb: Y0,V0,Y1,U0; Y2,V2,Y3,U2; Y4,V4,Y5,U5 */
#define IMG_FMT_YVYU 0x55595659
/** Similar to UYVY - but the image is fliped vertically
 *  also is made from lowercase letters ! */
#define IMG_FMT_cyuv 0x76757963

/** Compressed Formats */
#define IMG_FMT_MPEGPES (('M'<<24)|('P'<<16)|('E'<<8)|('S'))

#endif // AVIFILE_FORMATS_H
