#!/bin/bash
#
# SPDX-License-Identifier: GPL-2.0-only
#
# Copyright (C) 2019 Philippe Proulx <pproulx@efficios.com>
# Copyright (C) 2019 Francis Deslauriers <francis.deslauriers@efficios.com>
#

# This test validates that a `src.ctf.fs` component successfully reads
# specific CTF traces and creates the expected messages.
#
# Such CTF traces to open either exist (in `tests/ctf-traces/1/succeed`)
# or are generated by this test using local trace generators.

SH_TAP=1

if [ -n "${BT_TESTS_SRCDIR:-}" ]; then
	UTILSSH="$BT_TESTS_SRCDIR/utils/utils.sh"
else
	UTILSSH="$(dirname "$0")/../../utils/utils.sh"
fi

# shellcheck source=../../utils/utils.sh
source "$UTILSSH"

this_dir_relative="plugins/flt.lttng-utils.debug-info"
succeed_trace_dir="$BT_CTF_TRACES_PATH/1/succeed"
expect_dir="$BT_TESTS_DATADIR/$this_dir_relative"
binary_artefact_dir="$BT_TESTS_DATADIR/$this_dir_relative"

test_debug_info() {
	local name="$1"
	local local_args=(
		"-c" "flt.lttng-utils.debug-info"
		"-p" "target-prefix=\"$binary_artefact_dir/x86-64-linux-gnu/dwarf-full\""
		"-c" "sink.text.details"
		"-p" "with-trace-name=no,with-stream-name=no"
	)

	for mip_version in 0 1; do
		bt_test_cli "MIP '$mip_version': Trace '$name' gives the expected output" \
			--expect-stdout "$expect_dir/trace-$name-mip$mip_version.expect" -- \
			"$succeed_trace_dir/$name" "${local_args[@]}" \
			"--allowed-mip-versions=$mip_version"
	done
}

test_compare_to_ctf_fs() {
	# Compare the `sink.text.details` output of a graph with and without a
	# `flt.lttng-utils.debug-info` component. Both should be identical for
	# traces without LTTng debugging fields.
	local mip_version=$1
	local test_name=$2
	shift 2
	local cli_args=("$@")
	local debug_info_cli_args=("-c" "flt.lttng-utils.debug-info")
	local details_cli_args=(
		"-c" "sink.text.details"
		"--params" "with-trace-name=false,with-stream-name=false,with-uuid=false,with-uid=false"
	)
	local actual_stdout
	local actual_stderr
	local expected_stdout
	local expected_stderr
	local ret=0

	actual_stdout=$(mktemp -t test-debug-info-stdout-actual.XXXXXX)
	actual_stderr=$(mktemp -t test-debug-info-stderr-actual.XXXXXX)
	expected_stdout=$(mktemp -t test-debug-info-stdout-expected.XXXXXX)
	expected_stderr=$(mktemp -t test-debug-info-stderr-expected.XXXXXX)

	# Create expected files using a graph without a `debug-info` component.
	bt_cli \
		--stdout-file "$expected_stdout" \
		--stderr-file "$expected_stderr" \
		-- \
		"${cli_args[@]}" \
		--allowed-mip-versions="$mip_version" \
		"${details_cli_args[@]}"

	# Read the same trace with a `debug-info` component in the graph.
	bt_cli \
		--stdout-file "$actual_stdout" \
		--stderr-file "$actual_stderr" \
		-- \
		"${cli_args[@]}" \
		--allowed-mip-versions="$mip_version" \
		"${details_cli_args[@]}" \
		"${debug_info_cli_args[@]}"

	bt_diff "$expected_stdout" "$actual_stdout"
	ok $? "MIP '$mip_version': Input '$test_name' gives the expected stdout"

	bt_diff "$expected_stderr" "$actual_stderr"
	ok $? "MIP '$mip_version': Input '$test_name' gives the expected stderr"

	rm -f "$actual_stdout"
	rm -f "$actual_stderr"
	rm -f "$expected_stdout"
	rm -f "$expected_stderr"
}

test_compare_ctf_src_trace() {
	local trace_name=$1
	local trace_path="$succeed_trace_dir/$trace_name"
	local cli_args=("$trace_path")

	diag "Comparing output with and without 'flt.lttng-utils.debug-info' on '$trace_name'"

	for mip_version in 0 1; do
		test_compare_to_ctf_fs "$mip_version" "src.ctf.fs with $trace_name trace" \
			"${cli_args[@]}"
	done
}

test_compare_complete_src_trace() {

	local source_name="src.trace-ir-test.AllFields"

	for mip_version in 0 1; do
		local cli_args=("--component=$source_name" "--allowed-mip-versions=$mip_version")
		test_compare_to_ctf_fs "$mip_version" "$source_name" "${cli_args[@]}"
	done
}

plan_tests 22

test_debug_info debug-info

test_compare_ctf_src_trace smalltrace
test_compare_ctf_src_trace 2packets
test_compare_ctf_src_trace session-rotation

test_compare_complete_src_trace
