/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "gbemol-app.h"
#include "gbemol-mpd.h"
#include "gbemol-playlist.h"
#include "gbemol-utils.h"
#include "gbemol-library.h"
#include "gbemol-info.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <unistd.h>

static void on_mpd_state_changed_cb (GbemolMpd* mpd, GbemolApp* app);
static void on_mpd_song_changed_cb (GbemolMpd* mpd, GbemolApp* app);
static void on_mpd_refresh_cb (GbemolMpd* mpd, GbemolApp* app);
static void on_mpd_playlist_changed_cb (GbemolMpd *mpd, GbemolApp* app);
static void on_mpd_error_cb (GbemolMpd* mpd, gint error, gint error_at, gchar* error_str, GbemolApp* app);

int main (int argc, char *argv[])	
{
	GbemolApp* app;
	char opt;

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gtk_set_locale ();
	g_thread_init(NULL);
	gdk_threads_init();
	
	gtk_init (&argc, &argv);
	
	/* Remove horizontal separator from tree view */
	gtk_rc_parse_string ("style \"gbemol\" { GtkTreeView::horizontal_separator = 0 \
			GtkButton::inner-border = {0, 0, 0, 0} } \
			widget \"*.*.tv_songs\" style \"gbemol\" ");
	
	app = gbemol_app_new ();

	/* MPD Callbacks */
	g_signal_connect (G_OBJECT (app->mpd), "state-changed", G_CALLBACK (on_mpd_state_changed_cb), app);
	g_signal_connect (G_OBJECT (app->mpd), "song-changed", G_CALLBACK (on_mpd_song_changed_cb), app);
	g_signal_connect (G_OBJECT (app->mpd), "refresh", G_CALLBACK (on_mpd_refresh_cb), app);
	g_signal_connect (G_OBJECT (app->mpd), "playlist-changed", G_CALLBACK (on_mpd_playlist_changed_cb), app);
	g_signal_connect (G_OBJECT (app->mpd), "error", G_CALLBACK (on_mpd_error_cb), app);

	gbemol_app_connect_mpd (app);

	if (!gbemol_cfg_get_int ("win_main", "visible"))
		gbemol_app_show (app);

	/* Parse command-line options */
	while ((opt = getopt(argc, argv, "mclpfh")) != -1)
	{
		switch (opt)
		{
			/* Start minimized to tray */
			case 'm':
				gbemol_app_hide (app);
				break;
			/* Start on full mode */
			case 'f':
				gbemol_app_full_mode (app);
				break;
			/* Start on compact mode */
			case 'c':
				gbemol_app_compact_mode (app);
				break;
			/* Start with library selected */
			case 'l':
				gtk_notebook_set_current_page (GTK_NOTEBOOK (app->notebook), 1);
				break;
			/* Start with playlist selected */
			case 'p':
				gtk_notebook_set_current_page (GTK_NOTEBOOK (app->notebook), 0);
				break;
			/* Help message */
			case 'h':
				g_print ("Usage %s -mclphf\n", argv[0]);
				return 0;
				break;
			default:
				break;
		}
	}

	/* Update library on start */
	if (gbemol_mpd_is_connected (app->mpd) && gbemol_cfg_get_int ("MPD", "update-library"))
		gbemol_mpd_database_update (app->mpd, "/");

	gtk_main();

	return 0;
}

static void
on_mpd_state_changed_cb (GbemolMpd* mpd, GbemolApp* app)
{
	if (!gbemol_mpd_is_connected (mpd)  || !GBEMOL_IS_MPD (mpd) || !GBEMOL_IS_APP (app)  || (mpd->status == NULL))
		return;

	gbemol_app_set_state (app);
	on_mpd_song_changed_cb (mpd, app);
}

static void 
on_mpd_song_changed_cb (GbemolMpd* mpd, GbemolApp* app)
{	
	GbemolMpdSong* song;

	if (!gbemol_mpd_is_connected (mpd) || !GBEMOL_IS_MPD (mpd) || !GBEMOL_IS_APP (app))
		return;

	if (!mpd->status)
		return;
	
	/* The playlist is empty */
	if ((mpd->status->playlistLength < 0) || (mpd->status->state == MPD_STATUS_STATE_STOP))
		gbemol_app_set_song (app, NULL);
	else
	{	
		/* Set the song title */
		song = gbemol_mpd_get_current_song (mpd);
		if (song >= 0)
		{
			gbemol_app_set_song (app, song);	
			switch (gtk_notebook_get_current_page (GTK_NOTEBOOK (app->notebook)))
			{
				case 0:
					gbemol_playlist_bold_song (app->pls, song->id);
					break;
				case 1:
					break;
				case 2:
					break;

			}
		}
		gbemol_mpd_free_song (song);
	}

}

void
on_mpd_refresh_cb (GbemolMpd* mpd, GbemolApp* app)
{
	if (!gbemol_mpd_check_permission (mpd, "status")  || !GBEMOL_IS_MPD(mpd) || !GBEMOL_IS_APP(app))
		return;

	if (mpd->status == NULL)
		return;
	
	/* Update random/repeat */
	gbemol_app_set_random (app, mpd->status->random);
	gbemol_app_set_repeat (app, mpd->status->repeat);	

	/* Update the time */
	if (mpd->status->state == MPD_STATUS_STATE_PLAY)
		gbemol_app_set_time (app, mpd->status->elapsedTime, mpd->status->totalTime);

	/* Update volume */
	gbemol_app_set_volume (app, mpd->status->volume);

	/* Is it updating the DB? */
	gbemol_library_set_updating (app->lib, mpd->status->updatingDb);
}

static void
on_mpd_playlist_changed_cb (GbemolMpd* mpd, GbemolApp* app)
{
	if (!gbemol_mpd_is_connected (mpd) || (mpd->status == NULL))
		return;

	if (gbemol_playlist_current_is_selected (app->pls))
	{
		gbemol_playlist_update_songs (app->pls);
			if (app->mpd->status != NULL && app->mpd->status->songid != -1)
				gbemol_playlist_bold_song (app->pls, app->mpd->status->songid);
	}
}

static void
on_mpd_error_cb (GbemolMpd* mpd, gint error, gint error_at, gchar* error_str, GbemolApp* app)
{
	g_message ("Error");
	switch (error)
	{
		case MPD_ERROR_CONNCLOSED:
			gbemol_app_reconnect_dialog_new (app, error_str);
			break;
		case MPD_ERROR_TIMEOUT:
			gbemol_app_reconnect_dialog_new (app, error_str);
			break;
		case MPD_ERROR_ACK:
			switch (error_at) 
			{
				case MPD_ACK_ERROR_PASSWORD:
					gbemol_app_error_dialog_new (app, _("Error"), _("Wrong password"));
					gbemol_app_set_connected (app, FALSE);
				break;
				case MPD_ACK_ERROR_PERMISSION:
					gbemol_app_error_dialog_new (app, _("Permission Error"),
						       	_("You don't have permission to execute that action"));
				break;
			}
			break;
		case MPD_ERROR_UNKHOST:
			gbemol_app_reconnect_dialog_new (app, error_str);
			break;
		default:
			gbemol_app_error_dialog_new (app, _("MPD Error"), error_str);
			break;

	}
}


