/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gbemol-preferences.h"
#include "gbemol-utils.h"

struct _GbemolPreferencesPrivate {
	GtkListStore *lst_outputs;

	GtkWidget *notebook;

	/* MPD */
	GtkWidget *ent_host;
	GtkWidget *spn_port;
	GtkWidget *ent_pass;
	GtkWidget *chk_auth;
	GtkWidget *spn_timeout;
	GtkWidget *lbl_mpd;
	GtkWidget *ent_dir;
	GtkWidget *chk_update;

	/* Audio */
	GtkWidget *spn_fade;
	GtkWidget *tvw_outputs;

	/* Interface */
	GtkWidget *chk_playlist_sensitive;
	GtkWidget *chk_library_sensitive;
	GtkWidget *rad_playlist_mode[N_PLAYLIST_MODES];
	GtkWidget *chk_use_status_icon;
	GtkWidget *chk_close_tray;
	GtkWidget *chk_stick;
	GtkWidget *chk_top;
	
	/* Cover Art */
	GtkWidget *chk_cover_art_amazon;
	GtkWidget *chk_cover_art_file;

	GtkTooltips *tips;
};

enum {
	COLUMN_OUTPUT_ID,
	COLUMN_OUTPUT_ACTIVE,
	COLUMN_OUTPUT_NAME,
	N_COLUMNS_OUTPUT
};

static void gbemol_preferences_init (GbemolPreferences *obj);
static void gbemol_preferences_class_init (GObjectClass* g_class);
static void gbemol_preferences_finalize (GObject *obj);

static void gbemol_preferences_populate_outputs (GbemolPreferences* obj);
static void gbemol_preferences_populate (GbemolPreferences *obj);

/* Callbacks */
static void on_renderer_active_toggled (GtkCellRendererToggle *renderer, gchar *path, GbemolPreferences* obj);
static void on_btn_cancel_clicked (GtkButton *b, GbemolPreferences* obj);
static void on_btn_save_clicked (GtkButton *b, GbemolPreferences* obj);
static void on_btn_save_reconnect_clicked (GtkButton *b, GbemolPreferences* obj);
static void on_chk_auth_toggled (GtkToggleButton *b, GbemolPreferences* obj);
static void on_spn_fade_value_changed (GtkSpinButton *b, GbemolPreferences* obj);
static void on_btn_dir_activate (GtkButton *b, GbemolPreferences* obj);
static void on_rad_playlist_mode_toggled (GtkToggleButton *btn, GbemolPreferences *obj);
static void on_chk_use_status_icon_toggled (GtkToggleButton *btn, GbemolPreferences *obj);

GType
gbemol_preferences_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (GbemolPreferencesClass),
			NULL,   /* base_init */
			NULL,   /* base_finalize */
			(GClassInitFunc)gbemol_preferences_class_init,   /* class_init */
			NULL,   /* class_finalize */
			NULL,   /* class_data */
			sizeof (GbemolPreferences),
			0,      /* n_preallocs */
			(GInstanceInitFunc)gbemol_preferences_init    /* instance_init */
		};

		type = g_type_register_static (G_TYPE_OBJECT,
				"GbemolPreferences",
				&info, 0);
	}
	return type;
}


static void
gbemol_preferences_init (GbemolPreferences *obj)               
{
	obj->priv = g_new0 (GbemolPreferencesPrivate, 1);
	obj->priv->tips = gtk_tooltips_new ();

	obj->app = NULL;

}

static void
gbemol_preferences_class_init (GObjectClass* g_class)
{
	g_class->finalize = gbemol_preferences_finalize;
}

static void
gbemol_preferences_finalize (GObject *obj)
{
	g_free (GBEMOL_PREFERENCES (obj)->priv);
}

GbemolPreferences*
gbemol_preferences_new (GbemolApp* app)
{
	GbemolPreferences* obj;
	GtkTreeView* tvw;
	GtkCellRenderer* renderer;
	GtkTreeViewColumn* column;
	GtkTreeSelection *select;
	GtkWidget *vbox1, *vbox2, *vbox3, *hbox1, *table, *lbl, *scr, *btn, *frame;
	gint i;

	obj = GBEMOL_PREFERENCES (g_object_new (GBEMOL_TYPE_PREFERENCES, NULL));
	obj->app = app;

		obj->win_main = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (obj->win_main), _("gbemol - Preferences"));
	gtk_window_set_position (GTK_WINDOW (obj->win_main), GTK_WIN_POS_CENTER);
	gtk_container_set_border_width (GTK_CONTAINER (obj->win_main), 5);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (obj->win_main), vbox1);

	obj->priv->notebook = gtk_notebook_new ();
	gtk_box_pack_start (GTK_BOX (vbox1), obj->priv->notebook, TRUE, TRUE, 5);

	/*** MPD Tab ***/
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 20);
	lbl = gtk_label_new (_("mpd"));
	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook), vbox2, lbl);

	obj->priv->lbl_mpd = gtk_label_new ("mpd");
	gtk_misc_set_alignment (GTK_MISC (obj->priv->lbl_mpd), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox2), obj->priv->lbl_mpd, FALSE, FALSE, 0);
	
	table = gtk_table_new (7, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 5);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);
	gtk_box_pack_start (GTK_BOX (vbox2), table, TRUE, TRUE, 5);

	/* Host Entry */
	lbl = gtk_label_new (_("Host:"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), lbl, 0, 1, 0, 1);

	obj->priv->ent_host = gtk_entry_new ();
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->ent_host, _("Enter MPD Host address"), NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), obj->priv->ent_host, 1, 2, 0, 1);

	/* Port */
	lbl = gtk_label_new (_("Port:"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), lbl, 0, 1, 1, 2);

	obj->priv->spn_port = gtk_spin_button_new_with_range (1.0, 65535.0, 1.0);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->spn_port, _("MPD connection port"), NULL);
	gtk_spin_button_set_digits (GTK_SPIN_BUTTON (obj->priv->spn_port), 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (obj->priv->spn_port), 6600.0);
	gtk_table_attach_defaults (GTK_TABLE (table), obj->priv->spn_port, 1, 2, 1, 2);

	/* Pass */
	lbl = gtk_label_new (_("Pass:"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), lbl, 0, 1, 2, 3);

	obj->priv->ent_pass = gtk_entry_new ();
	gtk_entry_set_visibility (GTK_ENTRY (obj->priv->ent_pass), FALSE);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->ent_pass, _("MPD password, if any"), NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), obj->priv->ent_pass, 1, 2, 2, 3);

	/* Use pass? */
	obj->priv->chk_auth = gtk_check_button_new_with_label (_("Use authentication?"));
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->chk_auth, _("Does MPD need a password?"), NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), obj->priv->chk_auth, 1, 2, 3, 4);
	g_signal_connect (G_OBJECT (obj->priv->chk_auth), "toggled", G_CALLBACK (on_chk_auth_toggled), obj);

	/* Timeout */
	lbl = gtk_label_new (_("Timeout:"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), lbl, 0, 1, 4, 5);

	obj->priv->spn_timeout = gtk_spin_button_new_with_range (0.5, 100000.0, 0.5);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->spn_timeout, _("Timeout, in seconds"), NULL);
	gtk_spin_button_set_digits (GTK_SPIN_BUTTON (obj->priv->spn_timeout), 2);
	gtk_table_attach_defaults (GTK_TABLE (table), obj->priv->spn_timeout, 1, 2, 4, 5);

	/* Music Dir */
	lbl = gtk_label_new (_("Music folder:"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), lbl, 0, 1, 5, 6);

	hbox1 = gtk_hbox_new (FALSE, 5);
	gtk_table_attach_defaults (GTK_TABLE (table), hbox1, 1, 2, 5, 6);

	obj->priv->ent_dir = gtk_entry_new ();
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->ent_dir, _("Folder where are the music files, same as in MPD config"), NULL);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->ent_dir, TRUE, TRUE, 0);

	btn = gbemol_utils_make_button (NULL, GTK_STOCK_DIRECTORY);
	gtk_box_pack_start (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_dir_activate), obj);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_dir_activate), obj);

	/* Update library on start */
	obj->priv->chk_update = gtk_check_button_new_with_label (_("Update library on start?"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->chk_update, 
			_("Wheter the music library should be updated when the program is started"), NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), obj->priv->chk_update, 0, 1, 6, 7);

	/**** Audio Tab ****/
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 20);
	lbl = gtk_label_new (_("Audio"));
	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook), vbox2, lbl);

	/* Crossfade */
	hbox1 = gtk_hbox_new (TRUE, 5);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox1, FALSE, FALSE, 5);

	lbl = gtk_label_new (_("Crossfade"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox1), lbl, TRUE, TRUE, 5);

	obj->priv->spn_fade = gtk_spin_button_new_with_range (0.0, 100000.0, 1.0);
	gtk_spin_button_set_digits (GTK_SPIN_BUTTON (obj->priv->spn_fade), 2);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->spn_fade, TRUE, TRUE, 5);
	g_signal_connect (G_OBJECT (obj->priv->spn_fade), "change-value", G_CALLBACK (on_spn_fade_value_changed), obj);

	/* The outputs view */
	/* Create the outputs list */
	obj->priv->lst_outputs = gtk_list_store_new (N_COLUMNS_OUTPUT,
			G_TYPE_INT,
			G_TYPE_BOOLEAN,
			G_TYPE_STRING
			);

	lbl = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (lbl), _("<span weight=\"bold\">Audio Outputs</span>"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox2), lbl, FALSE, FALSE, 5);

	obj->priv->tvw_outputs = gtk_tree_view_new ();
	tvw = GTK_TREE_VIEW (obj->priv->tvw_outputs);
	scr = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scr), GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scr), obj->priv->tvw_outputs);
	gtk_box_pack_start (GTK_BOX (vbox2), scr, TRUE, TRUE, 5);

	gtk_tree_view_set_model (tvw, GTK_TREE_MODEL (obj->priv->lst_outputs));
	g_object_unref (G_OBJECT (obj->priv->lst_outputs));
	gtk_tree_view_set_search_column (tvw, COLUMN_OUTPUT_NAME);

	renderer = gtk_cell_renderer_toggle_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Active?"), renderer, 
			"active", COLUMN_OUTPUT_ACTIVE,
			NULL);
	gtk_tree_view_append_column (tvw, column);
	g_signal_connect (G_OBJECT (renderer), "toggled", G_CALLBACK (on_renderer_active_toggled), obj);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Name"), renderer,
			"text", COLUMN_OUTPUT_NAME,
			NULL);
	gtk_tree_view_append_column (tvw, column);
	
	select = gtk_tree_view_get_selection (tvw);
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

	/****** INTERFACE TAB *******/
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 20);
	lbl = gtk_label_new (_("Interface"));
	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook), vbox2, lbl);

	/** Playlist frame **/
	frame = gtk_frame_new (_("Playlist"));
	gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, TRUE, 0);
	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox3);

	/* Playlist search case sensitive */
	obj->priv->chk_playlist_sensitive = gtk_check_button_new_with_mnemonic 
		(_("Search in the _playlist is case sensitive?"));
	gtk_box_pack_start (GTK_BOX (vbox3), obj->priv->chk_playlist_sensitive, FALSE, FALSE, 10);

	/* Playlist modes */
	lbl = gtk_label_new (_("Playlist mode:"));
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox3), lbl, FALSE, FALSE, 4);

	hbox1 = gtk_hbox_new (TRUE, 5);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox1, FALSE, TRUE, 0);

	/* Modes */
	obj->priv->rad_playlist_mode[0] = gtk_radio_button_new_with_mnemonic (NULL, obj->app->pls->playlist_modes[0]);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->rad_playlist_mode[0], FALSE, FALSE, 0);
	g_object_set_data (G_OBJECT (obj->priv->rad_playlist_mode[0]), "mode", GINT_TO_POINTER (0));
	g_signal_connect (G_OBJECT (obj->priv->rad_playlist_mode[0]), "toggled",
			G_CALLBACK (on_rad_playlist_mode_toggled), obj);

	for (i = 1; i < N_PLAYLIST_MODES; i++)
	{
		obj->priv->rad_playlist_mode[i] = gtk_radio_button_new_with_mnemonic_from_widget 
			(GTK_RADIO_BUTTON (obj->priv->rad_playlist_mode[i-1]), obj->app->pls->playlist_modes[i]);
		gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->rad_playlist_mode[i], FALSE, FALSE, 0);
		g_object_set_data (G_OBJECT (obj->priv->rad_playlist_mode[i]), "mode", GINT_TO_POINTER (i));
		g_signal_connect (G_OBJECT (obj->priv->rad_playlist_mode[i]), "toggled",
				G_CALLBACK (on_rad_playlist_mode_toggled), obj);
	}

	/** Library frame **/
	frame = gtk_frame_new (_("Library"));
	gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, FALSE, 0);
	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox3);

	/* Library search case sensitive */
	obj->priv->chk_library_sensitive = gtk_check_button_new_with_mnemonic 
		(_("Search in the li_brary is case sensitive?"));
	gtk_box_pack_start (GTK_BOX (vbox3), obj->priv->chk_library_sensitive, FALSE, FALSE, 10);

	/****** WINDOW TAB *******/
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 20);
	lbl = gtk_label_new (_("Window Behavior"));
	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook), vbox2, lbl);

	/** Placement frame **/
	frame = gtk_frame_new (_("Placement"));
	gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, TRUE, 0);
	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox3);

	/* Always on top */
	obj->priv->chk_top = gtk_check_button_new_with_mnemonic 
		(_("Always on top"));
	gtk_box_pack_start (GTK_BOX (vbox3), obj->priv->chk_top, FALSE, FALSE, 10);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->chk_top, 
			_("Always keep the main window above other windows"), NULL);

	/* Show on all workspaces */
	obj->priv->chk_stick = gtk_check_button_new_with_mnemonic 
		(_("Show on all workspaces"));
	gtk_box_pack_start (GTK_BOX (vbox3), obj->priv->chk_stick, FALSE, FALSE, 10);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->chk_stick, 
			_("Show the main window on alll workspaces"), NULL);

	/** Status Icon frame **/
	frame = gtk_frame_new (_("Status Icon"));
	gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, FALSE, 0);
	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox3);

	/* Use status icon */
	obj->priv->chk_use_status_icon = gtk_check_button_new_with_mnemonic 
		(_("Use a status _icon"));
	gtk_box_pack_start (GTK_BOX (vbox3), obj->priv->chk_use_status_icon, FALSE, FALSE, 5);
	g_signal_connect (G_OBJECT (obj->priv->chk_use_status_icon), "toggled", 
			G_CALLBACK (on_chk_use_status_icon_toggled), obj);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->chk_use_status_icon, 
			_("Show an icon in the \"tray area\" of your desktop manager"), NULL);

	/* Close to tray */
	obj->priv->chk_close_tray = gtk_check_button_new_with_mnemonic 
		(_("Close to system _tray"));
	gtk_box_pack_start (GTK_BOX (vbox3), obj->priv->chk_close_tray, FALSE, FALSE, 5);
	gtk_tooltips_set_tip (obj->priv->tips, obj->priv->chk_close_tray, 
			_("Hide to the system tray instead of finishing the program"), NULL);
	
	/****** COVER ART TAB *******/
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 20);
	lbl = gtk_label_new (_("Cover Art"));
	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook), vbox2, lbl);

	/* Amazon Cover Art */
	obj->priv->chk_cover_art_amazon = gtk_check_button_new_with_mnemonic 
		(_("Try to fetch covers from _Amazon.com"));
	gtk_box_pack_start (GTK_BOX (vbox2), obj->priv->chk_cover_art_amazon, FALSE, FALSE, 10);
	
	obj->priv->chk_cover_art_file = gtk_check_button_new_with_mnemonic 
		(_("Try to load_ covers from ~/.covers"));
	gtk_box_pack_start (GTK_BOX (vbox2), obj->priv->chk_cover_art_file, FALSE, FALSE, 10);
	
	/***** BOTTOM BUTTONS ******/
	hbox1 = gtk_hbox_new (TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 0);

	btn = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_box_pack_start (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_cancel_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_cancel_clicked), obj);

	btn = gtk_button_new_from_stock (GTK_STOCK_SAVE);
	gtk_box_pack_end (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_save_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_save_clicked), obj);

	btn = gbemol_utils_make_button (_("Save & _Reconnect"), GTK_STOCK_REFRESH);
	gtk_box_pack_end (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_save_reconnect_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_save_reconnect_clicked), obj);

	gtk_widget_show_all (obj->win_main);

	gbemol_preferences_populate_outputs (obj);
	gbemol_preferences_populate (obj);

	return obj;
}

static void gbemol_preferences_save_entry(GtkWidget* entry, gchar* group, gchar* key)
{
	gbemol_cfg_set_string(group, key, gtk_entry_get_text(GTK_ENTRY(entry)));
}

static void gbemol_preferences_save_spin(GtkWidget* spin, gchar* group, gchar* key)
{
	 gbemol_cfg_set_int(group, key, gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin)));
}

static void gbemol_preferences_save_spin_double(GtkWidget* spin, gchar* group, gchar* key)
{
	 gbemol_cfg_set_double(group, key, gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)));
}

static void gbemol_preferences_save_toggle(GtkWidget* toggle, gchar* group, gchar* key)
{
	gbemol_cfg_set_bool(group, key, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle)));
}

static void gbemol_preferences_load_entry(GtkWidget* entry, gchar* group, gchar* key)
{
	gchar *txt;
	if ((txt = gbemol_cfg_get_string(group, key))) {		
		gtk_entry_set_text(GTK_ENTRY(entry), txt);
		g_free(txt);
	}
}

static void gbemol_preferences_load_entry_with_default(GtkWidget* entry, gchar* group, gchar* key, gchar* def)
{
	gchar *txt;
	if ((txt = gbemol_cfg_get_string(group, key))) {
		gtk_entry_set_text(GTK_ENTRY(entry), txt);
		g_free(txt);
	} else {
		gtk_entry_set_text(GTK_ENTRY(entry), def);
	}
}

static void gbemol_preferences_load_spin(GtkWidget* spin, gchar* group, gchar* key)
{
	gdouble v;
	v = gbemol_cfg_get_double(group, key);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), v);
}

static void gbemol_preferences_load_spin_with_default(GtkWidget* spin, gchar* group, gchar* key, gdouble def)
{
	gdouble v;
	if (!(v = gbemol_cfg_get_double(group, key))) {
		v = def;
	}
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), v);
}


static void gbemol_preferences_load_toggle(GtkWidget* toggle, gchar* group, gchar* key)
{
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle), gbemol_cfg_get_bool(group, key));
}

static void gbemol_preferences_load_toggle_with_default(GtkWidget* toggle, gchar* group, gchar* key, gboolean def)
{
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle), gbemol_cfg_get_bool_with_default(group, key, def));
}


/*
 * Populate the audio outputs
 */
static void 
gbemol_preferences_populate_outputs (GbemolPreferences* obj)
{
	GList* l;
	GbemolMpdOutput *out;
	GtkTreeIter iter;

	if (!gbemol_mpd_is_connected (obj->app->mpd))
		return;
	l = gbemol_mpd_output_get_list (obj->app->mpd);

	do
	{
		out = l->data;
		gtk_list_store_append (obj->priv->lst_outputs, &iter);
		gtk_list_store_set (obj->priv->lst_outputs, &iter,
				COLUMN_OUTPUT_ID, out->id,
				COLUMN_OUTPUT_ACTIVE, out->enabled,
				COLUMN_OUTPUT_NAME, out->name,
				-1);
	} while ((l = g_list_next (l)));

	gbemol_mpd_free_output_list (l);
}

/*
 * Populates the window with the values in the config file
 */
static void 
gbemol_preferences_populate (GbemolPreferences *obj)
{
	gchar *str, *str2;
	gint mode;

	/* MPD version */
	if (gbemol_mpd_is_connected (obj->app->mpd))
	{
		str2 = gbemol_mpd_get_version (obj->app->mpd);
		str = g_markup_printf_escaped ("<span weight=\"bold\">MPD %s</span>", str2);	
		gtk_label_set_markup (GTK_LABEL (obj->priv->lbl_mpd), str);
		g_free (str);
		g_free (str2);
	}
	else
		gtk_label_set_markup (GTK_LABEL (obj->priv->lbl_mpd), "<span weight=\"bold\">MPD</span>");

	/* MPD */	
	gbemol_preferences_load_entry_with_default(obj->priv->ent_host, "mpd", "host", "localhost");
	
	gbemol_preferences_load_spin_with_default(obj->priv->spn_port, "mpd", "port", 6600.0);

	/* Update library on start */
	gbemol_preferences_load_toggle(obj->priv->chk_update, "mpd", "update-library");

	/* Password */
	gbemol_preferences_load_toggle(obj->priv->chk_auth, "mpd", "use_auth");
	gtk_widget_set_sensitive (obj->priv->ent_pass, gbemol_cfg_get_bool("mpd", "use_auth"));
	
	gbemol_preferences_load_entry(obj->priv->ent_pass, "mpd", "pass");
	
	/* Timeout */
	gbemol_preferences_load_spin_with_default(obj->priv->spn_timeout, "mpd", "timeout", 1.0);

	/* Music dir */
	gbemol_preferences_load_entry(obj->priv->ent_dir, "mpd", "music_dir");
	
	/* Fadeout */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON (obj->priv->spn_fade), gbemol_mpd_get_crossfade(obj->app->mpd));
	
	/* Case sensitive searches */
	gbemol_preferences_load_toggle(obj->priv->chk_playlist_sensitive, "playlist", "case-sensitive");
	gbemol_preferences_load_toggle(obj->priv->chk_library_sensitive, "library", "case-sensitive");

	/* Playlist mode */
	mode = gbemol_cfg_get_int ("playlist", "mode");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (obj->priv->rad_playlist_mode[mode]), TRUE);
	
	/* Tray icon */
	gbemol_preferences_load_toggle(obj->priv->chk_use_status_icon, "app", "status-icon");
	if (!gbemol_cfg_get_bool("app", "status-icon"))
		gtk_widget_set_sensitive (obj->priv->chk_close_tray, FALSE);
	gbemol_preferences_load_toggle(obj->priv->chk_close_tray, "app", "close-tray");


	/* Cover art */
	gbemol_preferences_load_toggle_with_default(obj->priv->chk_cover_art_amazon, "cover_art", "amazon", TRUE);
	gbemol_preferences_load_toggle_with_default(obj->priv->chk_cover_art_file, "cover_art", "file", TRUE);
	
	/* Window */
	gbemol_preferences_load_toggle(obj->priv->chk_stick, "win_main", "stick");
	gbemol_preferences_load_toggle(obj->priv->chk_top, "win_main", "top");
}

static void 
on_renderer_active_toggled (GtkCellRendererToggle *renderer, gchar *path, GbemolPreferences* obj)
{
	GtkTreeIter iter;
	gint id;
	gboolean active;

	gtk_tree_model_get_iter_from_string (GTK_TREE_MODEL (obj->priv->lst_outputs), &iter, path);
	gtk_tree_model_get (GTK_TREE_MODEL (obj->priv->lst_outputs), &iter,
			COLUMN_OUTPUT_ID, &id,
			COLUMN_OUTPUT_ACTIVE, &active,
			-1);

	gbemol_mpd_output_set_active (obj->app->mpd, id, !active);

	gtk_list_store_set (GTK_LIST_STORE (obj->priv->lst_outputs), &iter,
			COLUMN_OUTPUT_ACTIVE, !active,
			-1);
}

static void on_btn_cancel_clicked (GtkButton *b, GbemolPreferences* obj)
{
	gtk_widget_destroy (obj->win_main);
}

static void on_btn_save_clicked (GtkButton *b, GbemolPreferences* obj)
{
	int a;

	gbemol_preferences_save_entry(obj->priv->ent_host, "mpd", "host");
	gbemol_mpd_set_host (obj->app->mpd, gtk_entry_get_text (GTK_ENTRY (obj->priv->ent_host)));

	gbemol_preferences_save_spin(obj->priv->spn_port, "mpd", "port");
	gbemol_mpd_set_port (obj->app->mpd, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (obj->priv->spn_port)));

	gbemol_preferences_save_toggle(obj->priv->chk_auth, "mpd", "use_auth");
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (obj->priv->chk_auth)))
	{
		gbemol_preferences_save_entry(obj->priv->ent_pass, "mpd", "pass");
		/* Set pass in the current mpd object */
		gbemol_mpd_set_pass (obj->app->mpd, gtk_entry_get_text (GTK_ENTRY (obj->priv->ent_pass)));
	}
	else
		gbemol_mpd_set_pass (obj->app->mpd, NULL);

	gbemol_preferences_save_spin_double(obj->priv->spn_timeout, "mpd", "timeout");
	gbemol_preferences_save_entry(obj->priv->ent_dir, "mpd", "music_dir");
	gbemol_preferences_save_toggle(obj->priv->chk_update, "mpd", "update-library");
	
	gbemol_preferences_save_toggle(obj->priv->chk_library_sensitive, "library", "case-sensitive");
	gbemol_preferences_save_toggle(obj->priv->chk_playlist_sensitive, "playlist", "case-sensitive");

	/* Status icon */
	gbemol_preferences_save_toggle(obj->priv->chk_use_status_icon, "app", "status-icon");
	gbemol_preferences_save_toggle(obj->priv->chk_close_tray, "app", "close-tray");

	/* Window */
	a = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (obj->priv->chk_stick));
	gbemol_app_set_stick (obj->app, a);
	gbemol_preferences_save_toggle(obj->priv->chk_stick, "win_main", "stick");

	a = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (obj->priv->chk_top));
	gbemol_app_set_top (obj->app, a);
	gbemol_preferences_save_toggle(obj->priv->chk_top, "win_main", "top");
	
	/* Cover art */
	gbemol_preferences_save_toggle(obj->priv->chk_cover_art_amazon, "cover_art", "amazon");
	gbemol_preferences_save_toggle(obj->priv->chk_cover_art_file, "cover_art", "file");

	gtk_widget_destroy (obj->win_main);
}


static void on_btn_save_reconnect_clicked (GtkButton *b, GbemolPreferences* obj)
{
	on_btn_save_clicked (b, obj);	
	gbemol_mpd_disconnect (obj->app->mpd);
	gbemol_app_connect_mpd (obj->app);
}

static void on_chk_auth_toggled (GtkToggleButton *b, GbemolPreferences* obj)
{
	gtk_widget_set_sensitive (obj->priv->ent_pass, gtk_toggle_button_get_active (b));
}

static void on_spn_fade_value_changed (GtkSpinButton *b, GbemolPreferences* obj)
{
	gbemol_mpd_crossfade (obj->app->mpd, gtk_spin_button_get_value_as_int (b));
}

static void 
on_btn_dir_activate (GtkButton *b, GbemolPreferences* obj)
{
	GtkWidget *dialog;

	dialog = gtk_file_chooser_dialog_new ("Choose music dir",
			GTK_WINDOW (obj->win_main),
			GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
	{
		char *dir;

		dir = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		gtk_entry_set_text (GTK_ENTRY (obj->priv->ent_dir), dir);
		g_free (dir);
	}

	gtk_widget_destroy (dialog);
}

static void 
on_rad_playlist_mode_toggled (GtkToggleButton *btn, GbemolPreferences *obj)
{
	int mode = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (btn), "mode"));
	if (gtk_toggle_button_get_active (btn))
	{
		gbemol_cfg_set_int ("playlist", "mode", mode);
		gbemol_playlist_set_mode (obj->app->pls, mode);
	}
}

static void
on_chk_use_status_icon_toggled (GtkToggleButton *btn, GbemolPreferences *obj)
{
	gboolean active = gtk_toggle_button_get_active (btn);
	if (active)
		gbemol_app_create_status_icon (obj->app);
	else
		gbemol_app_destroy_status_icon (obj->app);
	gtk_widget_set_sensitive (obj->priv->chk_close_tray, active);
}


