/*
  Copyright (c) 2007 Gordon Gremme <gordon@gremme.org>
  Copyright (c) 2007 Center for Bioinformatics, University of Hamburg

  Permission to use, copy, modify, and distribute this software for any
  purpose with or without fee is hereby granted, provided that the above
  copyright notice and this permission notice appear in all copies.

  THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
  WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
  ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

#include <string.h>
#include "core/assert_api.h"
#include "core/compat_api.h"
#include "core/fileutils_api.h"
#include "core/gtdatapath.h"
#include "core/log_api.h"

#ifndef _WIN32
#define GTDATADIR "/gtdata"
#define UPDIR     "/.."
#else
#define GTDATADIR "\\gtdata"
#define UPDIR     "\\.."
#endif

/* XXX: how to define the default path on Windows? */
static const char* GTDATA_DEFAULT_PATHS[]
  = { "/usr/share/genometools" GTDATADIR,
      "/usr/local/share/genometools" GTDATADIR,
      NULL };

static const char* GTDATA_RELATIVE_SEARCH_PATHS[]
  = { UPDIR "/share/genometools" GTDATADIR,
      NULL };

GtStr* gt_get_gtdata_path(const char *prog, GtError *err)
{
  GtStr *path;
  const char **defaultpath, **spath;
  int had_err = 0;
  gt_error_check(err);
  gt_assert(prog);
  path = gt_str_new();
  had_err = gt_file_find_exec_in_path(path, prog, err);
  if (!had_err) {
    if (gt_file_exists_and_is_dir(gt_str_get(path))) {
      gt_str_append_cstr(path, GTDATADIR);
      if (gt_file_exists_and_is_dir(gt_str_get(path)))
        return path;
      gt_str_set_length(path, gt_str_length(path) - strlen(GTDATADIR));
      gt_str_append_cstr(path, UPDIR);
      gt_str_append_cstr(path, GTDATADIR);
      if (gt_file_exists_and_is_dir(gt_str_get(path)))
        return path;
    }
    for (spath = GTDATA_RELATIVE_SEARCH_PATHS; *spath; spath++) {
      had_err = gt_file_find_exec_in_path(path, prog, err);
      if (!had_err) {
        gt_str_append_cstr(path, *spath);
        if (gt_file_exists_and_is_dir(gt_str_get(path)))
          return path;
      }
    }
    for (defaultpath = GTDATA_DEFAULT_PATHS; *defaultpath; defaultpath++) {
      gt_str_reset(path);
      gt_str_append_cstr(path, *defaultpath);
      if (gt_file_exists_and_is_dir(gt_str_get(path)))
        return path;
    }
    if (!gt_file_exists_and_is_dir(gt_str_get(path))) {
      gt_error_set(err, "could not find gtdata%c directory", GT_PATH_SEPARATOR);
      had_err = -1;
    }
  }
  if (had_err) {
    gt_str_delete(path);
    return NULL;
  }
  return path;
}
