// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	forgejo_sdk "code.forgejo.org/f3/gof3/v3/forges/forgejo/sdk"
)

type label struct {
	common

	forgejoLabel *forgejo_sdk.Label
}

var _ f3_tree.ForgeDriverInterface = &label{}

func newLabel() generic.NodeDriverInterface {
	return &label{}
}

func (o *label) SetNative(label any) {
	o.forgejoLabel = label.(*forgejo_sdk.Label)
}

func (o *label) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoLabel.ID)
}

func (o *label) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

func (o *label) ToFormat() f3.Interface {
	if o.forgejoLabel == nil {
		return o.NewFormat()
	}

	return &f3.Label{
		Common:      f3.NewCommon(o.GetNativeID()),
		Name:        o.forgejoLabel.Name,
		Description: o.forgejoLabel.Description,
		Color:       o.forgejoLabel.Color,
	}
}

func (o *label) FromFormat(content f3.Interface) {
	label := content.(*f3.Label)
	o.forgejoLabel = &forgejo_sdk.Label{
		ID:          util.ParseInt(label.GetID()),
		Name:        label.Name,
		Description: label.Description,
		Color:       label.Color,
	}
}

func (o *label) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	label, resp, err := o.getClient().GetRepoLabel(owner, project, node.GetID().Int64())
	if resp.StatusCode == 404 {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("label %v %w", o, err))
	}
	o.forgejoLabel = label
	return true
}

func (o *label) Put(ctx context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	label, _, err := o.getClient().CreateLabel(owner, project, forgejo_sdk.CreateLabelOption{
		Name:        o.forgejoLabel.Name,
		Color:       o.forgejoLabel.Color,
		Description: o.forgejoLabel.Description,
	})
	if err != nil {
		panic(err)
	}
	o.forgejoLabel = label
	return id.NewNodeID(o.GetNativeID())
}

func (o *label) Patch(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, _, err := o.getClient().EditLabel(owner, project, node.GetID().Int64(), forgejo_sdk.EditLabelOption{
		Name:        &o.forgejoLabel.Name,
		Color:       &o.forgejoLabel.Color,
		Description: &o.forgejoLabel.Description,
	})
	if err != nil {
		panic(err)
	}
}

func (o *label) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, err := o.getClient().DeleteLabel(owner, project, node.GetID().Int64())
	if err != nil {
		panic(err)
	}
}
