/**
 * collectd - src/meta_data.c
 * Copyright (C) 2008-2011  Florian octo Forster
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *   Florian octo Forster <octo at collectd.org>
 **/

#include "collectd.h"
#include "plugin.h"

#include <stdbool.h>

#define MD_MAX_NONSTRING_CHARS 128

/*
 * Data types
 */
union meta_value_u {
  char *mv_string;
  int64_t mv_signed_int;
  uint64_t mv_unsigned_int;
  double mv_double;
  bool mv_boolean;
};
typedef union meta_value_u meta_value_t;

struct meta_entry_s;
typedef struct meta_entry_s meta_entry_t;
struct meta_entry_s {
  char *key;
  meta_value_t value;
  int type;
  meta_entry_t *next;
};

struct meta_data_s {
  meta_entry_t *head;
  pthread_mutex_t lock;
};

/*
 * Private functions
 */
static meta_entry_t *md_entry_alloc(const char *key) /* {{{ */
{
  meta_entry_t *e;

  e = calloc(1, sizeof(*e));
  if (e == NULL) {
    ERROR("md_entry_alloc: calloc failed.");
    return NULL;
  }

  e->key = strdup(key);
  if (e->key == NULL) {
    free(e);
    ERROR("md_entry_alloc: strdup failed.");
    return NULL;
  }

  e->type = 0;
  e->next = NULL;

  return e;
} /* }}} meta_entry_t *md_entry_alloc */

static void md_entry_free(meta_entry_t *e) /* {{{ */
{
  if (e == NULL)
    return;

  free(e->key);

  if (e->type == MD_TYPE_STRING)
    free(e->value.mv_string);

  if (e->next != NULL)
    md_entry_free(e->next);

  free(e);
} /* }}} void md_entry_free */

static int md_entry_insert(meta_data_t *md, meta_entry_t *e) /* {{{ */
{
  meta_entry_t *this;
  meta_entry_t *prev;

  if ((md == NULL) || (e == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  prev = NULL;
  this = md->head;
  while (this != NULL) {
    if (strcasecmp(e->key, this->key) == 0)
      break;

    prev = this;
    this = this->next;
  }

  if (this == NULL) {
    /* This key does not exist yet. */
    if (md->head == NULL)
      md->head = e;
    else {
      assert(prev != NULL);
      prev->next = e;
    }

    e->next = NULL;
  } else /* (this != NULL) */
  {
    if (prev == NULL)
      md->head = e;
    else
      prev->next = e;

    e->next = this->next;
  }

  pthread_mutex_unlock(&md->lock);

  if (this != NULL) {
    this->next = NULL;
    md_entry_free(this);
  }

  return 0;
} /* }}} int md_entry_insert */

/* XXX: The lock on md must be held while calling this function! */
static meta_entry_t *md_entry_lookup(meta_data_t *md, /* {{{ */
                                     const char *key) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL))
    return NULL;

  for (e = md->head; e != NULL; e = e->next)
    if (strcasecmp(key, e->key) == 0)
      break;

  return e;
} /* }}} meta_entry_t *md_entry_lookup */

/*
 * Each value_list_t*, as it is going through the system, is handled by exactly
 * one thread. Plugins which pass a value_list_t* to another thread, e.g. the
 * rrdtool plugin, must create a copy first. The meta data within a
 * value_list_t* is not thread safe and doesn't need to be.
 *
 * The meta data associated with cache entries are a different story. There, we
 * need to ensure exclusive locking to prevent leaks and other funky business.
 * This is ensured by the uc_meta_data_get_*() functions.
 */

/*
 * Public functions
 */
meta_data_t *meta_data_create(void) /* {{{ */
{
  meta_data_t *md;

  md = calloc(1, sizeof(*md));
  if (md == NULL) {
    ERROR("meta_data_create: calloc failed.");
    return NULL;
  }

  pthread_mutex_init(&md->lock, /* attr = */ NULL);

  return md;
} /* }}} meta_data_t *meta_data_create */

void meta_data_destroy(meta_data_t *md) /* {{{ */
{
  if (md == NULL)
    return;

  md_entry_free(md->head);
  pthread_mutex_destroy(&md->lock);
  free(md);
} /* }}} void meta_data_destroy */

int meta_data_type(meta_data_t *md, const char *key) /* {{{ */
{
  if ((md == NULL) || (key == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  for (meta_entry_t *e = md->head; e != NULL; e = e->next) {
    if (strcasecmp(key, e->key) == 0) {
      pthread_mutex_unlock(&md->lock);
      return e->type;
    }
  }

  pthread_mutex_unlock(&md->lock);
  return 0;
} /* }}} int meta_data_type */

int meta_data_toc(meta_data_t *md, char ***toc) /* {{{ */
{
  int i = 0, count = 0;

  if ((md == NULL) || (toc == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  for (meta_entry_t *e = md->head; e != NULL; e = e->next)
    ++count;

  if (count == 0) {
    pthread_mutex_unlock(&md->lock);
    return count;
  }

  *toc = calloc(count, sizeof(**toc));
  for (meta_entry_t *e = md->head; e != NULL; e = e->next)
    (*toc)[i++] = strdup(e->key);

  pthread_mutex_unlock(&md->lock);
  return count;
} /* }}} int meta_data_toc */

/*
 * Add functions
 */
int meta_data_add_string(meta_data_t *md, /* {{{ */
                         const char *key, const char *value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL) || (value == NULL))
    return -EINVAL;

  e = md_entry_alloc(key);
  if (e == NULL)
    return -ENOMEM;

  e->value.mv_string = strdup(value);
  if (e->value.mv_string == NULL) {
    ERROR("meta_data_add_string: strdup failed.");
    md_entry_free(e);
    return -ENOMEM;
  }
  e->type = MD_TYPE_STRING;

  return md_entry_insert(md, e);
} /* }}} int meta_data_add_string */

int meta_data_add_signed_int(meta_data_t *md, /* {{{ */
                             const char *key, int64_t value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL))
    return -EINVAL;

  e = md_entry_alloc(key);
  if (e == NULL)
    return -ENOMEM;

  e->value.mv_signed_int = value;
  e->type = MD_TYPE_SIGNED_INT;

  return md_entry_insert(md, e);
} /* }}} int meta_data_add_signed_int */

int meta_data_add_unsigned_int(meta_data_t *md, /* {{{ */
                               const char *key, uint64_t value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL))
    return -EINVAL;

  e = md_entry_alloc(key);
  if (e == NULL)
    return -ENOMEM;

  e->value.mv_unsigned_int = value;
  e->type = MD_TYPE_UNSIGNED_INT;

  return md_entry_insert(md, e);
} /* }}} int meta_data_add_unsigned_int */

int meta_data_add_double(meta_data_t *md, /* {{{ */
                         const char *key, double value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL))
    return -EINVAL;

  e = md_entry_alloc(key);
  if (e == NULL)
    return -ENOMEM;

  e->value.mv_double = value;
  e->type = MD_TYPE_DOUBLE;

  return md_entry_insert(md, e);
} /* }}} int meta_data_add_double */

int meta_data_add_boolean(meta_data_t *md, /* {{{ */
                          const char *key, bool value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL))
    return -EINVAL;

  e = md_entry_alloc(key);
  if (e == NULL)
    return -ENOMEM;

  e->value.mv_boolean = value;
  e->type = MD_TYPE_BOOLEAN;

  return md_entry_insert(md, e);
} /* }}} int meta_data_add_boolean */

/*
 * Get functions
 */
int meta_data_get_string(meta_data_t *md, /* {{{ */
                         const char *key, char **value) {
  meta_entry_t *e;
  char *temp;

  if ((md == NULL) || (key == NULL) || (value == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  e = md_entry_lookup(md, key);
  if (e == NULL) {
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  if (e->type != MD_TYPE_STRING) {
    ERROR("meta_data_get_string: Type mismatch for key `%s'", e->key);
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  temp = strdup(e->value.mv_string);
  if (temp == NULL) {
    pthread_mutex_unlock(&md->lock);
    ERROR("meta_data_get_string: strdup failed.");
    return -ENOMEM;
  }

  pthread_mutex_unlock(&md->lock);

  *value = temp;

  return 0;
} /* }}} int meta_data_get_string */

int meta_data_get_signed_int(meta_data_t *md, /* {{{ */
                             const char *key, int64_t *value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL) || (value == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  e = md_entry_lookup(md, key);
  if (e == NULL) {
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  if (e->type != MD_TYPE_SIGNED_INT) {
    ERROR("meta_data_get_signed_int: Type mismatch for key `%s'", e->key);
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  *value = e->value.mv_signed_int;

  pthread_mutex_unlock(&md->lock);
  return 0;
} /* }}} int meta_data_get_signed_int */

int meta_data_get_unsigned_int(meta_data_t *md, /* {{{ */
                               const char *key, uint64_t *value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL) || (value == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  e = md_entry_lookup(md, key);
  if (e == NULL) {
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  if (e->type != MD_TYPE_UNSIGNED_INT) {
    ERROR("meta_data_get_unsigned_int: Type mismatch for key `%s'", e->key);
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  *value = e->value.mv_unsigned_int;

  pthread_mutex_unlock(&md->lock);
  return 0;
} /* }}} int meta_data_get_unsigned_int */

int meta_data_get_double(meta_data_t *md, /* {{{ */
                         const char *key, double *value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL) || (value == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  e = md_entry_lookup(md, key);
  if (e == NULL) {
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  if (e->type != MD_TYPE_DOUBLE) {
    ERROR("meta_data_get_double: Type mismatch for key `%s'", e->key);
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  *value = e->value.mv_double;

  pthread_mutex_unlock(&md->lock);
  return 0;
} /* }}} int meta_data_get_double */

int meta_data_get_boolean(meta_data_t *md, /* {{{ */
                          const char *key, bool *value) {
  meta_entry_t *e;

  if ((md == NULL) || (key == NULL) || (value == NULL))
    return -EINVAL;

  pthread_mutex_lock(&md->lock);

  e = md_entry_lookup(md, key);
  if (e == NULL) {
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  if (e->type != MD_TYPE_BOOLEAN) {
    ERROR("meta_data_get_boolean: Type mismatch for key `%s'", e->key);
    pthread_mutex_unlock(&md->lock);
    return -ENOENT;
  }

  *value = e->value.mv_boolean;

  pthread_mutex_unlock(&md->lock);
  return 0;
} /* }}} int meta_data_get_boolean */
