// Copyright 2012-2015 Oliver Eilhard. All rights reserved.
// Use of this source code is governed by a MIT-license.
// See http://olivere.mit-license.org/license.txt for details.

package elastic

// MaxAggregation is a single-value metrics aggregation that keeps track and
// returns the maximum value among the numeric values extracted from
// the aggregated documents. These values can be extracted either from
// specific numeric fields in the documents, or be generated by
// a provided script.
// See: http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-aggregations-metrics-max-aggregation.html
type MaxAggregation struct {
	field           string
	script          string
	scriptFile      string
	lang            string
	format          string
	params          map[string]interface{}
	subAggregations map[string]Aggregation
}

func NewMaxAggregation() MaxAggregation {
	a := MaxAggregation{
		params:          make(map[string]interface{}),
		subAggregations: make(map[string]Aggregation),
	}
	return a
}

func (a MaxAggregation) Field(field string) MaxAggregation {
	a.field = field
	return a
}

func (a MaxAggregation) Script(script string) MaxAggregation {
	a.script = script
	return a
}

func (a MaxAggregation) ScriptFile(scriptFile string) MaxAggregation {
	a.scriptFile = scriptFile
	return a
}

func (a MaxAggregation) Lang(lang string) MaxAggregation {
	a.lang = lang
	return a
}

func (a MaxAggregation) Format(format string) MaxAggregation {
	a.format = format
	return a
}

func (a MaxAggregation) Param(name string, value interface{}) MaxAggregation {
	a.params[name] = value
	return a
}

func (a MaxAggregation) SubAggregation(name string, subAggregation Aggregation) MaxAggregation {
	a.subAggregations[name] = subAggregation
	return a
}

func (a MaxAggregation) Source() interface{} {
	// Example:
	//	{
	//    "aggs" : {
	//      "max_price" : { "max" : { "field" : "price" } }
	//    }
	//	}
	// This method returns only the { "max" : { "field" : "price" } } part.

	source := make(map[string]interface{})
	opts := make(map[string]interface{})
	source["max"] = opts

	// ValuesSourceAggregationBuilder
	if a.field != "" {
		opts["field"] = a.field
	}
	if a.script != "" {
		opts["script"] = a.script
	}
	if a.scriptFile != "" {
		opts["script_file"] = a.scriptFile
	}
	if a.lang != "" {
		opts["lang"] = a.lang
	}
	if a.format != "" {
		opts["format"] = a.format
	}
	if len(a.params) > 0 {
		opts["params"] = a.params
	}

	// AggregationBuilder (SubAggregations)
	if len(a.subAggregations) > 0 {
		aggsMap := make(map[string]interface{})
		source["aggregations"] = aggsMap
		for name, aggregate := range a.subAggregations {
			aggsMap[name] = aggregate.Source()
		}
	}

	return source
}
