/****************************************************************************
**
** Copyright (C) 2016 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of Qt Creator.
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
****************************************************************************/

#include "googletest.h"

#include "clangiasyncjob.h"
#include "dummyclangipcclient.h"
#include "processevents-utilities.h"

#include <clangdocument.h>
#include <clangdocumentprocessor.h>
#include <clangdocumentprocessors.h>
#include <clangdocuments.h>
#include <clangexceptions.h>
#include <clangjobrequest.h>
#include <clangjobs.h>
#include <unsavedfiles.h>

using testing::Eq;

using namespace ClangBackEnd;

namespace {

class DocumentProcessors : public ::testing::Test
{
protected:
    void SetUp() override;
    void TearDown() override;

    bool waitUntilAllJobsFinished(int timeOutInMs = 10000) const;

protected:
    ClangBackEnd::UnsavedFiles unsavedFiles;
    ClangBackEnd::Documents documents{unsavedFiles};
    ClangBackEnd::Document document;

    DummyIpcClient dummyIpcClient;

    Utf8String filePath{Utf8StringLiteral(TESTDATA_DIR"/translationunits.cpp")};

    ClangBackEnd::JobRequest jobRequest;
    ClangBackEnd::JobContext jobContext;

    ClangBackEnd::DocumentProcessors documentProcessors{documents,
                                                        unsavedFiles,
                                                        dummyIpcClient};
};

using DocumentProcessorsSlowTest = DocumentProcessors;

TEST_F(DocumentProcessors, HasNoItemsInitially)
{
    ASSERT_TRUE(documentProcessors.processors().empty());
}

TEST_F(DocumentProcessors, CreateAddsADocumentProcessor)
{
    documentProcessors.create(document);

    ASSERT_THAT(documentProcessors.processors().size(), Eq(1));
}

TEST_F(DocumentProcessors, CreateReturnsDocumentProcessor)
{
    const DocumentProcessor documentProcessor = documentProcessors.create(document);

    ASSERT_THAT(documentProcessor.document(), Eq(document));
}

TEST_F(DocumentProcessors, CreateThrowsForAlreadyExisting)
{
    documentProcessors.create(document);

    ASSERT_THROW(documentProcessors.create(document),
                 ClangBackEnd::DocumentProcessorAlreadyExists);
}

TEST_F(DocumentProcessors, Access)
{
    documentProcessors.create(document);

    const DocumentProcessor documentProcessor = documentProcessors.processor(document);

    ASSERT_THAT(documentProcessor.document(), Eq(document));
}

TEST_F(DocumentProcessors, AccessThrowsForNotExisting)
{
    ASSERT_THROW(documentProcessors.processor(document),
                 ClangBackEnd::DocumentProcessorDoesNotExist);
}

TEST_F(DocumentProcessors, Remove)
{
    documentProcessors.create(document);

    documentProcessors.remove(document);

    ASSERT_TRUE(documentProcessors.processors().empty());
}

TEST_F(DocumentProcessors, ResetTakesOverJobsInQueue)
{
    documentProcessors.create(document);
    documentProcessors.processor(document).addJob(JobRequest::Type::RequestReferences);
    documents.remove({document.fileContainer()});
    const auto newDocument = *documents.create({document.fileContainer()}).begin();

    documentProcessors.reset(document, newDocument);

    ASSERT_THAT(documentProcessors.processor(document).queue().first().type,
                JobRequest::Type::RequestReferences);
}

TEST_F(DocumentProcessors, RemoveThrowsForNotExisting)
{
    ASSERT_THROW(documentProcessors.remove(document),
                 ClangBackEnd::DocumentProcessorDoesNotExist);
}

TEST_F(DocumentProcessors, ProcessEmpty)
{
    documentProcessors.create(document);

    const JobRequests jobsStarted = documentProcessors.process();

    ASSERT_TRUE(jobsStarted.isEmpty());
}

TEST_F(DocumentProcessorsSlowTest, ProcessSingle)
{
    DocumentProcessor documentProcessor = documentProcessors.create(document);
    documentProcessor.addJob(JobRequest::Type::UpdateAnnotations);

    const JobRequests jobsStarted = documentProcessors.process();

     ASSERT_THAT(jobsStarted.size(), 1);
}

void DocumentProcessors::SetUp()
{
    const QVector<FileContainer> fileContainer{FileContainer(filePath)};
    document = documents.create(fileContainer).front();
    documents.setVisibleInEditors({filePath});
    documents.setUsedByCurrentEditor(filePath);
}

void DocumentProcessors::TearDown()
{
    ASSERT_TRUE(waitUntilAllJobsFinished()); // QFuture/QFutureWatcher is implemented with events
}

bool DocumentProcessors::waitUntilAllJobsFinished(int timeOutInMs) const
{
    const auto noJobsRunningAnymore = [this](){ return documentProcessors.runningJobs().isEmpty(); };

    return ProcessEventUtilities::processEventsUntilTrue(noJobsRunningAnymore, timeOutInMs);
}

} // anonymous
