use lang:bs;
use core:lang;
use core:asm;
use core:debug;

// Get the name of the SrcName type.
SrcName srcName(SrcPos pos) on Compiler {
	SrcName name(pos);
	name.add("core");
	name.add("lang");
	name.add("SrcName");
	name;
}

// Get the name of the TypePart type.
SrcName namePart(SrcPos pos) on Compiler {
	SrcName name(pos);
	name.add("core");
	name.add("lang");
	name.add("RecPart");
	name;
}

// Create something that gives you a TypePart.
Expr createTypePart(SrcPos pos, Block block, NamePart part) on Compiler {
	ExprBlock b(pos, block);
	SrcName partT = namePart(pos);

	Expr str = strConstant(pos, part.name);
	Var base(b, partT, SStr("part", pos), Actuals(str));
	b.add(base);
	LocalVarAccess bAccess(pos, base.var);

	if (part as SimplePart) {
		for (Nat i = 0; i < part.params.count; i++) {
			Expr created = if (type = part.params[i].type) {
				createType(pos, b, Name(type.path));
			} else {
				createType(pos, b, Name("void"));
			};
			b.add(namedExpr(b, SStr("add", pos), bAccess, Actuals(created)));
		}
	} else if (part as RecPart) {
		for (Nat i = 0; i < part.params.count; i++) {
			Expr created = createType(pos, b, part.params[i]);
			b.add(namedExpr(b, SStr("add", pos), bAccess, Actuals(created)));
		}
	}

	b.add(bAccess);
	b;
}

// Create something that gives you a TypeName.
Expr createType(SrcPos pos, Block block, Name type) on Compiler {
	ExprBlock b(pos, block);
	SrcName nameT = srcName(pos);
	SrcName partT = namePart(pos);

	Var base(b, nameT, SStr("name", pos), Actuals());
	b.add(base);

	LocalVarAccess bAccess(pos, base.var);
	for (Nat i = 0; i < type.count; i++) {
		Expr created = createTypePart(pos, b, type[i]);
		b.add(namedExpr(b, SStr("add", pos), bAccess, Actuals(created)));
	}

	b.add(bAccess);
	b;
}

// Create a static reference to a Named object in the compiler.
Expr referNamed(SrcPos pos, Block block, SrcName type) on Compiler {
	Named? referTo = block.scope.find(type);
	if (referTo) {
		ReferNamed(pos, referTo);
	} else {
		throw SyntaxError(type.pos, "Could not resolve the name " + type.toS);
	}
}

Expr referNamedAbs(SrcPos pos, Block block, SrcName type) on Compiler {
	Named? referTo = rootScope().find(type);
	if (referTo) {
		ReferNamed(pos, referTo);
	} else {
		throw SyntaxError(type.pos, "Could not resolve the absolute name " + type.toS);
	}
}

// Special case for the current package.
Expr referNamed(SrcPos pos, Block block) on Compiler {
	ReferNamed(pos, ScopeLookup:firstPkg(block.lookup));
}

// Refer to a named object somewhere.
class ReferNamed extends Expr {

	// Object we're referring to.
	Named referTo;

	init(SrcPos pos, Named to) {
		init(pos) {
			referTo = to;
		}
	}

	ExprResult result() {
		ExprResult(Value(typeOf(referTo)));
	}

	void code(CodeGen state, CodeResult result) {
		if (!result.needed) {
			return;
		}

		// This solution is also bad when we want to serialize the code and load it later.
		// We should use a reference to the named item instead.
		var v = result.location(state);
		state.l << mov(v, objPtr(referTo));
		result.created(state);
	}

	Str toS() {
		"named{" # referTo.identifier # "}";
	}
}

Expr exploreType(SrcPos pos, Block block, SrcName typeName) on Compiler {
	Expr type = createType(pos, block, typeName);

	Expr rootScope = namedExpr(block, name{core:lang:rootScope}, Actuals());
	Actuals params;
	params.add(rootScope);
	params.add(type);
	Expr named = namedExpr(block, name{core:lang:Scope:find}, params);
	namedExpr(block, name{lang:bs:macro:exploreType}, Actuals(named));
}

void exploreType(Maybe<Named> named) on Compiler {
	if (named) {
		print("Contents of " # named.identifier() # ":\n");
		print(explore(named));
	} else {
		print("Unknown name");
	}
}

Str explore(Named named) on Compiler {
	if (named as NameSet) {
		named.forceLoad();

		StrBuf out;
		for (o in named) {
			out << o.toS() << "\n";
		}
		out.toS();
	} else {
		named.toS();
	}
}

Expr reloadName(Block block, SrcName name) on Compiler {
	if (found = rootScope().find(name) as Package) {
		ReferNamed ref(name.pos, found);
		namedExpr(block, SStr("reload", name.pos), ref);
	} else {
		throw SyntaxError(name.pos, "The name " + name.toS + " does not refer to a package.");
	}
}


Expr printExpr(Str text, Expr expr) on Compiler {
	if (expr as Operator) {
		print(text.removeIndentation # " <=> " # expr.meaning);
	} else {
	    print(text.removeIndentation # " <=> " # expr);
	}
	expr;
}

void summary(NameSet s) on Compiler {
	Type[] types;
	Function[] fns;

	for (i in s) {
		if (i as Type) {
			types << i;
		} else if (i as Function) {
			fns << i;
		}
	}

	print("Types:");
	for (i in types) {
		print(" " # i.path);
	}

	print("");
	print("Functions:");
	for (i in fns) {
		print(" " # i.path);
	}
}

class PrintAsm extends Expr {
	init(SrcPos pos) {
		init(pos);
	}

	ExprResult result() : override {
		Value();
	}

	void code(CodeGen state, CodeResult result) : override {
		print(state.toS);
	}

	Str toS() : override {
		"dumpAsm";
	}
}


/**
 * Custom version of RecPart that supports having the first parameter to be 'this'.
 * This is to make it possible to refer to member functions/variables easier.
 */
class MacroRecPart extends RecPart {
	// Create.
	init(SStr name) {
		init(name) {}
	}
	init(Str name) {
		init(name) {}
	}

	// Should the first parameter be "this"?
	Bool parentFirst;

	// Called from syntax to set us to have 'this' first.
	MacroRecPart addParentParam() {
		parentFirst = true;
		this;
	}

	// Resolve.
	SimplePart? find(Scope scope) : override {
		if (simplified = super:find(scope)) {
			if (parentFirst) {
				return ParentSimplePart(simplified);
			} else {
				return simplified;
			}
		} else {
			return null;
		}
	}
}

/**
 * Custom version of SimplePart that always matches 'this' as the first parameter.
 */
class ParentSimplePart extends SimplePart {
	// Create.
	init(SimplePart part) {
		init(part) {}

		params.insert(0, Value());
	}

	// See if we match a particular part.
	Int matches(Named candidate, Scope source) {
		unless (parent = candidate.parentLookup as Type)
			return -1;

		// Note: We keep params[0] as Value() to make the auto-generation of overloads happy.
		params[0].type = parent;
		Int result = super:matches(candidate, source);
		params[0].type = null;
		result;
	}
}


// Generate a name part from a string.
MacroRecPart strRecPart(Str name) {
	MacroRecPart(name.unescape('"'));
}
