use core:lang;
use core:asm;
use lang:bs:macro;

/**
 * Extension of the Value class in Storm to also include information about 'const'. Used as a
 * top-level construct in combination with pointers to model the type system in C/C++.
 */
value CppValue extends Value {
	// Is this value a const value?
	Bool const;

	cast(Value x) {
		init(x) { const = false; }
	}

	init(Value x, Bool const) {
		init(x) { const = const; }
	}
}


/**
 * Local variable in C++.
 *
 * Variables are always a pointer to an array allocation the size of one element, just like value
 * types are stored in a Variant.
 *
 * If the referred type is a class-type in Storm (i.e. it is foreign, we don't use them in the C++
 * world), it is just a pointer to the object.
 *
 * Parameters are explicitly copied at the start of the function, so that all variables follow the
 * above convention.
 */
class LocalVar extends Named {
	init(Str name, Type val, SrcPos pos, Bool const, Bool array) {
		init(pos, name) {
			type = val;
			array = array;
			const = const;
		}
	}

	// Value stored.
	Type type;

	// Is this variable marked 'const'? If it is an array, then the contents are const.
	Bool const;

	// Is this an array?
	Bool array;

	// The variable. Created during code generation.
	CppVar var;

	// Block to notify. Updated during code generation.
	Block? notify;

	// Initializer, if initialized to a constant value. In case of arrays, set to a number equal to the
	// size of the variable if its size is known.
	Variant initializer;

	// Create the variable. OK to call twice.
	void create(CodeGen gen) {
		if (var.v == Var()) {
			var = allocType(gen, type);
		}

		Listing:VarInfo info(name, type, true, pos);
		gen.l.varInfo(var.v, info);
	}
}

// Extract the type part 'safely'.
Type safeType(Value v, SrcPos pos) on Compiler {
	if (t = v.type)
		t;
	else
		throw SyntaxError(pos, "Not possible to use 'void' here.");
}

/**
 * Access to local variables.
 */
class LocalVarAccess extends Expr {
	LocalVar var;

	init(SrcPos pos, LocalVar var) {
		init(pos) { var = var; }
	}

	Bool temporary() : override {
		false;
	}

	Bool constant() : override {
		var.const;
	}

	Variant staticValue(Type t) : override {
		if (var.const)
			if (Value(t).mayReferTo(var.type))
				return var.initializer;

		Variant();
	}

	ExprResult result() : override {
		Value(var.type, true);
	}

	void code(CodeGen gen, CodeResult result) : override {
		if (!result.needed())
			return;

		if (result.type.ref) {
			var.var.adjust(gen, result.location(gen));
		} else {
			var r = result.location(gen);
			var.var.adjust(gen, ptrA);
			if (result.type.isAsmType) {
				gen.l << mov(r, xRel(r.size, ptrA));
			} else {
				gen.l << lea(ptrB, r);
				gen.l << fnParam(ptrDesc, ptrB);
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnCall(result.type.copyCtor, true);
			}
		}
	}

	void codePtr(CodeGen gen, Type type) : override {
		var.var.ptr(gen);
	}
}

/**
 * A local var access that appears to be a temporary. Used when compiling "return" statements.
 */
class LocalVarTmpAccess extends LocalVarAccess {
	init(LocalVarAccess access) {
		init(access.pos, access.var) {}
	}

	Bool temporary() : override {
		true;
	}
}

/**
 * Variable declaration.
 */
class VarDecl extends Stmt {
	init(SrcPos pos, Block parent, CppValue type, SStr name) {
		init(pos) {
			var(name.v, type.safeType(pos), name.pos, type.const, false);
		}

		if (t = type.type) {
			// C++ constructor.
			if (ctor = t.cppCtor([], parent.scope)) {
				initExpr = CtorCall(pos, ctor, [], true);
			} else if (ctor = t.defaultCtor) {
				initExpr = CtorCall(pos, ctor, [], false);
			}
		}

		if (initExpr.empty) {
			// print("TODO: Require the default constructor to be called!");
		}

		parent.add(var);
	}

	// Parameter 'original' is to allow computing constant values if needed.
	init(SrcPos pos, Block parent, CppValue type, SStr name, ExprRef init, Expr? original) {
		init(pos) {
			var(name.v, type.safeType(pos), name.pos, type.const, false);
			initExpr = init;
		}

		if (type.const) {
			if (original) {
				var.initializer = original.staticValue(var.type);
			}
		}

		parent.add(var);
	}

	// Create from an = expression.
	VarDecl assign(SrcPos pos, Block parent, CppValue type, SStr name, Expr init) : static {
		if (t = type.type) {
			init = expectCast(init, t);

			if (init as ExprRef) {
				// Just use the expression to construct the object for us.
				return VarDecl(pos, parent, type, name, init, init);
			}

			// We need to call the copy-ctor...
			return VarDecl(pos, parent, type, name, copyCtorCall(t, init, parent.scope), init);
		}

		throw SyntaxError(pos, "The type ${type} does not have a copy constructor!");
	}

	// Create from an () expression.
	VarDecl parens(SrcPos pos, Block parent, CppValue type, SStr name, Expr[] actuals) : static {
		if (t = type.type) {
			if (actuals.count == 1 & t is actuals[0].result.type.type) {
				// If there is only a single expression, we can use it to initialize the variable directly.
				if (create = actuals[0] as ExprRef)
					return VarDecl(pos, parent, type, name, create, create);
			}

			if (ctor = t.cppCtor(actuals, parent.scope)) {
				return VarDecl(pos, parent, type, name, CtorCall(pos, ctor, actuals, true), null);
			}

			if (ctor = t.stormCtor(actuals, parent.scope)) {
				return VarDecl(pos, parent, type, name, CtorCall(pos, ctor, actuals, false), null);
			}
		}

		throw SyntaxError(pos, "No suitable constructor for ${type}.");
	}

	// The variable we created.
	LocalVar var;

	// Initialization expression. This always supports allocating the type in-place.
	ExprRef? initExpr;

	// Generate code.
	Bool code(CodeGen gen) : override {
		if (initExpr) {
			initExpr.codeRef(gen, var.var);
		} else {
			var.var.created(gen);
		}

		true;
	}
}

/**
 * Array variable declaration. No initialization.
 */
class ArrayDecl extends Stmt {
	init(SrcPos pos, Block parent, CppValue type, SStr name, Expr count) {
		init(pos) {
			var(name.v, type.safeType(pos), name.pos, type.const, true);
			countExpr = count;
		}

		var.initializer = count.staticValue();

		if (t = type.type) {
			if (ctor = t.cppCtor([], parent.scope)) {
				initExpr = CtorCall(pos, ctor, [], true);
			} else if (ctor = t.defaultCtor) {
				initExpr = CtorCall(pos, ctor, [], false);
			}
		}

		parent.add(var);
	}

	// Created variable.
	LocalVar var;

	// Count expression.
	Expr countExpr;

	// Initialization expression.
	CtorCall? initExpr;

	// Generate code.
	Bool code(CodeGen gen) : override {
		Var count = gen.l.createVar(gen.block, sPtr);

		CodeResult r(Value(named{Int}), gen.block);
		countExpr.code(gen, r);

		// Check if the value is way too big.
		gen.l << fnParam(intDesc, r.location(gen));
		gen.l << fnCall(named{rangeCheck<Int>}.ref, false);

		// Proceed.
		gen.l << ucast(count, r.location(gen));

		// Allocate memory.
		gen.l << fnParam(ptrDesc, var.type.typeRef);
		gen.l << fnParam(ptrDesc, count);
		gen.l << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, ptrA);
		gen.l << mov(var.var.v, ptrA);

		var.var.created(gen);
		gen.l << location(SrcPos());

		// Set 'filled'
		Nat mask = AllocFlags:arrayAlloc.v;
		gen.l << mov(ptrB, count);
		gen.l << or(ptrB, ptrConst(mask));
		gen.l << mov(ptrRel(ptrA, Offset(sPtr)), ptrB);

		if (initExpr) {
			// Call the constructors.
			Var current = gen.l.createVar(gen.block, sPtr);
			Label start = gen.l.label();
			Label again = gen.l.label();

			// gen.l << mov(current, ptrConst(0)); // Note: this is the default.
			gen.l << jmp(start);

			gen.l << again;

			initExpr.codeRef(gen, var.var, current);

			gen.l << add(current, ptrConst(1));

			gen.l << start;
			gen.l << cmp(current, count);
			gen.l << jmp(again, CondFlag:ifBelow);
		}

		true;
	}

	// Preform a range-check on the allocation.
	private void rangeCheck(Int val) : static {
		// Some more or less sensible values. We don't allow more than ~65536 entries in an array to
		// be at least somewhat safe.
		if (val < 0 | val > 65535)
			throw InternalError("Improper size of an array allocation: ${val}");
	}
}

/**
 * Initialized array declaration. Does not support dynamic size of arrays.
 */
class InitArrayDecl extends Stmt {
	init(SrcPos pos, Block parent, CppValue type, SStr name, Nat count, Expr[] init) {
		Type t = type.safeType(pos);
		init(pos) {
			var(name.v, t, name.pos, type.const, true);
		}

		var.initializer = count;

		if (count < init.count)
			throw SyntaxError(pos, "Too many initializers. The array has size ${count} but ${init.count} initializers.");

		for (x in init) {
			if (x as CtorCall) {
				initExprs << x;
			} else {
				initExprs << copyCtorCall(t, x, parent.scope);
			}
		}

		if (initExprs.count < count) {
			CtorCall def = defCtorCall(pos, t, parent.scope);
			while (initExprs.count < count) {
				initExprs << def;
			}
		}

		parent.add(var);
	}

	// Created variable.
	LocalVar var;

	// Initializers. One for each element.
	CtorCall[] initExprs;

	// Generate code.
	Bool code(CodeGen gen) : override {
		var count = natConst(initExprs.count);

		// Allocate memory.
		gen.l << fnParam(ptrDesc, var.type.typeRef);
		gen.l << fnParam(ptrDesc, ptrConst(initExprs.count));
		gen.l << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, ptrA);
		gen.l << mov(var.var.v, ptrA);

		var.var.created(gen);
		gen.l << location(SrcPos());

		// Set 'filled'
		Nat mask = AllocFlags:arrayAlloc.v;
		gen.l << mov(intRel(ptrA, Offset(sPtr)), natConst(mask | initExprs.count));

		// Call the constructors.
		for (i, x in initExprs) {
			x.codeRef(gen, var.var, ptrConst(i));
		}

		true;
	}
}

Stmt arrayDecl(SrcPos pos, Block block, Value type, SStr name, Expr count, Maybe<Expr[]> init) on Compiler {
	if (init) {
		// Initializers present. Make sure that the expression is constant.
		if (staticCount = variantToInt(count.staticValue())) {
			if (staticCount < 0)
				throw SyntaxError(pos, "Array size can not be negative (it is ${staticCount}).");
			InitArrayDecl(pos, block, type, name, staticCount.nat, init);
		} else {
			throw SyntaxError(pos, "Can not use initializers with dynamically sized arrays!");
		}
	} else {
		// No initializers. Allocate it with dynamic size.
		ArrayDecl(pos, block, type, name, count);
	}
}

Stmt arrayDecl(SrcPos pos, Block block, Value type, SStr name, Expr[] init) on Compiler {
	InitArrayDecl(pos, block, type, name, init.count, init);
}

Expr[] createExprs(SStr str) on Compiler {
	Expr[] result;
	for (ch in str.v.unescape()) {
		result << IntLiteral(str.pos, ch.codepoint.int);
	}
	result << IntLiteral(str.pos, 0);
	result;
}
