#!/usr/bin/env python3
# DESCRIPTION: Verilator: Verilog Test driver/expect definition
#
# Copyright 2024 by Wilson Snyder. This program is free software; you
# can redistribute it and/or modify it under the terms of either the GNU
# Lesser General Public License Version 3 or the Perl Artistic License
# Version 2.0.
# SPDX-License-Identifier: LGPL-3.0-only OR Artistic-2.0

import vltest_bootstrap

test.scenarios('simulator')
test.top_filename = test.obj_dir + "/t_vlprocess_missing.v"

# Number of tests to generate
NUM_TESTS = 200

# Testbench header template
HEADER = """\
module Testbench;

  logic clk;
  logic reset;

  // Clock driver
  initial begin
    clk = 0;
    forever begin
      #5 clk = ~clk;
    end
  end

  task automatic advance_clock(int n = 1);
    repeat (n) @(posedge clk);
  endtask

"""

# Test task template
TEST_TASK_TEMPLATE = """
  task automatic test_{num}();
    int counter = 0;
    int expected_value = {num};

    // Timeout wait
    fork
      begin
        advance_clock(10000);
        $error("Timeout");
      end
    join_none
    wait (counter == expected_value);
    disable fork;

    while (counter < expected_value) begin
      advance_clock();
      counter++;
    end
  endtask
"""

# Testbench footer template
FOOTER = "  initial begin"

# Call template for invoking each test task
CALL_TEMPLATE = "    test_{num}();\n"

# Footer end
FOOTER_END = """
    $finish;
  end

endmodule
"""


def gen(filename, num_tests):
    """
    Generates a SystemVerilog testbench with the specified number of tests.

    Args:
        filename (str): The output file name for the generated testbench.
        num_tests (int): The number of test tasks to generate.
    """
    with open(filename, 'w', encoding="utf-8") as fh:
        fh.write("// Generated by t_vlprocess_missing.py\n")

        # Write the header
        fh.write(HEADER)

        # Generate the test tasks
        for i in range(1, num_tests + 1):
            fh.write(TEST_TASK_TEMPLATE.format(num=i))

        # Write the initial block with test calls
        fh.write(FOOTER)
        for i in range(1, num_tests + 1):
            fh.write(CALL_TEMPLATE.format(num=i))
        fh.write(FOOTER_END)


gen(test.top_filename, NUM_TESTS)

test.compile(verilator_flags2=["--binary"])

test.passes()
