// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System;
using System.Threading;
using System.Threading.Tasks;
using FluentAssertions;
using NuGet.Configuration;
using NuGet.Packaging;
using NuGet.Protocol.Core.Types;
using NuGet.Protocol.Providers;
using Xunit;

namespace NuGet.Protocol.Tests.Providers
{
    public class VulnerabilityInfoResourceV3ProviderTests
    {
        [Fact]
        public async Task TryCreate_NoResourceInServiceIndex_ReturnsFalseAsync()
        {
            // Arrange
            var serviceIndexProvider = MockServiceIndexResourceV3Provider.Create();
            var target = new VulnerabilityInfoResourceV3Provider();
            INuGetResourceProvider[] providers = new INuGetResourceProvider[] { serviceIndexProvider, target };

            PackageSource packageSource = new PackageSource("https://nuget.test/v3/index.json");
            SourceRepository sourceRepository = new SourceRepository(packageSource, providers);

            // Act
            var vulnResource = await target.TryCreate(sourceRepository, CancellationToken.None);

            // Assert
            vulnResource.Item1.Should().BeFalse();
            vulnResource.Item2.Should().BeNull();
        }

        [Fact]
        public async Task TryCreate_ResourceInServiceIndex_ReturnsTrueAsync()
        {
            // Arrange
            var vulnerabilityResourceEntry = new ServiceIndexEntry(new System.Uri("https://nuget.test/v3/vuln/index.json"), ServiceTypes.VulnerabilityInfo[0], MinClientVersionUtility.GetNuGetClientVersion());
            var serviceIndexProvider = MockServiceIndexResourceV3Provider.Create(vulnerabilityResourceEntry);
            var target = new VulnerabilityInfoResourceV3Provider();
            INuGetResourceProvider[] providers = new INuGetResourceProvider[] { serviceIndexProvider, target };

            PackageSource packageSource = new PackageSource("https://nuget.test/v3/index.json");
            SourceRepository sourceRepository = new SourceRepository(packageSource, providers);

            // Act
            var vulnResource = await target.TryCreate(sourceRepository, CancellationToken.None);

            // Assert
            vulnResource.Item1.Should().BeTrue();
            vulnResource.Item2.Should().NotBeNull();
        }

        [Fact]
        public async Task TryCreate_LocalFeed_ReturnsFalseAsync()
        {
            // Arrange
            var target = new VulnerabilityInfoResourceV3Provider();
            // sourceRepository.GetResourceAsync<ServiceIndexResourceV3>() returns null for feeds that are not V3 Http.
            INuGetResourceProvider[] providers = Array.Empty<INuGetResourceProvider>();

            PackageSource packageSource = new PackageSource("z:\\pkg\\source");
            SourceRepository sourceRepository = new SourceRepository(packageSource, providers);

            // Act
            var vulnResource = await target.TryCreate(sourceRepository, CancellationToken.None);

            // Assert
            vulnResource.Item1.Should().BeFalse();
            vulnResource.Item2.Should().BeNull();
        }
    }
}
