/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/Atom.h>

#include "AtomDCProperties.h"
#include "AtomDC.h"

#include <Application.h>
using namespace camitk;
#include <QMessageBox>

// ---------------------- Constructor ----------------------------
AtomDCProperties::AtomDCProperties(AtomDC *adc) : QObject(NULL) {
    setObjectName("Atom Properties");
    updateProperties(adc);
    // install a filter to get the modification of the dynamic properties
    installEventFilter(this);

}


// ---------------------- updateProperties ----------------------------
void AtomDCProperties::updateProperties(AtomDC *adc) {
    // get the DC (who will get the messages)
    myDC = adc;

    if (myDC != NULL) {
        // message on the statusbar
        Application::showStatusBarMessage("Atom #" + QString::number(adc->getAtom()->getIndex()) + " selected.", 2000);
    }

    // check for extra properties...
    AtomProperties *aProp = adc->getAtom()->getProperties();

    std::string propName;

    for (unsigned int i = 0; i < aProp->numberOfFields(); i++) {
        propName = aProp->getField(i);

        if (propName != "") {
            setProperty(propName.c_str(), QVariant(aProp->getString(propName).c_str()));
        }
    }

    /* debug loop (pour vérifier...)
    if (aProp->numberOfFields() > 0) {
      std::cout << "Atom #" << adc->getAtom()->getIndex() << ": ";
      const QMetaObject *metaobject = metaObject();
      int count = metaobject->propertyCount();

      for (int i = 0; i < count; ++i) {
        QMetaProperty metaproperty = metaobject->property (i);
        const char *name = metaproperty.name();
        QVariant value = property (name);
        std::cout << "property \"" <<  name << "=" << value.toString().toStdString() << "\" ";
      }

      std::cout << " added." << std::endl << "Other properties (dyn):" << std::endl;

      QList<QByteArray> dynProp = dynamicPropertyNames();

      for (int i = 0; i < dynProp.size(); ++i) {
        QVariant value = property (dynProp.at (i));
        std::cout << QString (dynProp.at (i)).toStdString() << "=" << value.toString().toStdString() << std::endl;
      }

      std::cout << "END" << std::endl;
    }
    */
}

// ---------------------- setIndex ----------------------------
void AtomDCProperties::setIndex(int id) {
    // TODO check that it is still unique (if the new id is in use, show a QMessageBox !)
    if (myDC)
        myDC->getAtom()->setIndex(id);
}

// ---------------------- getIndex ----------------------------
int AtomDCProperties::getIndex() const {
    if (myDC)
        return myDC->getAtom()->getIndex();
    else
        return -1;
}

// ---------------------- setPosition ----------------------------
void  AtomDCProperties::setPosition(QVector3D newCoord) {
    // reset the atom flag
    myDC->resetAlreadyMovedFlag();

    //std::cout << "setPosition(" << newCoord["x"].toDouble() << "," << newCoord["y"].toDouble() << "," << newCoord["z"].toDouble() << ")" << std::endl;

    // just get the different values from the map and set the new position
    myDC->setPosition(newCoord.x(), newCoord.y(), newCoord.z());

    myDC->refresh();
}

// ---------------------- getPosition ----------------------------
QVector3D AtomDCProperties::getPosition() const {
    double pos[3];
    myDC->getAtom()->getPosition(pos);

    return QVector3D(pos[0], pos[1], pos[2]);
}

// ---------------------- event ----------------------------
bool AtomDCProperties::event(QEvent * e) {
    if (e->type() == QEvent::DynamicPropertyChange) {
        e->accept();
        QDynamicPropertyChangeEvent *chev = dynamic_cast<QDynamicPropertyChangeEvent *>(e);

        if (! chev)
            return false;

        // set the corresponding property (field)
        myDC->getAtom()->getProperties()->set(QString(chev->propertyName()).toStdString(), std::string(property(chev->propertyName()).toString().toStdString()));

        return true;
    }

    return QObject::event(e);
}
