//# MeasTable.cc: MeasTable provides Measure computing database data
//# Copyright (C) 1995-1999,2000-2004,2016
//# Associated Universities, Inc. Washington DC, USA.
//#
//# This library is free software; you can redistribute it and/or modify it
//# under the terms of the GNU Library General Public License as published by
//# the Free Software Foundation; either version 2 of the License, or (at your
//# option) any later version.
//#
//# This library is distributed in the hope that it will be useful, but WITHOUT
//# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
//# License for more details.
//#
//# You should have received a copy of the GNU Library General Public License
//# along with this library; if not, write to the Free Software Foundation,
//# Inc., 675 Massachusetts Ave, Cambridge, MA 02139, USA.
//#
//# Correspondence concerning AIPS++ should be addressed as follows:
//#        Internet email: casa-feedback@nrao.edu.
//#        Postal address: AIPS++ Project Office
//#                        National Radio Astronomy Observatory
//#                        520 Edgemont Road
//#                        Charlottesville, VA 22903-2475 USA

//# Includes
#include <casacore/measures/Measures/MeasTable.h>
#include <casacore/casa/Arrays/ArrayMath.h>
#include <casacore/casa/Containers/RecordField.h>
#include <casacore/casa/Logging/LogIO.h>
#include <casacore/casa/BasicSL/Constants.h>
#include <casacore/casa/BasicMath/Math.h>
#include <casacore/measures/Measures/MPosition.h>
#include <casacore/measures/Measures/MDirection.h>
#include <casacore/measures/Measures/MeasIERS.h>
#include <casacore/measures/Measures/MeasJPL.h>
#include <casacore/casa/OS/Time.h>
#include <casacore/casa/OS/Path.h>
#include <casacore/casa/Quanta/UnitVal.h>
#include <casacore/casa/Quanta/RotMatrix.h>
#include <casacore/casa/Quanta/Euler.h>
#include <casacore/casa/Utilities/MUString.h>
#include <casacore/casa/Quanta/MVAngle.h>
#include <casacore/casa/Quanta/MVEpoch.h>
#include <casacore/tables/Tables/Table.h>
#include <casacore/tables/Tables/TableRecord.h>
#include <casacore/tables/Tables/TableRow.h>
#include <casacore/tables/Tables/ScalarColumn.h>
#include <casacore/tables/Tables/ArrayColumn.h>
#include <casacore/casa/System/Aipsrc.h>
#include <casacore/casa/System/AipsrcValue.h>
#include <casacore/casa/Utilities/Assert.h>

#include <atomic>

namespace casacore { //# NAMESPACE CASACORE - BEGIN

#ifndef CASADATA
#define CASADATA "/usr/local/share/data/casacore"
#endif


//# Constants

//# Static class Data
MeasTableMulSC         MeasTable::theirMulSC;
MeasTableMulSC1950     MeasTable::theirMulSC1950;
MeasTableMulSC2000A    MeasTable::theirMulSC2000A;
MeasTableMulSC2000B    MeasTable::theirMulSC2000B;
MeasTableMulAber       MeasTable::theirMulAber;
MeasTableMulAber1950   MeasTable::theirMulAber1950;
MeasTableMulPosSunXY   MeasTable::theirMulPosSunXY;
MeasTableMulPosSunZ    MeasTable::theirMulPosSunZ;
MeasTableMulPosEarthXY MeasTable::theirMulPosEarthXY;
MeasTableMulPosEarthZ  MeasTable::theirMulPosEarthZ;
std::once_flag MeasTable::theirPlanetaryInitOnceFlag;
std::once_flag MeasTable::theirPlanetaryConstantsInitOnceFlag;
std::once_flag MeasTable::theirObsInitOnceFlag;
Vector<String> MeasTable::obsNams;
Vector<MPosition> MeasTable::obsPos;
Vector<String> MeasTable::antResponsesPath;
std::once_flag MeasTable::theirLinesInitOnceFlag;
Vector<String> MeasTable::lineNams;
Vector<MFrequency> MeasTable::linePos;
std::once_flag MeasTable::theirSrcInitOnceFlag;
Vector<String> MeasTable::srcNams;
Vector<MDirection> MeasTable::srcPos;
std::once_flag MeasTable::theirIGRFInitOnceFlag;
Double MeasTable::dtimeIGRF = 0;
Double MeasTable::firstIGRF = 0;
std::vector<Vector<Double>> MeasTable::coefIGRF;
std::vector<Vector<Double>> MeasTable::dIGRF;
  ///#if !defined(USE_THREADS) || defined(__APPLE__)
  ///std::mutex MeasTable::theirdUT1Mutex;
  ///#endif

//# Member functions
Bool MeasTable::useIAU2000() {
  // Aipsrc registration (for speed) of use of iau2000 and if so the 2000a version.
  static const uInt iau2000_reg =
    AipsrcValue<Bool>::registerRC("measures.iau2000.b_use", False);
  return AipsrcValue<Bool>::get(iau2000_reg);
}

Bool MeasTable::useIAU2000A() {
  // comment as above in useIAU2000()
  static const uInt iau2000a_reg =
    AipsrcValue<Bool>::registerRC("measures.iau2000.b_use2000a", False);
  return AipsrcValue<Bool>::get(iau2000a_reg);
}

Double MeasTable::precRate00(const uInt which) {
  static const Double preoblcor[3] = { -0.29965*C::arcsec,
				       -0.02524*C::arcsec,
				       0*C::arcsec };
  DebugAssert(which < 3, AipsError);
  return preoblcor[which];
}

RotMatrix MeasTable::frameBias00() {
  static const Double bias[3] = { -0.041775*C::arcsec,
				  -0.0068192*C::arcsec,
				  -0.0146*C::arcsec};
  static const RotMatrix rbias (Euler(bias[2], 3,
                                bias[0]*sin(MeasData::eps0J2000()),
                                2, -bias[1], 1));
  return rbias;
}

void MeasTable::precessionCoef(Double T, Polynomial<Double> result[3]) {
  static const Double PCOEF[3][6] = {
    {+2306.2181,+1.39656,-0.000139,+0.30188,-0.000344,+0.017998},
    {+2004.3109,-0.85330,-0.000217,-0.42665,-0.000217,-0.041833},
    {+2306.2181,+1.39656,-0.000139,+1.09468,-0.000066,+0.018203}
  };
  calcPrecesCoef(T, result, &PCOEF[0]);
}

void MeasTable::precessionCoef2000(Polynomial<Double> result[3]) {
  static const Double PCOEF[3][6] = {
    { 2.5976176,2306.0809506, 0.3019015, 0.0179663,-0.0000327,-0.0000002},
    { 0.0,      2004.1917476,-0.4269353,-0.0418251,-0.0000601,-0.0000001},
    {-2.5976176,2306.0803226, 1.0947790, 0.0182273,-0.0000470,-0.0000003}
  };
  calcPrecesCoef2000(result, &PCOEF[0]);
}

void MeasTable::precessionCoef1950(Double T, Polynomial<Double> result[3]) {
  static const Double PCOEF[3][6] = {
    {2303.5545,+1.39720,0.000060,+0.30240,-0.000270,+0.017995},
    {2005.1120,-0.85290,-0.00037,-0.42650,-0.000370,-0.041800},
    {2303.5545,+1.39720,0.000060,+1.09480,+0.000390,+0.018325}
  };
  calcPrecesCoef(T, result, &PCOEF[0]);
}

void MeasTable::calcPrecesCoef(Double T, Polynomial<Double> result[3],
			       const Double coef[3][6]) {
  Int l; Int m=1;
  for (uInt i=0; i<3; i++) {
    m = -m;
    l = 0;
    for (uInt j=0; j<3; j++) {
      Polynomial<Double> poly(2-j);
      for (uInt k=0; k<3-j; k++, l++) {
	poly.setCoefficient(k,coef[i][l]);
      }
      result[i].setCoefficient(j+1,m*poly(T) * C::arcsec);
    }
  }
}

void MeasTable::calcPrecesCoef2000(Polynomial<Double> result[3],
				   const Double coef[3][6]) {
  Int m=1;
  for (uInt i=0; i<3; i++) {
    m = -m;
    for (uInt j=0; j<6; j++) {
      result[i].setCoefficient(j, m*coef[i][j] * C::arcsec);
    }
  }
}

const Polynomial<Double> &MeasTable::fundArg(uInt which) {
  static const Double FUND[6][4] = {
    {  84381.448,        -46.8150,-0.0059, 0.001813}, 
    { 485866.733, 1717915922.633, 31.310,  0.064}, 
    {1287099.804,  129596581.224, -0.577, -0.012}, 
    { 335778.877, 1739527263.137,-13.257,  0.011}, 
    {1072261.307, 1602961601.328, -6.891,  0.019}, 
    { 450160.280,   -6962890.539,  7.455,  0.008}
  };
  static const std::vector<Polynomial<Double>> polyArray(calcFundArg(&FUND[0]));
  DebugAssert(which < 6, AipsError);
  return polyArray[which];
}

const Polynomial<Double> &MeasTable::fundArg1950(uInt which) {
  static const Double FUND[6][4] = {
    {  84428.26,        -46.846,-0.0059, 0.00181},
    {1065976.59, 1717915856.79, 33.09,   0.0518},
    { 1290513.0,  129596579.1,  -0.54,  -0.0120},
    {   40503.2, 1739527290.54,-11.56,  -0.0012},
    { 1262654.95,1602961611.18, -5.17,   0.0068},
    {  933059.79,  -6962911.23,  7.48,   0.0080}
  };
  static const std::vector<Polynomial<Double>> polyArray(calcFundArg(&FUND[0]));
  DebugAssert(which < 6, AipsError);
  return polyArray[which];
}

const Polynomial<Double> &MeasTable::fundArg2000(uInt which) {
  static const Double FUND[6][5] = {
    {  84381.448,    -46.8150-0.02524, -0.0059,  0.001813,  0.0},
    { 485868.249036, 1717915923.2178,  31.8792,  0.051635, -0.00024470},
    {1287104.79305,   129596581.0481,  -0.5532,  0.000136, -0.00001149},
    { 335779.526232, 1739527262.8478, -12.7512, -0.001037,  0.00000417},
    {1072260.70369,  1602961601.2090, - 6.3706,  0.006593, -0.00003169},
    { 450160.398036,   -6962890.5431,   7.4722,  0.007702, -0.00005939}
  };
  static const std::vector<Polynomial<Double>> polyArray(calcFundArg00(&FUND[0]));
  DebugAssert(which < 6, AipsError);
  return polyArray[which];
}

const Polynomial<Double> &MeasTable::planetaryArg2000(uInt which) {
  static const Double FUND[8][2] = {
    { 4.402608842, 2608.7903141574 },
    { 3.176146697, 1021.3285546211 },
    { 1.753470314,  628.3075849991 },
    { 6.203480913,  334.0612426700 },
    { 0.599546497,   52.9690962641 },
    { 0.874016757,   21.3299104960 },
    { 5.481293871,    7.4781598567 },
    { 5.321159000,    3.8127774000 }
  };
  static const std::vector<Polynomial<Double>> polyArray(calcPlanArg00(&FUND[0]));
  DebugAssert(which < 14, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcFundArg(const Double coeff[6][4]) {
  std::vector<Polynomial<Double>> result(6);
  Int i,j;
  for (i=0; i<6; i++) {
    result[i] = Polynomial<Double>(3);
    for (j=0; j<4; j++) {
      result[i].setCoefficient(j, coeff[i][j]*C::arcsec);
    }
  }
  return result;
}    

std::vector<Polynomial<Double>> MeasTable::calcFundArg00(const Double coeff[6][5]) {
  std::vector<Polynomial<Double>> result(6);
  Int i,j;
  for (i=0; i<6; i++) {
    result[i] = Polynomial<Double>(4);
    for (j=0; j<5; j++) {
      result[i].setCoefficient(j, coeff[i][j]*C::arcsec);
    }
  }
  return result;
}    

std::vector<Polynomial<Double>> MeasTable::calcPlanArg00(const Double coeff[8][2]) {
  static const Double APA[3] = { 0.0, 0.02438175, 0.00000538691 };
  std::vector<Polynomial<Double>> result(14);
  for (uInt i=0; i<5; i++) {
    result[i] = fundArg2000(i+1);
  }
  for (uInt i=5; i<13; i++) {
    result[i] = Polynomial<Double>(1);
    for (uInt j=0; j<2; j++) {
      result[i].setCoefficient(j, coeff[i-5][j]);
    }
  }
  result[13] = Polynomial<Double>(2);
  for (uInt j=0; j<3; j++) {
    result[13].setCoefficient(j, APA[j]);
  }
  return result;
}    

const Double* MeasTable::mulArg(uInt which) {
  static const Double ARG[106][5] = {
    {0	,0	,0	,0	,1	},
    {0	,0	,0	,0	,2	},
    {-2	,0	,2	,0	,1	},
    {2	,0	,-2	,0	,0	},
    {-2	,0	,2	,0	,2	},
    
    {1	,-1	,0	,-1	,0	},
    {0	,-2	,2	,-2	,1	},
    {2	,0	,-2	,0	,1	},
    {0	,0	,2	,-2	,2	},
    {0	,1	,0	,0	,0	},
    
    {0	,1	,2	,-2	,2	},
    {0	,-1	,2	,-2	,2	},
    {0	,0	,2	,-2	,1	},
    {2	,0	,0	,-2	,0	},
    {0	,0	,2	,-2	,0	},
    
    {0	,2	,0	,0	,0	},
    {0	,1	,0	,0	,1	},
    {0	,2	,2	,-2	,2	},
    {0	,-1	,0	,0	,1	},
    {-2	,0	,0	,2	,1	},
    
    {0	,-1	,2	,-2	,1	},
    {2	,0	,0	,-2	,1	},
    {0	,1	,2	,-2	,1	},
    {1	,0	,0	,-1	,0	},
    {2	,1	,0	,-2	,0	},
    
    {0	,0	,-2	,2	,1	},
    {0	,1	,-2	,2	,0	},
    {0	,1	,0	,0	,2	},
    {-1	,0	,0	,1	,1	},
    {0	,1	,2	,-2	,0	},
    
    {0	,0	,2	,0	,2	},
    {1	,0	,0	,0	,0	},
    {0	,0	,2	,0	,1	},
    {1	,0	,2	,0	,2	},
    {1	,0	,0	,-2	,0	},
    
    {-1	,0	,2	,0	,2	},
    {0	,0	,0	,2	,0	},
    {1	,0	,0	,0	,1	},
    {-1	,0	,0	,0	,1	},
    {-1	,0	,2	,2	,2	},
    
    {1	,0	,2	,0	,1	},
    {0	,0	,2	,2	,2	},
    {2	,0	,0	,0	,0	},
    {1	,0	,2	,-2	,2	},
    {2	,0	,2	,0	,2	},
    
    {0	,0	,2	,0	,0	},
    {-1	,0	,2	,0	,1	},
    {-1	,0	,0	,2	,1	},
    {1	,0	,0	,-2	,1	},
    {-1	,0	,2	,2	,1	},
    
    {1	,1	,0	,-2	,0	},
    {0	,1	,2	,0	,2	},
    {0	,-1	,2	,0	,2	},
    {1	,0	,2	,2	,2	},
    {1	,0	,0	,2	,0	},
    
    {2	,0	,2	,-2	,2	},
    {0	,0	,0	,2	,1	},
    {0	,0	,2	,2	,1	},
    {1	,0	,2	,-2	,1	},
    {0	,0	,0	,-2	,1	},
    
    {1	,-1	,0	,0	,0	},
    {2	,0	,2	,0	,1	},
    {0	,1	,0	,-2	,0	},
    {1	,0	,-2	,0	,0	},
    {0	,0	,0	,1	,0	},
    
    {1	,1	,0	,0	,0	},
    {1	,0	,2	,0	,0	},
    {1	,-1	,2	,0	,2	},
    {-1	,-1	,2	,2	,2	},
    {-2	,0	,0	,0	,1	},
    
    {3	,0	,2	,0	,2	},
    {0	,-1	,2	,2	,2	},
    {1	,1	,2	,0	,2	},
    {-1	,0	,2	,-2	,1	},
    {2	,0	,0	,0	,1	},
    
    {1	,0	,0	,0	,2	},
    {3	,0	,0	,0	,0	},
    {0	,0	,2	,1	,2	},
    {-1	,0	,0	,0	,2	},
    {1	,0	,0	,-4	,0	},
    
    {-2	,0	,2	,2	,2	},
    {-1	,0	,2	,4	,2	},
    {2	,0	,0	,-4	,0	},
    {1	,1	,2	,-2	,2	},
    {1	,0	,2	,2	,1	},
    
    {-2	,0	,2	,4	,2	},
    {-1	,0	,4	,0	,2	},
    {1	,-1	,0	,-2	,0	},
    {2	,0	,2	,-2	,1	},
    {2	,0	,2	,2	,2	},
    
    {1	,0	,0	,2	,1	},
    {0	,0	,4	,-2	,2	},
    {3	,0	,2	,-2	,2	},
    {1	,0	,2	,-2	,0	},
    {0	,1	,2	,0	,1	},
    
    {-1	,-1	,0	,2	,1	},
    {0	,0	,-2	,0	,1	},
    {0	,0	,2	,-1	,2	},
    {0	,1	,0	,2	,0	},
    {1	,0	,-2	,-2	,0	},
    
    {0	,-1	,2	,0	,1	},
    {1	,1	,0	,-2	,1	},
    {1	,0	,-2	,2	,0	},
    {2	,0	,0	,2	,0	},
    {0	,0	,2	,4	,2	},
    
    {0	,1	,0	,1	,0	}
  };
  DebugAssert(which < 106, AipsError);
  return &(ARG[which][0]);
}

const Double* MeasTable::mulArg2000A(uInt which) {
  static const Double ARG[678][5] = {
  //         Multiple of        
  //    L     L'    F     D   Omega  
     {  0,    0,    0,    0,    1},   //    1
     {  0,    0,    2,   -2,    2},   //    2
     {  0,    0,    2,    0,    2},   //    3
     {  0,    0,    0,    0,    2},   //    4
     {  0,    1,    0,    0,    0},   //    5
     {  0,    1,    2,   -2,    2},   //    6
     {  1,    0,    0,    0,    0},   //    7
     {  0,    0,    2,    0,    1},   //    8
     {  1,    0,    2,    0,    2},   //    9
     {  0,   -1,    2,   -2,    2},   //   10
     {  0,    0,    2,   -2,    1},   //   11
     { -1,    0,    2,    0,    2},   //   12
     { -1,    0,    0,    2,    0},   //   13
     {  1,    0,    0,    0,    1},   //   14
     { -1,    0,    0,    0,    1},   //   15
     { -1,    0,    2,    2,    2},   //   16
     {  1,    0,    2,    0,    1},   //   17
     { -2,    0,    2,    0,    1},   //   18
     {  0,    0,    0,    2,    0},   //   19
     {  0,    0,    2,    2,    2},   //   20
     {  0,   -2,    2,   -2,    2},   //   21
     { -2,    0,    0,    2,    0},   //   22
     {  2,    0,    2,    0,    2},   //   23
     {  1,    0,    2,   -2,    2},   //   24
     { -1,    0,    2,    0,    1},   //   25
     {  2,    0,    0,    0,    0},   //   26
     {  0,    0,    2,    0,    0},   //   27
     {  0,    1,    0,    0,    1},   //   28
     { -1,    0,    0,    2,    1},   //   29
     {  0,    2,    2,   -2,    2},   //   30
     {  0,    0,   -2,    2,    0},   //   31
     {  1,    0,    0,   -2,    1},   //   32
     {  0,   -1,    0,    0,    1},   //   33
     { -1,    0,    2,    2,    1},   //   34
     {  0,    2,    0,    0,    0},   //   35
     {  1,    0,    2,    2,    2},   //   36
     { -2,    0,    2,    0,    0},   //   37
     {  0,    1,    2,    0,    2},   //   38
     {  0,    0,    2,    2,    1},   //   39
     {  0,   -1,    2,    0,    2},   //   40
     {  0,    0,    0,    2,    1},   //   41
     {  1,    0,    2,   -2,    1},   //   42
     {  2,    0,    2,   -2,    2},   //   43
     { -2,    0,    0,    2,    1},   //   44
     {  2,    0,    2,    0,    1},   //   45
     {  0,   -1,    2,   -2,    1},   //   46
     {  0,    0,    0,   -2,    1},   //   47
     { -1,   -1,    0,    2,    0},   //   48
     {  2,    0,    0,   -2,    1},   //   49
     {  1,    0,    0,    2,    0},   //   50
     {  0,    1,    2,   -2,    1},   //   51
     {  1,   -1,    0,    0,    0},   //   52
     { -2,    0,    2,    0,    2},   //   53
     {  3,    0,    2,    0,    2},   //   54
     {  0,   -1,    0,    2,    0},   //   55
     {  1,   -1,    2,    0,    2},   //   56
     {  0,    0,    0,    1,    0},   //   57
     { -1,   -1,    2,    2,    2},   //   58
     { -1,    0,    2,    0,    0},   //   59
     {  0,   -1,    2,    2,    2},   //   60
     { -2,    0,    0,    0,    1},   //   61
     {  1,    1,    2,    0,    2},   //   62
     {  2,    0,    0,    0,    1},   //   63
     { -1,    1,    0,    1,    0},   //   64
     {  1,    1,    0,    0,    0},   //   65
     {  1,    0,    2,    0,    0},   //   66
     { -1,    0,    2,   -2,    1},   //   67
     {  1,    0,    0,    0,    2},   //   68
     { -1,    0,    0,    1,    0},   //   69
     {  0,    0,    2,    1,    2},   //   70
     { -1,    0,    2,    4,    2},   //   71
     { -1,    1,    0,    1,    1},   //   72
     {  0,   -2,    2,   -2,    1},   //   73
     {  1,    0,    2,    2,    1},   //   74
     { -2,    0,    2,    2,    2},   //   75
     { -1,    0,    0,    0,    2},   //   76
     {  1,    1,    2,   -2,    2},   //   77
     { -2,    0,    2,    4,    2},   //   78
     { -1,    0,    4,    0,    2},   //   79
     {  2,    0,    2,   -2,    1},   //   80
     {  2,    0,    2,    2,    2},   //   81
     {  1,    0,    0,    2,    1},   //   82
     {  3,    0,    0,    0,    0},   //   83
     {  3,    0,    2,   -2,    2},   //   84
     {  0,    0,    4,   -2,    2},   //   85
     {  0,    1,    2,    0,    1},   //   86
     {  0,    0,   -2,    2,    1},   //   87
     {  0,    0,    2,   -2,    3},   //   88
     { -1,    0,    0,    4,    0},   //   89
     {  2,    0,   -2,    0,    1},   //   90
     { -2,    0,    0,    4,    0},   //   91
     { -1,   -1,    0,    2,    1},   //   92
     { -1,    0,    0,    1,    1},   //   93
     {  0,    1,    0,    0,    2},   //   94
     {  0,    0,   -2,    0,    1},   //   95
     {  0,   -1,    2,    0,    1},   //   96
     {  0,    0,    2,   -1,    2},   //   97
     {  0,    0,    2,    4,    2},   //   98
     { -2,   -1,    0,    2,    0},   //   99
     {  1,    1,    0,   -2,    1},   //  100
     { -1,    1,    0,    2,    0},   //  101
     { -1,    1,    0,    1,    2},   //  102
     {  1,   -1,    0,    0,    1},   //  103
     {  1,   -1,    2,    2,    2},   //  104
     { -1,    1,    2,    2,    2},   //  105
     {  3,    0,    2,    0,    1},   //  106
     {  0,    1,   -2,    2,    0},   //  107
     { -1,    0,    0,   -2,    1},   //  108
     {  0,    1,    2,    2,    2},   //  109
     { -1,   -1,    2,    2,    1},   //  110
     {  0,   -1,    0,    0,    2},   //  111
     {  1,    0,    2,   -4,    1},   //  112
     { -1,    0,   -2,    2,    0},   //  113
     {  0,   -1,    2,    2,    1},   //  114
     {  2,   -1,    2,    0,    2},   //  115
     {  0,    0,    0,    2,    2},   //  116
     {  1,   -1,    2,    0,    1},   //  117
     { -1,    1,    2,    0,    2},   //  118
     {  0,    1,    0,    2,    0},   //  119
     {  0,   -1,   -2,    2,    0},   //  120
     {  0,    3,    2,   -2,    2},   //  121
     {  0,    0,    0,    1,    1},   //  122
     { -1,    0,    2,    2,    0},   //  123
     {  2,    1,    2,    0,    2},   //  124
     {  1,    1,    0,    0,    1},   //  125
     {  1,    1,    2,    0,    1},   //  126
     {  2,    0,    0,    2,    0},   //  127
     {  1,    0,   -2,    2,    0},   //  128
     { -1,    0,    0,    2,    2},   //  129
     {  0,    1,    0,    1,    0},   //  130
     {  0,    1,    0,   -2,    1},   //  131
     { -1,    0,    2,   -2,    2},   //  132
     {  0,    0,    0,   -1,    1},   //  133
     { -1,    1,    0,    0,    1},   //  134
     {  1,    0,    2,   -1,    2},   //  135
     {  1,   -1,    0,    2,    0},   //  136
     {  0,    0,    0,    4,    0},   //  137
     {  1,    0,    2,    1,    2},   //  138
     {  0,    0,    2,    1,    1},   //  139
     {  1,    0,    0,   -2,    2},   //  140
     { -1,    0,    2,    4,    1},   //  141
     {  1,    0,   -2,    0,    1},   //  142
     {  1,    1,    2,   -2,    1},   //  143
     {  0,    0,    2,    2,    0},   //  144
     { -1,    0,    2,   -1,    1},   //  145
     { -2,    0,    2,    2,    1},   //  146
     {  4,    0,    2,    0,    2},   //  147
     {  2,   -1,    0,    0,    0},   //  148
     {  2,    1,    2,   -2,    2},   //  149
     {  0,    1,    2,    1,    2},   //  150
     {  1,    0,    4,   -2,    2},   //  151
     { -1,   -1,    0,    0,    1},   //  152
     {  0,    1,    0,    2,    1},   //  153
     { -2,    0,    2,    4,    1},   //  154
     {  2,    0,    2,    0,    0},   //  155
     {  1,    0,    0,    1,    0},   //  156
     { -1,    0,    0,    4,    1},   //  157
     { -1,    0,    4,    0,    1},   //  158
     {  2,    0,    2,    2,    1},   //  159
     {  0,    0,    2,   -3,    2},   //  160
     { -1,   -2,    0,    2,    0},   //  161
     {  2,    1,    0,    0,    0},   //  162
     {  0,    0,    4,    0,    2},   //  163
     {  0,    0,    0,    0,    3},   //  164
     {  0,    3,    0,    0,    0},   //  165
     {  0,    0,    2,   -4,    1},   //  166
     {  0,   -1,    0,    2,    1},   //  167
     {  0,    0,    0,    4,    1},   //  168
     { -1,   -1,    2,    4,    2},   //  169
     {  1,    0,    2,    4,    2},   //  170
     { -2,    2,    0,    2,    0},   //  171
     { -2,   -1,    2,    0,    1},   //  172
     { -2,    0,    0,    2,    2},   //  173
     { -1,   -1,    2,    0,    2},   //  174
     {  0,    0,    4,   -2,    1},   //  175
     {  3,    0,    2,   -2,    1},   //  176
     { -2,   -1,    0,    2,    1},   //  177
     {  1,    0,    0,   -1,    1},   //  178
     {  0,   -2,    0,    2,    0},   //  179
     { -2,    0,    0,    4,    1},   //  180
     { -3,    0,    0,    0,    1},   //  181
     {  1,    1,    2,    2,    2},   //  182
     {  0,    0,    2,    4,    1},   //  183
     {  3,    0,    2,    2,    2},   //  184
     { -1,    1,    2,   -2,    1},   //  185
     {  2,    0,    0,   -4,    1},   //  186
     {  0,    0,    0,   -2,    2},   //  187
     {  2,    0,    2,   -4,    1},   //  188
     { -1,    1,    0,    2,    1},   //  189
     {  0,    0,    2,   -1,    1},   //  190
     {  0,   -2,    2,    2,    2},   //  191
     {  2,    0,    0,    2,    1},   //  192
     {  4,    0,    2,   -2,    2},   //  193
     {  2,    0,    0,   -2,    2},   //  194
     {  0,    2,    0,    0,    1},   //  195
     {  1,    0,    0,   -4,    1},   //  196
     {  0,    2,    2,   -2,    1},   //  197
     { -3,    0,    0,    4,    0},   //  198
     { -1,    1,    2,    0,    1},   //  199
     { -1,   -1,    0,    4,    0},   //  200
     { -1,   -2,    2,    2,    2},   //  201
     { -2,   -1,    2,    4,    2},   //  202
     {  1,   -1,    2,    2,    1},   //  203
     { -2,    1,    0,    2,    0},   //  204
     { -2,    1,    2,    0,    1},   //  205
     {  2,    1,    0,   -2,    1},   //  206
     { -3,    0,    2,    0,    1},   //  207
     { -2,    0,    2,   -2,    1},   //  208
     { -1,    1,    0,    2,    2},   //  209
     {  0,   -1,    2,   -1,    2},   //  210
     { -1,    0,    4,   -2,    2},   //  211
     {  0,   -2,    2,    0,    2},   //  212
     { -1,    0,    2,    1,    2},   //  213
     {  2,    0,    0,    0,    2},   //  214
     {  0,    0,    2,    0,    3},   //  215
     { -2,    0,    4,    0,    2},   //  216
     { -1,    0,   -2,    0,    1},   //  217
     { -1,    1,    2,    2,    1},   //  218
     {  3,    0,    0,    0,    1},   //  219
     { -1,    0,    2,    3,    2},   //  220
     {  2,   -1,    2,    0,    1},   //  221
     {  0,    1,    2,    2,    1},   //  222
     {  0,   -1,    2,    4,    2},   //  223
     {  2,   -1,    2,    2,    2},   //  224
     {  0,    2,   -2,    2,    0},   //  225
     { -1,   -1,    2,   -1,    1},   //  226
     {  0,   -2,    0,    0,    1},   //  227
     {  1,    0,    2,   -4,    2},   //  228
     {  1,   -1,    0,   -2,    1},   //  229
     { -1,   -1,    2,    0,    1},   //  230
     {  1,   -1,    2,   -2,    2},   //  231
     { -2,   -1,    0,    4,    0},   //  232
     { -1,    0,    0,    3,    0},   //  233
     { -2,   -1,    2,    2,    2},   //  234
     {  0,    2,    2,    0,    2},   //  235
     {  1,    1,    0,    2,    0},   //  236
     {  2,    0,    2,   -1,    2},   //  237
     {  1,    0,    2,    1,    1},   //  238
     {  4,    0,    0,    0,    0},   //  239
     {  2,    1,    2,    0,    1},   //  240
     {  3,   -1,    2,    0,    2},   //  241
     { -2,    2,    0,    2,    1},   //  242
     {  1,    0,    2,   -3,    1},   //  243
     {  1,    1,    2,   -4,    1},   //  244
     { -1,   -1,    2,   -2,    1},   //  245
     {  0,   -1,    0,   -1,    1},   //  246
     {  0,   -1,    0,   -2,    1},   //  247
     { -2,    0,    0,    0,    2},   //  248
     { -2,    0,   -2,    2,    0},   //  249
     { -1,    0,   -2,    4,    0},   //  250
     {  1,   -2,    0,    0,    0},   //  251
     {  0,    1,    0,    1,    1},   //  252
     { -1,    2,    0,    2,    0},   //  253
     {  1,   -1,    2,   -2,    1},   //  254
     {  1,    2,    2,   -2,    2},   //  255
     {  2,   -1,    2,   -2,    2},   //  256
     {  1,    0,    2,   -1,    1},   //  257
     {  2,    1,    2,   -2,    1},   //  258
     { -2,    0,    0,   -2,    1},   //  259
     {  1,   -2,    2,    0,    2},   //  260
     {  0,    1,    2,    1,    1},   //  261
     {  1,    0,    4,   -2,    1},   //  262
     { -2,    0,    4,    2,    2},   //  263
     {  1,    1,    2,    1,    2},   //  264
     {  1,    0,    0,    4,    0},   //  265
     {  1,    0,    2,    2,    0},   //  266
     {  2,    0,    2,    1,    2},   //  267
     {  3,    1,    2,    0,    2},   //  268
     {  4,    0,    2,    0,    1},   //  269
     { -2,   -1,    2,    0,    0},   //  270
     {  0,    1,   -2,    2,    1},   //  271
     {  1,    0,   -2,    1,    0},   //  272
     {  0,   -1,   -2,    2,    1},   //  273
     {  2,   -1,    0,   -2,    1},   //  274
     { -1,    0,    2,   -1,    2},   //  275
     {  1,    0,    2,   -3,    2},   //  276
     {  0,    1,    2,   -2,    3},   //  277
     {  0,    0,    2,   -3,    1},   //  278
     { -1,    0,   -2,    2,    1},   //  279
     {  0,    0,    2,   -4,    2},   //  280
     { -2,    1,    0,    0,    1},   //  281
     { -1,    0,    0,   -1,    1},   //  282
     {  2,    0,    2,   -4,    2},   //  283
     {  0,    0,    4,   -4,    4},   //  284
     {  0,    0,    4,   -4,    2},   //  285
     { -1,   -2,    0,    2,    1},   //  286
     { -2,    0,    0,    3,    0},   //  287
     {  1,    0,   -2,    2,    1},   //  288
     { -3,    0,    2,    2,    2},   //  289
     { -3,    0,    2,    2,    1},   //  290
     { -2,    0,    2,    2,    0},   //  291
     {  2,   -1,    0,    0,    1},   //  292
     { -2,    1,    2,    2,    2},   //  293
     {  1,    1,    0,    1,    0},   //  294
     {  0,    1,    4,   -2,    2},   //  295
     { -1,    1,    0,   -2,    1},   //  296
     {  0,    0,    0,   -4,    1},   //  297
     {  1,   -1,    0,    2,    1},   //  298
     {  1,    1,    0,    2,    1},   //  299
     { -1,    2,    2,    2,    2},   //  300
     {  3,    1,    2,   -2,    2},   //  301
     {  0,   -1,    0,    4,    0},   //  302
     {  2,   -1,    0,    2,    0},   //  303
     {  0,    0,    4,    0,    1},   //  304
     {  2,    0,    4,   -2,    2},   //  305
     { -1,   -1,    2,    4,    1},   //  306
     {  1,    0,    0,    4,    1},   //  307
     {  1,   -2,    2,    2,    2},   //  308
     {  0,    0,    2,    3,    2},   //  309
     { -1,    1,    2,    4,    2},   //  310
     {  3,    0,    0,    2,    0},   //  311
     { -1,    0,    4,    2,    2},   //  312
     {  1,    1,    2,    2,    1},   //  313
     { -2,    0,    2,    6,    2},   //  314
     {  2,    1,    2,    2,    2},   //  315
     { -1,    0,    2,    6,    2},   //  316
     {  1,    0,    2,    4,    1},   //  317
     {  2,    0,    2,    4,    2},   //  318
     {  1,    1,   -2,    1,    0},   //  319
     { -3,    1,    2,    1,    2},   //  320
     {  2,    0,   -2,    0,    2},   //  321
     { -1,    0,    0,    1,    2},   //  322
     { -4,    0,    2,    2,    1},   //  323
     { -1,   -1,    0,    1,    0},   //  324
     {  0,    0,   -2,    2,    2},   //  325
     {  1,    0,    0,   -1,    2},   //  326
     {  0,   -1,    2,   -2,    3},   //  327
     { -2,    1,    2,    0,    0},   //  328
     {  0,    0,    2,   -2,    4},   //  329
     { -2,   -2,    0,    2,    0},   //  330
     { -2,    0,   -2,    4,    0},   //  331
     {  0,   -2,   -2,    2,    0},   //  332
     {  1,    2,    0,   -2,    1},   //  333
     {  3,    0,    0,   -4,    1},   //  334
     { -1,    1,    2,   -2,    2},   //  335
     {  1,   -1,    2,   -4,    1},   //  336
     {  1,    1,    0,   -2,    2},   //  337
     { -3,    0,    2,    0,    0},   //  338
     { -3,    0,    2,    0,    2},   //  339
     { -2,    0,    0,    1,    0},   //  340
     {  0,    0,   -2,    1,    0},   //  341
     { -3,    0,    0,    2,    1},   //  342
     { -1,   -1,   -2,    2,    0},   //  343
     {  0,    1,    2,   -4,    1},   //  344
     {  2,    1,    0,   -4,    1},   //  345
     {  0,    2,    0,   -2,    1},   //  346
     {  1,    0,    0,   -3,    1},   //  347
     { -2,    0,    2,   -2,    2},   //  348
     { -2,   -1,    0,    0,    1},   //  349
     { -4,    0,    0,    2,    0},   //  350
     {  1,    1,    0,   -4,    1},   //  351
     { -1,    0,    2,   -4,    1},   //  352
     {  0,    0,    4,   -4,    1},   //  353
     {  0,    3,    2,   -2,    2},   //  354
     { -3,   -1,    0,    4,    0},   //  355
     { -3,    0,    0,    4,    1},   //  356
     {  1,   -1,   -2,    2,    0},   //  357
     { -1,   -1,    0,    2,    2},   //  358
     {  1,   -2,    0,    0,    1},   //  359
     {  1,   -1,    0,    0,    2},   //  360
     {  0,    0,    0,    1,    2},   //  361
     { -1,   -1,    2,    0,    0},   //  362
     {  1,   -2,    2,   -2,    2},   //  363
     {  0,   -1,    2,   -1,    1},   //  364
     { -1,    0,    2,    0,    3},   //  365
     {  1,    1,    0,    0,    2},   //  366
     { -1,    1,    2,    0,    0},   //  367
     {  1,    2,    0,    0,    0},   //  368
     { -1,    2,    2,    0,    2},   //  369
     { -1,    0,    4,   -2,    1},   //  370
     {  3,    0,    2,   -4,    2},   //  371
     {  1,    2,    2,   -2,    1},   //  372
     {  1,    0,    4,   -4,    2},   //  373
     { -2,   -1,    0,    4,    1},   //  374
     {  0,   -1,    0,    2,    2},   //  375
     { -2,    1,    0,    4,    0},   //  376
     { -2,   -1,    2,    2,    1},   //  377
     {  2,    0,   -2,    2,    0},   //  378
     {  1,    0,    0,    1,    1},   //  379
     {  0,    1,    0,    2,    2},   //  380
     {  1,   -1,    2,   -1,    2},   //  381
     { -2,    0,    4,    0,    1},   //  382
     {  2,    1,    0,    0,    1},   //  383
     {  0,    1,    2,    0,    0},   //  384
     {  0,   -1,    4,   -2,    2},   //  385
     {  0,    0,    4,   -2,    4},   //  386
     {  0,    2,    2,    0,    1},   //  387
     { -3,    0,    0,    6,    0},   //  388
     { -1,   -1,    0,    4,    1},   //  389
     {  1,   -2,    0,    2,    0},   //  390
     { -1,    0,    0,    4,    2},   //  391
     { -1,   -2,    2,    2,    1},   //  392
     { -1,    0,    0,   -2,    2},   //  393
     {  1,    0,   -2,   -2,    1},   //  394
     {  0,    0,   -2,   -2,    1},   //  395
     { -2,    0,   -2,    0,    1},   //  396
     {  0,    0,    0,    3,    1},   //  397
     {  0,    0,    0,    3,    0},   //  398
     { -1,    1,    0,    4,    0},   //  399
     { -1,   -1,    2,    2,    0},   //  400
     { -2,    0,    2,    3,    2},   //  401
     {  1,    0,    0,    2,    2},   //  402
     {  0,   -1,    2,    1,    2},   //  403
     {  3,   -1,    0,    0,    0},   //  404
     {  2,    0,    0,    1,    0},   //  405
     {  1,   -1,    2,    0,    0},   //  406
     {  0,    0,    2,    1,    0},   //  407
     {  1,    0,    2,    0,    3},   //  408
     {  3,    1,    0,    0,    0},   //  409
     {  3,   -1,    2,   -2,    2},   //  410
     {  2,    0,    2,   -1,    1},   //  411
     {  1,    1,    2,    0,    0},   //  412
     {  0,    0,    4,   -1,    2},   //  413
     {  1,    2,    2,    0,    2},   //  414
     { -2,    0,    0,    6,    0},   //  415
     {  0,   -1,    0,    4,    1},   //  416
     { -2,   -1,    2,    4,    1},   //  417
     {  0,   -2,    2,    2,    1},   //  418
     {  0,   -1,    2,    2,    0},   //  419
     { -1,    0,    2,    3,    1},   //  420
     { -2,    1,    2,    4,    2},   //  421
     {  2,    0,    0,    2,    2},   //  422
     {  2,   -2,    2,    0,    2},   //  423
     { -1,    1,    2,    3,    2},   //  424
     {  3,    0,    2,   -1,    2},   //  425
     {  4,    0,    2,   -2,    1},   //  426
     { -1,    0,    0,    6,    0},   //  427
     { -1,   -2,    2,    4,    2},   //  428
     { -3,    0,    2,    6,    2},   //  429
     { -1,    0,    2,    4,    0},   //  430
     {  3,    0,    0,    2,    1},   //  431
     {  3,   -1,    2,    0,    1},   //  432
     {  3,    0,    2,    0,    0},   //  433
     {  1,    0,    4,    0,    2},   //  434
     {  5,    0,    2,   -2,    2},   //  435
     {  0,   -1,    2,    4,    1},   //  436
     {  2,   -1,    2,    2,    1},   //  437
     {  0,    1,    2,    4,    2},   //  438
     {  1,   -1,    2,    4,    2},   //  439
     {  3,   -1,    2,    2,    2},   //  440
     {  3,    0,    2,    2,    1},   //  441
     {  5,    0,    2,    0,    2},   //  442
     {  0,    0,    2,    6,    2},   //  443
     {  4,    0,    2,    2,    2},   //  444
     {  0,   -1,    1,   -1,    1},   //  445
     { -1,    0,    1,    0,    3},   //  446
     {  0,   -2,    2,   -2,    3},   //  447
     {  1,    0,   -1,    0,    1},   //  448
     {  2,   -2,    0,   -2,    1},   //  449
     { -1,    0,    1,    0,    2},   //  450
     { -1,    0,    1,    0,    1},   //  451
     { -1,   -1,    2,   -1,    2},   //  452
     { -2,    2,    0,    2,    2},   //  453
     { -1,    0,    1,    0,    0},   //  454
     { -4,    1,    2,    2,    2},   //  455
     { -3,    0,    2,    1,    1},   //  456
     { -2,   -1,    2,    0,    2},   //  457
     {  1,    0,   -2,    1,    1},   //  458
     {  2,   -1,   -2,    0,    1},   //  459
     { -4,    0,    2,    2,    0},   //  460
     { -3,    1,    0,    3,    0},   //  461
     { -1,    0,   -1,    2,    0},   //  462
     {  0,   -2,    0,    0,    2},   //  463
     {  0,   -2,    0,    0,    2},   //  464
     { -3,    0,    0,    3,    0},   //  465
     { -2,   -1,    0,    2,    2},   //  466
     { -1,    0,   -2,    3,    0},   //  467
     { -4,    0,    0,    4,    0},   //  468
     {  2,    1,   -2,    0,    1},   //  469
     {  2,   -1,    0,   -2,    2},   //  470
     {  0,    0,    1,   -1,    0},   //  471
     { -1,    2,    0,    1,    0},   //  472
     { -2,    1,    2,    0,    2},   //  473
     {  1,    1,    0,   -1,    1},   //  474
     {  1,    0,    1,   -2,    1},   //  475
     {  0,    2,    0,    0,    2},   //  476
     {  1,   -1,    2,   -3,    1},   //  477
     { -1,    1,    2,   -1,    1},   //  478
     { -2,    0,    4,   -2,    2},   //  479
     { -2,    0,    4,   -2,    1},   //  480
     { -2,   -2,    0,    2,    1},   //  481
     { -2,    0,   -2,    4,    0},   //  482
     {  1,    2,    2,   -4,    1},   //  483
     {  1,    1,    2,   -4,    2},   //  484
     { -1,    2,    2,   -2,    1},   //  485
     {  2,    0,    0,   -3,    1},   //  486
     { -1,    2,    0,    0,    1},   //  487
     {  0,    0,    0,   -2,    0},   //  488
     { -1,   -1,    2,   -2,    2},   //  489
     { -1,    1,    0,    0,    2},   //  490
     {  0,    0,    0,   -1,    2},   //  491
     { -2,    1,    0,    1,    0},   //  492
     {  1,   -2,    0,   -2,    1},   //  493
     {  1,    0,   -2,    0,    2},   //  494
     { -3,    1,    0,    2,    0},   //  495
     { -1,    1,   -2,    2,    0},   //  496
     { -1,   -1,    0,    0,    2},   //  497
     { -3,    0,    0,    2,    0},   //  498
     { -3,   -1,    0,    2,    0},   //  499
     {  2,    0,    2,   -6,    1},   //  500
     {  0,    1,    2,   -4,    2},   //  501
     {  2,    0,    0,   -4,    2},   //  502
     { -2,    1,    2,   -2,    1},   //  503
     {  0,   -1,    2,   -4,    1},   //  504
     {  0,    1,    0,   -2,    2},   //  505
     { -1,    0,    0,   -2,    0},   //  506
     {  2,    0,   -2,   -2,    1},   //  507
     { -4,    0,    2,    0,    1},   //  508
     { -1,   -1,    0,   -1,    1},   //  509
     {  0,    0,   -2,    0,    2},   //  510
     { -3,    0,    0,    1,    0},   //  511
     { -1,    0,   -2,    1,    0},   //  512
     { -2,    0,   -2,    2,    1},   //  513
     {  0,    0,   -4,    2,    0},   //  514
     { -2,   -1,   -2,    2,    0},   //  515
     {  1,    0,    2,   -6,    1},   //  516
     { -1,    0,    2,   -4,    2},   //  517
     {  1,    0,    0,   -4,    2},   //  518
     {  2,    1,    2,   -4,    2},   //  519
     {  2,    1,    2,   -4,    1},   //  520
     {  0,    1,    4,   -4,    4},   //  521
     {  0,    1,    4,   -4,    2},   //  522
     { -1,   -1,   -2,    4,    0},   //  523
     { -1,   -3,    0,    2,    0},   //  524
     { -1,    0,   -2,    4,    1},   //  525
     { -2,   -1,    0,    3,    0},   //  526
     {  0,    0,   -2,    3,    0},   //  527
     { -2,    0,    0,    3,    1},   //  528
     {  0,   -1,    0,    1,    0},   //  529
     { -3,    0,    2,    2,    0},   //  530
     {  1,    1,   -2,    2,    0},   //  531
     { -1,    1,    0,    2,    2},   //  532
     {  1,   -2,    2,   -2,    1},   //  533
     {  0,    0,    1,    0,    2},   //  534
     {  0,    0,    1,    0,    1},   //  535
     {  0,    0,    1,    0,    0},   //  536
     { -1,    2,    0,    2,    1},   //  537
     {  0,    0,    2,    0,    2},   //  538
     { -2,    0,    2,    0,    2},   //  539
     {  2,    0,    0,   -1,    1},   //  540
     {  3,    0,    0,   -2,    1},   //  541
     {  1,    0,    2,   -2,    3},   //  542
     {  1,    2,    0,    0,    1},   //  543
     {  2,    0,    2,   -3,    2},   //  544
     { -1,    1,    4,   -2,    2},   //  545
     { -2,   -2,    0,    4,    0},   //  546
     {  0,   -3,    0,    2,    0},   //  547
     {  0,    0,   -2,    4,    0},   //  548
     { -1,   -1,    0,    3,    0},   //  549
     { -2,    0,    0,    4,    2},   //  550
     { -1,    0,    0,    3,    1},   //  551
     {  2,   -2,    0,    0,    0},   //  552
     {  1,   -1,    0,    1,    0},   //  553
     { -1,    0,    0,    2,    0},   //  554
     {  0,   -2,    2,    0,    1},   //  555
     { -1,    0,    1,    2,    1},   //  556
     { -1,    1,    0,    3,    0},   //  557
     { -1,   -1,    2,    1,    2},   //  558
     {  0,   -1,    2,    0,    0},   //  559
     { -2,    1,    2,    2,    1},   //  560
     {  2,   -2,    2,   -2,    2},   //  561
     {  1,    1,    0,    1,    1},   //  562
     {  1,    0,    1,    0,    1},   //  563
     {  1,    0,    1,    0,    0},   //  564
     {  0,    2,    0,    2,    0},   //  565
     {  2,   -1,    2,   -2,    1},   //  566
     {  0,   -1,    4,   -2,    1},   //  567
     {  0,    0,    4,   -2,    3},   //  568
     {  0,    1,    4,   -2,    1},   //  569
     {  4,    0,    2,   -4,    2},   //  570
     {  2,    2,    2,   -2,    2},   //  571
     {  2,    0,    4,   -4,    2},   //  572
     { -1,   -2,    0,    4,    0},   //  573
     { -1,   -3,    2,    2,    2},   //  574
     { -3,    0,    2,    4,    2},   //  575
     { -3,    0,    2,   -2,    1},   //  576
     { -1,   -1,    0,   -2,    1},   //  577
     { -3,    0,    0,    0,    2},   //  578
     { -3,    0,   -2,    2,    0},   //  579
     {  0,    1,    0,   -4,    1},   //  580
     { -2,    1,    0,   -2,    1},   //  581
     { -4,    0,    0,    0,    1},   //  582
     { -1,    0,    0,   -4,    1},   //  583
     { -3,    0,    0,   -2,    1},   //  584
     {  0,    0,    0,    3,    2},   //  585
     { -1,    1,    0,    4,    1},   //  586
     {  1,   -2,    2,    0,    1},   //  587
     {  0,    1,    0,    3,    0},   //  588
     { -1,    0,    2,    2,    3},   //  589
     {  0,    0,    2,    2,    2},   //  590
     { -2,    0,    2,    2,    2},   //  591
     { -1,    1,    2,    2,    0},   //  592
     {  3,    0,    0,    0,    2},   //  593
     {  2,    1,    0,    1,    0},   //  594
     {  2,   -1,    2,   -1,    2},   //  595
     {  0,    0,    2,    0,    1},   //  596
     {  0,    0,    3,    0,    3},   //  597
     {  0,    0,    3,    0,    2},   //  598
     { -1,    2,    2,    2,    1},   //  599
     { -1,    0,    4,    0,    0},   //  600
     {  1,    2,    2,    0,    1},   //  601
     {  3,    1,    2,   -2,    1},   //  602
     {  1,    1,    4,   -2,    2},   //  603
     { -2,   -1,    0,    6,    0},   //  604
     {  0,   -2,    0,    4,    0},   //  605
     { -2,    0,    0,    6,    1},   //  606
     { -2,   -2,    2,    4,    2},   //  607
     {  0,   -3,    2,    2,    2},   //  608
     {  0,    0,    0,    4,    2},   //  609
     { -1,   -1,    2,    3,    2},   //  610
     { -2,    0,    2,    4,    0},   //  611
     {  2,   -1,    0,    2,    1},   //  612
     {  1,    0,    0,    3,    0},   //  613
     {  0,    1,    0,    4,    1},   //  614
     {  0,    1,    0,    4,    0},   //  615
     {  1,   -1,    2,    1,    2},   //  616
     {  0,    0,    2,    2,    3},   //  617
     {  1,    0,    2,    2,    2},   //  618
     { -1,    0,    2,    2,    2},   //  619
     { -2,    0,    4,    2,    1},   //  620
     {  2,    1,    0,    2,    1},   //  621
     {  2,    1,    0,    2,    0},   //  622
     {  2,   -1,    2,    0,    0},   //  623
     {  1,    0,    2,    1,    0},   //  624
     {  0,    1,    2,    2,    0},   //  625
     {  2,    0,    2,    0,    3},   //  626
     {  3,    0,    2,    0,    2},   //  627
     {  1,    0,    2,    0,    2},   //  628
     {  1,    0,    3,    0,    3},   //  629
     {  1,    1,    2,    1,    1},   //  630
     {  0,    2,    2,    2,    2},   //  631
     {  2,    1,    2,    0,    0},   //  632
     {  2,    0,    4,   -2,    1},   //  633
     {  4,    1,    2,   -2,    2},   //  634
     { -1,   -1,    0,    6,    0},   //  635
     { -3,   -1,    2,    6,    2},   //  636
     { -1,    0,    0,    6,    1},   //  637
     { -3,    0,    2,    6,    1},   //  638
     {  1,   -1,    0,    4,    1},   //  639
     {  1,   -1,    0,    4,    0},   //  640
     { -2,    0,    2,    5,    2},   //  641
     {  1,   -2,    2,    2,    1},   //  642
     {  3,   -1,    0,    2,    0},   //  643
     {  1,   -1,    2,    2,    0},   //  644
     {  0,    0,    2,    3,    1},   //  645
     { -1,    1,    2,    4,    1},   //  646
     {  0,    1,    2,    3,    2},   //  647
     { -1,    0,    4,    2,    1},   //  648
     {  2,    0,    2,    1,    1},   //  649
     {  5,    0,    0,    0,    0},   //  650
     {  2,    1,    2,    1,    2},   //  651
     {  1,    0,    4,    0,    1},   //  652
     {  3,    1,    2,    0,    1},   //  653
     {  3,    0,    4,   -2,    2},   //  654
     { -2,   -1,    2,    6,    2},   //  655
     {  0,    0,    0,    6,    0},   //  656
     {  0,   -2,    2,    4,    2},   //  657
     { -2,    0,    2,    6,    1},   //  658
     {  2,    0,    0,    4,    1},   //  659
     {  2,    0,    0,    4,    0},   //  660
     {  2,   -2,    2,    2,    2},   //  661
     {  0,    0,    2,    4,    0},   //  662
     {  1,    0,    2,    3,    2},   //  663
     {  4,    0,    0,    2,    0},   //  664
     {  2,    0,    2,    2,    0},   //  665
     {  0,    0,    4,    2,    2},   //  666
     {  4,   -1,    2,    0,    2},   //  667
     {  3,    0,    2,    1,    2},   //  668
     {  2,    1,    2,    2,    1},   //  669
     {  4,    1,    2,    0,    2},   //  670
     { -1,   -1,    2,    6,    2},   //  671
     { -1,    0,    2,    6,    1},   //  672
     {  1,   -1,    2,    4,    1},   //  673
     {  1,    1,    2,    4,    2},   //  674
     {  3,    1,    2,    2,    2},   //  675
     {  5,    0,    2,    0,    1},   //  676
     {  2,   -1,    2,    4,    2},   //  677
     {  2,    0,    2,    4,    1}    //  678
  };
  DebugAssert(which < 678, AipsError);
  return &(ARG[which][0]);
}

const Double* MeasTable::mulPlanArg2000A(uInt which) {
  static const Double ARG[687][14] = {
  // L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
  {  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0},   //    1
  {  0,  0,  0,  0,  0,  0,  0, -8, 16, -4, -5,  0,  0,  2},   //    2
  {  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2},   //    3
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  2,  2},   //    4
  {  0,  0,  0,  0,  0,  0,  0, -4,  8, -1, -5,  0,  0,  2},   //    5
  {  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1},   //    6
  {  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0},   //    7
  { -1,  0,  0,  0,  0,  0, 10, -3,  0,  0,  0,  0,  0,  0},   //    8
  {  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  6, -3,  0,  2},   //    9
  {  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //   10
  {  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0},   //   11
  {  0,  0,  0,  0,  0,  0,  0, -4,  8, -3,  0,  0,  0,  1},   //   12
  {  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2},   //   13
  {  0,  0,  0,  0,  0,  0, -5,  6,  4,  0,  0,  0,  0,  2},   //   14
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  2},   //   15
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1},   //   16
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0},   //   17
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0},   //   18
  {  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0},   //   19
  {  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  1},   //   20
  {  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  2},   //   21
  {  2,  0, -1, -1,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0},   //   22
  {  1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0},   //   23
  {  0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0},   //   24
  {  1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0},   //   25
  {  0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0},   //   26
  { -2,  0,  0,  2,  1,  0,  0,  2,  0,  0, -5,  0,  0,  0},   //   27
  {  0,  0,  0,  0,  0,  0,  3, -7,  4,  0,  0,  0,  0,  0},   //   28
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0},   //   29
  { -2,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //   30
  { -1,  0,  0,  0,  0,  0, 18,-16,  0,  0,  0,  0,  0,  0},   //   31
  { -2,  0,  1,  1,  2,  0,  0,  1,  0, -2,  0,  0,  0,  0},   //   32
  { -1,  0,  1, -1,  1,  0, 18,-17,  0,  0,  0,  0,  0,  0},   //   33
  { -1,  0,  0,  1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0},   //   34
  {  0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  2},   //   35
  {  0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0},   //   36
  {  0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  1},   //   37
  {  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0},   //   38
  {  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0},   //   39
  {  0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0},   //   40
  {  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1},   //   41
  { -2,  0,  0,  2,  1,  0,  0,  2,  0, -4,  5,  0,  0,  0},   //   42
  { -2,  0,  0,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //   43
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  1,  0,  0,  0},   //   44
  {  0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0},   //   45
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -4,  3,  0,  0,  0},   //   46
  {  0,  0, -1,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0},   //   47
  {  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0},   //   48
  {  0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0},   //   49
  { -1,  0,  1,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0},   //   50
  { -1,  0,  0,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0},   //   51
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -2, -2,  0,  0,  0},   //   52
  { -2,  0,  2,  0,  2,  0,  0, -5,  9,  0,  0,  0,  0,  0},   //   53
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0},   //   54
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0},   //   55
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0},   //   56
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1},   //   57
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2},   //   58
  { -1,  0,  0,  1,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0},   //   59
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  2,  0,  0,  0},   //   60
  {  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0},   //   61
  {  0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0},   //   62
  {  0,  0,  0,  0,  2,  0, -3,  5,  0,  0,  0,  0,  0,  0},   //   63
  {  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0},   //   64
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0},   //   65
  {  1,  0,  0, -2,  0,  0, 17,-16,  0, -2,  0,  0,  0,  0},   //   66
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0},   //   67
  { -2,  0,  0,  2,  1,  0,  0,  5, -6,  0,  0,  0,  0,  0},   //   68
  {  0,  0, -2,  2,  0,  0,  0,  9,-13,  0,  0,  0,  0,  0},   //   69
  {  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0},   //   70
  {  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0},   //   71
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0},   //   72
  {  0,  0, -2,  2,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0},   //   73
  {  0,  0, -1,  1,  1,  0,  5, -7,  0,  0,  0,  0,  0,  0},   //   74
  { -2,  0,  0,  2,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0},   //   75
  {  2,  0,  1, -3,  1,  0, -6,  7,  0,  0,  0,  0,  0,  0},   //   76
  {  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0},   //   77
  {  0,  0, -1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0},   //   78
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0},   //   79
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1},   //   80
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2},   //   81
  {  0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  2},   //   82
  {  0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  1},   //   83
  {  0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0},   //   84
  {  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0},   //   85
  {  1,  0, -1, -1,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0},   //   86
  {  2,  0,  0, -2,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0},   //   87
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -5,  5,  0,  0,  0},   //   88
  {  2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0},   //   89
  {  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0},   //   90
  { -2,  0,  1,  1,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0},   //   91
  { -2,  0,  1,  1,  1,  0,  0,  1,  0, -3,  0,  0,  0,  0},   //   92
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //   93
  { -2,  0,  0,  2,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0},   //   94
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -1, -5,  0,  0,  0},   //   95
  { -1,  0,  0,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //   96
  { -1,  0,  1,  1,  1,  0,-20, 20,  0,  0,  0,  0,  0,  0},   //   97
  {  1,  0,  0, -2,  0,  0, 20,-21,  0,  0,  0,  0,  0,  0},   //   98
  {  0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0},   //   99
  {  0,  0,  2, -2,  1,  0,  0,-10, 15,  0,  0,  0,  0,  0},   //  100
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0},   //  101
  {  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0},   //  102
  {  0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  103
  {  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0},   //  104
  {  2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0},   //  105
  {  0,  0, -2,  2,  1,  0,  5, -6,  0,  0,  0,  0,  0,  0},   //  106
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1},   //  107
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0},   //  108
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0},   //  109
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0},   //  110
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1},   //  111
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2},   //  112
  {  0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0},   //  113
  {  0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0},   //  114
  { -2,  0,  0,  2,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0},   //  115
  {  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0,  0},   //  116
  {  0,  0,  0,  0,  0,  0,  0, -9, 17,  0,  0,  0,  0,  2},   //  117
  {  1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0},   //  118
  {  1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0},   //  119
  {  0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0},   //  120
  {  0,  0, -1,  1,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0},   //  121
  {  0,  0, -2,  2,  0,  1,  0, -2,  0,  0,  0,  0,  0,  0},   //  122
  {  0,  0,  0,  0,  0,  0,  3, -5,  0,  2,  0,  0,  0,  0},   //  123
  { -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  1,  0,  0,  0},   //  124
  { -2,  0,  0,  2,  1,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  125
  {  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0},   //  126
  {  0,  0, -1,  1,  0,  0,  8,-12,  0,  0,  0,  0,  0,  0},   //  127
  {  0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0},   //  128
  { -1,  0,  0,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0},   //  129
  { -1,  0,  0,  0,  1,  0, 18,-16,  0,  0,  0,  0,  0,  0},   //  130
  {  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0},   //  131
  {  0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0},   //  132
  { -2,  0,  1,  1,  1,  0,  0, -3,  7,  0,  0,  0,  0,  0},   //  133
  {  0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0},   //  134
  {  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0},   //  135
  {  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0},   //  136
  {  1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0},   //  137
  {  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0},   //  138
  { -1,  0,  0,  0,  1,  0, 10, -3,  0,  0,  0,  0,  0,  0},   //  139
  {  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  140
  {  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0},   //  141
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0},   //  142
  {  2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0},   //  143
  { -2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -5,  0,  0,  0},   //  144
  {  0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0},   //  145
  { -2,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  146
  {  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0},   //  147
  { -2,  0,  1,  1,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0},   //  148
  {  0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0},   //  149
  {  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0},   //  150
  {  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1},   //  151
  {  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0},   //  152
  {  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0},   //  153
  {  0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0},   //  154
  {  0,  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  1},   //  155
  { -1,  0,  0,  1,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0},   //  156
  { -1,  0,  0,  1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0},   //  157
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0},   //  158
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0},   //  159
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1},   //  160
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2},   //  161
  {  0,  0,  1, -1,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0},   //  162
  {  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0},   //  163
  {  0,  0,  1, -1,  2,  0, -3,  4,  0,  0,  0,  0,  0,  0},   //  164
  {  0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0},   //  165
  {  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0},   //  166
  {  0,  0, -1,  1,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0},   //  167
  {  0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0},   //  168
  { -2,  0,  0,  2,  1,  0,  6, -8,  0,  0,  0,  0,  0,  0},   //  169
  {  0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0},   //  170
  { -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  171
  { -2,  0,  0,  2,  1,  0,  0,  6, -8,  0,  0,  0,  0,  0},   //  172
  {  1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  173
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0},   //  174
  {  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0},   //  175
  {  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  1},   //  176
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0},   //  177
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1},   //  178
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  179
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1},   //  180
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2},   //  181
  {  0,  0,  1, -1,  2,  0,  0, -1,  0,  0, -1,  0,  0,  0},   //  182
  {  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0},   //  183
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0},   //  184
  {  0,  0,  0,  0,  0,  0,  0, -7, 13,  0,  0,  0,  0,  2},   //  185
  {  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0,  0},   //  186
  {  2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0},   //  187
  {  0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0},   //  188
  {  0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0},   //  189
  { -2,  0,  0,  2,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0},   //  190
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0},   //  191
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0},   //  192
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1},   //  193
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2},   //  194
  { -2,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  195
  {  0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0},   //  196
  {  0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  197
  {  2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0},   //  198
  {  0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0},   //  199
  {  0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0},   //  200
  {  0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0},   //  201
  {  0,  0, -1,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0},   //  202
  {  0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0},   //  203
  {  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0},   //  204
  {  0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0},   //  205
  {  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1},   //  206
  {  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0},   //  207
  {  0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0},   //  208
  {  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  1},   //  209
  {  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2},   //  210
  {  0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0},   //  211
  {  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2},   //  212
  {  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1},   //  213
  {  0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0},   //  214
  {  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0},   //  215
  {  0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1},   //  216
  {  0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0},   //  217
  {  0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1},   //  218
  {  0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  2},   //  219
  {  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2},   //  220
  {  0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0},   //  221
  {  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2},   //  222
  {  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1},   //  223
  {  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0},   //  224
  {  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1},   //  225
  {  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0},   //  226
  {  0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0},   //  227
  {  0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0},   //  228
  {  0,  0, -1,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  229
  {  0,  0,  2, -2,  1,  0,  0, -2,  0,  1,  0,  0,  0,  0},   //  230
  {  0,  0,  0,  0,  0,  0,  0, -6, 11,  0,  0,  0,  0,  2},   //  231
  {  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0,  0},   //  232
  {  0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  0,  0,  2},   //  233
  {  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0},   //  234
  {  2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0},   //  235
  { -2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0},   //  236
  {  0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0},   //  237
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2},   //  238
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0},   //  239
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1},   //  240
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0},   //  241
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1},   //  242
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2},   //  243
  {  0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0},   //  244
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2},   //  245
  {  0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0},   //  246
  {  0,  0, -1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0},   //  247
  {  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0},   //  248
  {  0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0},   //  249
  {  0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0},   //  250
  {  0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0},   //  251
  {  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0},   //  252
  {  0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1},   //  253
  {  0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0},   //  254
  {  0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1},   //  255
  {  0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  2},   //  256
  {  0,  0, -1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  257
  {  0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0},   //  258
  {  0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  2},   //  259
  {  0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  1},   //  260
  {  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0},   //  261
  {  0,  0, -1,  1,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0},   //  262
  {  0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0},   //  263
  { -2,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  264
  {  0,  0, -2,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  265
  {  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  1},   //  266
  {  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  2},   //  267
  {  0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  2},   //  268
  {  0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  1},   //  269
  {  0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0},   //  270
  {  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0},   //  271
  {  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1},   //  272
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1},   //  273
  {  0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0},   //  274
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1},   //  275
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2},   //  276
  {  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0},   //  277
  {  0,  0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  2},   //  278
  {  0,  0, -2,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  279
  {  0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  2},   //  280
  {  0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  1},   //  281
  {  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0},   //  282
  {  0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0},   //  283
  {  0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0},   //  284
  {  0,  0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  2},   //  285
  {  0,  0,  0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0},   //  286
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2},   //  287
  {  0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  2},   //  288
  {  0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  1},   //  289
  {  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0},   //  290
  {  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1},   //  291
  {  0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0},   //  292
  {  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0},   //  293
  {  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  1},   //  294
  {  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  2},   //  295
  {  0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  2},   //  296
  {  0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  1},   //  297
  {  0,  0, -2,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0},   //  298
  {  0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0},   //  299
  {  0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0},   //  300
  {  0,  0, -1,  1,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0},   //  301
  {  0,  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0},   //  302
  {  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  1},   //  303
  {  0,  0,  1, -1,  1,  0, -4,  6,  0,  0,  0,  0,  0,  0},   //  304
  {  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  2},   //  305
  {  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  2},   //  306
  {  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1},   //  307
  {  0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0},   //  308
  {  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1},   //  309
  {  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0},   //  310
  { -2,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  311
  {  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0},   //  312
  {  0,  0, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},   //  313
  {  0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  314
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  5,  0,  0,  0,  2},   //  315
  {  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0},   //  316
  {  0,  0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  2},   //  317
  {  0,  0,  0,  0,  0,  0,  0, -7, 12,  0,  0,  0,  0,  2},   //  318
  {  0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  2},   //  319
  {  0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  1},   //  320
  {  0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0},   //  321
  {  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  322
  {  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1},   //  323
  {  0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0},   //  324
  {  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  2},   //  325
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  2},   //  326
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0},   //  327
  {  0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0},   //  328
  {  0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  2},   //  329
  {  0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0},   //  330
  {  0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0},   //  331
  {  0,  0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  2},   //  332
  { -2,  0,  0,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0},   //  333
  {  0,  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  2},   //  334
  {  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0},   //  335
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  2},   //  336
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  1},   //  337
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0},   //  338
  {  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0},   //  339
  {  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  1},   //  340
  {  0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0},   //  341
  {  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  2},   //  342
  {  0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  2},   //  343
  {  0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  1},   //  344
  {  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,  0},   //  345
  {  0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0},   //  346
  {  0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0},   //  347
  {  0,  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  2},   //  348
  {  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0},   //  349
  {  0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0},   //  350
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  2,  0,  0,  0,  2},   //  351
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0},   //  352
  {  0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  353
  {  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  0,  2},   //  354
  {  0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0},   //  355
  {  0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  2},   //  356
  {  0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  1},   //  357
  {  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0},   //  358
  {  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1},   //  359
  {  0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0},   //  360
  {  0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0},   //  361
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -3,  0,  0,  0},   //  362
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -5,  0,  0,  0},   //  363
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  1},   //  364
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  365
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1},   //  366
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0},   //  367
  {  0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0},   //  368
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0},   //  369
  {  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0},   //  370
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0},   //  371
  {  0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  372
  {  0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0},   //  373
  {  0,  0,  0,  0,  0,  0, -8, 14,  0,  0,  0,  0,  0,  2},   //  374
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0},   //  375
  {  0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  0},   //  376
  {  0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  2},   //  377
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  1},   //  378
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  379
  {  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0,  0},   //  380
  {  0,  0,  0,  0,  0,  0,  0, -3,  8, -3,  0,  0,  0,  2},   //  381
  {  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2},   //  382
  {  0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  2},   //  383
  {  0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  0},   //  384
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -2,  0,  0,  0},   //  385
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2},   //  386
  {  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0},   //  387
  {  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2},   //  388
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2},   //  389
  {  0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0},   //  390
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0},   //  391
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1},   //  392
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2},   //  393
  {  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0},   //  394
  {  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  1},   //  395
  {  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  2},   //  396
  {  0,  0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  2},   //  397
  {  0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  2},   //  398
  {  0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  1},   //  399
  {  0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0},   //  400
  {  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0},   //  401
  {  0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  402
  {  0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  403
  {  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  0,  0,  2},   //  404
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2},   //  405
  {  0,  0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  2},   //  406
  {  0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  407
  {  0,  0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  2},   //  408
  {  0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0},   //  409
  {  0,  0,  0,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,  1},   //  410
  {  0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0},   //  411
  {  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  412
  {  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1},   //  413
  {  0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2},   //  414
  {  0,  0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  2},   //  415
  {  0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0},   //  416
  {  0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0},   //  417
  {  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0},   //  418
  {  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0},   //  419
  {  0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  1},   //  420
  {  0,  0,  1, -1,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0},   //  421
  {  0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  2},   //  422
  {  0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  2},   //  423
  {  0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  1},   //  424
  {  0,  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0},   //  425
  {  0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  0,  2},   //  426
  {  0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  2},   //  427
  {  0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  1},   //  428
  {  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0},   //  429
  {  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2},   //  430
  {  0,  0,  0,  0,  0,  0,  0, -2,  0,  5,  0,  0,  0,  2},   //  431
  {  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2},   //  432
  {  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},   //  433
  {  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2},   //  434
  {  0,  0,  0,  0,  0,  0, -9, 13,  0,  0,  0,  0,  0,  2},   //  435
  {  0,  0,  0,  0,  0,  0,  0, -1,  5,  0,  0,  0,  0,  2},   //  436
  {  0,  0,  0,  0,  0,  0,  0, -2,  0,  4,  0,  0,  0,  2},   //  437
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0,  0},   //  438
  {  0,  0,  0,  0,  0,  0,  0, -2,  7,  0,  0,  0,  0,  2},   //  439
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  440
  {  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  1},   //  441
  {  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  2},   //  442
  {  0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  2},   //  443
  {  0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  1},   //  444
  {  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0},   //  445
  {  0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  446
  {  0,  0,  0,  0,  0,  0,  0, -3,  9,  0,  0,  0,  0,  2},   //  447
  {  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0},   //  448
  {  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2},   //  449
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  450
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1},   //  451
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2},   //  452
  {  0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  0,  2},   //  453
  {  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0},   //  454
  {  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2},   //  455
  {  0,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  1},   //  456
  {  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  457
  {  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1},   //  458
  {  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2},   //  459
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -3,  0,  0,  0},   //  460
  {  0,  0,  0,  0,  0,  0,  0, -5, 13,  0,  0,  0,  0,  2},   //  461
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0},   //  462
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2},   //  463
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0},   //  464
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1},   //  465
  {  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0},   //  466
  {  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2},   //  467
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2},   //  468
  {  0,  0,  0,  0,  0,  0,  0, -6, 15,  0,  0,  0,  0,  2},   //  469
  {  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  0,  2},   //  470
  {  0,  0,  0,  0,  0,  0, -3,  9, -4,  0,  0,  0,  0,  2},   //  471
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2},   //  472
  {  0,  0,  0,  0,  0,  0,  0, -2,  8, -1, -5,  0,  0,  2},   //  473
  {  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2},   //  474
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0},   //  475
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0},   //  476
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1},   //  477
  {  0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  478
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1},   //  479
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2},   //  480
  {  0,  0,  0,  0,  0,  0,  0, -6, 16, -4, -5,  0,  0,  2},   //  481
  {  0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2},   //  482
  {  0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2},   //  483
  {  0,  0,  0,  0,  0,  0,  0,  6, -8,  1,  5,  0,  0,  2},   //  484
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2},   //  485
  {  0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2},   //  486
  {  0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2},   //  487
  {  0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  1},   //  488
  {  0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2},   //  489
  {  0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2},   //  490
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2},   //  491
  {  0,  0,  0,  0,  0,  0,  3, -3,  0,  2,  0,  0,  0,  2},   //  492
  {  0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  493
  {  0,  0,  1, -1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  494
  {  0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0},   //  495
  {  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2},   //  496
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2},   //  497
  {  0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  0,  2},   //  498
  {  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2},   //  499
  {  0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  2},   //  500
  {  0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  1},   //  501
  {  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0},   //  502
  {  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2},   //  503
  {  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  0,  0,  2},   //  504
  {  0,  0,  0,  0,  0,  0,  0, -1,  6,  0,  0,  0,  0,  2},   //  505
  {  0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2},   //  506
  {  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0},   //  507
  {  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2},   //  508
  {  0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2},   //  509
  {  0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2},   //  510
  {  0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  1},   //  511
  {  0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  2},   //  512
  {  0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  2},   //  513
  {  0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  1},   //  514
  {  0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2},   //  515
  {  0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2},   //  516
  {  0,  0,  0,  0,  0,  0, -4,  4,  0,  0,  0,  0,  0,  1},   //  517
  {  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0},   //  518
  {  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1},   //  519
  {  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2},   //  520
  {  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2},   //  521
  {  0,  0,  0,  0,  0,  0,  0, -3,  0,  5,  0,  0,  0,  2},   //  522
  {  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0},   //  523
  {  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1},   //  524
  {  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2},   //  525
  {  0,  0,  0,  0,  0,  0, -9, 12,  0,  0,  0,  0,  0,  2},   //  526
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -4,  0,  0,  0,  0},   //  527
  {  0,  0,  2, -2,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  528
  {  0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2},   //  529
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  0},   //  530
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2},   //  531
  {  0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  0,  2},   //  532
  {  0,  0,  0,  0,  0,  0, -6,  7,  0,  0,  0,  0,  0,  1},   //  533
  {  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0,  0},   //  534
  {  0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2},   //  535
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  0},   //  536
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2},   //  537
  {  0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2},   //  538
  {  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  0},   //  539
  {  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2},   //  540
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2},   //  541
  {  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2},   //  542
  {  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2},   //  543
  {  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2},   //  544
  {  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2},   //  545
  {  0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  546
  {  0,  0,  0,  0,  0,  0, -8, 16,  0,  0,  0,  0,  0,  2},   //  547
  {  0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2},   //  548
  {  0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2},   //  549
  {  0,  0,  0,  0,  0,  0,  0, -5, 16, -4, -5,  0,  0,  2},   //  550
  {  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2},   //  551
  {  0,  0,  0,  0,  0,  0,  0, -1,  8, -3,  0,  0,  0,  2},   //  552
  {  0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2},   //  553
  {  0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  1},   //  554
  {  0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2},   //  555
  {  0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2},   //  556
  {  0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2},   //  557
  {  0,  0,  0,  0,  0,  0, -3,  8,  0,  0,  0,  0,  0,  2},   //  558
  {  0,  0,  0,  0,  0,  0, -5,  5,  0,  0,  0,  0,  0,  1},   //  559
  {  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0},   //  560
  {  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1},   //  561
  {  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2},   //  562
  {  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0},   //  563
  {  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1},   //  564
  {  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2},   //  565
  {  0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2},   //  566
  {  0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2},   //  567
  {  0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2},   //  568
  {  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  0,  0},   //  569
  {  0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2},   //  570
  {  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2},   //  571
  {  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2},   //  572
  {  0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  2},   //  573
  {  0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  1},   //  574
  {  0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2},   //  575
  {  0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2},   //  576
  {  0,  0,  0,  0,  0,  0, -6,  6,  0,  0,  0,  0,  0,  1},   //  577
  {  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0},   //  578
  {  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1},   //  579
  {  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2},   //  580
  {  0,  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  2},   //  581
  {  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0},   //  582
  {  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1},   //  583
  {  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2},   //  584
  {  0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2},   //  585
  {  0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2},   //  586
  {  0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2},   //  587
  {  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0},   //  588
  {  0,  0,  0,  0,  0,  0,  8, -9,  0,  0,  0,  0,  0,  0},   //  589
  {  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  0,  2},   //  590
  {  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2},   //  591
  {  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1},   //  592
  {  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1},   //  593
  {  0,  0,  0,  0,  0,  0, -7,  7,  0,  0,  0,  0,  0,  1},   //  594
  {  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0,  0},   //  595
  {  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1},   //  596
  {  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2},   //  597
  {  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0},   //  598
  {  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0},   //  599
  {  0,  0,  0,  0,  0,  0,  0,  5,  0, -4,  0,  0,  0,  2},   //  600
  {  0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2},   //  601
  {  0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2},   //  602
  {  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  2},   //  603
  {  0,  0,  0,  0,  0,  0, -8,  8,  0,  0,  0,  0,  0,  1},   //  604
  {  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0,  0},   //  605
  {  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1},   //  606
  {  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  2},   //  607
  {  0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1},   //  608
  {  0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1},   //  609
  {  0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1},   //  610
  {  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0,  0},   //  611
  {  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1},   //  612
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2},   //  613
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0},   //  614
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0},   //  615
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1},   //  616
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2},   //  617
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0},   //  618
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1},   //  619
  {  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2},   //  620
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2},   //  621
  {  1,  0,  0, -2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  622
  {  1,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  623
  {  1,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  624
  {  1,  0,  0, -2,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  625
  { -1,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  626
  { -1,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  627
  { -1,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  628
  {  1,  0,  0, -2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  629
  { -2,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  630
  { -1,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  631
  { -1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  632
  { -1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  633
  { -1,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  634
  {  1,  0, -1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  635
  { -1,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  636
  { -2,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  637
  {  1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  638
  { -1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0},   //  639
  {  1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0},   //  640
  { -1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  641
  { -1,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  642
  {  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  643
  { -1,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  644
  { -1,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  645
  {  1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0},   //  646
  {  1,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0},   //  647
  {  1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0},   //  648
  {  1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  649
  {  1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  650
  {  0,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  651
  {  0,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  652
  {  0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0},   //  653
  {  0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  654
  {  0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0},   //  655
  {  0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0},   //  656
  {  0,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  657
  {  0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  658
  {  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  659
  { -1,  0,  2,  0,  2,  0, 10, -3,  0,  0,  0,  0,  0,  0},   //  660
  {  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  661
  {  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  662
  {  0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  663
  {  0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0},   //  664
  { -1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0},   //  665
  {  2,  0,  2, -2,  2,  0,  0, -2,  0,  3,  0,  0,  0,  0},   //  666
  {  1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0},   //  667
  {  0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  668
  { -1,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  669
  { -2,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  670
  {  0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0},   //  671
  {  0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  672
  {  0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0},   //  673
  {  0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0},   //  674
  { -1,  0,  2,  2,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0},   //  675
  {  1,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0},   //  676
  { -1,  0,  2,  2,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  677
  {  2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0},   //  678
  {  1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0},   //  679
  {  1,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0},   //  680
  {  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  681
  {  0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  682
  {  2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},   //  683
  { -1,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0},   //  684
  { -1,  0,  2,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0},   //  685
  {  1,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0},   //  686
  {  0,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0}    //  687
  };
  DebugAssert(which < 687, AipsError);
  return &(ARG[which][0]);
}

const Double* MeasTable::mulArg2000B(uInt which) {
  static const Double ARG[77][5] = {
  //         Multiple of        
  //      L     L'    F     D   Omega  
       {  0,    0,    0,    0,    1},                 //   1
       {  0,    0,    2,   -2,    2},                 //   2
       {  0,    0,    2,    0,    2},                 //   3
       {  0,    0,    0,    0,    2},                 //   4
       {  0,    1,    0,    0,    0},                 //   5
       {  0,    1,    2,   -2,    2},                 //   6
       {  1,    0,    0,    0,    0},                 //   7
       {  0,    0,    2,    0,    1},                 //   8
       {  1,    0,    2,    0,    2},                 //   9
       {  0,   -1,    2,   -2,    2},                 //  10
       {  0,    0,    2,   -2,    1},                 //  11
       { -1,    0,    2,    0,    2},                 //  12
       { -1,    0,    0,    2,    0},                 //  13
       {  1,    0,    0,    0,    1},                 //  14
       { -1,    0,    0,    0,    1},                 //  15
       { -1,    0,    2,    2,    2},                 //  16
       {  1,    0,    2,    0,    1},                 //  17
       { -2,    0,    2,    0,    1},                 //  18
       {  0,    0,    0,    2,    0},                 //  19
       {  0,    0,    2,    2,    2},                 //  20
       {  0,   -2,    2,   -2,    2},                 //  21
       { -2,    0,    0,    2,    0},                 //  22
       {  2,    0,    2,    0,    2},                 //  23
       {  1,    0,    2,   -2,    2},                 //  24
       { -1,    0,    2,    0,    1},                 //  25
       {  2,    0,    0,    0,    0},                 //  26
       {  0,    0,    2,    0,    0},                 //  27
       {  0,    1,    0,    0,    1},                 //  28
       { -1,    0,    0,    2,    1},                 //  29
       {  0,    2,    2,   -2,    2},                 //  30
       {  0,    0,   -2,    2,    0},                 //  31
       {  1,    0,    0,   -2,    1},                 //  32
       {  0,   -1,    0,    0,    1},                 //  33
       { -1,    0,    2,    2,    1},                 //  34
       {  0,    2,    0,    0,    0},                 //  35
       {  1,    0,    2,    2,    2},                 //  36
       { -2,    0,    2,    0,    0},                 //  37
       {  0,    1,    2,    0,    2},                 //  38
       {  0,    0,    2,    2,    1},                 //  39
       {  0,   -1,    2,    0,    2},                 //  40
       {  0,    0,    0,    2,    1},                 //  41
       {  1,    0,    2,   -2,    1},                 //  42
       {  2,    0,    2,   -2,    2},                 //  43
       { -2,    0,    0,    2,    1},                 //  44
       {  2,    0,    2,    0,    1},                 //  45
       {  0,   -1,    2,   -2,    1},                 //  46
       {  0,    0,    0,   -2,    1},                 //  47
       { -1,   -1,    0,    2,    0},                 //  48
       {  2,    0,    0,   -2,    1},                 //  49
       {  1,    0,    0,    2,    0},                 //  50
       {  0,    1,    2,   -2,    1},                 //  51
       {  1,   -1,    0,    0,    0},                 //  52
       { -2,    0,    2,    0,    2},                 //  53
       {  3,    0,    2,    0,    2},                 //  54
       {  0,   -1,    0,    2,    0},                 //  55
       {  1,   -1,    2,    0,    2},                 //  56
       {  0,    0,    0,    1,    0},                 //  57
       { -1,   -1,    2,    2,    2},                 //  58
       { -1,    0,    2,    0,    0},                 //  59
       {  0,   -1,    2,    2,    2},                 //  60
       { -2,    0,    0,    0,    1},                 //  61
       {  1,    1,    2,    0,    2},                 //  62
       {  2,    0,    0,    0,    1},                 //  63
       { -1,    1,    0,    1,    0},                 //  64
       {  1,    1,    0,    0,    0},                 //  65
       {  1,    0,    2,    0,    0},                 //  66
       { -1,    0,    2,   -2,    1},                 //  67
       {  1,    0,    0,    0,    2},                 //  68
       { -1,    0,    0,    1,    0},                 //  69
       {  0,    0,    2,    1,    2},                 //  70
       { -1,    0,    2,    4,    2},                 //  71
       { -1,    1,    0,    1,    1},                 //  72
       {  0,   -2,    2,   -2,    1},                 //  73
       {  1,    0,    2,    2,    1},                 //  74
       { -2,    0,    2,    2,    2},                 //  75
       { -1,    0,    0,    0,    2},                 //  76
       {  1,    1,    2,   -2,    2}                  //  77
  };
  DebugAssert(which < 77, AipsError);
  return &(ARG[which][0]);
}

const Double* MeasTable::mulArgEqEqCT2000(uInt which) {
  static const Double ARG[34][14] = {
  // L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
   { 0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  1
   { 0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  2
   { 0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  3
   { 0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  4
   { 0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  5
   { 0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  6
   { 0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  7
   { 0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  8
   { 0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   //  9
   { 0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 10
   { 1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 11
   { 1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 12
   { 0,  1,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 13
   { 0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 14
   { 0,  0,  4, -4,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 15
   { 0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0 },   // 16
   { 0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 17
   { 0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 18
   { 1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 19
   { 1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 20
   { 0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 21
   { 0,  1, -2,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 22
   { 0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 23
   { 0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1 },   // 24
   { 0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 25
   { 2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 26
   { 1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 27
   { 0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 28
   { 1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 29
   { 0,  0,  4, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 30
   { 0,  0,  2, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 31
   { 1,  0, -2,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 32
   { 1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 },   // 33
   { 0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 }    // T^1 term
  };
  DebugAssert(which < 34, AipsError);
  return &(ARG[which][0]);
}

const Double* MeasTable::mulArg1950(uInt which) {
  static const Double ARG[69][5] = {
    {0	,0	,0	,0	,1	},
    {0	,0	,0	,0	,2	},
    {-2	,0	,2	,0	,1	},
    {2	,0	,-2	,0	,0	},
    {0  ,-2     ,2	,-2	,1	},
    
    {-2	,0	,2	,0	,2	},
    {1	,-1	,0	,-1	,0	},
    {0	,0	,2	,-2	,2	},
    {0	,1	,0	,0	,0	},
    {0	,1	,2	,-2	,2	},
    
    {0	,-1	,2	,-2	,2	},
    {0	,0	,2	,-2	,1	},
    {2	,0	,0	,-2	,0	},
    {0	,0	,2	,-2	,0	},
    {0	,2	,0	,0	,0	},
    
    {0	,1	,0	,0	,1	},
    {0	,2	,2	,-2	,2	},
    {0	,-1	,0	,0	,1	},
    {-2	,0	,0	,2	,1	},
    {0	,-1	,2	,-2	,1	},
    
    {2	,0	,0	,-2	,1	},
    {0	,1	,2	,-2	,1	},
    {1	,0	,0	,-1	,0	},
    {0	,0	,2	,0	,2	},
    {1	,0	,0	,0	,0	},
    
    {0	,0	,2	,0	,1	},
    {1	,0	,2	,0	,2	},
    {1	,0	,0	,-2	,0	},
    {-1	,0	,2	,0	,2	},
    {0	,0	,0	,2	,0	},
    
    {1	,0	,0	,0	,1	},
    {-1	,0	,0	,0	,1	},
    {-1	,0	,2	,2	,2	},
    {1	,0	,2	,0	,1	},
    {0	,0	,2	,2	,2	},
    
    {2	,0	,0	,0	,0	},
    {1	,0	,2	,-2	,2	},
    {2	,0	,2	,0	,2	},
    {0	,0	,2	,0	,0	},
    {-1	,0	,2	,0	,1	},
    
    {-1	,0	,0	,2	,1	},
    {1	,0	,0	,-2	,1	},
    {-1	,0	,2	,2	,1	},
    {1	,1	,0	,-2	,0	},
    {0	,1	,2	,0	,2	},
    
    {1	,0	,0	,2	,0	},
    {0	,0	,0	,2	,1	},
    {0	,-1	,2	,0	,2	},
    {1	,0	,2	,2	,2	},
    {2	,0	,2	,-2	,2	},
    
    {0	,0	,0	,-2	,1	},
    {0	,0	,2	,2	,1	},
    {1	,0	,2	,-2	,1	},
    {0	,0	,0	,1	,0	},
    {0	,1	,0	,-2	,0	},
    
    {1	,-1	,0	,0	,0	},
    {1	,0	,-2	,0	,0	},
    {2	,0	,2	,0	,1	},
    {1	,0	,2	,0	,0	},
    {1	,1	,0	,0	,0	},
    
    {1	,-1	,2	,0	,2	},
    {-2	,0	,0	,0	,1	},
    {-1	,0	,2	,-2	,1	},
    {2	,0	,0	,0	,1	},
    {-1	,-1	,2	,2	,2	},
    
    {0	,-1	,2	,2	,2	},
    {1	,0	,0	,0	,2	},
    {1	,1	,2	,0	,2	},
    {3	,0	,2	,0	,2	}
  };
  DebugAssert(which < 69, AipsError);
  return &(ARG[which][0]);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulSC(Double time, Double epsilon)
{
  return theirMulSC.getArray (time, epsilon);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulSC2000A(Double time, Double epsilon)
{
  return theirMulSC2000A.getArray (time, epsilon);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulSC2000B(Double time, Double epsilon)
{
  return theirMulSC2000B.getArray (time, epsilon);
}

const Double* MeasTable::mulPlanSC2000A(uInt which) {
  // Luni-Solar nutation coefficients, unit 1e-7 arcsec
  static const Double MULSC[687][4] = {
  //        Longitude               Obliquity
  //    sin          cos         sin         cos
     {  1440,          0,          0,          0},   //    1
     {    56,       -117,        -42,        -40},   //    2
     {   125,        -43,          0,        -54},   //    3
     {     0,          5,          0,          0},   //    4
     {     3,         -7,         -3,          0},   //    5
     {     3,          0,          0,         -2},   //    6
     {  -114,          0,          0,         61},   //    7
     {  -219,         89,          0,          0},   //    8
     {    -3,          0,          0,          0},   //    9
     {  -462,       1604,          0,          0},   //   10
     {    99,          0,          0,        -53},   //   11
     {    -3,          0,          0,          2},   //   12
     {     0,          6,          2,          0},   //   13
     {     3,          0,          0,          0},   //   14
     {   -12,          0,          0,          0},   //   15
     {    14,       -218,        117,          8},   //   16
     {    31,       -481,       -257,        -17},   //   17
     {  -491,        128,          0,          0},   //   18
     { -3084,       5123,       2735,       1647},   //   19
     { -1444,       2409,      -1286,       -771},   //   20
     {    11,        -24,        -11,         -9},   //   21
     {    26,         -9,          0,          0},   //   22
     {   103,        -60,          0,          0},   //   23
     {     0,        -13,         -7,          0},   //   24
     {   -26,        -29,        -16,         14},   //   25
     {     9,        -27,        -14,         -5},   //   26
     {    12,          0,          0,         -6},   //   27
     {    -7,          0,          0,          0},   //   28
     {     0,         24,          0,          0},   //   29
     {   284,          0,          0,       -151},   //   30
     {   226,        101,          0,          0},   //   31
     {     0,         -8,         -2,          0},   //   32
     {     0,         -6,         -3,          0},   //   33
     {     5,          0,          0,         -3},   //   34
     {   -41,        175,         76,         17},   //   35
     {     0,         15,          6,          0},   //   36
     {   425,        212,       -133,        269},   //   37
     {  1200,        598,        319,       -641},   //   38
     {   235,        334,          0,          0},   //   39
     {    11,        -12,         -7,         -6},   //   40
     {     5,         -6,          3,          3},   //   41
     {    -5,          0,          0,          3},   //   42
     {     6,          0,          0,         -3},   //   43
     {    15,          0,          0,          0},   //   44
     {    13,          0,          0,         -7},   //   45
     {    -6,         -9,          0,          0},   //   46
     {   266,        -78,          0,          0},   //   47
     {  -460,       -435,       -232,        246},   //   48
     {     0,         15,          7,          0},   //   49
     {    -3,          0,          0,          2},   //   50
     {     0,        131,          0,          0},   //   51
     {     4,          0,          0,          0},   //   52
     {     0,          3,          0,          0},   //   53
     {     0,          4,          2,          0},   //   54
     {     0,          3,          0,          0},   //   55
     {   -17,        -19,        -10,          9},   //   56
     {    -9,        -11,          6,         -5},   //   57
     {    -6,          0,          0,          3},   //   58
     {   -16,          8,          0,          0},   //   59
     {     0,          3,          0,          0},   //   60
     {    11,         24,         11,         -5},   //   61
     {    -3,         -4,         -2,          1},   //   62
     {     3,          0,          0,         -1},   //   63
     {     0,         -8,         -4,          0},   //   64
     {     0,          3,          0,          0},   //   65
     {     0,          5,          0,          0},   //   66
     {     0,          3,          2,          0},   //   67
     {    -6,          4,          2,          3},   //   68
     {    -3,         -5,          0,          0},   //   69
     {    -5,          0,          0,          2},   //   70
     {     4,         24,         13,         -2},   //   71
     {   -42,         20,          0,          0},   //   72
     {   -10,        233,          0,          0},   //   73
     {    -3,          0,          0,          1},   //   74
     {    78,        -18,          0,          0},   //   75
     {     0,          3,          1,          0},   //   76
     {     0,         -3,         -1,          0},   //   77
     {     0,         -4,         -2,          1},   //   78
     {     0,         -8,         -4,         -1},   //   79
     {     0,         -5,          3,          0},   //   80
     {    -7,          0,          0,          3},   //   81
     {   -14,          8,          3,          6},   //   82
     {     0,          8,         -4,          0},   //   83
     {     0,         19,         10,          0},   //   84
     {    45,        -22,          0,          0},   //   85
     {    -3,          0,          0,          0},   //   86
     {     0,         -3,          0,          0},   //   87
     {     0,          3,          0,          0},   //   88
     {     3,          5,          3,         -2},   //   89
     {    89,        -16,         -9,        -48},   //   90
     {     0,          3,          0,          0},   //   91
     {    -3,          7,          4,          2},   //   92
     {  -349,        -62,          0,          0},   //   93
     {   -15,         22,          0,          0},   //   94
     {    -3,          0,          0,          0},   //   95
     {   -53,          0,          0,          0},   //   96
     {     5,          0,          0,         -3},   //   97
     {     0,         -8,          0,          0},   //   98
     {    15,         -7,         -4,         -8},   //   99
     {    -3,          0,          0,          1},   //  100
     {   -21,        -78,          0,          0},   //  101
     {    20,        -70,        -37,        -11},   //  102
     {     0,          6,          3,          0},   //  103
     {     5,          3,          2,         -2},   //  104
     {   -17,         -4,         -2,          9},   //  105
     {     0,          6,          3,          0},   //  106
     {    32,         15,         -8,         17},   //  107
     {   174,         84,         45,        -93},   //  108
     {    11,         56,          0,          0},   //  109
     {   -66,        -12,         -6,         35},   //  110
     {    47,          8,          4,        -25},   //  111
     {     0,          8,          4,          0},   //  112
     {    10,        -22,        -12,         -5},   //  113
     {    -3,          0,          0,          2},   //  114
     {   -24,         12,          0,          0},   //  115
     {     5,         -6,          0,          0},   //  116
     {     3,          0,          0,         -2},   //  117
     {     4,          3,          1,         -2},   //  118
     {     0,         29,         15,          0},   //  119
     {    -5,         -4,         -2,          2},   //  120
     {     8,         -3,         -1,         -5},   //  121
     {     0,         -3,          0,          0},   //  122
     {    10,          0,          0,          0},   //  123
     {     3,          0,          0,         -2},   //  124
     {    -5,          0,          0,          3},   //  125
     {    46,         66,         35,        -25},   //  126
     {   -14,          7,          0,          0},   //  127
     {     0,          3,          2,          0},   //  128
     {    -5,          0,          0,          0},   //  129
     {   -68,        -34,        -18,         36},   //  130
     {     0,         14,          7,          0},   //  131
     {    10,         -6,         -3,         -5},   //  132
     {    -5,         -4,         -2,          3},   //  133
     {    -3,          5,          2,          1},   //  134
     {    76,         17,          9,        -41},   //  135
     {    84,        298,        159,        -45},   //  136
     {     3,          0,          0,         -1},   //  137
     {    -3,          0,          0,          2},   //  138
     {    -3,          0,          0,          1},   //  139
     {   -82,        292,        156,         44},   //  140
     {   -73,         17,          9,         39},   //  141
     {    -9,        -16,          0,          0},   //  142
     {     3,          0,         -1,         -2},   //  143
     {    -3,          0,          0,          0},   //  144
     {    -9,         -5,         -3,          5},   //  145
     {  -439,          0,          0,          0},   //  146
     {    57,        -28,        -15,        -30},   //  147
     {     0,         -6,         -3,          0},   //  148
     {    -4,          0,          0,          2},   //  149
     {   -40,         57,         30,         21},   //  150
     {    23,          7,          3,        -13},   //  151
     {   273,         80,         43,       -146},   //  152
     {  -449,        430,          0,          0},   //  153
     {    -8,        -47,        -25,          4},   //  154
     {     6,         47,         25,         -3},   //  155
     {     0,         23,         13,          0},   //  156
     {    -3,          0,          0,          2},   //  157
     {     3,         -4,         -2,         -2},   //  158
     {   -48,       -110,        -59,         26},   //  159
     {    51,        114,         61,        -27},   //  160
     {  -133,          0,          0,         57},   //  161
     {     0,          4,          0,          0},   //  162
     {   -21,         -6,         -3,         11},   //  163
     {     0,         -3,         -1,          0},   //  164
     {   -11,        -21,        -11,          6},   //  165
     {   -18,       -436,       -233,          9},   //  166
     {    35,         -7,          0,          0},   //  167
     {     0,          5,          3,          0},   //  168
     {    11,         -3,         -1,         -6},   //  169
     {    -5,         -3,         -1,          3},   //  170
     {   -53,         -9,         -5,         28},   //  171
     {     0,          3,          2,          1},   //  172
     {     4,          0,          0,         -2},   //  173
     {     0,         -4,          0,          0},   //  174
     {   -50,        194,        103,         27},   //  175
     {   -13,         52,         28,          7},   //  176
     {   -91,        248,          0,          0},   //  177
     {     6,         49,         26,         -3},   //  178
     {    -6,        -47,        -25,          3},   //  179
     {     0,          5,          3,          0},   //  180
     {    52,         23,         10,        -23},   //  181
     {    -3,          0,          0,          1},   //  182
     {     0,          5,          3,          0},   //  183
     {    -4,          0,          0,          0},   //  184
     {    -4,          8,          3,          2},   //  185
     {    10,          0,          0,          0},   //  186
     {     3,          0,          0,         -2},   //  187
     {     0,          8,          4,          0},   //  188
     {     0,          8,          4,          1},   //  189
     {    -4,          0,          0,          0},   //  190
     {    -4,          0,          0,          0},   //  191
     {    -8,          4,          2,          4},   //  192
     {     8,         -4,         -2,         -4},   //  193
     {     0,         15,          7,          0},   //  194
     {  -138,          0,          0,          0},   //  195
     {     0,         -7,         -3,          0},   //  196
     {     0,         -7,         -3,          0},   //  197
     {    54,          0,          0,        -29},   //  198
     {     0,         10,          4,          0},   //  199
     {    -7,          0,          0,          3},   //  200
     {   -37,         35,         19,         20},   //  201
     {     0,          4,          0,          0},   //  202
     {    -4,          9,          0,          0},   //  203
     {     8,          0,          0,         -4},   //  204
     {    -9,        -14,         -8,          5},   //  205
     {    -3,         -9,         -5,          3},   //  206
     {  -145,         47,          0,          0},   //  207
     {   -10,         40,         21,          5},   //  208
     {    11,        -49,        -26,         -7},   //  209
     { -2150,          0,          0,        932},   //  210
     {   -12,          0,          0,          5},   //  211
     {    85,          0,          0,        -37},   //  212
     {     4,          0,          0,         -2},   //  213
     {     3,          0,          0,         -2},   //  214
     {   -86,        153,          0,          0},   //  215
     {    -6,          9,          5,          3},   //  216
     {     9,        -13,         -7,         -5},   //  217
     {    -8,         12,          6,          4},   //  218
     {   -51,          0,          0,         22},   //  219
     {   -11,       -268,       -116,          5},   //  220
     {     0,         12,          5,          0},   //  221
     {     0,          7,          3,          0},   //  222
     {    31,          6,          3,        -17},   //  223
     {   140,         27,         14,        -75},   //  224
     {    57,         11,          6,        -30},   //  225
     {   -14,        -39,          0,          0},   //  226
     {     0,         -6,         -2,          0},   //  227
     {     4,         15,          8,         -2},   //  228
     {     0,          4,          0,          0},   //  229
     {    -3,          0,          0,          1},   //  230
     {     0,         11,          5,          0},   //  231
     {     9,          6,          0,          0},   //  232
     {    -4,         10,          4,          2},   //  233
     {     5,          3,          0,          0},   //  234
     {    16,          0,          0,         -9},   //  235
     {    -3,          0,          0,          0},   //  236
     {     0,          3,          2,         -1},   //  237
     {     7,          0,          0,         -3},   //  238
     {   -25,         22,          0,          0},   //  239
     {    42,        223,        119,        -22},   //  240
     {   -27,       -143,        -77,         14},   //  241
     {     9,         49,         26,         -5},   //  242
     { -1166,          0,          0,        505},   //  243
     {    -5,          0,          0,          2},   //  244
     {    -6,          0,          0,          3},   //  245
     {    -8,          0,          1,          4},   //  246
     {     0,         -4,          0,          0},   //  247
     {   117,          0,          0,        -63},   //  248
     {    -4,          8,          4,          2},   //  249
     {     3,          0,          0,         -2},   //  250
     {    -5,          0,          0,          2},   //  251
     {     0,         31,          0,          0},   //  252
     {    -5,          0,          1,          3},   //  253
     {     4,          0,          0,         -2},   //  254
     {    -4,          0,          0,          2},   //  255
     {   -24,        -13,         -6,         10},   //  256
     {     3,          0,          0,          0},   //  257
     {     0,        -32,        -17,          0},   //  258
     {     8,         12,          5,         -3},   //  259
     {     3,          0,          0,         -1},   //  260
     {     7,         13,          0,          0},   //  261
     {    -3,         16,          0,          0},   //  262
     {    50,          0,          0,        -27},   //  263
     {     0,         -5,         -3,          0},   //  264
     {    13,          0,          0,          0},   //  265
     {     0,          5,          3,          1},   //  266
     {    24,          5,          2,        -11},   //  267
     {     5,        -11,         -5,         -2},   //  268
     {    30,         -3,         -2,        -16},   //  269
     {    18,          0,          0,         -9},   //  270
     {     8,        614,          0,          0},   //  271
     {     3,         -3,         -1,         -2},   //  272
     {     6,         17,          9,         -3},   //  273
     {    -3,         -9,         -5,          2},   //  274
     {     0,          6,          3,         -1},   //  275
     {  -127,         21,          9,         55},   //  276
     {     3,          5,          0,          0},   //  277
     {    -6,        -10,         -4,          3},   //  278
     {     5,          0,          0,          0},   //  279
     {    16,          9,          4,         -7},   //  280
     {     3,          0,          0,         -2},   //  281
     {     0,         22,          0,          0},   //  282
     {     0,         19,         10,          0},   //  283
     {     7,          0,          0,         -4},   //  284
     {     0,         -5,         -2,          0},   //  285
     {     0,          3,          1,          0},   //  286
     {    -9,          3,          1,          4},   //  287
     {    17,          0,          0,         -7},   //  288
     {     0,         -3,         -2,         -1},   //  289
     {   -20,         34,          0,          0},   //  290
     {   -10,          0,          1,          5},   //  291
     {    -4,          0,          0,          2},   //  292
     {    22,        -87,          0,          0},   //  293
     {    -4,          0,          0,          2},   //  294
     {    -3,         -6,         -2,          1},   //  295
     {   -16,         -3,         -1,          7},   //  296
     {     0,         -3,         -2,          0},   //  297
     {     4,          0,          0,          0},   //  298
     {   -68,         39,          0,          0},   //  299
     {    27,          0,          0,        -14},   //  300
     {     0,         -4,          0,          0},   //  301
     {   -25,          0,          0,          0},   //  302
     {   -12,         -3,         -2,          6},   //  303
     {     3,          0,          0,         -1},   //  304
     {     3,         66,         29,         -1},   //  305
     {   490,          0,          0,       -213},   //  306
     {   -22,         93,         49,         12},   //  307
     {    -7,         28,         15,          4},   //  308
     {    -3,         13,          7,          2},   //  309
     {   -46,         14,          0,          0},   //  310
     {    -5,          0,          0,          0},   //  311
     {     2,          1,          0,          0},   //  312
     {     0,         -3,          0,          0},   //  313
     {   -28,          0,          0,         15},   //  314
     {     5,          0,          0,         -2},   //  315
     {     0,          3,          0,          0},   //  316
     {   -11,          0,          0,          5},   //  317
     {     0,          3,          1,          0},   //  318
     {    -3,          0,          0,          1},   //  319
     {    25,        106,         57,        -13},   //  320
     {     5,         21,         11,         -3},   //  321
     {  1485,          0,          0,          0},   //  322
     {    -7,        -32,        -17,          4},   //  323
     {     0,          5,          3,          0},   //  324
     {    -6,         -3,         -2,          3},   //  325
     {    30,         -6,         -2,        -13},   //  326
     {    -4,          4,          0,          0},   //  327
     {   -19,          0,          0,         10},   //  328
     {     0,          4,          2,         -1},   //  329
     {     0,          3,          0,          0},   //  330
     {     4,          0,          0,         -2},   //  331
     {     0,         -3,         -1,          0},   //  332
     {    -3,          0,          0,          0},   //  333
     {     5,          3,          1,         -2},   //  334
     {     0,         11,          0,          0},   //  335
     {   118,          0,          0,        -52},   //  336
     {     0,         -5,         -3,          0},   //  337
     {   -28,         36,          0,          0},   //  338
     {     5,         -5,          0,          0},   //  339
     {    14,        -59,        -31,         -8},   //  340
     {     0,          9,          5,          1},   //  341
     {  -458,          0,          0,        198},   //  342
     {     0,        -45,        -20,          0},   //  343
     {     9,          0,          0,         -5},   //  344
     {     0,         -3,          0,          0},   //  345
     {     0,         -4,         -2,         -1},   //  346
     {    11,          0,          0,         -6},   //  347
     {     6,          0,          0,         -2},   //  348
     {   -16,         23,          0,          0},   //  349
     {     0,         -4,         -2,          0},   //  350
     {    -5,          0,          0,          2},   //  351
     {  -166,        269,          0,          0},   //  352
     {    15,          0,          0,         -8},   //  353
     {    10,          0,          0,         -4},   //  354
     {   -78,         45,          0,          0},   //  355
     {     0,         -5,         -2,          0},   //  356
     {     7,          0,          0,         -4},   //  357
     {    -5,        328,          0,          0},   //  358
     {     3,          0,          0,         -2},   //  359
     {     5,          0,          0,         -2},   //  360
     {     0,          3,          1,          0},   //  361
     {    -3,          0,          0,          0},   //  362
     {    -3,          0,          0,          0},   //  363
     {     0,         -4,         -2,          0},   //  364
     { -1223,        -26,          0,          0},   //  365
     {     0,          7,          3,          0},   //  366
     {     3,          0,          0,          0},   //  367
     {     0,          3,          2,          0},   //  368
     {    -6,         20,          0,          0},   //  369
     {  -368,          0,          0,          0},   //  370
     {   -75,          0,          0,          0},   //  371
     {    11,          0,          0,         -6},   //  372
     {     3,          0,          0,         -2},   //  373
     {    -3,          0,          0,          1},   //  374
     {   -13,        -30,          0,          0},   //  375
     {    21,          3,          0,          0},   //  376
     {    -3,          0,          0,          1},   //  377
     {    -4,          0,          0,          2},   //  378
     {     8,        -27,          0,          0},   //  379
     {   -19,        -11,          0,          0},   //  380
     {    -4,          0,          0,          2},   //  381
     {     0,          5,          2,          0},   //  382
     {    -6,          0,          0,          2},   //  383
     {    -8,          0,          0,          0},   //  384
     {    -1,          0,          0,          0},   //  385
     {   -14,          0,          0,          6},   //  386
     {     6,          0,          0,          0},   //  387
     {   -74,          0,          0,         32},   //  388
     {     0,         -3,         -1,          0},   //  389
     {     4,          0,          0,         -2},   //  390
     {     8,         11,          0,          0},   //  391
     {     0,          3,          2,          0},   //  392
     {  -262,          0,          0,        114},   //  393
     {     0,         -4,          0,          0},   //  394
     {    -7,          0,          0,          4},   //  395
     {     0,        -27,        -12,          0},   //  396
     {   -19,         -8,         -4,          8},   //  397
     {   202,          0,          0,        -87},   //  398
     {    -8,         35,         19,          5},   //  399
     {     0,          4,          2,          0},   //  400
     {    16,         -5,          0,          0},   //  401
     {     5,          0,          0,         -3},   //  402
     {     0,         -3,          0,          0},   //  403
     {     1,          0,          0,          0},   //  404
     {   -35,        -48,        -21,         15},   //  405
     {    -3,         -5,         -2,          1},   //  406
     {     6,          0,          0,         -3},   //  407
     {     3,          0,          0,         -1},   //  408
     {     0,         -5,          0,          0},   //  409
     {    12,         55,         29,         -6},   //  410
     {     0,          5,          3,          0},   //  411
     {  -598,          0,          0,          0},   //  412
     {    -3,        -13,         -7,          1},   //  413
     {    -5,         -7,         -3,          2},   //  414
     {     3,          0,          0,         -1},   //  415
     {     5,         -7,          0,          0},   //  416
     {     4,          0,          0,         -2},   //  417
     {    16,         -6,          0,          0},   //  418
     {     8,         -3,          0,          0},   //  419
     {     8,        -31,        -16,         -4},   //  420
     {     0,          3,          1,          0},   //  421
     {   113,          0,          0,        -49},   //  422
     {     0,        -24,        -10,          0},   //  423
     {     4,          0,          0,         -2},   //  424
     {    27,          0,          0,          0},   //  425
     {    -3,          0,          0,          1},   //  426
     {     0,         -4,         -2,          0},   //  427
     {     5,          0,          0,         -2},   //  428
     {     0,         -3,          0,          0},   //  429
     {   -13,          0,          0,          6},   //  430
     {     5,          0,          0,         -2},   //  431
     {   -18,        -10,         -4,          8},   //  432
     {    -4,        -28,          0,          0},   //  433
     {    -5,          6,          3,          2},   //  434
     {    -3,          0,          0,          1},   //  435
     {    -5,         -9,         -4,          2},   //  436
     {    17,          0,          0,         -7},   //  437
     {    11,          4,          0,          0},   //  438
     {     0,         -6,         -2,          0},   //  439
     {    83,         15,          0,          0},   //  440
     {    -4,          0,          0,          2},   //  441
     {     0,       -114,        -49,          0},   //  442
     {   117,          0,          0,        -51},   //  443
     {    -5,         19,         10,          2},   //  444
     {    -3,          0,          0,          0},   //  445
     {    -3,          0,          0,          2},   //  446
     {     0,         -3,         -1,          0},   //  447
     {     3,          0,          0,          0},   //  448
     {     0,         -6,         -2,          0},   //  449
     {   393,          3,          0,          0},   //  450
     {    -4,         21,         11,          2},   //  451
     {    -6,          0,         -1,          3},   //  452
     {    -3,          8,          4,          1},   //  453
     {     8,          0,          0,          0},   //  454
     {    18,        -29,        -13,         -8},   //  455
     {     8,         34,         18,         -4},   //  456
     {    89,          0,          0,          0},   //  457
     {     3,         12,          6,         -1},   //  458
     {    54,        -15,         -7,        -24},   //  459
     {     0,          3,          0,          0},   //  460
     {     3,          0,          0,         -1},   //  461
     {     0,         35,          0,          0},   //  462
     {  -154,        -30,        -13,         67},   //  463
     {    15,          0,          0,          0},   //  464
     {     0,          4,          2,          0},   //  465
     {     0,          9,          0,          0},   //  466
     {    80,        -71,        -31,        -35},   //  467
     {     0,        -20,         -9,          0},   //  468
     {    11,          5,          2,         -5},   //  469
     {    61,        -96,        -42,        -27},   //  470
     {    14,          9,          4,         -6},   //  471
     {   -11,         -6,         -3,          5},   //  472
     {     0,         -3,         -1,          0},   //  473
     {   123,       -415,       -180,        -53},   //  474
     {     0,          0,          0,        -35},   //  475
     {    -5,          0,          0,          0},   //  476
     {     7,        -32,        -17,         -4},   //  477
     {     0,         -9,         -5,          0},   //  478
     {     0,         -4,          2,          0},   //  479
     {   -89,          0,          0,         38},   //  480
     {     0,        -86,        -19,         -6},   //  481
     {     0,          0,        -19,          6},   //  482
     {  -123,       -416,       -180,         53},   //  483
     {     0,         -3,         -1,          0},   //  484
     {    12,         -6,         -3,         -5},   //  485
     {   -13,          9,          4,          6},   //  486
     {     0,        -15,         -7,          0},   //  487
     {     3,          0,          0,         -1},   //  488
     {   -62,        -97,        -42,         27},   //  489
     {   -11,          5,          2,          5},   //  490
     {     0,        -19,         -8,          0},   //  491
     {    -3,          0,          0,          1},   //  492
     {     0,          4,          2,          0},   //  493
     {     0,          3,          0,          0},   //  494
     {     0,          4,          2,          0},   //  495
     {   -85,        -70,        -31,         37},   //  496
     {   163,        -12,         -5,        -72},   //  497
     {   -63,        -16,         -7,         28},   //  498
     {   -21,        -32,        -14,          9},   //  499
     {     0,         -3,         -1,          0},   //  500
     {     3,          0,          0,         -2},   //  501
     {     0,          8,          0,          0},   //  502
     {     3,         10,          4,         -1},   //  503
     {     3,          0,          0,         -1},   //  504
     {     0,         -7,         -3,          0},   //  505
     {     0,         -4,         -2,          0},   //  506
     {     6,         19,          0,          0},   //  507
     {     5,       -173,        -75,         -2},   //  508
     {     0,         -7,         -3,          0},   //  509
     {     7,        -12,         -5,         -3},   //  510
     {    -3,          0,          0,          2},   //  511
     {     3,         -4,         -2,         -1},   //  512
     {    74,          0,          0,        -32},   //  513
     {    -3,         12,          6,          2},   //  514
     {    26,        -14,         -6,        -11},   //  515
     {    19,          0,          0,         -8},   //  516
     {     6,         24,         13,         -3},   //  517
     {    83,          0,          0,          0},   //  518
     {     0,        -10,         -5,          0},   //  519
     {    11,         -3,         -1,         -5},   //  520
     {     3,          0,          1,         -1},   //  521
     {     3,          0,          0,         -1},   //  522
     {    -4,          0,          0,          0},   //  523
     {     5,        -23,        -12,         -3},   //  524
     {  -339,          0,          0,        147},   //  525
     {     0,        -10,         -5,          0},   //  526
     {     5,          0,          0,          0},   //  527
     {     3,          0,          0,         -1},   //  528
     {     0,         -4,         -2,          0},   //  529
     {    18,         -3,          0,          0},   //  530
     {     9,        -11,         -5,         -4},   //  531
     {    -8,          0,          0,          4},   //  532
     {     3,          0,          0,         -1},   //  533
     {     0,          9,          0,          0},   //  534
     {     6,         -9,         -4,         -2},   //  535
     {    -4,        -12,          0,          0},   //  536
     {    67,        -91,        -39,        -29},   //  537
     {    30,        -18,         -8,        -13},   //  538
     {     0,          0,          0,          0},   //  539
     {     0,       -114,        -50,          0},   //  540
     {     0,          0,          0,         23},   //  541
     {   517,         16,          7,       -224},   //  542
     {     0,         -7,         -3,          0},   //  543
     {   143,         -3,         -1,        -62},   //  544
     {    29,          0,          0,        -13},   //  545
     {    -4,          0,          0,          2},   //  546
     {    -6,          0,          0,          3},   //  547
     {     5,         12,          5,         -2},   //  548
     {   -25,          0,          0,         11},   //  549
     {    -3,          0,          0,          1},   //  550
     {     0,          4,          2,          0},   //  551
     {   -22,         12,          5,         10},   //  552
     {    50,          0,          0,        -22},   //  553
     {     0,          7,          4,          0},   //  554
     {     0,          3,          1,          0},   //  555
     {    -4,          4,          2,          2},   //  556
     {    -5,        -11,         -5,          2},   //  557
     {     0,          4,          2,          0},   //  558
     {     4,         17,          9,         -2},   //  559
     {    59,          0,          0,          0},   //  560
     {     0,         -4,         -2,          0},   //  561
     {    -8,          0,          0,          4},   //  562
     {    -3,          0,          0,          0},   //  563
     {     4,        -15,         -8,         -2},   //  564
     {   370,         -8,          0,       -160},   //  565
     {     0,          0,         -3,          0},   //  566
     {     0,          3,          1,          0},   //  567
     {    -6,          3,          1,          3},   //  568
     {     0,          6,          0,          0},   //  569
     {   -10,          0,          0,          4},   //  570
     {     0,          9,          4,          0},   //  571
     {     4,         17,          7,         -2},   //  572
     {    34,          0,          0,        -15},   //  573
     {     0,          5,          3,          0},   //  574
     {    -5,          0,          0,          2},   //  575
     {   -37,         -7,         -3,         16},   //  576
     {     3,         13,          7,         -2},   //  577
     {    40,          0,          0,          0},   //  578
     {     0,         -3,         -2,          0},   //  579
     {  -184,         -3,         -1,         80},   //  580
     {    -3,          0,          0,          1},   //  581
     {    -3,          0,          0,          0},   //  582
     {     0,        -10,         -6,         -1},   //  583
     {    31,         -6,          0,        -13},   //  584
     {    -3,        -32,        -14,          1},   //  585
     {    -7,          0,          0,          3},   //  586
     {     0,         -8,         -4,          0},   //  587
     {     3,         -4,          0,          0},   //  588
     {     0,          4,          0,          0},   //  589
     {     0,          3,          1,          0},   //  590
     {    19,        -23,        -10,          2},   //  591
     {     0,          0,          0,        -10},   //  592
     {     0,          3,          2,          0},   //  593
     {     0,          9,          5,         -1},   //  594
     {    28,          0,          0,          0},   //  595
     {     0,         -7,         -4,          0},   //  596
     {     8,         -4,          0,         -4},   //  597
     {     0,          0,         -2,          0},   //  598
     {     0,          3,          0,          0},   //  599
     {    -3,          0,          0,          1},   //  600
     {    -9,          0,          1,          4},   //  601
     {     3,         12,          5,         -1},   //  602
     {    17,         -3,         -1,          0},   //  603
     {     0,          7,          4,          0},   //  604
     {    19,          0,          0,          0},   //  605
     {     0,         -5,         -3,          0},   //  606
     {    14,         -3,          0,         -1},   //  607
     {     0,          0,         -1,          0},   //  608
     {     0,          0,          0,         -5},   //  609
     {     0,          5,          3,          0},   //  610
     {    13,          0,          0,          0},   //  611
     {     0,         -3,         -2,          0},   //  612
     {     2,          9,          4,          3},   //  613
     {     0,          0,          0,         -4},   //  614
     {     8,          0,          0,          0},   //  615
     {     0,          4,          2,          0},   //  616
     {     6,          0,          0,         -3},   //  617
     {     6,          0,          0,          0},   //  618
     {     0,          3,          1,          0},   //  619
     {     5,          0,          0,         -2},   //  620
     {     3,          0,          0,         -1},   //  621
     {    -3,          0,          0,          0},   //  622
     {     6,          0,          0,          0},   //  623
     {     7,          0,          0,          0},   //  624
     {    -4,          0,          0,          0},   //  625
     {     4,          0,          0,          0},   //  626
     {     6,          0,          0,          0},   //  627
     {     0,         -4,          0,          0},   //  628
     {     0,         -4,          0,          0},   //  629
     {     5,          0,          0,          0},   //  630
     {    -3,          0,          0,          0},   //  631
     {     4,          0,          0,          0},   //  632
     {    -5,          0,          0,          0},   //  633
     {     4,          0,          0,          0},   //  634
     {     0,          3,          0,          0},   //  635
     {    13,          0,          0,          0},   //  636
     {    21,         11,          0,          0},   //  637
     {     0,         -5,          0,          0},   //  638
     {     0,         -5,         -2,          0},   //  639
     {     0,          5,          3,          0},   //  640
     {     0,         -5,          0,          0},   //  641
     {    -3,          0,          0,          2},   //  642
     {    20,         10,          0,          0},   //  643
     {   -34,          0,          0,          0},   //  644
     {   -19,          0,          0,          0},   //  645
     {     3,          0,          0,         -2},   //  646
     {    -3,          0,          0,          1},   //  647
     {    -6,          0,          0,          3},   //  648
     {    -4,          0,          0,          0},   //  649
     {     3,          0,          0,          0},   //  650
     {     3,          0,          0,          0},   //  651
     {     4,          0,          0,          0},   //  652
     {     3,          0,          0,         -1},   //  653
     {     6,          0,          0,         -3},   //  654
     {    -8,          0,          0,          3},   //  655
     {     0,          3,          1,          0},   //  656
     {    -3,          0,          0,          0},   //  657
     {     0,         -3,         -2,          0},   //  658
     {   126,        -63,        -27,        -55},   //  659
     {    -5,          0,          1,          2},   //  660
     {    -3,         28,         15,          2},   //  661
     {     5,          0,          1,         -2},   //  662
     {     0,          9,          4,          1},   //  663
     {     0,          9,          4,         -1},   //  664
     {  -126,        -63,        -27,         55},   //  665
     {     3,          0,          0,         -1},   //  666
     {    21,        -11,         -6,        -11},   //  667
     {     0,         -4,          0,          0},   //  668
     {   -21,        -11,         -6,         11},   //  669
     {    -3,          0,          0,          1},   //  670
     {     0,          3,          1,          0},   //  671
     {     8,          0,          0,         -4},   //  672
     {    -6,          0,          0,          3},   //  673
     {    -3,          0,          0,          1},   //  674
     {     3,          0,          0,         -1},   //  675
     {    -3,          0,          0,          1},   //  676
     {    -5,          0,          0,          2},   //  677
     {    24,        -12,         -5,        -11},   //  678
     {     0,          3,          1,          0},   //  679
     {     0,          3,          1,          0},   //  680
     {     0,          3,          2,          0},   //  681
     {   -24,        -12,         -5,         10},   //  682
     {     4,          0,         -1,         -2},   //  683
     {    13,          0,          0,         -6},   //  684
     {     7,          0,          0,         -3},   //  685
     {     3,          0,          0,         -1},   //  686
     {     3,          0,          0,         -1}    //  687
  };
  DebugAssert(which < 687, AipsError);
  return &(MULSC[which][0]);
}

const Double* MeasTable::mulSCEqEqCT2000(uInt which) {
  // Equation of Equinox complementary terms
  static const Double MULSC[34][2] = {
  //        sin                 cos
     {  +2640.96e-6,          -0.39e-6 },     //  1
     {    +63.52e-6,          -0.02e-6 },     //  2
     {    +11.75e-6,          +0.01e-6 },     //  3
     {    +11.21e-6,          +0.01e-6 },     //  4
     {     -4.55e-6,          +0.00e-6 },     //  5
     {     +2.02e-6,          +0.00e-6 },     //  6
     {     +1.98e-6,          +0.00e-6 },     //  7
     {     -1.72e-6,          +0.00e-6 },     //  8
     {     -1.41e-6,          -0.01e-6 },     //  9
     {     -1.26e-6,          -0.01e-6 },     // 10
     {     -0.63e-6,          +0.00e-6 },     // 11
     {     -0.63e-6,          +0.00e-6 },     // 12
     {     +0.46e-6,          +0.00e-6 },     // 13
     {     +0.45e-6,          +0.00e-6 },     // 14
     {     +0.36e-6,          +0.00e-6 },     // 15
     {     -0.24e-6,          -0.12e-6 },     // 16
     {     +0.32e-6,          +0.00e-6 },     // 17
     {     +0.28e-6,          +0.00e-6 },     // 18
     {     +0.27e-6,          +0.00e-6 },     // 19
     {     +0.26e-6,          +0.00e-6 },     // 20
     {     -0.21e-6,          +0.00e-6 },     // 21
     {     +0.19e-6,          +0.00e-6 },     // 22
     {     +0.18e-6,          +0.00e-6 },     // 23
     {     -0.10e-6,          +0.05e-6 },     // 24
     {     +0.15e-6,          +0.00e-6 },     // 25
     {     -0.14e-6,          +0.00e-6 },     // 26
     {     +0.14e-6,          +0.00e-6 },     // 27
     {     -0.14e-6,          +0.00e-6 },     // 28
     {     +0.14e-6,          +0.00e-6 },     // 29
     {     +0.13e-6,          +0.00e-6 },     // 30
     {     -0.11e-6,          +0.00e-6 },     // 31
     {     +0.11e-6,          +0.00e-6 },     // 32
     {     +0.11e-6,          +0.00e-6 },     // 33
     {     -0.87e-6,          +0.00e-6 }      // T^1 term
  };
  DebugAssert(which < 34, AipsError);
  return &(MULSC[which][0]);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulSC1950(Double time, Double epsilon) {
  return theirMulSC1950.getArray (time, epsilon);
}


Double MeasTable::dPsiEps(uInt which, Double T) {
#if defined(USE_THREADS)
  static std::atomic<Bool> msgDone;
#else
  static Bool msgDone;
#endif
  DebugAssert(which < 2, AipsError);
  Double r = 0;
  MeasIERS::Types type = (which==1  ?  MeasIERS::dEps : MeasIERS::dPsi);
  if (!MeasIERS::get(r, MeasIERS::MEASURED, type, T)) {
    // It is harmless if the message accidentally appears multiple times.
    if (!msgDone) {
      LogIO os(LogOrigin("MeasTable", "dPsiEps(uInt, Double)", WHERE));
      os << LogIO::NORMAL3
         << "High precision nutation information not available."
         << LogIO::POST;
      msgDone = True;
    }
  }
  ///  cout << "psieps " << r << endl;
  return (r * C::arcsec);
}

// Planetary data
Vector<Double> MeasTable::Planetary(MeasTable::Types which, Double T) {
  static MeasJPL::Files fil(MeasJPL::DE200);
  std::call_once(theirPlanetaryInitOnceFlag, calcPlanetary, &fil);

  Vector<Double> res(6);
  if (!MeasJPL::get(res, fil, (MeasJPL::Types)which, MVEpoch(T))) {
    const String tnam[2] = {"DE200", "DE405"};
    LogIO os(LogOrigin("MeasTable",
                       "Planetary(MeasTable::Types, Double)", WHERE));
    os << "Cannot find the planetary data for MeasJPL object number "
       << (Int) which << " at UT day " << T << " in table " << tnam[fil]
       << LogIO::WARN;
    res = 0.;
  }
  return res;
}

void MeasTable::calcPlanetary(MeasJPL::Files *fil) {
  const String tnam[2] = {"DE200", "DE405"};
  uInt t;
  Aipsrc::find(t, "measures.jpl.ephemeris", 2, tnam, "DE200");
  *fil = (MeasJPL::Files)t;
}

// Planetary constants
Double MeasTable::Planetary(MeasTable::JPLconst what) {
  static Double cn[MeasTable::N_JPLconst];
  std::call_once(theirPlanetaryConstantsInitOnceFlag, calcPlanetaryConstants, cn);
  return cn[what];
}

void MeasTable::calcPlanetaryConstants(Double cn[MeasTable::N_JPLconst]) {
  const String tnam[2] = {"DE200", "DE405"};
  uInt t;
  Aipsrc::find(t, "measures.jpl.ephemeris", 2, tnam, "DE200");
  MeasJPL::Files fil = (MeasJPL::Files)t;

  for (uInt i=0; i<MeasTable::N_JPLconst; i++) {
    if (!MeasJPL::getConst(cn[i], fil, (MeasJPL::Codes) i)) {
      LogIO os(LogOrigin("MeasTable",
                         "Planetary(MeasTable::JPLconst)", WHERE));
      os << "Cannot find the planetary data table " + tnam[fil]
         << LogIO::EXCEPTION;
    }
  }
}

// Observatory data
void MeasTable::initObservatories() {
  std::call_once(theirObsInitOnceFlag, doInitObservatories);
}

void MeasTable::doInitObservatories()
{
  Table t;
  ROTableRow row;
  TableRecord kws;
  String rfn[3] = {"Long", "Lat", "Height"};
  RORecordFieldPtr<Double> rfp[3];
  Double dt;
  String vs;	
  if (!MeasIERS::getTable(t, kws, row, rfp, vs, dt, 3, rfn, "Observatories",
                          "measures.observatory.directory", "geodetic")) {
    LogIO os(LogOrigin("MeasTable", "doInitObservatories()", WHERE));
    os << "Cannot read table of Observatories" << LogIO::EXCEPTION;
  }
  Int N = t.nrow();
  if (N<1) {
    LogIO os(LogOrigin("MeasTable", "doInitObservatories()", WHERE));
    os << "No entries in table of Observatories" << LogIO::EXCEPTION;
  }
  obsNams.resize(N);
  obsPos.resize(N);
  antResponsesPath.resize(N);
  Bool hasAntResp = False;
  if(row.record().isDefined("AntennaResponses")){
    hasAntResp = True;
  }

  MPosition::Ref mr;
  MPosition tmp;
  for (Int i=0; i<N; i++) {
    row.get(i);
    obsNams(i) = *RORecordFieldPtr<String>(row.record(), "Name");
    if(hasAntResp){
      antResponsesPath(i) = *RORecordFieldPtr<String>(row.record(), "AntennaResponses");
    }
    if (!tmp.giveMe(mr, *RORecordFieldPtr<String>(row.record(), "Type"))) {
      LogIO os(LogOrigin("MeasTable", "doInitObservatories()", WHERE));
      os << "Illegal position type in Observatories" << LogIO::EXCEPTION;
    }
    obsPos(i) = MPosition(MVPosition(Quantity(*(rfp[2]), "m"),
                                     Quantity(*(rfp[0]), "deg"),
                                     Quantity(*(rfp[1]), "deg")), mr);
  }
}

const Vector<String> &MeasTable::Observatories() {
  std::call_once(theirObsInitOnceFlag, doInitObservatories);
  return MeasTable::obsNams;
}

Bool MeasTable::Observatory(MPosition &obs, const String &nam) {
  std::call_once(theirObsInitOnceFlag, doInitObservatories);
  uInt i=MUString::minimaxNC(nam, MeasTable::obsNams);
  if (i < MeasTable::obsNams.nelements()) {
    obs = MeasTable::obsPos[i];
    return True;
  }
  return False;
}

Bool MeasTable::AntennaResponsesPath(String &antRespPath, const String &nam) {
  std::call_once(theirObsInitOnceFlag, doInitObservatories);
  uInt i=MUString::minimaxNC(nam, MeasTable::obsNams);
  if (i < MeasTable::obsNams.nelements()) {
    antRespPath = MeasTable::antResponsesPath(i);
    if(antRespPath.empty()){ // i.e. there is no table for this observatory
      return False; 
    }
    else if(antRespPath[0] == '/'){ // path is absolute
      Path lPath(antRespPath);
      if(!Table::isReadable(lPath.absoluteName())){
	return False;
      }
    }
    else{ // path is relative
      // find and prepend the path to the data repository
      String absPathName;
      Bool isValid = False;
      {
	String mdir;
	Aipsrc::find(mdir, "measures.directory");
	mdir.trim();
	Path lPath(mdir);
	lPath.append(antRespPath);
	absPathName = lPath.absoluteName();
	isValid = Table::isReadable(absPathName);
      }
      if(!isValid){
        String casadata=String(CASADATA);
        casadata.gsub("%CASAROOT%", Aipsrc::aipsRoot());
        casadata.gsub("%CASAHOME%", Aipsrc::aipsHome());
        Path lPath(casadata + "/" + antRespPath);
        isValid = Table::isReadable(absPathName);
      }
      if(!isValid){
	return False; // table not found
      }
      antRespPath = absPathName;
    }
    return True;
  }
  return False; // observatory not found
}

// Line data
void MeasTable::initLines() {
  std::call_once(theirLinesInitOnceFlag, doInitLines);
}

void MeasTable::doInitLines()
{
  Table t;
  ROTableRow row;
  TableRecord kws;
  String rfn[1] = {"Freq"};
  RORecordFieldPtr<Double> rfp[1];
  Double dt;
  String vs;	
  if (!MeasIERS::getTable(t, kws, row, rfp, vs, dt, 1, rfn, "Lines",
                          "measures.line.directory", "ephemerides")) {
    LogIO os(LogOrigin("MeasTable", "doInitLines()", WHERE));
    os << "Cannot read table of spectral Lines" << LogIO::EXCEPTION;
  }
  Int N = t.nrow();
  if (N<1) {
    LogIO os(LogOrigin("MeasTable", "doInitLines()", WHERE));
    os << "No entries in table of spectral Lines" << LogIO::EXCEPTION;
  }
  lineNams.resize(N);
  linePos.resize(N);
  MFrequency::Ref mr(MFrequency::REST);
  MFrequency tmp;
  for (Int i=0; i<N; i++) {
    row.get(i);
    lineNams(i) = *RORecordFieldPtr<String>(row.record(), "Name");
    linePos(i) = MFrequency(MVFrequency(Quantity(*(rfp[0]), "GHz")), mr);
    if (lineNams(i) == "HI") linePos(i) = MFrequency(QC::HI( ), mr);
  }
}

const Vector<String> &MeasTable::Lines() {
  std::call_once(theirLinesInitOnceFlag, doInitLines);
  return MeasTable::lineNams;
}

Bool MeasTable::Line(MFrequency &obs, const String &nam) {
  std::call_once(theirLinesInitOnceFlag, doInitLines);
  uInt i=MUString::minimaxNC(nam, MeasTable::lineNams);
  if (i < MeasTable::lineNams.nelements()) {
    obs = MeasTable::linePos(i);
    return True;
  }
  return False;
}

// Source data
void MeasTable::initSources() {
  std::call_once(theirSrcInitOnceFlag, doInitSources);
}

void MeasTable::doInitSources()
{
  Table t;
  ROTableRow row;
  TableRecord kws;
  String rfn[2] = {"Long", "Lat"};
  RORecordFieldPtr<Double> rfp[2];
  Double dt;
  String vs;	
  if (!MeasIERS::getTable(t, kws, row, rfp, vs, dt, 2, rfn, "Sources",
                          "measures.sources.directory", "ephemerides")) {
    LogIO os(LogOrigin("MeasTable", "doInitSources()", WHERE));
    os << "Cannot read table of Sources" << LogIO::EXCEPTION;
  }
  Int N = t.nrow();
  if (N<1) {
    LogIO os(LogOrigin("MeasTable", "doInitSources()", WHERE));
    os << "No entries in table of Sources" << LogIO::EXCEPTION;
  }
  srcNams.resize(N);
  srcPos.resize(N);
  MDirection::Ref mr;
  MDirection tmp;
  for (Int i=0; i<N; i++) {
    row.get(i);
    srcNams(i) = *RORecordFieldPtr<String>(row.record(), "Name");
    if (!tmp.giveMe(mr, *RORecordFieldPtr<String>(row.record(), "Type"))) {
      LogIO os(LogOrigin("MeasTable", "doInitSources()", WHERE));
      os << "Illegal direction type in Sources" << LogIO::EXCEPTION;
    }
    srcPos(i) = MDirection(MVDirection(Quantity(*(rfp[0]), "deg"),
                                       Quantity(*(rfp[1]), "deg")), mr);
  }
}

const Vector<String> &MeasTable::Sources() {
  std::call_once(theirSrcInitOnceFlag, doInitSources);
  return MeasTable::srcNams;
}

Bool MeasTable::Source(MDirection &obs, const String &nam) {
  std::call_once(theirSrcInitOnceFlag, doInitSources);
  uInt i=MUString::minimaxNC(nam, MeasTable::srcNams);
  if (i < MeasTable::srcNams.nelements()) {
    obs = MeasTable::srcPos(i);
    return True;
  }
  return False;
}

// Magnetic field (IGRF) function
Vector<Double> MeasTable::IGRF(Double tm) {
  std::call_once(theirIGRFInitOnceFlag, doInitIGRF);
  // Look up closest MJD interval. Note that each interval has same width.
  Int indx = Int((tm-firstIGRF) / dtimeIGRF) - 1;
  if (indx >= Int(coefIGRF.size())) {
    indx = coefIGRF.size() - 1;
  } else if (indx < 0) {
    indx = 0;
  }
  // Interpolate using the d value.
  /// What is factor 5 meaning?
  double mjd = tm - (firstIGRF + (indx+1)*dtimeIGRF);
  return coefIGRF[indx] + dIGRF[indx] * (5*mjd/dtimeIGRF);
}

void MeasTable::initIGRF() {
  std::call_once(theirIGRFInitOnceFlag, doInitIGRF);
}

void MeasTable::doInitIGRF()
{
  Table t;
  TableRecord kws;
  ROTableRow row;
  const String rfn[1] = {"MJD"};
  RORecordFieldPtr<Double> rfp[1];
  Double dt;
  String vs;	
  if (!MeasIERS::getTable(t, kws, row, rfp, vs, dt, 1, rfn, "IGRF",
                          "measures.igrf.directory", "geodetic")) {
    LogIO os(LogOrigin("MeasTable", "doInitIGRF()", WHERE));
    os << "Cannot read table of IGRF models" << LogIO::EXCEPTION;
  }
  Int N = t.nrow();
  if (N<10 || !kws.isDefined("MJD0") || kws.asDouble("MJD0") < 10000 ||
      !kws.isDefined("dMJD") || kws.asDouble("dMJD") < 300) {
    LogIO os(LogOrigin("MeasTable", "doInitIGRF()", WHERE));
    os << "Incorrect entries in table of IGRF models" << LogIO::EXCEPTION;
  }
  firstIGRF = kws.asDouble("MJD0");
  dtimeIGRF = kws.asDouble("dMJD");
  coefIGRF.reserve (N);
  dIGRF.reserve (N);
  ScalarColumn<Double> accmjd(t, "MJD");
  ArrayColumn<Double> acc(t, "COEF");
  ArrayColumn<Double> accd(t, "dCOEF");
  for (Int i=0; i<N; ++i) {
    double igrfmjd = accmjd(i);
    if (! near(igrfmjd, firstIGRF+(i+1)*dtimeIGRF)) {
      LogIO os(LogOrigin("MeasTable", "doInitIGRF()", WHERE));
      os << "Non-constant MJD increment in IGRF models table"
         << LogIO::EXCEPTION;
    }
    coefIGRF.push_back (acc(i));
    dIGRF.push_back (accd(i));
  }
}

// Aberration function
const Polynomial<Double> &MeasTable::aberArg(uInt which) {
  static const std::vector<Polynomial<Double>> polyArray(calcAberArg());
  DebugAssert(which < 13, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcAberArg() {
  static const Double ABERFUND[13][2] = {
    {4.4026088,	2608.7903142},
    {3.1761467,	1021.3285546},
    {1.7534703,	 628.3075849},
    {6.2034809,	 334.0612431},
    {0.5995465,	  52.9690965},
    {0.8740168,	  21.3299095},
    {5.4812939,	   7.4781599},
    {5.3118863,	   3.8133036},
    {3.8103444,	8399.6847337},
    {5.1984667,	7771.3771486},
    {2.3555559,	8328.6914289},
    {6.2400601,	 628.3019553},
    {1.6279052,	8433.4661601}
  };
  std::vector<Polynomial<Double>> polyArray(13);
  Int i,j;
  for (i=0; i<13; i++) {
    polyArray[i] = Polynomial<Double>(1);
    for (j=0; j<2; j++) {
      polyArray[i].setCoefficient(j, ABERFUND[i][j]);
    }
  }
  return polyArray;
}

// Derivative aber
const Polynomial<Double> &MeasTable::aberArgDeriv(uInt which) {
  static const std::vector<Polynomial<Double>> polyArray(calcAberArgDeriv());
  DebugAssert(which < 13, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcAberArgDeriv() {
  std::vector<Polynomial<Double>> polyArray(13);
  for (int i=0; i<13; i++) {
    const Polynomial<Double> *polyPtr = &aberArg(i);
    polyArray[i] = polyPtr->derivative();
  }
  return polyArray;
}

const Polynomial<Double> &MeasTable::aber1950Arg(uInt which) {
  static const std::vector<Polynomial<Double>> polyArray(calcAber1950Arg());
  DebugAssert(which < 12, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcAber1950Arg() {
  static const Double ABERFUND[12][4] = {
    {1065976.59, 1717915856.79, 33.09,   0.0518},
    {1290513.0,  129596579.1,   -0.54,   -0.0120},
    {40503.2,    1739527290.54, -11.56,  -0.0012},
    {1262654.95, 1602961611.18, -5.17,   0.0068},
    {933059.79,  -6962911.23,   7.48,    0.0080},
    {764820.00,	210662974.800,	0,	0},
    {1150495.2,	68903917.200,	0,	0},
    {811011.60,	10924498.800,	0,	0},
    {632145.60,	4398458.400,	0,	0},
    {0,		8128.800,	0,	0},
    {260701.20,	1542164.400,	0,	0},
    {135831.60,	786459.600,	0,	0}
  };
  std::vector<Polynomial<Double>> polyArray(12);
  Int i,j;
  for (i=0; i<12; i++) {
    polyArray[i] = Polynomial<Double>(3);
    for (j=0; j<4; j++) {
      polyArray[i].setCoefficient(j, ABERFUND[i][j]*C::arcsec);
    }
  }
  return polyArray;
}

// Derivative aber1950
const Polynomial<Double> &MeasTable::aber1950ArgDeriv(uInt which) {
  static const std::vector<Polynomial<Double>> polyArray(calcAber1950ArgDeriv());
  DebugAssert(which < 12, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcAber1950ArgDeriv() {
  std::vector<Polynomial<Double>> polyArray(12);
  for (int i=0; i<12; i++) {
    const Polynomial<Double> *polyPtr = &aber1950Arg(i);
    polyArray[i] = polyPtr->derivative();
  }
  return polyArray;
}

const Double* MeasTable::mulAberArg(uInt which) {
  static const Double ABERARG[80][6] = {
    {	0,	0,	1,	0,	0,	0},
    {	0,	0,	2,	0,	0,	0},
    {	0,	0,	3,	0,	0,	0},
    {	0,	0,	2,	0,	-1,	0},
    {	0,	0,	3,	-8,	3,	0},
    {	0,	0,	5,	-8,	3,	0},
    {	0,	1,	0,	0,	0,	0},
    {	0,	2,	-1,	0,	0,	0},
    {	0,	0,	1,	0,	-2,	0},
    {	0,	0,	1,	0,	1,	0},
    {	0,	2,	-2,	0,	0,	0},
    {	0,	0,	1,	0,	-1,	0},
    {	0,	0,	4,	0,	0,	0},
    {	0,	0,	3,	0,	-2,	0},
    {	0,	1,	-2,	0,	0,	0},
    {	0,	2,	-3,	0,	0,	0},
    {	0,	2,	-4,	0,	0,	0},
    {	0,	0,	3,	-2,	0,	0},
    {	0,	8,	-12,	0,	0,	0},
    {	0,	8,	-14,	0,	0,	0},
    {	0,	0,	0,	2,	0,	0},
    {	0,	3,	-4,	0,	0,	0},
    {	0,	0,	2,	0,	-2,	0},
    {	0,	3,	-3,	0,	0,	0},
    {	0,	0,	2,	-2,	0,	0},
    {	0,	3,	-6,	0,	0,	0},
    {	0,	0,	0,	0,	1,	0},
    {	0,	0,	9,	-16,	4,	5},
    {	0,	0,	7,	-16,	4,	5},
    {	0,	0,	1,	0,	-3,	0},
    {	0,	0,	2,	0,	-3,	0},
    {	0,	4,	-5,	0,	0,	0},
    {	0,	0,	1,	-4,	0,	0},
    {	0,	0,	3,	0,	-3,	0},
    {	0,	0,	3,	-4,	0,	0},
    {	0,	3,	-2,	0,	0,	0},
    {	0,	0,	4,	-4,	0,	0},
    {	0,	0,	2,	0,	0,	-1},
    {	0,	0,	3,	-3,	0,	0},
    {	0,	0,	3,	0,	-1,	0}, //40
    {	0,	0,	1,	0,	0,	1},
    {	0,	0,	0,	0,	2,	0},
    {	0,	0,	2,	-1,	0,	0},
    {	0,	0,	1,	0,	0,	-1},
    {	0,	5,	-6,	0,	0,	0},
    {	0,	0,	1,	-3,	0,	0},
    {	0,	3,	-6,	4,	0,	0},
    {	0,	3,	-8,	4,	0,	0},
    {	0,	0,	4,	-5,	0,	0},
    {	0,	1,	1,	0,	0,	0},
    {	0,	3,	-5,	0,	0,	0},
    {	0,	6,	-7,	0,	0,	0},
    {	0,	10,	-9,	0,	0,	0},
    {	0,	0,	2,	-8,	3,	0},
    {	0,	0,	6,	-8,	3,	0},
    {	0,	0,	1,	-2,	0,	0},
    {	0,	0,	9,	-15,	0,	0},
    {	0,	0,	1,	0,	-2,	5},
    {	0,	0,	1,	0,	2,	-5},
    {	0,	0,	1,	0,	0,	-2}, //60
    {	0,	0,	0,	1,	0,	0},
    {	0,	0,	7,	-15,	0,	0},
    {	0,	2,	0,	0,	0,	0},
    {	0,	0,	2,	0,	2,	-5},
    {	2,	0,	-2,	0,	0,	0},
    {	0,	0,	9,	-19,	0,	3},
    {	0,	0,	11,	-19,	0,	3},
    {	0,	0,	2,	-5,	0,	0},
    {	0,	5,	-9,	0,	0,	0},
    {	0,	11,	-10,	0,	0,	0},
    {	0,	4,	-4,	0,	0,	0},
    {	0,	0,	2,	0,	-4,	0},
    {	0,	0,	5,	-6,	0,	0},
    {	0,	5,	-5,	0,	0,	0},
    {	0,	0,	4,	0,	-3,	0},
    {	0,	4,	-6,	0,	0,	0},
    {	0,	5,	-7,	0,	0,	0},
    {	0,	0,	4,	0,	-2,	0},
    {	0,	0,	3,	0,	-4,	0},
    {	0,	7,	-8,	0,	0,	0}
  };
  DebugAssert(which < 80, AipsError);
  return &(ABERARG[which][0]);
}

const Double* MeasTable::mulAber1950Arg(uInt which) {
  static const Double ABERARG[132][12] = {
    { 0, 0, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0,-1,-1, 1,-1, 0, 0, 1, 0, 0, 0, 0},
    { 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    { 1, 0,-1,-2,-1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 1,-1, 1,-1,-1, 0, 0, 0, 0, 0, 0},
    { 0, 4, 1,-1, 1, 0,-8, 3, 0, 0, 0, 0},	// 10
    { 0,-4, 1,-1, 1, 0, 8,-3, 0, 0, 0, 0},
    { 0, 2,-1, 1,-1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 2,-1, 1,-1, 0, 0,-2, 0, 0, 0, 0},
    { 0,-4, 1,-1, 1, 0, 8,-3, 0, 0, 0, 0},
    { 0, 4, 1,-1, 1, 0,-8, 3, 0, 0, 0, 0},
    { 0, 1,-1, 1,-1,-1, 0, 0, 0, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 1, 0, 0, 0, 0},
    { 0, 2,-1, 1,-1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 0, 1, 2, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0,-3,-1, 1,-1, 0, 0, 0, 0, 0, 0, 0},	// 20
    { 0, 0,-1, 1,-1, 0, 0, 1, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1, 0, 0,-2, 0, 0, 0, 0},
    { 0, 3,-1, 1,-1,-2, 0, 0, 0, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 2, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1,-1, 0, 0, 0, 0, 0, 0},
    { 0, 3,-1, 1,-1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1, 0,-2, 0, 0, 0, 0, 0},
    { 0,-1, 1,-1, 1, 0, 2, 0, 0, 0, 0, 0},
    { 0, 3, 1,-1, 1,-2, 0, 0, 0, 0, 0, 0},	// 30
    { 0,-1,-1, 1,-1, 0, 0, 2, 0, 0, 0, 0},
    { 0, 4,-1, 1,-1,-3, 0, 0, 0, 0, 0, 0},
    { 0, 3, 1,-1, 1,-2, 0, 0, 0, 0, 0, 0},
    { 0,13, 1,-1, 1,-8, 0, 0, 0, 2, 0, 0},
    {0,-13, 1,-1, 1, 8, 0, 0, 0,-2, 0, 0},
    { 0,13, 1,-1, 1,-8, 0, 0, 0, 2, 0, 0},
    {0,-13, 1,-1, 1, 8, 0, 0, 0,-2, 0, 0},
    { 0, 0,-1, 2,-1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1,-1, 0, 0, 0, 0, 0, 0},	// 40
    { 0, 0,-1, 2, 0, 0, 0, 0, 0, 0, 0, 0},
    { 0, 3, 1,-1, 1,-3, 0, 0, 0, 0, 0, 0},
    { 0,-1, 1,-1, 1, 0, 0, 1, 0, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 5, 1,-1, 1,-3, 0, 0, 0, 0, 0, 0},
    { 0, 3,-1, 1,-1, 0, 0,-3, 0, 0, 0, 0},
    { 0,-5, 1,-1, 1, 3, 0, 0, 0, 0, 0, 0},
    { 0, 4, 1,-1, 1,-4, 0, 0, 0, 0, 0, 0},
    { 0, 4, 0, 0, 0,-3, 0, 0, 0, 0, 0, 0},
    { 0, 5, 1,-1, 1,-3, 0, 0, 0, 0, 0, 0},	// 50
    { 0,-5, 1,-1, 1, 3, 0, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1, 0, 0,-2, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 0, 3, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0,-1, 1,-1, 1, 0, 2, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1, 0,-2, 0, 0, 0, 0, 0},
    { 0, 3, 1,-1, 1, 0,-4, 0, 0, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 2, 0, 0, 0, 0, 0},
    { 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0},	// 60
    { 0,-1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 3, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 4, 0, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1, 0,-4, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1, 0, 0, 0,-1, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 0, 3, 0, 0, 0, 0},
    { 0, 3,-1, 1,-1,-3, 0, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 0, 1, 0, 0, 0, 0},
    { 0, 4, 1,-1, 1,-4, 0, 0, 0, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 1, 0, 0, 0, 0},	// 70
    { 0,-2, 1,-1, 1, 0, 0, 2, 0, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 0, 1, 0, 0, 0, 0},
    { 0, 5, 1,-1, 1,-5, 0, 0, 0, 0, 0, 0},
    { 0,-1, 1,-1, 1, 0, 0, 2, 0, 0, 0, 0},
    { 0, 4,-1, 1,-1,-3, 0, 0, 0, 0, 0, 0},
    {-1, 0,-1, 2,-1, 0, 0, 0, 0, 0, 0, 0},
    { 0, 5, 1,-1, 1,-4, 0, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 0, 0, 2, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 0, 1, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 0,-1, 0, 0, 0},	// 80
    { 0, 5,-1,-1, 1,-5, 0, 0, 0, 0, 0, 0},
    { 0, 6, 1,-1, 1,-6, 0, 0, 0, 0, 0, 0},
    { 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0},
    { 0,-7, 1,-1, 1, 3, 4, 0, 0, 0, 0, 0},
    { 0, 7, 1,-1, 1,-3,-4, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 4, 0, 0, 0, 0, 0},
    { 0, 2, 1,-1, 1, 0,-4, 0, 0, 0, 0, 0},
    { 0, 4, 1,-1, 1,-3, 0, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 3, 0, 0, 0, 0, 0},
    { 0, 1, 1,-1, 1,-2, 0, 0, 0, 0, 0, 0},	// 90
    { 0, 1,-1, 1,-1,-2, 0, 0, 0, 0, 0, 0},
    { 1, 0,-1, 2,-1, 0, 0, 0, 0, 0, 0, 0},
    { 0,-1, 1,-1, 1, 0, 0, 0, 1, 0, 0, 0},
    { 0,-3, 1,-1, 1, 0, 0, 2, 0, 0, 0, 0},
    { 0,-3,-1, 1,-1, 0, 0, 2, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 2, 0, 0, 0, 0, 0},
    { 0,-3, 1,-1, 1, 0, 3, 0, 0, 0, 0, 0},
    { 0,-3,-1, 1,-1, 0, 3, 0, 0, 0, 0, 0},
    { 0,-1, 1,-1, 1, 0, 0, 3, 0, 0, 0, 0},
    { 0,-1,-1, 1,-1, 0, 0, 3, 0, 0, 0, 0},	// 100
    { 0, 1, 1,-1, 1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 1,-1, 1,-1,-2, 0, 0, 0, 0, 0, 0},
    { 0, 7, 1,-1, 1,-5, 0, 0, 0, 0, 0, 0},
    { 0, 7,-1, 1,-1,-5, 0, 0, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 0, 4, 0, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 0, 4, 0, 0, 0, 0},
    { 0,-2, 1,-1, 1, 0, 0, 3, 0, 0, 0, 0},
    { 0,-2,-1, 1,-1, 0, 0, 3, 0, 0, 0, 0},
    //                  g2,g3,g4,g5, v, g6,g7
    { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0},	// 110
    { 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
    { 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0},	// 120
    { 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0},
    { 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0},
    { 0, 0, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},	// 130
    { 0, 1, 1,-1, 1, 0, 0, 0, 0, 0, 0, 0},
  };
  DebugAssert(which < 132, AipsError);
  return &(ABERARG[which][0]);
}

const Double* MeasTable::mulAberSunArg(uInt which) {
  static const Double ABERSUNARG[17][7] = {
    {	0,	0,	0,	1,	0,	0,	0},
    {	0,	0,	0,	0,	1,	0,	0},
    {	0,	0,	0,	2,	0,	0,	0},
    {	0,	0,	0,	0,	0,	1,	0},
    {	0,	0,	0,	0,	0,	0,	1},
    {	0,	0,	0,	0,	2,	0,	0},
    {	1,	0,	0,	0,	0,	0,	0},
    {	0,	1,	0,	0,	0,	0,	0},
    {	0,	0,	0,	3,	0,	0,	0},
    {	0,	0,	0,	1,	-5,	0,	0},
    {	0,	0,	0,	3,	-5,	0,	0},
    {	1,	0,	0,	0,	0,	0,	-2},
    {	0,	0,	0,	0,	3,	0,	0},
    {	0,	0,	0,	2,	-6,	0,	0},
    {	0,	0,	0,	2,	-4,	0,	0},
    {	0,	0,	0,	0,	0,	2,	0},
    {	0,	0,	0,	1,	0,	0,	-2}
  };
  DebugAssert(which < 17, AipsError);
  return &(ABERSUNARG[which][0]);
}

const Double* MeasTable::mulAberEarthArg(uInt which) {
  static const Double ABEREARTHARG[17][5] = {
    {	1,	0,	0,	0,	0},
    {	0,	0,	0,	0,	1},
    {	1,	0,	1,	0,	0},
    {	1,	2,	-1,	0,	0},
    {	1,	-2,	0,	0,	0},
    {	1,	2,	0,	0,	0},
    {	0,	0,	1,	0,	1},
    {	1,	-2,	1,	0,	0},
    {	1,	0,	2,	0,	0},
    {	0,	2,	0,	0,	-1},
    {	1,	0,	0,	0,	-2},
    {	1,	0,	0,	1,	0},
    {	1,	0,	0,	-1,	0},
    {	1,	4,	-2,	0,	0},
    {	1,	-2,	2,	0,	0},
    {	1,	2,	1,	0,	0},
    {	0,	2,	-1,	0,	1}
  };
  DebugAssert(which < 17, AipsError);
  return &(ABEREARTHARG[which][0]);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulAber(Double time, Double epsilon) {
  return theirMulAber.getArray (time, epsilon);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulAber1950(Double time, Double epsilon) {
  return theirMulAber1950.getArray (time, epsilon);
}

const Vector<Double> &MeasTable::mulSunAber(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcMulSunAber());
  DebugAssert(which < 17, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcMulSunAber() {
  static const Short MSUNABER[17][6] = {
    {	719,	0,	6,	-660,	-15,	-283},
    {	159,	0,	2,	-147,	-6,	-61},
    {	34,	-9,	-8,	-31,	-4,	-13},
    {	17,	0,	0,	-16,	0,	-7},
    {	16,	0,	1,	-15,	-3,	-6},
    {	0,	-9,	-8,	0,	-3,	1},
    {	6,	0,	0,	-6,	0,	-2},
    {	5,	0,	0,	-5,	0,	-2},
    {	2,	-1,	-1,	-2,	0,	-1},
    {	-2,	0,	0,	-2,	0,	-1},
    {	-2,	0,	0,	2,	0,	1},
    {	-1,	0,	0,	-1,	0,	0},
    {	-1,	0,	0,	1,	0,	0},
    {	1,	0,	0,	1,	0,	0},
    {	1,	0,	0,	-1,	0,	0},
    {	-1,	0,	0,	1,	0,	0},
    {	1,	0,	0,	0,	0,	0}
  };
  std::vector<Vector<Double>> argArray(17);
  UnitVal AUperDay(1e-8,"AU/d");
  Double factor = AUperDay.getFac();
  Int i,j;
  for (i=0; i<17; i++) {
    argArray[i].resize(6);
    for (j=0; j<6; j++) {
      argArray[i](j) = MSUNABER[i][j] * factor;
    }
  }
  return argArray;
}

const Vector<Double> &MeasTable::mulEarthAber(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcMulEarthAber());
  DebugAssert(which < 17, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcMulEarthAber() {
  static const Short MEARTHABER[17][3] = {
    {	715,	-656,	-285},
    {	0,	26,	-59},
    {	39,	-36,	-16},
    {	8,	-7,	-3},
    {	5,	-5,	-2},
    {	4,	-4,	-2},
    {	0,	1,	-3},
    {	-2,	2,	1},
    {	2,	-2,	-1},
    {	0,	1,	-2},
    {	-1,	1,	1},
    {	-1,	1,	0},
    {	1,	-1,	0},
    {	1,	-1,	0},
    {	-1,	1,	0},
    {	1,	0,	0},
    {	0,	0,	-1}
  };
  std::vector<Vector<Double>> argArray(17);
  UnitVal AUperDay(1e-8,"AU/d");
  Double factor = AUperDay.getFac();
  Int i,j;
  for (i=0; i<17; i++) {
    argArray[i].resize(3);
    for (j=0; j<3; j++) {
      argArray[i](j) = MEARTHABER[i][j] * factor;
    }
  }
  return argArray;
}

const Vector<Double> &MeasTable::AberETerm(uInt which) {
  static const std::vector<Vector<Double>> termArray(calcAberETerm());
  DebugAssert(which < 2, AipsError);
  return termArray[which];
}

std::vector<Vector<Double>> MeasTable::calcAberETerm() {
  static const Double TERM[2][3] = {
    {-1.62557,  -0.31919, -0.13843},
    {+1.245,    -1.580,   -0.659}
  };
  std::vector<Vector<Double>> termArray(2);
  Int i;
  for (i=0; i<2; i++) {
    termArray[i].resize(3);
  }
  for (i=0; i<3; i++) {
    termArray[0](i) = TERM[0][i] * 1e-6;
    termArray[1](i) = TERM[1][i] * 1e-3;
  }
  return termArray;
}

// Diurnal Aberration factor
Double MeasTable::diurnalAber(Double radius, Double T) {
  ///  static Double res;
  ///  res = (2.0*M_PI) * radius / MeasData::SECinDAY *
  ///    MeasTable::UTtoST(T)/C::c;
  ///  return res;
  return (2.0*M_PI) * radius / MeasData::SECinDAY *
    MeasTable::UTtoST(T)/C::c;
}

// LSR velocity (kinematical)
const Vector<Double> &MeasTable::velocityLSRK(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcVelocityLSRK());
  DebugAssert(which < 2, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcVelocityLSRK() {
  static const Double LSR[2][3] = {
    {0.0145021,  -0.865863, 0.500071},
    {0.00724658, -0.865985, 0.500018}
  };
  std::vector<Vector<Double>> argArray(2);
  Double v = 20.0*1000.;
  for (Int i=0; i<2; i++) {
    argArray[i].resize(3);
    for (Int j=0; j<3; j++) {
      argArray[i](j) = v * LSR[i][j];
    }
  }
  return argArray;
}

// LSR velocity (dynamical)
const Vector<Double> &MeasTable::velocityLSR(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcVelocityLSR());
  DebugAssert(which < 2, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcVelocityLSR() {
  static const Double LSR[2][3] = {
    {-0.0385568, -0.881138, 0.471285},
    {-0.0461164, -0.880664, 0.471491}
  };
  std::vector<Vector<Double>> argArray(2);
  Double v = sqrt(81.+144.+49.)*1000.;
  for (Int i=0; i<2; i++) {
    argArray[i].resize(3);
    for (Int j=0; j<3; j++) {
      argArray[i](j) = v * LSR[i][j];
    }
  }
  return argArray;
}

// LSR velocity wrt galactic centre
const Vector<Double> &MeasTable::velocityLSRGal(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcVelocityLSRGal());
  DebugAssert(which < 2, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcVelocityLSRGal() {
  static const Double LSR[2][3] = {
    {0.494109, -0.44483 , 0.746982},
    {0.492728, -0.450347, 0.744585}
  };
  std::vector<Vector<Double>> argArray(2);
  Double v = 220.*1000.;
  for (Int i=0; i<2; i++) {
    argArray[i].resize(3);
    for (Int j=0; j<3; j++) {
      argArray[i](j) = v * LSR[i][j];
    }
  }
  return argArray;
}

// LGROUP velocity wrt bary center
const Vector<Double> &MeasTable::velocityLGROUP(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcVelocityLGROUP());
  DebugAssert(which < 2, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcVelocityLGROUP() {
  static const Double LGROUP[2][3] = {
    {0.593553979227, -0.177954636914, 0.784873124106},
    {0.5953342407,   -0.184600136022, 0.781984610866} 
  };
  std::vector<Vector<Double>> argArray(2);
  Double v = 308.*1000.;
  for (Int i=0; i<2; i++) {
    argArray[i].resize(3);
    for (Int j=0; j<3; j++) {
      argArray[i](j) = v * LGROUP[i][j];
    }
  }
  return argArray;
}

// CMB velocity wrt bary center
const Vector<Double> &MeasTable::velocityCMB(uInt which) {
  static const std::vector<Vector<Double>> argArray(calcVelocityCMB());
  DebugAssert(which < 2, AipsError);
  return argArray[which];
}

std::vector<Vector<Double>> MeasTable::calcVelocityCMB() {
  static const Double CMB[2][3] = {
    {-0.97176985257,  0.202393953108, -0.121243727187},
    {-0.970024232022, 0.213247954272, -0.11652595972}
  };
  std::vector<Vector<Double>> argArray(2);
  Double v = 369.5*1000.;
  for (Int i=0; i<2; i++) {
    argArray[i].resize(3);
    for (Int j=0; j<3; j++) {
      argArray[i](j) = v * CMB[i][j];
    }
  }
  return argArray;
}

// Earth and Sun position
const Polynomial<Double> &MeasTable::posArg(uInt which) { 
  static const std::vector<Polynomial<Double>> polyArray(calcPosArg());
  DebugAssert(which < 12, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcPosArg() {
  static const Double POSFUND[12][2] = {
    {252.25,       149472.67},      // Q
    {181.9798,     58517.8157},     // V
    {100.46644851, 35999.37285186}, // E
    {355.43327,    19140.29933},    // M
    { 34.351484,   3034.905675},    // J
    { 50.077471,   1222.113794},    // S
    {314.055005,   428.466998},     // U
    {304.348665,   218.486200},     // N
    {238.47,       145.28},         // P
    {297.850206,   445267.111519},  // D
    { 93.27210,    483202.01753},   // F
    {134.9634,     477198.8676}     // l
  };
  std::vector<Polynomial<Double>> polyArray(12);
  Int i,j;
  for (i=0; i<12; i++) {
    polyArray[i] = Polynomial<Double>(1);
    for (j=0; j<2; j++) {
      polyArray[i].setCoefficient(j, POSFUND[i][j]*C::degree);
    }
  }
  return polyArray;
}

// Derivative of Earth and Sun position polynomial
const Polynomial<Double> &MeasTable::posArgDeriv(uInt which) {
  static const std::vector<Polynomial<Double>> polyArray(calcPosArgDeriv());
  DebugAssert(which < 12, AipsError);
  return polyArray[which];
}

std::vector<Polynomial<Double>> MeasTable::calcPosArgDeriv() {
  std::vector<Polynomial<Double>> polyArray(12);
  for (int i=0; i<12; i++) {
    const Polynomial<Double> *polyPtr = &posArg(i);
    polyArray[i] = polyPtr->derivative();
  }
  return polyArray;
}

const Double* MeasTable::mulPosEarthXYArg(uInt which) {
  static const Double POSXYARG[189][12] = {
    //X,Y(ecliptic) factors
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0},
    {0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -8,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -8,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0},	// 11
    {0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0},
    {0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -5,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 21
    {0,  8,-14,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  8,-12,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0, -1},
    {0,  0,  3, -2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0},	// 31
    {0,  3, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0, -2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  9,-16,  4,  5,  0,  0,  0,  0,  0,  0},
    {0,  0,  7,-16,  4,  5,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  1, -4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1, -3,  0,  0,  0,  0,  0,  0,  0,  0},	// 41
    {0,  4, -5,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  1},
    {0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0, -3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -3,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -8,  4,  0,  0,  0,  0,  0,  0,  0,  0},	// 51
    {0,  3, -6,  4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -5,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  4, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -6,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0, -1,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -2,  5,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0, -1,  0,  1},
    {0,  0,  1,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  7,-15,  0,  0,  0,  0,  0,  0,  0,  0},	// 61
    {0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -8,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  9,-15,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  4, -7,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0, -3,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -9,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -5,  0,  0,  0,  0,  0,  0,  0,  0},	// 71
    {0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0},
    {0,  5, -7,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0, 11,-19,  0,  3,  0,  0,  0,  0,  0,  0},
    {0,  0,  9,-19,  0,  3,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -5,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -7,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -6,  0,  0,  0,  0,  0,  0,  0,  0},	// 81
    {0,  0,  2, -8,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  6, -8,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0, -1,  0,  0},
    {0,  8,-13,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0, -1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -7,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -6,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -6,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0},	// 91
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  3,  0, -1},
    {0,  6, -7,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -7,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -8,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -9,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  1, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 101
    {0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  6, -9,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -3,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -8,  1,  5,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -8,  1,  5,  0,  0,  0,  0,  0,  0},
    {0,  4, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  5,-11,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  6,-13,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -5,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 111
    {0,  3, -6,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -9,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  8,-13,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -4,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0, -4,  0,  0,  0,  0,  0,  0,  0},
    {0,  7, -8,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  7,-11,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -6,  2,  0,  0,  0,  0,  0,  0,  0},	// 121
    {0,  0,  2, -6,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -5,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0},
    {0,  4, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  6, -8,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  8,-15,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0,  0, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -8,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0, 10,-17,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4,  0, -3,  0,  0,  0,  0,  0,  0,  0},	// 131
    {0,  0,  0,  0,  1, -5,  0,  0,  0,  0,  0,  0},
    {0,  6,-11,  0,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  6, -9,  0,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  8,-17,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  8,-15,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0},
    {0,  6, -8,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {1,  0, -5,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4,  0, -2,  0,  0,  0,  0,  0,  0,  0},	// 141
    {0,  0,  5,-10,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  6,-11,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0},
    {1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  4, -2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -7, -4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -5, -4,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  1, -2,  0},
    {0,  0,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0},	// 151
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  3,  0,  0},
    {0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0},
    {1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0},
    {0,  6,-10,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0},
    {0,  8, -9,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 161
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0},
    {0,  0,  6, -7,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0,  0,  0,  0,  0,  1,  0,  0},
    {0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0},
    {0,  0,  7,-13,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -8,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  6, -9,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -5,  0,  0,  0,  0,  0,  0,  0},	// 171
    {0,  3, -7,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  2,  0, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -2,  0,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  5, -8,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  3, -8,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 181
    {0,  8,-14,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  8,-12,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  3, -5,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0}
  };
  DebugAssert(which < 189, AipsError);
  return &(POSXYARG[which][0]);
}

const Double* MeasTable::mulPosEarthZArg(uInt which) {
  static const Double POSZARG[32][12] = {
    //Z(ecliptic) factors
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0},
    {0,  3, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0},
    {0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -1},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0},
    {0,  4, -5,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 11
    {0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -1,  0},
    {0,  0,  1,  0, -3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -1,  0},
    {0,  0,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -7,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1},
    {0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 21
    {0,  3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  8,-12,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  5, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 31
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0}
  };
  DebugAssert(which < 32, AipsError);
  return &(POSZARG[which][0]);
}

const Double* MeasTable::mulPosSunXYArg(uInt which) {
  static const Double POSXYARG[98][12] = {
    //X,Y(ecliptic) factors
    {0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0},	// 11
    {0,  0,  0,  0,  1, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0},
    {0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0},
    {0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 21
    {0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0},
    {0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -7,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0},	// 31
    {0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0},	// 41
    {0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0},
    {0,  0,  0,  0,  4,-11,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -7,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -5,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0},	// 51
    {0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  5,-10,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -6,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -6,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4, -9,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3,-10,  0,  0,  0,  0,  0,  0},	// 61
    {0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4,-10,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4,-12,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  4,-10,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -8,  0,  0,  0,  0,  0,  0},	// 71
    {0,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -8,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -6,  2,  0,  0,  0,  0,  0},
    {1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -6,  4,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0},
    {0,  0,  0,  0,  5,-11,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3,-11,  3,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0},	// 81
    {0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -5,  0,  0,  0,  0,  0,  0},	// 91
    {0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -7,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0}
  };
  DebugAssert(which < 98, AipsError);
  return &(POSXYARG[which][0]);
}

const Double* MeasTable::mulPosSunZArg(uInt which) {
  static const Double POSZARG[29][12] = {
    //Z(ecliptic) factors
    {0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -5,  0,  0,  0,  0,  0,  0},	// 11
    {0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0},
    {0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},	// 12
    {0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0},
    {0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  2, -7,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0},
    {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0}
  };
  DebugAssert(which < 29, AipsError);
  return &(POSZARG[which][0]);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulPosEarthXY(Double time, Double epsilon)
{
  return theirMulPosEarthXY.getArray (time, epsilon);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulPosEarthZ(Double time, Double epsilon)
{
  return theirMulPosEarthZ.getArray (time, epsilon);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulPosSunXY(Double time, Double epsilon)
{
  return theirMulPosSunXY.getArray (time, epsilon);
}

std::shared_ptr<Matrix<Double>> MeasTable::mulPosSunZ(Double time, Double epsilon)
{
  return theirMulPosSunZ.getArray (time, epsilon);
}

const RotMatrix &MeasTable::posToRect() {
  static const RotMatrix rot(Euler(+84381.4091 * C::arcsec, 1,
                                   -0.0930 * C::arcsec, 3));
  return rot;
}

const RotMatrix &MeasTable::rectToPos() {
  static RotMatrix rot(calcRectToPos());
  return rot;
}

RotMatrix MeasTable::calcRectToPos() {
  RotMatrix rot(MeasTable::posToRect());
  rot.transpose();
  return rot;
}

const RotMatrix &MeasTable::galToSupergal() {
  static RotMatrix rot(Euler( -90*C::degree, 3, -83.68*C::degree,
                              2, -47.37*C::degree, 3));
  return rot;
}

const RotMatrix &MeasTable::ICRSToJ2000() {
  static RotMatrix rot(calcICRSToJ2000());
  return rot;
}

RotMatrix MeasTable::calcICRSToJ2000() {
  RotMatrix rot(MeasTable::frameBias00());
  rot.transpose();
  return rot;
}

// Position related routines
Double MeasTable::WGS84(uInt which) {
  static const Double data[2] = { 6378137, 298.257223563};

  DebugAssert(which < 2, AipsError);
  return data[which];
}

// Polar motion related routines
Euler MeasTable::polarMotion(Double ut) {
#if defined(USE_THREADS)
    static std::atomic<Bool> msgDone;
#else
    static Bool msgDone;
#endif
  Euler res(0.0, 2, 0.0, 1, 0.0, 3);
  if (!MeasIERS::get(res(0), MeasIERS::MEASURED, MeasIERS::X, ut) ||
      !MeasIERS::get(res(1), MeasIERS::MEASURED, MeasIERS::Y, ut)) {
    // It is harmless if the message accidentally appears multiple times.
    if (!msgDone) {
      LogIO os(LogOrigin("MeasTable", "PolarMotion(Double)", WHERE));
      os << LogIO::NORMAL3
         << "High precision polar motion information not available."
         << LogIO::POST;
      msgDone = True;
    }
  }
  ///    cout << "polarmotion " << res(0) << ' ' << res(1) << endl;
  res(0) *= -C::arcsec;
  res(1) *= -C::arcsec;
  return res;
}

// Time functions
Double MeasTable::dUTC(Double utc) {
  static const Statics_dUTC st(calc_dUTC());
  Double (* const &LEAP)[4] = st.LEAP; // alias to avoid more clutter below
  const int &N = st.N;                // idem

  Double val(0);
  if (utc < LEAP[0][0]) {
    val = LEAP[0][1] + (utc - LEAP[0][2])*LEAP[0][3];
  } else {
    for (Int i = N-1; i >= 0; i--) {
      if (utc >= LEAP[i][0]) {
	val = LEAP[i][1];
	if (LEAP[i][3] != 0) {
	  val += (utc - LEAP[i][2])*LEAP[i][3];
	}
	break;
      }
    }
  }
  return val;
}

MeasTable::Statics_dUTC MeasTable::calc_dUTC() {
  Statics_dUTC rv;

  Table t;
  ROTableRow row;
  TableRecord kws;
  const String rfn[4] = {"MJD", "dUTC", "Offset", "Multiplier"};
  RORecordFieldPtr<Double> rfp[4];
  Double dt;
  String vs;
  if (!MeasIERS::getTable(t, kws, row, rfp, vs, dt, 4, rfn, "TAI_UTC",
                          "measures.tai_utc.directory", "geodetic")) {
    LogIO os(LogOrigin("MeasTable", "dUTC(Double)", WHERE));
    os << "Cannot read leap second table TAI_UTC" << LogIO::EXCEPTION;
  }
  rv.N = t.nrow();
  if (rv.N < 35) {
    LogIO os(LogOrigin("MeasTable", "dUTC(Double)", WHERE));
    os << "Leap second table TAI_UTC corrupted" << LogIO::EXCEPTION;
  }
  if (Time().modifiedJulianDay() - dt > 180) {
    LogIO os(LogOrigin("MeasTable", "dUTC(Double)", WHERE));
    os << LogIO::SEVERE
       << "Leap second table TAI_UTC seems out-of-date.\n"
          "Until the table is updated (see the CASA documentation or your system admin),\n"
          "times and coordinates derived from UTC could be wrong by 1s or more."
       << LogIO::POST;
  }
  rv.LEAP = (Double (*)[4])(new Double[4*rv.N]);
  for (Int i=0; i < rv.N; i++) {
    row.get(i);
    for (Int j=0; j < 4; j++) {
      rv.LEAP[i][j] = *(rfp[j]);
    }
  }

  return rv;
}

Double MeasTable::dTAI(Double) {
  return (32.184);
}

Double MeasTable::dTDT(Double ut1) {
  Double g = (357.53 + 0.9856003*(ut1-MeasData::MJD2000))*C::degree;
  return (0.001658*sin(g) + 0.000014*sin(2*g));
}

Double MeasTable::dTDB(Double tai) {
  return(1.550505e-8*86400*(tai-43144.0));
}

Double MeasTable::dTCG(Double tai) {
  return(6.969291e-10*86400*(tai-43144.0));
}

Double MeasTable::GMST0(Double ut1) {
  static Polynomial<Double> stPoly(calcGMST0());
  return (stPoly((ut1-MeasData::MJD2000)/MeasData::JDCEN));
}

Polynomial<Double> MeasTable::calcGMST0() {
  Polynomial<Double> stPoly(3);
  stPoly.setCoefficient(0, 24110.54841);
  stPoly.setCoefficient(1, 8640184.812866);
  stPoly.setCoefficient(2, 0.093104);
  stPoly.setCoefficient(3, -6.2e-6);
  return stPoly;
}

Double MeasTable::GMST00(Double ut1, Double tt) {
  static Polynomial<Double> stPoly(calcGMST00());
  return (stPoly((tt-MeasData::MJD2000)/MeasData::JDCEN) +
	  MeasTable::ERA00(ut1));
}

Polynomial<Double> MeasTable::calcGMST00() {
  Polynomial<Double> stPoly(4);
  stPoly.setCoefficient(0, 0.014506*C::arcsec);
  stPoly.setCoefficient(1, 4612.15739966*C::arcsec+630.73514045148926);
  stPoly.setCoefficient(2, + 1.39667721*C::arcsec);
  stPoly.setCoefficient(3, - 0.00009344*C::arcsec);
  stPoly.setCoefficient(4, + 0.00001882*C::arcsec);
  return stPoly;
}

Double MeasTable::ERA00(Double ut1) {
  static Polynomial<Double> stPoly(calcERA00());
  ut1 -= MeasData::MJD2000;
  return MVAngle(stPoly(ut1)+ (2.0*M_PI)*fmod(ut1, 1.0))(0.0).radian();
}

Polynomial<Double> MeasTable::calcERA00() {
  Polynomial<Double> stPoly(1);
  stPoly.setCoefficient(0, 0.7790572732640*(2.0*M_PI));
  stPoly.setCoefficient(1, 0.00273781191135448*(2.0*M_PI));
  return stPoly;
}

Double MeasTable::sprime00(Double tt) {
  return ((tt-MeasData::MJD2000)/MeasData::JDCEN * -47e-6 * C::arcsec);
}

Double MeasTable::GMUT0(Double gmst1) {
  static Polynomial<Double> stPoly(calcGMUT0());
  return (stPoly((gmst1-MeasData::MJD2000-6713.)/MeasData::JDCEN));
}

Polynomial<Double> MeasTable::calcGMUT0() {
  Polynomial<Double> stPoly(3);
  stPoly.setCoefficient(0, -0.65830845056254866847);
  stPoly.setCoefficient(1, -235.90946916710752);
  stPoly.setCoefficient(2, -0.00000252822553597972);
  stPoly.setCoefficient(3, 0.0000000001679);
  return stPoly;
}

Double MeasTable::UTtoST(Double ut1) {
  static Polynomial<Double> UTSTPoly(calcUTtoST());
  return(UTSTPoly((ut1-MeasData::MJD2000)/MeasData::JDCEN));
}

Polynomial<Double> MeasTable::calcUTtoST() {
  Polynomial<Double> UTSTPoly(2);
  UTSTPoly.setCoefficient(0, 1.002737909350795);
  UTSTPoly.setCoefficient(1, +5.9006e-11);
  UTSTPoly.setCoefficient(2, -5.9e-15);
  return UTSTPoly;
}

Double MeasTable::dUT1(Double utc) {
#if defined(USE_THREADS)
  static std::atomic<Bool> msgDone;
#else
  static Bool msgDone;
#endif

  ///#if defined(USE_THREADS) && !defined(__APPLE__)
  static thread_local Double res = 0.0;
  static thread_local Double checkT = -1e6;
  ///#else // !USE_THREADS (empty Mutex impl) or __APPLE__
  ///  std::lock_guard<std::mutex> lock(theirdUT1Mutex); // Pity. Try to narrow blunt __APPLE__ cond.
  ///  static Double res = 0.0;
  ///  static Double checkT = -1e6;
  ///#endif
  if ( !nearAbs(utc, checkT, 0.04)) {
    checkT = utc;
    if (!MeasIERS::get(res, MeasIERS::MEASURED, MeasIERS::dUT1, utc)) {
      // It is harmless if the message accidentally appears multiple times.
      if (!msgDone) {
	LogIO os(LogOrigin("MeasTable", "dUT1(Double)", WHERE));
	os << LogIO::NORMAL3
	   << "High precision dUT1 information not available."
	   << LogIO::POST;
	msgDone = True;
      }
    }
  }
  ///  cout << "dutc1 " << res << endl;
  return res;
}

} //# NAMESPACE CASACORE - END

