# +==========================================================================+
# || CipUX::Dog                                                             ||
# ||                                                                        ||
# || CipUX CipUX::Dog                                                       ||
# ||                                                                        ||
# || Copyright 2008 - 2009 by Christian Kuelker. All rights reserved!       ||
# ||                                                                        ||
# || License: GNU GPL version 2 or any later version.                       ||
# ||                                                                        ||
# +==========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::Dog;

use 5.008001;
use strict;
use warnings;
use Carp;
use Class::Std;
use Data::Dumper;
use English qw( -no_match_vars);
use Log::Log4perl qw(get_logger :levels);
use Readonly;
use base qw(CipUX);

{    # BEGIN INSIDE-OUT CLASS

    use version; our $VERSION = qv('3.4.0.0');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # +======================================================================+
    # || CONST                                                              ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $WRITE_EXEC   => q{-|};

    # +======================================================================+
    # || OBJECT                                                             ||
    # +======================================================================+

    # If use af ATTR above OK, then sub BUILD is not needed:
    my %dog_hr_of : ATTR( init_arg => 'dog_hr');                # dog
    my %object_of : ATTR( init_arg => 'object');                # dog
    my %overwrite_hr_of : ATTR( init_arg => 'overwrite_hr');    # dog

    my %dog_cfg : ATTR( :get<dog_cfg> );

    # +======================================================================+
    # || GLOBAL                                                             ||
    # +======================================================================+
    # +======================================================================+
    # || CONSTRUCTOR                                                        ||
    # +======================================================================+
    sub BUILD {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident, $arg_r ) = @_;

        # add prefix for cfg, if needed
        my $pref
            = exists $arg_r->{pref}
            ? $self->l( $arg_r->{pref} )
            : $EMPTY_STRING;

        # +------------------------------------------------------------------+
        # | main
        $dog_cfg{$ident} = $self->cfg( { 'sub' => 'dog', pref => $pref } );

        # add them if needed:
        # $dog_of{$ident}  = $arg_r->{dog};
        # $object_of{$ident}    = $arg_r->{object};
        # $overwrite_hr_of{$ident} = $arg_r->{overwrite_hr};

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +======================================================================+
    # || DESTRUCTOR                                                         ||
    # +======================================================================+
    sub DEMOLISH {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident ) = @_;

        # +------------------------------------------------------------------+
        # | main
        delete $dog_cfg{$ident};
        delete $dog_hr_of{$ident};
        delete $object_of{$ident};
        delete $overwrite_hr_of{$ident};

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || MAIN                                                               ||
    # +======================================================================+

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # | PRIVATE INTERFACE                                                    |
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+

    # Module implementation here

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # | PUBLIC INTERFACE                                                     |
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+

    # +======================================================================+
    # || METHODS                                                            ||
    # +======================================================================+

    sub bite {

        # API
        my ($self) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $cfg_dog_cmd_hr = $dog_cfg{ ident $self};
        $logger->debug( 'cfg_dog_cmd_hr: ',
            { filter => \&Dumper, value => $cfg_dog_cmd_hr } );

        $logger->debug('starting dog action');
        my $cipux_dog  = $EMPTY_STRING;
        my $cmdargs_hr = {};

        # key: cipux_dog, user, role
        # val: create_homedir, @object@, role
        foreach my $key ( sort keys %{ $dog_hr_of{ ident $self } } ) {

            $logger->debug( 'dog_hr_of key',
                { filter => \&Dumper, value => $key } );
            $logger->debug(
                'dog_hr_of value',
                {
                    filter => \&Dumper,
                    value  => $dog_hr_of{ ident $self }{$key}
                }
            );
            if ( $key eq 'cipux_dog' ) {
                $cipux_dog = $dog_hr_of{ ident $self }{$key};
            }
            else {
                my $cli_opt = $self->l($key);
                $cli_opt =~ s/\@object\@/$object_of{ident $self}/gsmx;

                my $cli_val = $self->l( $dog_hr_of{ ident $self }{$key} );
                $cli_val =~ s/\@object\@/$object_of{ident $self}/gsmx;

                # {
                #    'user' => 'testuser12',
                #    'role' => 'role',
                # }
                $cmdargs_hr->{$cli_opt} = $cli_val;

            }
        }

        my $msg = 'The section "commands" is not defined';
        $msg .= " in cipux-dog.perl for $cipux_dog!\n";
        my %commands
            = exists $cfg_dog_cmd_hr->{cipux_dogs}->{$cipux_dog}->{commands}
            ? %{ $cfg_dog_cmd_hr->{cipux_dogs}->{$cipux_dog}->{commands} }
            : croak $msg;

        my $cmd         = $EMPTY_STRING;
        my $cmd_args_ar = [];

        # $cmd: 1, 2, 3, ...
        foreach my $count ( sort keys %commands ) {
            $logger->debug("count: $count");

            $cmd = $cfg_dog_cmd_hr->{cipux_dogs}->{$cipux_dog}->{commands}
                ->{$count}->{shell};
            $logger->debug("cmd: $cmd");

            my $args_ar
                = $cfg_dog_cmd_hr->{cipux_dogs}->{$cipux_dog}->{commands}
                ->{$count}->{args};
            $logger->debug( 'args_ar ',
                { filter => \&Dumper, value => $args_ar } );

            # arg: --user, @user@, --role, @role@,
            foreach my $arg ( @{$args_ar} ) {

                # if arg is a variable
                if ( $arg =~ m/^\@(.*)\@$/xms ) {
                    if ( defined $cmdargs_hr->{$1} ) {

                        # @user@ -> testuser12
                        # @role@ -> role
                        $arg = $cmdargs_hr->{$1};
                    }
                }

                # --user -> --user
                # @user@ -> testuser12
                push @{$cmd_args_ar}, $self->l($arg);

            }

        }

        $logger->debug("cmd: $cmd");
        $logger->debug( 'cmd_args_ar ',
            { filter => \&Dumper, value => $cmd_args_ar } );

        my $return = $self->execute(
            { command => $self->l($cmd), cmdargs_ar => $cmd_args_ar } );

        $logger->debug('END');
        return ( $cmd, $cmd_args_ar, $return );

    }

    sub execute {

        my ( $self, $arg_r ) = @_;

        my $command
            = exists $arg_r->{command}
            ? $self->l( $arg_r->{command} )
            : $self->perr('command');

        my $cmdargs_ar
            = exists $arg_r->{cmdargs_ar}
            ? $self->a( $arg_r->{cmdargs_ar} )
            : $self->perr('cmdargs_ar');

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("> command: $command");
        $logger->debug( '> cmdargs_ar ',
            { filter => \&Dumper, value => $cmdargs_ar } );

        my @sanitized_arg = ();
        foreach my $arg ( @{$cmdargs_ar} ) {
            $arg =~ s/[`]//gsmx;
            push @sanitized_arg, $arg;
        }

        my @return = ();
        my $return = $EMPTY_STRING;
        my $msg    = "Cannot execute [$command " . join q{ }, @sanitized_arg;
        $msg .= ']. Please provide this command or change [cipux-dog.perl].';
        $msg .= ' The error message was: ';

        if ( not -e $command ) {
            croak $msg . 'Command do not exist!';
        }

        if ( not -x $command ) {
            croak $msg . 'Command is not executable! (forget chmod +x ?)';
        }

        open my $GET_HANDLE, $WRITE_EXEC, $command, @sanitized_arg
            or croak $msg . $CHILD_ERROR;

        while (<$GET_HANDLE>) {
            if (wantarray) { push @return, $_; }
            else           { $return .= $_; }
        }

        close $GET_HANDLE or croak "Can not close command $command!\n";

        $logger->debug('END');

        return ( wantarray ? @return : $return );

    }

}    # END INSIDE-OUT CLASS

1;   # Magic true value required at end of module

__END__

=pod

=for stopwords CipUX Kuelker config Readonly webpage cipux-dog.perl

=head1 NAME

CipUX::Dog - Adds a hook to CipUX::Task to execute commands.


=head1 VERSION

This document describes CipUX::Dog version 3.4.0.0


=head1 SYNOPSIS

   use CipUX::Dog;

or


 use English qw( -no_match_vars);

 eval { require CipUX::Dog; };

 if ( not $EVAL_ERROR ) {

     my $dog = CipUX::Dog->new(
        {
             dog_hr       => $dog_hr,
             object       => $object,
             overwrite_hr => $overwrite_hr,
         }
     );

     $dog->bite();    # exec files system command

 }


=head1 DESCRIPTION

CipuX::Dog can be used as a hook for CipUX::Task. If CipUX::Dog is installed
it can be configured by the administrator to execute commands triggered by
execution of a given task.

=head1 SUBROUTINES/METHODS

=head2 BUILD

Create the Object, reads the config file

=head2 DEMOLISH

Removes the object

=head2 bite

Calls the dog to bite.

=head2 execute

Execute a command.

 my $return = $self->execute(
     { command => $self->l($cmd), cmdargs_ar => $cmd_args_ar } );


=head1 DIAGNOSTICS

This section list every single error and warning message that the module can
generate (even the ones that will "never happen"), with a full explanation of
each problem, one or more likely causes, and any suggested remedies.

=over

=item C<< The section "commands" is not defined in cipux-dog.perl for %s >>

The hash key 'commands' is not inside the configuration for a given dog hash
key in cipux-dog.perl configuration. The problem might look like this:

 $cfg = {
     cipux_dogs => {
         create_homedir => {
         },
     },
 };

It can be solved by adding a 'commands' key:

 $cfg = {
     cipux_dogs => {
         create_homedir => {
             commands => {
             },
         },
     },
 };


=item C<< Cannot execute [%s]. Please provide this command or change [cipux-dog.perl]. The error message was: %s >>

There can be different reasons for this message. It depends on the last %s.
The following reasons are foreseen:

C<< Command do not exist! >>

The command specified in cipux-dog.perl do not exist. You should copy,
provide or write this command.

C<< Command is not executable! (forget chmod +x ?) >>

The command specified in cipux-dog.perl exists but can not be executed.
Either it has no executable flags or you have not the right to execute it.
Check and/ro set the execution bit on this command or try do give you the
right to execute it.

C<< ? >>

Other reasons not be able to execute the command might come from the system
and will display after the message.


=item C<< Can not close command %s >>

If the file handle can not be closed this message will be displayed.

=back


=head1 CONFIGURATION AND ENVIRONMENT

CipUX::Dog requires a configuration file but no environment variables. It can
be configured by cipux-dog.$ext, where $ext is a valid and capable extension
from CipUX configuration space. The default configuration for CipUX::Dog is
in: /usr/share/cipux/etc/cipux-dog.perl


=head1 DEPENDENCIES

Carp
CipUX
Class::Std
Data::Dumper
English
Log::Log4perl
Readonly
version

=head1 INCOMPATIBILITIES

None reported.


=head1 BUGS AND LIMITATIONS

No bugs have been reported.


=head1 SEE ALSO

See the CipUX webpage and the manual at L<http://www.cipux.org>


=head1 AUTHOR

Christian Kuelker  C<< <christian.kuelker@cipworx.org> >>


=head1 LICENSE AND COPYRIGHT

Copyright (C) 2008 - 2009, Christian Kuelker. All rights reserved.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENSE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut
