// RUN: mlir-opt %s \
// RUN:  -gpu-lower-to-nvvm-pipeline="cubin-chip=sm_90 cubin-features=+ptx80 opt-level=3" \
// RUN:  | mlir-runner \
// RUN:   --shared-libs=%mlir_cuda_runtime \
// RUN:   --shared-libs=%mlir_runner_utils \
// RUN:   --entry-point-result=void \
// RUN:  | FileCheck %s

// Test swizzling with TMA load
// 128B Swizzle Each numbered cell is 16 byte 
// |-------------------------------|
// | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 
// | 1 | 0 | 3 | 2 | 5 | 4 | 7 | 6 |
// | 2 | 3 | 0 | 1 | 6 | 7 | 4 | 5 |
// | 3 | 2 | 1 | 0 | 7 | 6 | 5 | 4 | 
// | 4 | 5 | 6 | 7 | 0 | 1 | 2 | 3 |
// | 5 | 4 | 7 | 6 | 1 | 0 | 3 | 2 | 
// | 6 | 7 | 4 | 5 | 2 | 3 | 0 | 1 |
// |-------------------------------| 
// | ... pattern repeats ...       |
// |-------------------------------|


!barrierType = !nvgpu.mbarrier.group<memorySpace = #gpu.address_space<workgroup>>
!tokenType = !nvgpu.mbarrier.token

!lhs = memref<128x64xf16>
!shmemlhs = memref<128x64xf16, 3>
!lhsTensorMap = !nvgpu.tensormap.descriptor<tensor = !shmemlhs, swizzle = swizzle_128b, l2promo=none, oob=zero, interleave=none>

module @mymod {
  func.func private @printMemrefF32(memref<*xf32>)
  memref.global "private" @bufferLhsGlobal : !shmemlhs
  llvm.func @printf(!llvm.ptr, ...) -> i32
  func.func @main() {
    %c8192 = arith.constant 8192 : index
    %c-1_i32 = arith.constant -1 : i32
    %c10000000 = arith.constant 10000000 : index
    %c64 = arith.constant 64 : index
    %c1 = arith.constant 1 : index
    %c0 = arith.constant 0 : index
    %c128 = arith.constant 128 : index
    %c8 = arith.constant 8 : index
    
    // Step 1. Allocate host data and initialize it.
    %alloc = memref.alloc() : !lhs    
    %alloc_1 = memref.alloc() : memref<128x64xf32>
    scf.for %arg0 = %c0 to %c128 step %c1 {
      scf.for %arg1 = %c0 to %c64 step %c1 {
        %5 = arith.muli %arg0, %c64 : index
        %6 = arith.addi %5, %arg1 : index
        %7 = arith.divui %6, %c8 : index
        %8 = arith.index_cast %7 : index to i32
        %9 = arith.sitofp %8 : i32 to f16
        memref.store %9, %alloc[%arg0, %arg1] : !lhs
        %10 = arith.extf %9 : f16 to f32
        memref.store %10, %alloc_1[%arg0, %arg1] : memref<128x64xf32>
      }
    }

    // Step 2. Print on the host
    %cast = memref.cast %alloc_1 : memref<128x64xf32> to memref<*xf32>
    call @printMemrefF32(%cast) : (memref<*xf32>) -> ()
    
    // Step 3. Copy host to device
    %0 = gpu.wait async
    %memref, %asyncToken = gpu.alloc async [%0] () : !lhs    
    %1 = gpu.memcpy async [%0] %memref, %alloc : !lhs, !lhs    
    %cast_6 = memref.cast %memref : !lhs to memref<*xf16>
    
    // Step 4. Create TMA tensor descriptor
    %3 = nvgpu.tma.create.descriptor %cast_6 box[%c128, %c64] : memref<*xf16> -> !lhsTensorMap

    // Step 5. Launch a GPU kernel
    gpu.launch blocks(%arg0, %arg1, %arg2) in (%arg6 = %c1, %arg7 = %c1, %arg8 = %c1) threads(%arg3, %arg4, %arg5) in (%arg9 = %c128, %arg10 = %c1, %arg11 = %c1) {
      %5 = gpu.block_dim  x
      %6 = gpu.thread_id  x
      %7 = memref.get_global @bufferLhsGlobal : !shmemlhs
    
      // Step 6. Initialize the mbarrier
      %9 = nvgpu.mbarrier.create -> !barrierType
      nvgpu.mbarrier.init %9[%c0], %5 : !barrierType
      %10 = arith.cmpi eq, %6, %c0 : index
      
      
      // Step 7. First thread does TMA load
      scf.if %10 {
        gpu.printf "[GPU] TMA SIZE %d\0A", %c8192 : index
        nvgpu.tma.async.load %3[%c0, %c0], %9[%c0] to %7 : !lhsTensorMap, !barrierType -> !shmemlhs
        nvgpu.mbarrier.arrive.expect_tx %9[%c0], %c8192 : !barrierType
      } else {
        nvgpu.mbarrier.arrive.expect_tx %9[%c0], %c0 : !barrierType
      }

      // Step 8. Wait until TMA is done
      %phase_c0 = arith.constant 0 : i1
      nvgpu.mbarrier.try_wait.parity %9[%c0], %phase_c0, %c10000000 : !barrierType

      // Step 9. Print loaded data in 128b swizzled
      scf.if %10 {        
        gpu.printf "===--- Matrix A ---=== %d \0A", %c-1_i32 : i32
        scf.for %arg12 = %c0 to %c128 step %c1 {
          scf.for %arg13 = %c0 to %c64 step %c1 {
            %15 = memref.load %7[%arg12, %arg13] : !shmemlhs
            %16 = arith.extf %15 : f16 to f32
            gpu.printf "%.0f,   ", %16 : f32
          }
          gpu.printf "%d\0A", %c-1_i32 : i32
        }
        gpu.printf "===----------------=== %d \0A", %c-1_i32 : i32
      }
      gpu.terminator
    }
    return
  }
}


// CHECK: [GPU] TMA SIZE 8192
// CHECK: ===--- Matrix A ---=== -1 
// CHECK: 0,   0,   0,   0,   0,   0,   0,   0,   1,   1,   1,   1,   1,   1,   1,   1,   2,   2,   2,   2,   2,   2,   2,   2,   3,   3,   3,   3,   3,   3,   3,   3,   4,   4,   4,   4,   4,   4,   4,   4,   5,   5,   5,   5,   5,   5,   5,   5,   6,   6,   6,   6,   6,   6,   6,   6,   7,   7,   7,   7,   7,   7,   7,   7,   -1
// CHECK: 9,   9,   9,   9,   9,   9,   9,   9,   8,   8,   8,   8,   8,   8,   8,   8,   11,   11,   11,   11,   11,   11,   11,   11,   10,   10,   10,   10,   10,   10,   10,   10,   13,   13,   13,   13,   13,   13,   13,   13,   12,   12,   12,   12,   12,   12,   12,   12,   15,   15,   15,   15,   15,   15,   15,   15,   14,   14,   14,   14,   14,   14,   14,   14,   -1
// CHECK: 18,   18,   18,   18,   18,   18,   18,   18,   19,   19,   19,   19,   19,   19,   19,   19,   16,   16,   16,   16,   16,   16,   16,   16,   17,   17,   17,   17,   17,   17,   17,   17,   22,   22,   22,   22,   22,   22,   22,   22,   23,   23,   23,   23,   23,   23,   23,   23,   20,   20,   20,   20,   20,   20,   20,   20,   21,   21,   21,   21,   21,   21,   21,   21,   -1
// CHECK: 27,   27,   27,   27,   27,   27,   27,   27,   26,   26,   26,   26,   26,   26,   26,   26,   25,   25,   25,   25,   25,   25,   25,   25,   24,   24,   24,   24,   24,   24,   24,   24,   31,   31,   31,   31,   31,   31,   31,   31,   30,   30,   30,   30,   30,   30,   30,   30,   29,   29,   29,   29,   29,   29,   29,   29,   28,   28,   28,   28,   28,   28,   28,   28,   -1
// CHECK: 36,   36,   36,   36,   36,   36,   36,   36,   37,   37,   37,   37,   37,   37,   37,   37,   38,   38,   38,   38,   38,   38,   38,   38,   39,   39,   39,   39,   39,   39,   39,   39,   32,   32,   32,   32,   32,   32,   32,   32,   33,   33,   33,   33,   33,   33,   33,   33,   34,   34,   34,   34,   34,   34,   34,   34,   35,   35,   35,   35,   35,   35,   35,   35,   -1
// CHECK: 45,   45,   45,   45,   45,   45,   45,   45,   44,   44,   44,   44,   44,   44,   44,   44,   47,   47,   47,   47,   47,   47,   47,   47,   46,   46,   46,   46,   46,   46,   46,   46,   41,   41,   41,   41,   41,   41,   41,   41,   40,   40,   40,   40,   40,   40,   40,   40,   43,   43,   43,   43,   43,   43,   43,   43,   42,   42,   42,   42,   42,   42,   42,   42,   -1
// CHECK: 54,   54,   54,   54,   54,   54,   54,   54,   55,   55,   55,   55,   55,   55,   55,   55,   52,   52,   52,   52,   52,   52,   52,   52,   53,   53,   53,   53,   53,   53,   53,   53,   50,   50,   50,   50,   50,   50,   50,   50,   51,   51,   51,   51,   51,   51,   51,   51,   48,   48,   48,   48,   48,   48,   48,   48,   49,   49,   49,   49,   49,   49,   49,   49,   -1
// CHECK: 63,   63,   63,   63,   63,   63,   63,   63,   62,   62,   62,   62,   62,   62,   62,   62,   61,   61,   61,   61,   61,   61,   61,   61,   60,   60,   60,   60,   60,   60,   60,   60,   59,   59,   59,   59,   59,   59,   59,   59,   58,   58,   58,   58,   58,   58,   58,   58,   57,   57,   57,   57,   57,   57,   57,   57,   56,   56,   56,   56,   56,   56,   56,   56,   -1
// CHECK: 64,   64,   64,   64,   64,   64,   64,   64,   65,   65,   65,   65,   65,   65,   65,   65,   66,   66,   66,   66,   66,   66,   66,   66,   67,   67,   67,   67,   67,   67,   67,   67,   68,   68,   68,   68,   68,   68,   68,   68,   69,   69,   69,   69,   69,   69,   69,   69,   70,   70,   70,   70,   70,   70,   70,   70,   71,   71,   71,   71,   71,   71,   71,   71,   -1
// CHECK: 73,   73,   73,   73,   73,   73,   73,   73,   72,   72,   72,   72,   72,   72,   72,   72,   75,   75,   75,   75,   75,   75,   75,   75,   74,   74,   74,   74,   74,   74,   74,   74,   77,   77,   77,   77,   77,   77,   77,   77,   76,   76,   76,   76,   76,   76,   76,   76,   79,   79,   79,   79,   79,   79,   79,   79,   78,   78,   78,   78,   78,   78,   78,   78,   -1
// CHECK: 82,   82,   82,   82,   82,   82,   82,   82,   83,   83,   83,   83,   83,   83,   83,   83,   80,   80,   80,   80,   80,   80,   80,   80,   81,   81,   81,   81,   81,   81,   81,   81,   86,   86,   86,   86,   86,   86,   86,   86,   87,   87,   87,   87,   87,   87,   87,   87,   84,   84,   84,   84,   84,   84,   84,   84,   85,   85,   85,   85,   85,   85,   85,   85,   -1
// CHECK: 91,   91,   91,   91,   91,   91,   91,   91,   90,   90,   90,   90,   90,   90,   90,   90,   89,   89,   89,   89,   89,   89,   89,   89,   88,   88,   88,   88,   88,   88,   88,   88,   95,   95,   95,   95,   95,   95,   95,   95,   94,   94,   94,   94,   94,   94,   94,   94,   93,   93,   93,   93,   93,   93,   93,   93,   92,   92,   92,   92,   92,   92,   92,   92,   -1
// CHECK: 100,   100,   100,   100,   100,   100,   100,   100,   101,   101,   101,   101,   101,   101,   101,   101,   102,   102,   102,   102,   102,   102,   102,   102,   103,   103,   103,   103,   103,   103,   103,   103,   96,   96,   96,   96,   96,   96,   96,   96,   97,   97,   97,   97,   97,   97,   97,   97,   98,   98,   98,   98,   98,   98,   98,   98,   99,   99,   99,   99,   99,   99,   99,   99,   -1
// CHECK: 109,   109,   109,   109,   109,   109,   109,   109,   108,   108,   108,   108,   108,   108,   108,   108,   111,   111,   111,   111,   111,   111,   111,   111,   110,   110,   110,   110,   110,   110,   110,   110,   105,   105,   105,   105,   105,   105,   105,   105,   104,   104,   104,   104,   104,   104,   104,   104,   107,   107,   107,   107,   107,   107,   107,   107,   106,   106,   106,   106,   106,   106,   106,   106,   -1
// CHECK: 118,   118,   118,   118,   118,   118,   118,   118,   119,   119,   119,   119,   119,   119,   119,   119,   116,   116,   116,   116,   116,   116,   116,   116,   117,   117,   117,   117,   117,   117,   117,   117,   114,   114,   114,   114,   114,   114,   114,   114,   115,   115,   115,   115,   115,   115,   115,   115,   112,   112,   112,   112,   112,   112,   112,   112,   113,   113,   113,   113,   113,   113,   113,   113,   -1
// CHECK: 127,   127,   127,   127,   127,   127,   127,   127,   126,   126,   126,   126,   126,   126,   126,   126,   125,   125,   125,   125,   125,   125,   125,   125,   124,   124,   124,   124,   124,   124,   124,   124,   123,   123,   123,   123,   123,   123,   123,   123,   122,   122,   122,   122,   122,   122,   122,   122,   121,   121,   121,   121,   121,   121,   121,   121,   120,   120,   120,   120,   120,   120,   120,   120,   -1
// CHECK: 128,   128,   128,   128,   128,   128,   128,   128,   129,   129,   129,   129,   129,   129,   129,   129,   130,   130,   130,   130,   130,   130,   130,   130,   131,   131,   131,   131,   131,   131,   131,   131,   132,   132,   132,   132,   132,   132,   132,   132,   133,   133,   133,   133,   133,   133,   133,   133,   134,   134,   134,   134,   134,   134,   134,   134,   135,   135,   135,   135,   135,   135,   135,   135,   -1
// CHECK: 137,   137,   137,   137,   137,   137,   137,   137,   136,   136,   136,   136,   136,   136,   136,   136,   139,   139,   139,   139,   139,   139,   139,   139,   138,   138,   138,   138,   138,   138,   138,   138,   141,   141,   141,   141,   141,   141,   141,   141,   140,   140,   140,   140,   140,   140,   140,   140,   143,   143,   143,   143,   143,   143,   143,   143,   142,   142,   142,   142,   142,   142,   142,   142,   -1
// CHECK: 146,   146,   146,   146,   146,   146,   146,   146,   147,   147,   147,   147,   147,   147,   147,   147,   144,   144,   144,   144,   144,   144,   144,   144,   145,   145,   145,   145,   145,   145,   145,   145,   150,   150,   150,   150,   150,   150,   150,   150,   151,   151,   151,   151,   151,   151,   151,   151,   148,   148,   148,   148,   148,   148,   148,   148,   149,   149,   149,   149,   149,   149,   149,   149,   -1
// CHECK: 155,   155,   155,   155,   155,   155,   155,   155,   154,   154,   154,   154,   154,   154,   154,   154,   153,   153,   153,   153,   153,   153,   153,   153,   152,   152,   152,   152,   152,   152,   152,   152,   159,   159,   159,   159,   159,   159,   159,   159,   158,   158,   158,   158,   158,   158,   158,   158,   157,   157,   157,   157,   157,   157,   157,   157,   156,   156,   156,   156,   156,   156,   156,   156,   -1
// CHECK: 164,   164,   164,   164,   164,   164,   164,   164,   165,   165,   165,   165,   165,   165,   165,   165,   166,   166,   166,   166,   166,   166,   166,   166,   167,   167,   167,   167,   167,   167,   167,   167,   160,   160,   160,   160,   160,   160,   160,   160,   161,   161,   161,   161,   161,   161,   161,   161,   162,   162,   162,   162,   162,   162,   162,   162,   163,   163,   163,   163,   163,   163,   163,   163,   -1
// CHECK: 173,   173,   173,   173,   173,   173,   173,   173,   172,   172,   172,   172,   172,   172,   172,   172,   175,   175,   175,   175,   175,   175,   175,   175,   174,   174,   174,   174,   174,   174,   174,   174,   169,   169,   169,   169,   169,   169,   169,   169,   168,   168,   168,   168,   168,   168,   168,   168,   171,   171,   171,   171,   171,   171,   171,   171,   170,   170,   170,   170,   170,   170,   170,   170,   -1
// CHECK: 182,   182,   182,   182,   182,   182,   182,   182,   183,   183,   183,   183,   183,   183,   183,   183,   180,   180,   180,   180,   180,   180,   180,   180,   181,   181,   181,   181,   181,   181,   181,   181,   178,   178,   178,   178,   178,   178,   178,   178,   179,   179,   179,   179,   179,   179,   179,   179,   176,   176,   176,   176,   176,   176,   176,   176,   177,   177,   177,   177,   177,   177,   177,   177,   -1
// CHECK: 191,   191,   191,   191,   191,   191,   191,   191,   190,   190,   190,   190,   190,   190,   190,   190,   189,   189,   189,   189,   189,   189,   189,   189,   188,   188,   188,   188,   188,   188,   188,   188,   187,   187,   187,   187,   187,   187,   187,   187,   186,   186,   186,   186,   186,   186,   186,   186,   185,   185,   185,   185,   185,   185,   185,   185,   184,   184,   184,   184,   184,   184,   184,   184,   -1
// CHECK: 192,   192,   192,   192,   192,   192,   192,   192,   193,   193,   193,   193,   193,   193,   193,   193,   194,   194,   194,   194,   194,   194,   194,   194,   195,   195,   195,   195,   195,   195,   195,   195,   196,   196,   196,   196,   196,   196,   196,   196,   197,   197,   197,   197,   197,   197,   197,   197,   198,   198,   198,   198,   198,   198,   198,   198,   199,   199,   199,   199,   199,   199,   199,   199,   -1
// CHECK: 201,   201,   201,   201,   201,   201,   201,   201,   200,   200,   200,   200,   200,   200,   200,   200,   203,   203,   203,   203,   203,   203,   203,   203,   202,   202,   202,   202,   202,   202,   202,   202,   205,   205,   205,   205,   205,   205,   205,   205,   204,   204,   204,   204,   204,   204,   204,   204,   207,   207,   207,   207,   207,   207,   207,   207,   206,   206,   206,   206,   206,   206,   206,   206,   -1
// CHECK: 210,   210,   210,   210,   210,   210,   210,   210,   211,   211,   211,   211,   211,   211,   211,   211,   208,   208,   208,   208,   208,   208,   208,   208,   209,   209,   209,   209,   209,   209,   209,   209,   214,   214,   214,   214,   214,   214,   214,   214,   215,   215,   215,   215,   215,   215,   215,   215,   212,   212,   212,   212,   212,   212,   212,   212,   213,   213,   213,   213,   213,   213,   213,   213,   -1
// CHECK: 219,   219,   219,   219,   219,   219,   219,   219,   218,   218,   218,   218,   218,   218,   218,   218,   217,   217,   217,   217,   217,   217,   217,   217,   216,   216,   216,   216,   216,   216,   216,   216,   223,   223,   223,   223,   223,   223,   223,   223,   222,   222,   222,   222,   222,   222,   222,   222,   221,   221,   221,   221,   221,   221,   221,   221,   220,   220,   220,   220,   220,   220,   220,   220,   -1
// CHECK: 228,   228,   228,   228,   228,   228,   228,   228,   229,   229,   229,   229,   229,   229,   229,   229,   230,   230,   230,   230,   230,   230,   230,   230,   231,   231,   231,   231,   231,   231,   231,   231,   224,   224,   224,   224,   224,   224,   224,   224,   225,   225,   225,   225,   225,   225,   225,   225,   226,   226,   226,   226,   226,   226,   226,   226,   227,   227,   227,   227,   227,   227,   227,   227,   -1
// CHECK: 237,   237,   237,   237,   237,   237,   237,   237,   236,   236,   236,   236,   236,   236,   236,   236,   239,   239,   239,   239,   239,   239,   239,   239,   238,   238,   238,   238,   238,   238,   238,   238,   233,   233,   233,   233,   233,   233,   233,   233,   232,   232,   232,   232,   232,   232,   232,   232,   235,   235,   235,   235,   235,   235,   235,   235,   234,   234,   234,   234,   234,   234,   234,   234,   -1
// CHECK: 246,   246,   246,   246,   246,   246,   246,   246,   247,   247,   247,   247,   247,   247,   247,   247,   244,   244,   244,   244,   244,   244,   244,   244,   245,   245,   245,   245,   245,   245,   245,   245,   242,   242,   242,   242,   242,   242,   242,   242,   243,   243,   243,   243,   243,   243,   243,   243,   240,   240,   240,   240,   240,   240,   240,   240,   241,   241,   241,   241,   241,   241,   241,   241,   -1
// CHECK: 255,   255,   255,   255,   255,   255,   255,   255,   254,   254,   254,   254,   254,   254,   254,   254,   253,   253,   253,   253,   253,   253,   253,   253,   252,   252,   252,   252,   252,   252,   252,   252,   251,   251,   251,   251,   251,   251,   251,   251,   250,   250,   250,   250,   250,   250,   250,   250,   249,   249,   249,   249,   249,   249,   249,   249,   248,   248,   248,   248,   248,   248,   248,   248,   -1
// CHECK: 256,   256,   256,   256,   256,   256,   256,   256,   257,   257,   257,   257,   257,   257,   257,   257,   258,   258,   258,   258,   258,   258,   258,   258,   259,   259,   259,   259,   259,   259,   259,   259,   260,   260,   260,   260,   260,   260,   260,   260,   261,   261,   261,   261,   261,   261,   261,   261,   262,   262,   262,   262,   262,   262,   262,   262,   263,   263,   263,   263,   263,   263,   263,   263,   -1
// CHECK: 265,   265,   265,   265,   265,   265,   265,   265,   264,   264,   264,   264,   264,   264,   264,   264,   267,   267,   267,   267,   267,   267,   267,   267,   266,   266,   266,   266,   266,   266,   266,   266,   269,   269,   269,   269,   269,   269,   269,   269,   268,   268,   268,   268,   268,   268,   268,   268,   271,   271,   271,   271,   271,   271,   271,   271,   270,   270,   270,   270,   270,   270,   270,   270,   -1
// CHECK: 274,   274,   274,   274,   274,   274,   274,   274,   275,   275,   275,   275,   275,   275,   275,   275,   272,   272,   272,   272,   272,   272,   272,   272,   273,   273,   273,   273,   273,   273,   273,   273,   278,   278,   278,   278,   278,   278,   278,   278,   279,   279,   279,   279,   279,   279,   279,   279,   276,   276,   276,   276,   276,   276,   276,   276,   277,   277,   277,   277,   277,   277,   277,   277,   -1
// CHECK: 283,   283,   283,   283,   283,   283,   283,   283,   282,   282,   282,   282,   282,   282,   282,   282,   281,   281,   281,   281,   281,   281,   281,   281,   280,   280,   280,   280,   280,   280,   280,   280,   287,   287,   287,   287,   287,   287,   287,   287,   286,   286,   286,   286,   286,   286,   286,   286,   285,   285,   285,   285,   285,   285,   285,   285,   284,   284,   284,   284,   284,   284,   284,   284,   -1
// CHECK: 292,   292,   292,   292,   292,   292,   292,   292,   293,   293,   293,   293,   293,   293,   293,   293,   294,   294,   294,   294,   294,   294,   294,   294,   295,   295,   295,   295,   295,   295,   295,   295,   288,   288,   288,   288,   288,   288,   288,   288,   289,   289,   289,   289,   289,   289,   289,   289,   290,   290,   290,   290,   290,   290,   290,   290,   291,   291,   291,   291,   291,   291,   291,   291,   -1
// CHECK: 301,   301,   301,   301,   301,   301,   301,   301,   300,   300,   300,   300,   300,   300,   300,   300,   303,   303,   303,   303,   303,   303,   303,   303,   302,   302,   302,   302,   302,   302,   302,   302,   297,   297,   297,   297,   297,   297,   297,   297,   296,   296,   296,   296,   296,   296,   296,   296,   299,   299,   299,   299,   299,   299,   299,   299,   298,   298,   298,   298,   298,   298,   298,   298,   -1
// CHECK: 310,   310,   310,   310,   310,   310,   310,   310,   311,   311,   311,   311,   311,   311,   311,   311,   308,   308,   308,   308,   308,   308,   308,   308,   309,   309,   309,   309,   309,   309,   309,   309,   306,   306,   306,   306,   306,   306,   306,   306,   307,   307,   307,   307,   307,   307,   307,   307,   304,   304,   304,   304,   304,   304,   304,   304,   305,   305,   305,   305,   305,   305,   305,   305,   -1
// CHECK: 319,   319,   319,   319,   319,   319,   319,   319,   318,   318,   318,   318,   318,   318,   318,   318,   317,   317,   317,   317,   317,   317,   317,   317,   316,   316,   316,   316,   316,   316,   316,   316,   315,   315,   315,   315,   315,   315,   315,   315,   314,   314,   314,   314,   314,   314,   314,   314,   313,   313,   313,   313,   313,   313,   313,   313,   312,   312,   312,   312,   312,   312,   312,   312,   -1
// CHECK: 320,   320,   320,   320,   320,   320,   320,   320,   321,   321,   321,   321,   321,   321,   321,   321,   322,   322,   322,   322,   322,   322,   322,   322,   323,   323,   323,   323,   323,   323,   323,   323,   324,   324,   324,   324,   324,   324,   324,   324,   325,   325,   325,   325,   325,   325,   325,   325,   326,   326,   326,   326,   326,   326,   326,   326,   327,   327,   327,   327,   327,   327,   327,   327,   -1
// CHECK: 329,   329,   329,   329,   329,   329,   329,   329,   328,   328,   328,   328,   328,   328,   328,   328,   331,   331,   331,   331,   331,   331,   331,   331,   330,   330,   330,   330,   330,   330,   330,   330,   333,   333,   333,   333,   333,   333,   333,   333,   332,   332,   332,   332,   332,   332,   332,   332,   335,   335,   335,   335,   335,   335,   335,   335,   334,   334,   334,   334,   334,   334,   334,   334,   -1
// CHECK: 338,   338,   338,   338,   338,   338,   338,   338,   339,   339,   339,   339,   339,   339,   339,   339,   336,   336,   336,   336,   336,   336,   336,   336,   337,   337,   337,   337,   337,   337,   337,   337,   342,   342,   342,   342,   342,   342,   342,   342,   343,   343,   343,   343,   343,   343,   343,   343,   340,   340,   340,   340,   340,   340,   340,   340,   341,   341,   341,   341,   341,   341,   341,   341,   -1
// CHECK: 347,   347,   347,   347,   347,   347,   347,   347,   346,   346,   346,   346,   346,   346,   346,   346,   345,   345,   345,   345,   345,   345,   345,   345,   344,   344,   344,   344,   344,   344,   344,   344,   351,   351,   351,   351,   351,   351,   351,   351,   350,   350,   350,   350,   350,   350,   350,   350,   349,   349,   349,   349,   349,   349,   349,   349,   348,   348,   348,   348,   348,   348,   348,   348,   -1
// CHECK: 356,   356,   356,   356,   356,   356,   356,   356,   357,   357,   357,   357,   357,   357,   357,   357,   358,   358,   358,   358,   358,   358,   358,   358,   359,   359,   359,   359,   359,   359,   359,   359,   352,   352,   352,   352,   352,   352,   352,   352,   353,   353,   353,   353,   353,   353,   353,   353,   354,   354,   354,   354,   354,   354,   354,   354,   355,   355,   355,   355,   355,   355,   355,   355,   -1
// CHECK: 365,   365,   365,   365,   365,   365,   365,   365,   364,   364,   364,   364,   364,   364,   364,   364,   367,   367,   367,   367,   367,   367,   367,   367,   366,   366,   366,   366,   366,   366,   366,   366,   361,   361,   361,   361,   361,   361,   361,   361,   360,   360,   360,   360,   360,   360,   360,   360,   363,   363,   363,   363,   363,   363,   363,   363,   362,   362,   362,   362,   362,   362,   362,   362,   -1
// CHECK: 374,   374,   374,   374,   374,   374,   374,   374,   375,   375,   375,   375,   375,   375,   375,   375,   372,   372,   372,   372,   372,   372,   372,   372,   373,   373,   373,   373,   373,   373,   373,   373,   370,   370,   370,   370,   370,   370,   370,   370,   371,   371,   371,   371,   371,   371,   371,   371,   368,   368,   368,   368,   368,   368,   368,   368,   369,   369,   369,   369,   369,   369,   369,   369,   -1
// CHECK: 383,   383,   383,   383,   383,   383,   383,   383,   382,   382,   382,   382,   382,   382,   382,   382,   381,   381,   381,   381,   381,   381,   381,   381,   380,   380,   380,   380,   380,   380,   380,   380,   379,   379,   379,   379,   379,   379,   379,   379,   378,   378,   378,   378,   378,   378,   378,   378,   377,   377,   377,   377,   377,   377,   377,   377,   376,   376,   376,   376,   376,   376,   376,   376,   -1
// CHECK: 384,   384,   384,   384,   384,   384,   384,   384,   385,   385,   385,   385,   385,   385,   385,   385,   386,   386,   386,   386,   386,   386,   386,   386,   387,   387,   387,   387,   387,   387,   387,   387,   388,   388,   388,   388,   388,   388,   388,   388,   389,   389,   389,   389,   389,   389,   389,   389,   390,   390,   390,   390,   390,   390,   390,   390,   391,   391,   391,   391,   391,   391,   391,   391,   -1
// CHECK: 393,   393,   393,   393,   393,   393,   393,   393,   392,   392,   392,   392,   392,   392,   392,   392,   395,   395,   395,   395,   395,   395,   395,   395,   394,   394,   394,   394,   394,   394,   394,   394,   397,   397,   397,   397,   397,   397,   397,   397,   396,   396,   396,   396,   396,   396,   396,   396,   399,   399,   399,   399,   399,   399,   399,   399,   398,   398,   398,   398,   398,   398,   398,   398,   -1
// CHECK: 402,   402,   402,   402,   402,   402,   402,   402,   403,   403,   403,   403,   403,   403,   403,   403,   400,   400,   400,   400,   400,   400,   400,   400,   401,   401,   401,   401,   401,   401,   401,   401,   406,   406,   406,   406,   406,   406,   406,   406,   407,   407,   407,   407,   407,   407,   407,   407,   404,   404,   404,   404,   404,   404,   404,   404,   405,   405,   405,   405,   405,   405,   405,   405,   -1
// CHECK: 411,   411,   411,   411,   411,   411,   411,   411,   410,   410,   410,   410,   410,   410,   410,   410,   409,   409,   409,   409,   409,   409,   409,   409,   408,   408,   408,   408,   408,   408,   408,   408,   415,   415,   415,   415,   415,   415,   415,   415,   414,   414,   414,   414,   414,   414,   414,   414,   413,   413,   413,   413,   413,   413,   413,   413,   412,   412,   412,   412,   412,   412,   412,   412,   -1
// CHECK: 420,   420,   420,   420,   420,   420,   420,   420,   421,   421,   421,   421,   421,   421,   421,   421,   422,   422,   422,   422,   422,   422,   422,   422,   423,   423,   423,   423,   423,   423,   423,   423,   416,   416,   416,   416,   416,   416,   416,   416,   417,   417,   417,   417,   417,   417,   417,   417,   418,   418,   418,   418,   418,   418,   418,   418,   419,   419,   419,   419,   419,   419,   419,   419,   -1
// CHECK: 429,   429,   429,   429,   429,   429,   429,   429,   428,   428,   428,   428,   428,   428,   428,   428,   431,   431,   431,   431,   431,   431,   431,   431,   430,   430,   430,   430,   430,   430,   430,   430,   425,   425,   425,   425,   425,   425,   425,   425,   424,   424,   424,   424,   424,   424,   424,   424,   427,   427,   427,   427,   427,   427,   427,   427,   426,   426,   426,   426,   426,   426,   426,   426,   -1
// CHECK: 438,   438,   438,   438,   438,   438,   438,   438,   439,   439,   439,   439,   439,   439,   439,   439,   436,   436,   436,   436,   436,   436,   436,   436,   437,   437,   437,   437,   437,   437,   437,   437,   434,   434,   434,   434,   434,   434,   434,   434,   435,   435,   435,   435,   435,   435,   435,   435,   432,   432,   432,   432,   432,   432,   432,   432,   433,   433,   433,   433,   433,   433,   433,   433,   -1
// CHECK: 447,   447,   447,   447,   447,   447,   447,   447,   446,   446,   446,   446,   446,   446,   446,   446,   445,   445,   445,   445,   445,   445,   445,   445,   444,   444,   444,   444,   444,   444,   444,   444,   443,   443,   443,   443,   443,   443,   443,   443,   442,   442,   442,   442,   442,   442,   442,   442,   441,   441,   441,   441,   441,   441,   441,   441,   440,   440,   440,   440,   440,   440,   440,   440,   -1
// CHECK: 448,   448,   448,   448,   448,   448,   448,   448,   449,   449,   449,   449,   449,   449,   449,   449,   450,   450,   450,   450,   450,   450,   450,   450,   451,   451,   451,   451,   451,   451,   451,   451,   452,   452,   452,   452,   452,   452,   452,   452,   453,   453,   453,   453,   453,   453,   453,   453,   454,   454,   454,   454,   454,   454,   454,   454,   455,   455,   455,   455,   455,   455,   455,   455,   -1
// CHECK: 457,   457,   457,   457,   457,   457,   457,   457,   456,   456,   456,   456,   456,   456,   456,   456,   459,   459,   459,   459,   459,   459,   459,   459,   458,   458,   458,   458,   458,   458,   458,   458,   461,   461,   461,   461,   461,   461,   461,   461,   460,   460,   460,   460,   460,   460,   460,   460,   463,   463,   463,   463,   463,   463,   463,   463,   462,   462,   462,   462,   462,   462,   462,   462,   -1
// CHECK: 466,   466,   466,   466,   466,   466,   466,   466,   467,   467,   467,   467,   467,   467,   467,   467,   464,   464,   464,   464,   464,   464,   464,   464,   465,   465,   465,   465,   465,   465,   465,   465,   470,   470,   470,   470,   470,   470,   470,   470,   471,   471,   471,   471,   471,   471,   471,   471,   468,   468,   468,   468,   468,   468,   468,   468,   469,   469,   469,   469,   469,   469,   469,   469,   -1
// CHECK: 475,   475,   475,   475,   475,   475,   475,   475,   474,   474,   474,   474,   474,   474,   474,   474,   473,   473,   473,   473,   473,   473,   473,   473,   472,   472,   472,   472,   472,   472,   472,   472,   479,   479,   479,   479,   479,   479,   479,   479,   478,   478,   478,   478,   478,   478,   478,   478,   477,   477,   477,   477,   477,   477,   477,   477,   476,   476,   476,   476,   476,   476,   476,   476,   -1
// CHECK: 484,   484,   484,   484,   484,   484,   484,   484,   485,   485,   485,   485,   485,   485,   485,   485,   486,   486,   486,   486,   486,   486,   486,   486,   487,   487,   487,   487,   487,   487,   487,   487,   480,   480,   480,   480,   480,   480,   480,   480,   481,   481,   481,   481,   481,   481,   481,   481,   482,   482,   482,   482,   482,   482,   482,   482,   483,   483,   483,   483,   483,   483,   483,   483,   -1
// CHECK: 493,   493,   493,   493,   493,   493,   493,   493,   492,   492,   492,   492,   492,   492,   492,   492,   495,   495,   495,   495,   495,   495,   495,   495,   494,   494,   494,   494,   494,   494,   494,   494,   489,   489,   489,   489,   489,   489,   489,   489,   488,   488,   488,   488,   488,   488,   488,   488,   491,   491,   491,   491,   491,   491,   491,   491,   490,   490,   490,   490,   490,   490,   490,   490,   -1
// CHECK: 502,   502,   502,   502,   502,   502,   502,   502,   503,   503,   503,   503,   503,   503,   503,   503,   500,   500,   500,   500,   500,   500,   500,   500,   501,   501,   501,   501,   501,   501,   501,   501,   498,   498,   498,   498,   498,   498,   498,   498,   499,   499,   499,   499,   499,   499,   499,   499,   496,   496,   496,   496,   496,   496,   496,   496,   497,   497,   497,   497,   497,   497,   497,   497,   -1
// CHECK: 511,   511,   511,   511,   511,   511,   511,   511,   510,   510,   510,   510,   510,   510,   510,   510,   509,   509,   509,   509,   509,   509,   509,   509,   508,   508,   508,   508,   508,   508,   508,   508,   507,   507,   507,   507,   507,   507,   507,   507,   506,   506,   506,   506,   506,   506,   506,   506,   505,   505,   505,   505,   505,   505,   505,   505,   504,   504,   504,   504,   504,   504,   504,   504,   -1
// CHECK: 512,   512,   512,   512,   512,   512,   512,   512,   513,   513,   513,   513,   513,   513,   513,   513,   514,   514,   514,   514,   514,   514,   514,   514,   515,   515,   515,   515,   515,   515,   515,   515,   516,   516,   516,   516,   516,   516,   516,   516,   517,   517,   517,   517,   517,   517,   517,   517,   518,   518,   518,   518,   518,   518,   518,   518,   519,   519,   519,   519,   519,   519,   519,   519,   -1
// CHECK: 521,   521,   521,   521,   521,   521,   521,   521,   520,   520,   520,   520,   520,   520,   520,   520,   523,   523,   523,   523,   523,   523,   523,   523,   522,   522,   522,   522,   522,   522,   522,   522,   525,   525,   525,   525,   525,   525,   525,   525,   524,   524,   524,   524,   524,   524,   524,   524,   527,   527,   527,   527,   527,   527,   527,   527,   526,   526,   526,   526,   526,   526,   526,   526,   -1
// CHECK: 530,   530,   530,   530,   530,   530,   530,   530,   531,   531,   531,   531,   531,   531,   531,   531,   528,   528,   528,   528,   528,   528,   528,   528,   529,   529,   529,   529,   529,   529,   529,   529,   534,   534,   534,   534,   534,   534,   534,   534,   535,   535,   535,   535,   535,   535,   535,   535,   532,   532,   532,   532,   532,   532,   532,   532,   533,   533,   533,   533,   533,   533,   533,   533,   -1
// CHECK: 539,   539,   539,   539,   539,   539,   539,   539,   538,   538,   538,   538,   538,   538,   538,   538,   537,   537,   537,   537,   537,   537,   537,   537,   536,   536,   536,   536,   536,   536,   536,   536,   543,   543,   543,   543,   543,   543,   543,   543,   542,   542,   542,   542,   542,   542,   542,   542,   541,   541,   541,   541,   541,   541,   541,   541,   540,   540,   540,   540,   540,   540,   540,   540,   -1
// CHECK: 548,   548,   548,   548,   548,   548,   548,   548,   549,   549,   549,   549,   549,   549,   549,   549,   550,   550,   550,   550,   550,   550,   550,   550,   551,   551,   551,   551,   551,   551,   551,   551,   544,   544,   544,   544,   544,   544,   544,   544,   545,   545,   545,   545,   545,   545,   545,   545,   546,   546,   546,   546,   546,   546,   546,   546,   547,   547,   547,   547,   547,   547,   547,   547,   -1
// CHECK: 557,   557,   557,   557,   557,   557,   557,   557,   556,   556,   556,   556,   556,   556,   556,   556,   559,   559,   559,   559,   559,   559,   559,   559,   558,   558,   558,   558,   558,   558,   558,   558,   553,   553,   553,   553,   553,   553,   553,   553,   552,   552,   552,   552,   552,   552,   552,   552,   555,   555,   555,   555,   555,   555,   555,   555,   554,   554,   554,   554,   554,   554,   554,   554,   -1
// CHECK: 566,   566,   566,   566,   566,   566,   566,   566,   567,   567,   567,   567,   567,   567,   567,   567,   564,   564,   564,   564,   564,   564,   564,   564,   565,   565,   565,   565,   565,   565,   565,   565,   562,   562,   562,   562,   562,   562,   562,   562,   563,   563,   563,   563,   563,   563,   563,   563,   560,   560,   560,   560,   560,   560,   560,   560,   561,   561,   561,   561,   561,   561,   561,   561,   -1
// CHECK: 575,   575,   575,   575,   575,   575,   575,   575,   574,   574,   574,   574,   574,   574,   574,   574,   573,   573,   573,   573,   573,   573,   573,   573,   572,   572,   572,   572,   572,   572,   572,   572,   571,   571,   571,   571,   571,   571,   571,   571,   570,   570,   570,   570,   570,   570,   570,   570,   569,   569,   569,   569,   569,   569,   569,   569,   568,   568,   568,   568,   568,   568,   568,   568,   -1
// CHECK: 576,   576,   576,   576,   576,   576,   576,   576,   577,   577,   577,   577,   577,   577,   577,   577,   578,   578,   578,   578,   578,   578,   578,   578,   579,   579,   579,   579,   579,   579,   579,   579,   580,   580,   580,   580,   580,   580,   580,   580,   581,   581,   581,   581,   581,   581,   581,   581,   582,   582,   582,   582,   582,   582,   582,   582,   583,   583,   583,   583,   583,   583,   583,   583,   -1
// CHECK: 585,   585,   585,   585,   585,   585,   585,   585,   584,   584,   584,   584,   584,   584,   584,   584,   587,   587,   587,   587,   587,   587,   587,   587,   586,   586,   586,   586,   586,   586,   586,   586,   589,   589,   589,   589,   589,   589,   589,   589,   588,   588,   588,   588,   588,   588,   588,   588,   591,   591,   591,   591,   591,   591,   591,   591,   590,   590,   590,   590,   590,   590,   590,   590,   -1
// CHECK: 594,   594,   594,   594,   594,   594,   594,   594,   595,   595,   595,   595,   595,   595,   595,   595,   592,   592,   592,   592,   592,   592,   592,   592,   593,   593,   593,   593,   593,   593,   593,   593,   598,   598,   598,   598,   598,   598,   598,   598,   599,   599,   599,   599,   599,   599,   599,   599,   596,   596,   596,   596,   596,   596,   596,   596,   597,   597,   597,   597,   597,   597,   597,   597,   -1
// CHECK: 603,   603,   603,   603,   603,   603,   603,   603,   602,   602,   602,   602,   602,   602,   602,   602,   601,   601,   601,   601,   601,   601,   601,   601,   600,   600,   600,   600,   600,   600,   600,   600,   607,   607,   607,   607,   607,   607,   607,   607,   606,   606,   606,   606,   606,   606,   606,   606,   605,   605,   605,   605,   605,   605,   605,   605,   604,   604,   604,   604,   604,   604,   604,   604,   -1
// CHECK: 612,   612,   612,   612,   612,   612,   612,   612,   613,   613,   613,   613,   613,   613,   613,   613,   614,   614,   614,   614,   614,   614,   614,   614,   615,   615,   615,   615,   615,   615,   615,   615,   608,   608,   608,   608,   608,   608,   608,   608,   609,   609,   609,   609,   609,   609,   609,   609,   610,   610,   610,   610,   610,   610,   610,   610,   611,   611,   611,   611,   611,   611,   611,   611,   -1
// CHECK: 621,   621,   621,   621,   621,   621,   621,   621,   620,   620,   620,   620,   620,   620,   620,   620,   623,   623,   623,   623,   623,   623,   623,   623,   622,   622,   622,   622,   622,   622,   622,   622,   617,   617,   617,   617,   617,   617,   617,   617,   616,   616,   616,   616,   616,   616,   616,   616,   619,   619,   619,   619,   619,   619,   619,   619,   618,   618,   618,   618,   618,   618,   618,   618,   -1
// CHECK: 630,   630,   630,   630,   630,   630,   630,   630,   631,   631,   631,   631,   631,   631,   631,   631,   628,   628,   628,   628,   628,   628,   628,   628,   629,   629,   629,   629,   629,   629,   629,   629,   626,   626,   626,   626,   626,   626,   626,   626,   627,   627,   627,   627,   627,   627,   627,   627,   624,   624,   624,   624,   624,   624,   624,   624,   625,   625,   625,   625,   625,   625,   625,   625,   -1
// CHECK: 639,   639,   639,   639,   639,   639,   639,   639,   638,   638,   638,   638,   638,   638,   638,   638,   637,   637,   637,   637,   637,   637,   637,   637,   636,   636,   636,   636,   636,   636,   636,   636,   635,   635,   635,   635,   635,   635,   635,   635,   634,   634,   634,   634,   634,   634,   634,   634,   633,   633,   633,   633,   633,   633,   633,   633,   632,   632,   632,   632,   632,   632,   632,   632,   -1
// CHECK: 640,   640,   640,   640,   640,   640,   640,   640,   641,   641,   641,   641,   641,   641,   641,   641,   642,   642,   642,   642,   642,   642,   642,   642,   643,   643,   643,   643,   643,   643,   643,   643,   644,   644,   644,   644,   644,   644,   644,   644,   645,   645,   645,   645,   645,   645,   645,   645,   646,   646,   646,   646,   646,   646,   646,   646,   647,   647,   647,   647,   647,   647,   647,   647,   -1
// CHECK: 649,   649,   649,   649,   649,   649,   649,   649,   648,   648,   648,   648,   648,   648,   648,   648,   651,   651,   651,   651,   651,   651,   651,   651,   650,   650,   650,   650,   650,   650,   650,   650,   653,   653,   653,   653,   653,   653,   653,   653,   652,   652,   652,   652,   652,   652,   652,   652,   655,   655,   655,   655,   655,   655,   655,   655,   654,   654,   654,   654,   654,   654,   654,   654,   -1
// CHECK: 658,   658,   658,   658,   658,   658,   658,   658,   659,   659,   659,   659,   659,   659,   659,   659,   656,   656,   656,   656,   656,   656,   656,   656,   657,   657,   657,   657,   657,   657,   657,   657,   662,   662,   662,   662,   662,   662,   662,   662,   663,   663,   663,   663,   663,   663,   663,   663,   660,   660,   660,   660,   660,   660,   660,   660,   661,   661,   661,   661,   661,   661,   661,   661,   -1
// CHECK: 667,   667,   667,   667,   667,   667,   667,   667,   666,   666,   666,   666,   666,   666,   666,   666,   665,   665,   665,   665,   665,   665,   665,   665,   664,   664,   664,   664,   664,   664,   664,   664,   671,   671,   671,   671,   671,   671,   671,   671,   670,   670,   670,   670,   670,   670,   670,   670,   669,   669,   669,   669,   669,   669,   669,   669,   668,   668,   668,   668,   668,   668,   668,   668,   -1
// CHECK: 676,   676,   676,   676,   676,   676,   676,   676,   677,   677,   677,   677,   677,   677,   677,   677,   678,   678,   678,   678,   678,   678,   678,   678,   679,   679,   679,   679,   679,   679,   679,   679,   672,   672,   672,   672,   672,   672,   672,   672,   673,   673,   673,   673,   673,   673,   673,   673,   674,   674,   674,   674,   674,   674,   674,   674,   675,   675,   675,   675,   675,   675,   675,   675,   -1
// CHECK: 685,   685,   685,   685,   685,   685,   685,   685,   684,   684,   684,   684,   684,   684,   684,   684,   687,   687,   687,   687,   687,   687,   687,   687,   686,   686,   686,   686,   686,   686,   686,   686,   681,   681,   681,   681,   681,   681,   681,   681,   680,   680,   680,   680,   680,   680,   680,   680,   683,   683,   683,   683,   683,   683,   683,   683,   682,   682,   682,   682,   682,   682,   682,   682,   -1
// CHECK: 694,   694,   694,   694,   694,   694,   694,   694,   695,   695,   695,   695,   695,   695,   695,   695,   692,   692,   692,   692,   692,   692,   692,   692,   693,   693,   693,   693,   693,   693,   693,   693,   690,   690,   690,   690,   690,   690,   690,   690,   691,   691,   691,   691,   691,   691,   691,   691,   688,   688,   688,   688,   688,   688,   688,   688,   689,   689,   689,   689,   689,   689,   689,   689,   -1
// CHECK: 703,   703,   703,   703,   703,   703,   703,   703,   702,   702,   702,   702,   702,   702,   702,   702,   701,   701,   701,   701,   701,   701,   701,   701,   700,   700,   700,   700,   700,   700,   700,   700,   699,   699,   699,   699,   699,   699,   699,   699,   698,   698,   698,   698,   698,   698,   698,   698,   697,   697,   697,   697,   697,   697,   697,   697,   696,   696,   696,   696,   696,   696,   696,   696,   -1
// CHECK: 704,   704,   704,   704,   704,   704,   704,   704,   705,   705,   705,   705,   705,   705,   705,   705,   706,   706,   706,   706,   706,   706,   706,   706,   707,   707,   707,   707,   707,   707,   707,   707,   708,   708,   708,   708,   708,   708,   708,   708,   709,   709,   709,   709,   709,   709,   709,   709,   710,   710,   710,   710,   710,   710,   710,   710,   711,   711,   711,   711,   711,   711,   711,   711,   -1
// CHECK: 713,   713,   713,   713,   713,   713,   713,   713,   712,   712,   712,   712,   712,   712,   712,   712,   715,   715,   715,   715,   715,   715,   715,   715,   714,   714,   714,   714,   714,   714,   714,   714,   717,   717,   717,   717,   717,   717,   717,   717,   716,   716,   716,   716,   716,   716,   716,   716,   719,   719,   719,   719,   719,   719,   719,   719,   718,   718,   718,   718,   718,   718,   718,   718,   -1
// CHECK: 722,   722,   722,   722,   722,   722,   722,   722,   723,   723,   723,   723,   723,   723,   723,   723,   720,   720,   720,   720,   720,   720,   720,   720,   721,   721,   721,   721,   721,   721,   721,   721,   726,   726,   726,   726,   726,   726,   726,   726,   727,   727,   727,   727,   727,   727,   727,   727,   724,   724,   724,   724,   724,   724,   724,   724,   725,   725,   725,   725,   725,   725,   725,   725,   -1
// CHECK: 731,   731,   731,   731,   731,   731,   731,   731,   730,   730,   730,   730,   730,   730,   730,   730,   729,   729,   729,   729,   729,   729,   729,   729,   728,   728,   728,   728,   728,   728,   728,   728,   735,   735,   735,   735,   735,   735,   735,   735,   734,   734,   734,   734,   734,   734,   734,   734,   733,   733,   733,   733,   733,   733,   733,   733,   732,   732,   732,   732,   732,   732,   732,   732,   -1
// CHECK: 740,   740,   740,   740,   740,   740,   740,   740,   741,   741,   741,   741,   741,   741,   741,   741,   742,   742,   742,   742,   742,   742,   742,   742,   743,   743,   743,   743,   743,   743,   743,   743,   736,   736,   736,   736,   736,   736,   736,   736,   737,   737,   737,   737,   737,   737,   737,   737,   738,   738,   738,   738,   738,   738,   738,   738,   739,   739,   739,   739,   739,   739,   739,   739,   -1
// CHECK: 749,   749,   749,   749,   749,   749,   749,   749,   748,   748,   748,   748,   748,   748,   748,   748,   751,   751,   751,   751,   751,   751,   751,   751,   750,   750,   750,   750,   750,   750,   750,   750,   745,   745,   745,   745,   745,   745,   745,   745,   744,   744,   744,   744,   744,   744,   744,   744,   747,   747,   747,   747,   747,   747,   747,   747,   746,   746,   746,   746,   746,   746,   746,   746,   -1
// CHECK: 758,   758,   758,   758,   758,   758,   758,   758,   759,   759,   759,   759,   759,   759,   759,   759,   756,   756,   756,   756,   756,   756,   756,   756,   757,   757,   757,   757,   757,   757,   757,   757,   754,   754,   754,   754,   754,   754,   754,   754,   755,   755,   755,   755,   755,   755,   755,   755,   752,   752,   752,   752,   752,   752,   752,   752,   753,   753,   753,   753,   753,   753,   753,   753,   -1
// CHECK: 767,   767,   767,   767,   767,   767,   767,   767,   766,   766,   766,   766,   766,   766,   766,   766,   765,   765,   765,   765,   765,   765,   765,   765,   764,   764,   764,   764,   764,   764,   764,   764,   763,   763,   763,   763,   763,   763,   763,   763,   762,   762,   762,   762,   762,   762,   762,   762,   761,   761,   761,   761,   761,   761,   761,   761,   760,   760,   760,   760,   760,   760,   760,   760,   -1
// CHECK: 768,   768,   768,   768,   768,   768,   768,   768,   769,   769,   769,   769,   769,   769,   769,   769,   770,   770,   770,   770,   770,   770,   770,   770,   771,   771,   771,   771,   771,   771,   771,   771,   772,   772,   772,   772,   772,   772,   772,   772,   773,   773,   773,   773,   773,   773,   773,   773,   774,   774,   774,   774,   774,   774,   774,   774,   775,   775,   775,   775,   775,   775,   775,   775,   -1
// CHECK: 777,   777,   777,   777,   777,   777,   777,   777,   776,   776,   776,   776,   776,   776,   776,   776,   779,   779,   779,   779,   779,   779,   779,   779,   778,   778,   778,   778,   778,   778,   778,   778,   781,   781,   781,   781,   781,   781,   781,   781,   780,   780,   780,   780,   780,   780,   780,   780,   783,   783,   783,   783,   783,   783,   783,   783,   782,   782,   782,   782,   782,   782,   782,   782,   -1
// CHECK: 786,   786,   786,   786,   786,   786,   786,   786,   787,   787,   787,   787,   787,   787,   787,   787,   784,   784,   784,   784,   784,   784,   784,   784,   785,   785,   785,   785,   785,   785,   785,   785,   790,   790,   790,   790,   790,   790,   790,   790,   791,   791,   791,   791,   791,   791,   791,   791,   788,   788,   788,   788,   788,   788,   788,   788,   789,   789,   789,   789,   789,   789,   789,   789,   -1
// CHECK: 795,   795,   795,   795,   795,   795,   795,   795,   794,   794,   794,   794,   794,   794,   794,   794,   793,   793,   793,   793,   793,   793,   793,   793,   792,   792,   792,   792,   792,   792,   792,   792,   799,   799,   799,   799,   799,   799,   799,   799,   798,   798,   798,   798,   798,   798,   798,   798,   797,   797,   797,   797,   797,   797,   797,   797,   796,   796,   796,   796,   796,   796,   796,   796,   -1
// CHECK: 804,   804,   804,   804,   804,   804,   804,   804,   805,   805,   805,   805,   805,   805,   805,   805,   806,   806,   806,   806,   806,   806,   806,   806,   807,   807,   807,   807,   807,   807,   807,   807,   800,   800,   800,   800,   800,   800,   800,   800,   801,   801,   801,   801,   801,   801,   801,   801,   802,   802,   802,   802,   802,   802,   802,   802,   803,   803,   803,   803,   803,   803,   803,   803,   -1
// CHECK: 813,   813,   813,   813,   813,   813,   813,   813,   812,   812,   812,   812,   812,   812,   812,   812,   815,   815,   815,   815,   815,   815,   815,   815,   814,   814,   814,   814,   814,   814,   814,   814,   809,   809,   809,   809,   809,   809,   809,   809,   808,   808,   808,   808,   808,   808,   808,   808,   811,   811,   811,   811,   811,   811,   811,   811,   810,   810,   810,   810,   810,   810,   810,   810,   -1
// CHECK: 822,   822,   822,   822,   822,   822,   822,   822,   823,   823,   823,   823,   823,   823,   823,   823,   820,   820,   820,   820,   820,   820,   820,   820,   821,   821,   821,   821,   821,   821,   821,   821,   818,   818,   818,   818,   818,   818,   818,   818,   819,   819,   819,   819,   819,   819,   819,   819,   816,   816,   816,   816,   816,   816,   816,   816,   817,   817,   817,   817,   817,   817,   817,   817,   -1
// CHECK: 831,   831,   831,   831,   831,   831,   831,   831,   830,   830,   830,   830,   830,   830,   830,   830,   829,   829,   829,   829,   829,   829,   829,   829,   828,   828,   828,   828,   828,   828,   828,   828,   827,   827,   827,   827,   827,   827,   827,   827,   826,   826,   826,   826,   826,   826,   826,   826,   825,   825,   825,   825,   825,   825,   825,   825,   824,   824,   824,   824,   824,   824,   824,   824,   -1
// CHECK: 832,   832,   832,   832,   832,   832,   832,   832,   833,   833,   833,   833,   833,   833,   833,   833,   834,   834,   834,   834,   834,   834,   834,   834,   835,   835,   835,   835,   835,   835,   835,   835,   836,   836,   836,   836,   836,   836,   836,   836,   837,   837,   837,   837,   837,   837,   837,   837,   838,   838,   838,   838,   838,   838,   838,   838,   839,   839,   839,   839,   839,   839,   839,   839,   -1
// CHECK: 841,   841,   841,   841,   841,   841,   841,   841,   840,   840,   840,   840,   840,   840,   840,   840,   843,   843,   843,   843,   843,   843,   843,   843,   842,   842,   842,   842,   842,   842,   842,   842,   845,   845,   845,   845,   845,   845,   845,   845,   844,   844,   844,   844,   844,   844,   844,   844,   847,   847,   847,   847,   847,   847,   847,   847,   846,   846,   846,   846,   846,   846,   846,   846,   -1
// CHECK: 850,   850,   850,   850,   850,   850,   850,   850,   851,   851,   851,   851,   851,   851,   851,   851,   848,   848,   848,   848,   848,   848,   848,   848,   849,   849,   849,   849,   849,   849,   849,   849,   854,   854,   854,   854,   854,   854,   854,   854,   855,   855,   855,   855,   855,   855,   855,   855,   852,   852,   852,   852,   852,   852,   852,   852,   853,   853,   853,   853,   853,   853,   853,   853,   -1
// CHECK: 859,   859,   859,   859,   859,   859,   859,   859,   858,   858,   858,   858,   858,   858,   858,   858,   857,   857,   857,   857,   857,   857,   857,   857,   856,   856,   856,   856,   856,   856,   856,   856,   863,   863,   863,   863,   863,   863,   863,   863,   862,   862,   862,   862,   862,   862,   862,   862,   861,   861,   861,   861,   861,   861,   861,   861,   860,   860,   860,   860,   860,   860,   860,   860,   -1
// CHECK: 868,   868,   868,   868,   868,   868,   868,   868,   869,   869,   869,   869,   869,   869,   869,   869,   870,   870,   870,   870,   870,   870,   870,   870,   871,   871,   871,   871,   871,   871,   871,   871,   864,   864,   864,   864,   864,   864,   864,   864,   865,   865,   865,   865,   865,   865,   865,   865,   866,   866,   866,   866,   866,   866,   866,   866,   867,   867,   867,   867,   867,   867,   867,   867,   -1
// CHECK: 877,   877,   877,   877,   877,   877,   877,   877,   876,   876,   876,   876,   876,   876,   876,   876,   879,   879,   879,   879,   879,   879,   879,   879,   878,   878,   878,   878,   878,   878,   878,   878,   873,   873,   873,   873,   873,   873,   873,   873,   872,   872,   872,   872,   872,   872,   872,   872,   875,   875,   875,   875,   875,   875,   875,   875,   874,   874,   874,   874,   874,   874,   874,   874,   -1
// CHECK: 886,   886,   886,   886,   886,   886,   886,   886,   887,   887,   887,   887,   887,   887,   887,   887,   884,   884,   884,   884,   884,   884,   884,   884,   885,   885,   885,   885,   885,   885,   885,   885,   882,   882,   882,   882,   882,   882,   882,   882,   883,   883,   883,   883,   883,   883,   883,   883,   880,   880,   880,   880,   880,   880,   880,   880,   881,   881,   881,   881,   881,   881,   881,   881,   -1
// CHECK: 895,   895,   895,   895,   895,   895,   895,   895,   894,   894,   894,   894,   894,   894,   894,   894,   893,   893,   893,   893,   893,   893,   893,   893,   892,   892,   892,   892,   892,   892,   892,   892,   891,   891,   891,   891,   891,   891,   891,   891,   890,   890,   890,   890,   890,   890,   890,   890,   889,   889,   889,   889,   889,   889,   889,   889,   888,   888,   888,   888,   888,   888,   888,   888,   -1
// CHECK: 896,   896,   896,   896,   896,   896,   896,   896,   897,   897,   897,   897,   897,   897,   897,   897,   898,   898,   898,   898,   898,   898,   898,   898,   899,   899,   899,   899,   899,   899,   899,   899,   900,   900,   900,   900,   900,   900,   900,   900,   901,   901,   901,   901,   901,   901,   901,   901,   902,   902,   902,   902,   902,   902,   902,   902,   903,   903,   903,   903,   903,   903,   903,   903,   -1
// CHECK: 905,   905,   905,   905,   905,   905,   905,   905,   904,   904,   904,   904,   904,   904,   904,   904,   907,   907,   907,   907,   907,   907,   907,   907,   906,   906,   906,   906,   906,   906,   906,   906,   909,   909,   909,   909,   909,   909,   909,   909,   908,   908,   908,   908,   908,   908,   908,   908,   911,   911,   911,   911,   911,   911,   911,   911,   910,   910,   910,   910,   910,   910,   910,   910,   -1
// CHECK: 914,   914,   914,   914,   914,   914,   914,   914,   915,   915,   915,   915,   915,   915,   915,   915,   912,   912,   912,   912,   912,   912,   912,   912,   913,   913,   913,   913,   913,   913,   913,   913,   918,   918,   918,   918,   918,   918,   918,   918,   919,   919,   919,   919,   919,   919,   919,   919,   916,   916,   916,   916,   916,   916,   916,   916,   917,   917,   917,   917,   917,   917,   917,   917,   -1
// CHECK: 923,   923,   923,   923,   923,   923,   923,   923,   922,   922,   922,   922,   922,   922,   922,   922,   921,   921,   921,   921,   921,   921,   921,   921,   920,   920,   920,   920,   920,   920,   920,   920,   927,   927,   927,   927,   927,   927,   927,   927,   926,   926,   926,   926,   926,   926,   926,   926,   925,   925,   925,   925,   925,   925,   925,   925,   924,   924,   924,   924,   924,   924,   924,   924,   -1
// CHECK: 932,   932,   932,   932,   932,   932,   932,   932,   933,   933,   933,   933,   933,   933,   933,   933,   934,   934,   934,   934,   934,   934,   934,   934,   935,   935,   935,   935,   935,   935,   935,   935,   928,   928,   928,   928,   928,   928,   928,   928,   929,   929,   929,   929,   929,   929,   929,   929,   930,   930,   930,   930,   930,   930,   930,   930,   931,   931,   931,   931,   931,   931,   931,   931,   -1
// CHECK: 941,   941,   941,   941,   941,   941,   941,   941,   940,   940,   940,   940,   940,   940,   940,   940,   943,   943,   943,   943,   943,   943,   943,   943,   942,   942,   942,   942,   942,   942,   942,   942,   937,   937,   937,   937,   937,   937,   937,   937,   936,   936,   936,   936,   936,   936,   936,   936,   939,   939,   939,   939,   939,   939,   939,   939,   938,   938,   938,   938,   938,   938,   938,   938,   -1
// CHECK: 950,   950,   950,   950,   950,   950,   950,   950,   951,   951,   951,   951,   951,   951,   951,   951,   948,   948,   948,   948,   948,   948,   948,   948,   949,   949,   949,   949,   949,   949,   949,   949,   946,   946,   946,   946,   946,   946,   946,   946,   947,   947,   947,   947,   947,   947,   947,   947,   944,   944,   944,   944,   944,   944,   944,   944,   945,   945,   945,   945,   945,   945,   945,   945,   -1
// CHECK: 959,   959,   959,   959,   959,   959,   959,   959,   958,   958,   958,   958,   958,   958,   958,   958,   957,   957,   957,   957,   957,   957,   957,   957,   956,   956,   956,   956,   956,   956,   956,   956,   955,   955,   955,   955,   955,   955,   955,   955,   954,   954,   954,   954,   954,   954,   954,   954,   953,   953,   953,   953,   953,   953,   953,   953,   952,   952,   952,   952,   952,   952,   952,   952,   -1
// CHECK: 960,   960,   960,   960,   960,   960,   960,   960,   961,   961,   961,   961,   961,   961,   961,   961,   962,   962,   962,   962,   962,   962,   962,   962,   963,   963,   963,   963,   963,   963,   963,   963,   964,   964,   964,   964,   964,   964,   964,   964,   965,   965,   965,   965,   965,   965,   965,   965,   966,   966,   966,   966,   966,   966,   966,   966,   967,   967,   967,   967,   967,   967,   967,   967,   -1
// CHECK: 969,   969,   969,   969,   969,   969,   969,   969,   968,   968,   968,   968,   968,   968,   968,   968,   971,   971,   971,   971,   971,   971,   971,   971,   970,   970,   970,   970,   970,   970,   970,   970,   973,   973,   973,   973,   973,   973,   973,   973,   972,   972,   972,   972,   972,   972,   972,   972,   975,   975,   975,   975,   975,   975,   975,   975,   974,   974,   974,   974,   974,   974,   974,   974,   -1
// CHECK: 978,   978,   978,   978,   978,   978,   978,   978,   979,   979,   979,   979,   979,   979,   979,   979,   976,   976,   976,   976,   976,   976,   976,   976,   977,   977,   977,   977,   977,   977,   977,   977,   982,   982,   982,   982,   982,   982,   982,   982,   983,   983,   983,   983,   983,   983,   983,   983,   980,   980,   980,   980,   980,   980,   980,   980,   981,   981,   981,   981,   981,   981,   981,   981,   -1
// CHECK: 987,   987,   987,   987,   987,   987,   987,   987,   986,   986,   986,   986,   986,   986,   986,   986,   985,   985,   985,   985,   985,   985,   985,   985,   984,   984,   984,   984,   984,   984,   984,   984,   991,   991,   991,   991,   991,   991,   991,   991,   990,   990,   990,   990,   990,   990,   990,   990,   989,   989,   989,   989,   989,   989,   989,   989,   988,   988,   988,   988,   988,   988,   988,   988,   -1
// CHECK: 996,   996,   996,   996,   996,   996,   996,   996,   997,   997,   997,   997,   997,   997,   997,   997,   998,   998,   998,   998,   998,   998,   998,   998,   999,   999,   999,   999,   999,   999,   999,   999,   992,   992,   992,   992,   992,   992,   992,   992,   993,   993,   993,   993,   993,   993,   993,   993,   994,   994,   994,   994,   994,   994,   994,   994,   995,   995,   995,   995,   995,   995,   995,   995,   -1
// CHECK: 1005,   1005,   1005,   1005,   1005,   1005,   1005,   1005,   1004,   1004,   1004,   1004,   1004,   1004,   1004,   1004,   1007,   1007,   1007,   1007,   1007,   1007,   1007,   1007,   1006,   1006,   1006,   1006,   1006,   1006,   1006,   1006,   1001,   1001,   1001,   1001,   1001,   1001,   1001,   1001,   1000,   1000,   1000,   1000,   1000,   1000,   1000,   1000,   1003,   1003,   1003,   1003,   1003,   1003,   1003,   1003,   1002,   1002,   1002,   1002,   1002,   1002,   1002,   1002,   -1
// CHECK: 1014,   1014,   1014,   1014,   1014,   1014,   1014,   1014,   1015,   1015,   1015,   1015,   1015,   1015,   1015,   1015,   1012,   1012,   1012,   1012,   1012,   1012,   1012,   1012,   1013,   1013,   1013,   1013,   1013,   1013,   1013,   1013,   1010,   1010,   1010,   1010,   1010,   1010,   1010,   1010,   1011,   1011,   1011,   1011,   1011,   1011,   1011,   1011,   1008,   1008,   1008,   1008,   1008,   1008,   1008,   1008,   1009,   1009,   1009,   1009,   1009,   1009,   1009,   1009,   -1
// CHECK: 1023,   1023,   1023,   1023,   1023,   1023,   1023,   1023,   1022,   1022,   1022,   1022,   1022,   1022,   1022,   1022,   1021,   1021,   1021,   1021,   1021,   1021,   1021,   1021,   1020,   1020,   1020,   1020,   1020,   1020,   1020,   1020,   1019,   1019,   1019,   1019,   1019,   1019,   1019,   1019,   1018,   1018,   1018,   1018,   1018,   1018,   1018,   1018,   1017,   1017,   1017,   1017,   1017,   1017,   1017,   1017,   1016,   1016,   1016,   1016,   1016,   1016,   1016,   1016,   -1
