/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { AbsoluteTime, TalerMerchantApi } from "@gnu-taler/taler-util";
import { Time, useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ConfirmModal, Row } from "../../../../components/modal/index.js";
import { CreatePage } from "./CreatePage.js";


const TALER_SCREEN_ID = 30;

export type Entity = TalerMerchantApi.LoginTokenRequest;
interface Props {
  onBack?: () => void;
  onConfirm: () => void;
}

export default function AccessTokenCreatePage({
  onConfirm,
  onBack,
}: Props): VNode {
  const { i18n } = useTranslationContext();

  const [ok, setOk] = useState<{ token: string; expiration: AbsoluteTime }>();

  return (
    <Fragment>
      {!ok ? undefined : (
        <ConfirmModal
          active
          onCancel={onConfirm}
          description={i18n.str`Access token created`}
          label={i18n.str`Confirm`}
        >
          <div class="table-container">
            <table>
              <tbody>
                <tr>
                  <td colSpan={3}>
                    <i18n.Translate>
                      Copy the value of the access token and save it, as you cannot retrieve it again.
                    </i18n.Translate>
                  </td>
                </tr>
                <Row name={i18n.str`Token`} value={ok.token} literal />

                <tr>
                  <td colSpan={3}>
                    {AbsoluteTime.isNever(ok.expiration) ? (
                      <i18n.Translate>
                        This token will never expire
                      </i18n.Translate>
                    ) : (
                      <i18n.Translate>
                        This token will be available until{" "}
                        <Time
                          format="dd/MM/yyyy HH:mm:ss"
                          timestamp={ok.expiration}
                        />
                      </i18n.Translate>
                    )}
                  </td>
                </tr>
              </tbody>
            </table>
          </div>
        </ConfirmModal>
      )}
      <CreatePage
        onBack={onBack}
        onCreated={(c) => {
          setOk({
            expiration: AbsoluteTime.fromProtocolTimestamp(c.expiration),
            token: c.access_token,
          });
        }}
      />
    </Fragment>
  );
}
