//===- Serialization.h - MLIR SPIR-V (De)serialization ----------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file declares the entry point for serializing SPIR-V binary modules.
//
//===----------------------------------------------------------------------===//

#ifndef MLIR_TARGET_SPIRV_SERIALIZATION_H
#define MLIR_TARGET_SPIRV_SERIALIZATION_H

#include "mlir/Support/LLVM.h"
#include <cstdint>
#include <string>

namespace mlir {
class MLIRContext;

namespace spirv {
class ModuleOp;

struct SerializationOptions {
  /// Whether to emit `OpName` instructions for SPIR-V symbol ops.
  bool emitSymbolName = true;
  /// Whether to emit `OpLine` location information for SPIR-V ops.
  bool emitDebugInfo = false;
  /// Whether to store a module to an additional file during
  /// serialization. This is used to store the SPIR-V module to the
  /// file in addition to writing it to `os` passed from the calling
  /// tool. This saved file is later used for validation.
  bool saveModuleForValidation = false;
  /// A prefix prepended to the file used when `saveModuleForValidation`
  /// is set to `true`. This can either be a file prefix, or a relative or
  /// or an absolute path followed by the prefix. For example:
  ///
  ///   * "foo" - Create files with a `foo` prefix in the current working
  ///     directory. For example: `fooXYZ123.spv`, `fooABC456.spv` ...
  ///     `fooXXXXXX.spv`. The last 6 characters will be a unique combination
  ///     as generated by `llvm::sys::fs::createUniqueFile`.
  ///
  ///   * "my/dir/foo" - Create files in `my/dir` with a `foo` prefix. The
  ///     `my/dir` need to exists. For example: `fooXYZ123.spv`,
  ///     `fooABC456.spv` ... `fooXXXXXX.spv` will be created and stored in
  ///     `/my/dir`. Filenames follow the same pattern as above.
  ///
  ///   * "/home/user/my/dir" - Same as above but using an absolute path.
  std::string validationFilePrefix = "";
};

/// Serializes the given SPIR-V `moduleOp` and writes to `binary`. On failure,
/// reports errors to the error handler registered with the MLIR context for
/// `moduleOp`.
LogicalResult serialize(ModuleOp moduleOp, SmallVectorImpl<uint32_t> &binary,
                        const SerializationOptions &options = {});

} // namespace spirv
} // namespace mlir

#endif // MLIR_TARGET_SPIRV_SERIALIZATION_H
